// Admin Panel JavaScript
document.addEventListener('DOMContentLoaded', function() {
    // Populate preset-driven selects if presetManager is available
    if (window.presetManager && typeof window.presetManager.updateFormDropdowns === 'function') {
        setTimeout(function(){ window.presetManager.updateFormDropdowns(); }, 200);
    }
    
    // Flash Sale Management
    initializeFlashSales();
    // Sidebar toggle for mobile
    const sidebarToggle = document.querySelector('.sidebar-toggle');
    const sidebar = document.querySelector('.sidebar');
    
    if (sidebarToggle && sidebar) {
        sidebarToggle.addEventListener('click', function() {
            sidebar.classList.toggle('open');
        });
    }
    
    // Close sidebar when clicking outside on mobile
    document.addEventListener('click', function(e) {
        if (window.innerWidth <= 768) {
            if (!sidebar.contains(e.target) && !sidebarToggle.contains(e.target)) {
                sidebar.classList.remove('open');
            }
        }
    });
    
    // Image upload preview
    const imageUploads = document.querySelectorAll('.image-upload');
    imageUploads.forEach(upload => {
        const input = upload.querySelector('input[type="file"]');
        const preview = upload.querySelector('.image-preview');
        
        if (input && preview) {
            input.addEventListener('change', function(e) {
                const file = e.target.files[0];
                if (file) {
                    const reader = new FileReader();
                    reader.onload = function(e) {
                        preview.src = e.target.result;
                        preview.style.display = 'block';
                    };
                    reader.readAsDataURL(file);
                }
            });
        }
    });
    
    // Form validation
    const forms = document.querySelectorAll('form[data-validate]');
    forms.forEach(form => {
        form.addEventListener('submit', function(e) {
            const requiredFields = form.querySelectorAll('[required]');
            let isValid = true;
            
            requiredFields.forEach(field => {
                if (!field.value.trim()) {
                    field.style.borderColor = '#dc3545';
                    isValid = false;
                } else {
                    field.style.borderColor = '#ddd';
                }
            });
            
            if (!isValid) {
                e.preventDefault();
                alert('Please fill in all required fields.');
            }
        });
    });
    
    // Confirm delete actions
    const deleteButtons = document.querySelectorAll('[data-confirm-delete]');
    deleteButtons.forEach(button => {
        button.addEventListener('click', function(e) {
            const message = this.getAttribute('data-confirm-message') || 'Are you sure you want to delete this item?';
            if (!confirm(message)) {
                e.preventDefault();
            }
        });
    });
    
    // Auto-hide alerts
    const alerts = document.querySelectorAll('.alert');
    alerts.forEach(alert => {
        setTimeout(() => {
            alert.style.opacity = '0';
            setTimeout(() => {
                alert.remove();
            }, 300);
        }, 5000);
    });
    
    // Search functionality
    const searchInputs = document.querySelectorAll('.search-input');
    searchInputs.forEach(input => {
        input.addEventListener('input', function() {
            const searchTerm = this.value.toLowerCase();
            const table = this.closest('.search-filter').nextElementSibling;
            
            if (table && table.classList.contains('data-table')) {
                const rows = table.querySelectorAll('tbody tr');
                rows.forEach(row => {
                    const text = row.textContent.toLowerCase();
                    if (text.includes(searchTerm)) {
                        row.style.display = '';
                    } else {
                        row.style.display = 'none';
                    }
                });
            }
        });
    });
    
    // Status update buttons
    const statusButtons = document.querySelectorAll('[data-status-update]');
    statusButtons.forEach(button => {
        button.addEventListener('click', function() {
            const orderId = this.getAttribute('data-order-id');
            const newStatus = this.getAttribute('data-status-update');
            
            if (confirm(`Are you sure you want to update this order to ${newStatus}?`)) {
                updateOrderStatus(orderId, newStatus);
            }
        });
    });
    
    // Stock is now managed through volume variations in the edit modal
    // Removed direct stock input functionality
});

// Resolve base path for ajax endpoints depending on page depth
function getAjaxBasePath() {
    try {
        return window.location.pathname.includes('/admin/pages/') ? '../' : '';
    } catch (e) {
        return '';
    }
}

// Update order status
function updateOrderStatus(orderId, status) {
    fetch(getAjaxBasePath() + 'ajax/update_order_status.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            order_id: orderId,
            status: status
        })
    })
    .then(async response => {
        const text = await response.text();
        try { return JSON.parse(text); } catch (e) { throw new Error(text); }
    })
    .then(data => {
        if (data.success) {
            location.reload();
        } else {
            alert('Error updating order status: ' + data.message);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error updating order status');
    });
}

// Update product stock
function updateProductStock(productId, stock) {
    fetch(getAjaxBasePath() + 'ajax/update_product_stock.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            product_id: productId,
            stock: stock
        })
    })
    .then(async response => {
        const text = await response.text();
        try { return JSON.parse(text); } catch (e) { throw new Error(text); }
    })
    .then(data => {
        if (data.success) {
            location.reload();
        } else {
            alert('Error updating product stock: ' + data.message);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error updating product stock');
    });
}

// Show modal
function showModal(modalId) {
    const modal = document.getElementById(modalId);
    if (modal) {
        modal.style.display = 'flex';
        document.body.style.overflow = 'hidden'; // Prevent background scrolling
    } else {
        console.error('Modal not found:', modalId);
    }
}

// Hide modal
function hideModal(modalId) {
    const modal = document.getElementById(modalId);
    if (modal) {
        modal.style.display = 'none';
        document.body.style.overflow = 'auto'; // Restore background scrolling
    }
}

// Show modal overlay (for shipping page and other pages using modal-overlay)
function showModalOverlay(modalId) {
    const modal = document.getElementById(modalId);
    if (modal) {
        modal.style.display = 'flex';
        document.body.style.overflow = 'hidden'; // Prevent background scrolling
    } else {
        console.error('Modal overlay not found:', modalId);
    }
}

// Hide modal overlay
function hideModalOverlay(modalId) {
    const modal = document.getElementById(modalId);
    if (modal) {
        modal.style.display = 'none';
        document.body.style.overflow = 'auto'; // Restore background scrolling
    }
}

// Close modal when clicking outside
window.addEventListener('click', function(e) {
    if (e.target.classList.contains('modal') || e.target.classList.contains('modal-overlay')) {
        e.target.style.display = 'none';
        document.body.style.overflow = 'auto'; // Restore background scrolling
    }
});

// Format currency
function formatCurrency(amount) {
    return 'RM ' + parseFloat(amount).toFixed(2);
}

// Format date
function formatDate(dateString) {
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', {
        year: 'numeric',
        month: 'short',
        day: 'numeric'
    });
}

// Show loading state
function showLoading(element) {
    element.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Loading...';
    element.disabled = true;
}

// Hide loading state
function hideLoading(element, originalText) {
    element.innerHTML = originalText;
    element.disabled = false;
}

// Export data to CSV
function exportToCSV(data, filename) {
    const csv = convertToCSV(data);
    const blob = new Blob([csv], { type: 'text/csv' });
    const url = window.URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = filename;
    a.click();
    window.URL.revokeObjectURL(url);
}

// Convert data to CSV format
function convertToCSV(data) {
    if (data.length === 0) return '';
    
    const headers = Object.keys(data[0]);
    const csvContent = [
        headers.join(','),
        ...data.map(row => headers.map(header => `"${row[header] || ''}"`).join(','))
    ].join('\n');
    
    return csvContent;
}

// Print page
function printPage() {
    window.print();
}

// Refresh page
function refreshPage() {
    location.reload();
}

// Flash Sale Management Functions
function initializeFlashSales() {
    // Add flash sale buttons to products table
    addFlashSaleButtons();
    
    // Update flash sale timers every second
    setInterval(updateFlashSaleTimers, 1000);
}

function addFlashSaleButtons() {
    const productRows = document.querySelectorAll('tbody tr');
    productRows.forEach(row => {
        const productId = row.querySelector('button[onclick*="editProduct"]')?.onclick?.toString().match(/\d+/)?.[0];
        if (productId) {
            const actionCell = row.querySelector('td:last-child');
            if (actionCell && !actionCell.querySelector('.flash-sale-btn')) {
                const flashSaleBtn = document.createElement('button');
                flashSaleBtn.className = 'btn btn-sm btn-warning flash-sale-btn';
                flashSaleBtn.innerHTML = '<i class="fas fa-bolt"></i>';
                flashSaleBtn.title = 'Start Flash Sale';
                flashSaleBtn.onclick = () => startFlashSale(productId);
                actionCell.appendChild(flashSaleBtn);
            }
        }
    });
}

function startFlashSale(productId) {
    const duration = prompt('Enter flash sale duration in hours (1-168):', '24');
    if (duration && duration >= 1 && duration <= 168) {
        const ajaxBasePath = getAjaxBasePath();
        fetch(`${ajaxBasePath}start_flash_sale.php`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                product_id: productId,
                duration_hours: parseInt(duration)
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showNotification('Flash sale started successfully!', 'success');
                setTimeout(() => location.reload(), 1000);
            } else {
                showNotification(data.error || 'Failed to start flash sale', 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification('Error starting flash sale', 'error');
        });
    }
}

function updateFlashSaleTimers() {
    const flashSaleElements = document.querySelectorAll('[data-flash-sale-end]');
    flashSaleElements.forEach(element => {
        const endTime = new Date(element.dataset.flashSaleEnd);
        const now = new Date();
        const timeLeft = endTime - now;
        
        if (timeLeft > 0) {
            const hours = Math.floor(timeLeft / (1000 * 60 * 60));
            const minutes = Math.floor((timeLeft % (1000 * 60 * 60)) / (1000 * 60));
            const seconds = Math.floor((timeLeft % (1000 * 60)) / 1000);
            
            element.textContent = `${hours}h ${minutes}m ${seconds}s`;
        } else {
            element.textContent = 'Expired';
            element.style.color = '#e74c3c';
        }
    });
}

// Go back
function goBack() {
    history.back();
}