<?php
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../models/Product.php';
require_once __DIR__ . '/../../models/Order.php';
require_once __DIR__ . '/../../models/User.php';

class AdminFunctions {
    private $conn;
    private $product;
    private $order;
    private $user;
    
    public function __construct() {
        $this->conn = getDB();
        $this->product = new Product();
        $this->order = new Order();
        $this->user = new User();
    }
    
    // Dashboard Statistics
    public function getDashboardStats() {
        $stats = [];
        
        // Total products
        $stmt = $this->conn->query("SELECT COUNT(*) as total FROM products WHERE is_active = 1");
        $stats['total_products'] = $stmt->fetch()['total'];
        
        // Total orders
        $stmt = $this->conn->query("SELECT COUNT(*) as total FROM orders");
        $stats['total_orders'] = $stmt->fetch()['total'];
        
        // Total revenue
        $stmt = $this->conn->query("SELECT SUM(total_amount) as total FROM orders WHERE status != 'cancelled'");
        $stats['total_revenue'] = $stmt->fetch()['total'] ?? 0;
        
        // Total customers
        $stmt = $this->conn->query("SELECT COUNT(*) as total FROM users WHERE is_guest = 0");
        $stats['total_customers'] = $stmt->fetch()['total'];
        
        // Low stock products
        $stmt = $this->conn->query("SELECT COUNT(*) as total FROM products WHERE stock_quantity < 10 AND is_active = 1");
        $stats['low_stock'] = $stmt->fetch()['total'];
        
        // Pending orders
        $stmt = $this->conn->query("SELECT COUNT(*) as total FROM orders WHERE status = 'pending'");
        $stats['pending_orders'] = $stmt->fetch()['total'];
        
        return $stats;
    }
    
    // Recent orders
    public function getRecentOrders($limit = 10) {
        $query = "SELECT o.* 
                  FROM orders o 
                  ORDER BY o.created_at DESC 
                  LIMIT :limit";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll();
    }
    
    // Top selling products
    public function getTopSellingProducts($limit = 5) {
        $query = "SELECT p.name, p.brand, p.price, SUM(oi.quantity) as total_sold
                  FROM products p
                  JOIN order_items oi ON p.id = oi.product_id
                  JOIN orders o ON oi.order_id = o.id
                  WHERE o.status != 'cancelled'
                  GROUP BY p.id
                  ORDER BY total_sold DESC
                  LIMIT :limit";
        
        try {
            $stmt = $this->conn->prepare($query);
            $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
            $stmt->execute();
            return $stmt->fetchAll();
        } catch (Exception $e) {
            // If there are no orders or order_items, return empty array
            return [];
        }
    }
    
    // Monthly revenue chart data
    public function getMonthlyRevenue($months = 12) {
        // Try different column names for total
        $queries = [
            "SELECT 
                DATE_FORMAT(created_at, '%Y-%m') as month,
                SUM(total) as revenue
              FROM orders 
              WHERE status != 'cancelled' 
                AND created_at >= DATE_SUB(NOW(), INTERVAL :months MONTH)
              GROUP BY DATE_FORMAT(created_at, '%Y-%m')
              ORDER BY month ASC",
            "SELECT 
                DATE_FORMAT(created_at, '%Y-%m') as month,
                SUM(total_amount) as revenue
              FROM orders 
              WHERE status != 'cancelled' 
                AND created_at >= DATE_SUB(NOW(), INTERVAL :months MONTH)
              GROUP BY DATE_FORMAT(created_at, '%Y-%m')
              ORDER BY month ASC"
        ];
        
        foreach ($queries as $query) {
            try {
                $stmt = $this->conn->prepare($query);
                $stmt->bindValue(':months', $months, PDO::PARAM_INT);
                $stmt->execute();
                return $stmt->fetchAll();
            } catch (Exception $e) {
                continue;
            }
        }
        
        return [];
    }
    
    // Order status distribution
    public function getOrderStatusDistribution() {
        $query = "SELECT status, COUNT(*) as count 
                  FROM orders 
                  GROUP BY status";
        
        $stmt = $this->conn->query($query);
        return $stmt->fetchAll();
    }
    
    // Version variations management
    public function addVersionVariations($productId, $variations) {
        try {
            // First, delete existing variations for this product
            $deleteQuery = "DELETE FROM version_variations WHERE product_id = :product_id";
            $stmt = $this->conn->prepare($deleteQuery);
            $stmt->bindValue(':product_id', $productId);
            $stmt->execute();
            
            // Insert new variations
            if (!empty($variations)) {
                $insertQuery = "INSERT INTO version_variations (product_id, version_type, size, price, stock_quantity, is_active) VALUES (:product_id, :version_type, :size, :price, :stock_quantity, :is_active)";
                $stmt = $this->conn->prepare($insertQuery);
                
                foreach ($variations as $variation) {
                    $stmt->bindValue(':product_id', $productId);
                    $stmt->bindValue(':version_type', $variation['version_type']);
                    $stmt->bindValue(':size', $variation['size']);
                    $stmt->bindValue(':price', $variation['price']);
                    $stmt->bindValue(':stock_quantity', $variation['stock_quantity']);
                    $stmt->bindValue(':is_active', $variation['is_active'] ?? 1);
                    $stmt->execute();
                }
            }
            
            return true;
        } catch (Exception $e) {
            error_log("Error adding version variations: " . $e->getMessage());
            return false;
        }
    }
    
    public function getVersionVariations($productId) {
        try {
            $query = "SELECT * FROM version_variations WHERE product_id = :product_id AND is_active = 1 ORDER BY version_type, size";
            $stmt = $this->conn->prepare($query);
            $stmt->bindValue(':product_id', $productId);
            $stmt->execute();
            return $stmt->fetchAll();
        } catch (Exception $e) {
            error_log("Error getting version variations: " . $e->getMessage());
            return [];
        }
    }
    
    public function getProductWithVersionVariations($productId) {
        try {
            // Get main product data
            $query = "SELECT * FROM products WHERE id = :id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindValue(':id', $productId);
            $stmt->execute();
            $product = $stmt->fetch();
            
            if ($product) {
                // Get version variations
                $product['version_variations'] = $this->getVersionVariations($productId);
            }
            
            return $product;
        } catch (Exception $e) {
            error_log("Error getting product with version variations: " . $e->getMessage());
            return null;
        }
    }

    // Product management functions
    public function addProduct($data) {
        try {
            $this->conn->beginTransaction();
            
            // Insert main product
            $query = "INSERT INTO products (name, brand, description, currency, sale_price, is_on_sale, sale_start_date, sale_end_date, is_flash_sale, flash_sale_duration_hours, display_price_mode, scent, notes, season, recommendation, image, image_2, image_3, image_4, image_5, image_6, category_id, stock_quantity, is_bestseller, is_new_arrival, is_active, point_rewards) 
                      VALUES (:name, :brand, :description, :currency, :sale_price, :is_on_sale, :sale_start_date, :sale_end_date, :is_flash_sale, :flash_sale_duration_hours, :display_price_mode, :scent, :notes, :season, :recommendation, :image, :image_2, :image_3, :image_4, :image_5, :image_6, :category_id, :stock_quantity, :is_bestseller, :is_new_arrival, :is_active, :point_rewards)";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindValue(':name', $data['name']);
            $stmt->bindValue(':brand', $data['brand']);
            $stmt->bindValue(':description', $data['description']);
            $stmt->bindValue(':currency', $data['currency'] ?? 'RM');
            $stmt->bindValue(':sale_price', $data['sale_price']);
            $stmt->bindValue(':is_on_sale', $data['is_on_sale'] ?? 0);
            $stmt->bindValue(':sale_start_date', !empty($data['sale_start_date']) ? $data['sale_start_date'] : null);
            $stmt->bindValue(':sale_end_date', !empty($data['sale_end_date']) ? $data['sale_end_date'] : null);
            $stmt->bindValue(':is_flash_sale', $data['is_flash_sale'] ?? 0);
            $stmt->bindValue(':flash_sale_duration_hours', $data['flash_sale_duration_hours'] ?? 24);
            $stmt->bindValue(':display_price_mode', $data['display_price_mode'] ?? 'range');
            $stmt->bindValue(':scent', $data['scent']);
            $stmt->bindValue(':notes', $data['notes']);
            $stmt->bindValue(':season', $data['season']);
            $stmt->bindValue(':recommendation', $data['recommendation']);
            $stmt->bindValue(':image', $data['image']);
            $stmt->bindValue(':image_2', $data['image_2'] ?? '');
            $stmt->bindValue(':image_3', $data['image_3'] ?? '');
            $stmt->bindValue(':image_4', $data['image_4'] ?? '');
            $stmt->bindValue(':image_5', $data['image_5'] ?? '');
            $stmt->bindValue(':image_6', $data['image_6'] ?? '');
            $stmt->bindValue(':category_id', $data['category_id']);
            $stmt->bindValue(':stock_quantity', $data['stock_quantity']);
            $stmt->bindValue(':is_bestseller', $data['is_bestseller'] ?? 0);
            $stmt->bindValue(':is_new_arrival', $data['is_new_arrival'] ?? 0);
            $stmt->bindValue(':is_active', $data['is_active'] ?? 1);
            $stmt->bindValue(':point_rewards', $data['point_rewards'] ?? 0);
            
            $result = $stmt->execute();
            
            if ($result) {
                $productId = $this->conn->lastInsertId();
                
                // Insert version variations (unified system)
                if (isset($data['version_variations'])) {
                    $this->addVersionVariations($productId, $data['version_variations']);
                }
            }
            
            $this->conn->commit();
            return $result;
            
        } catch (Exception $e) {
            $this->conn->rollBack();
            return false;
        }
    }
    
    public function updateProduct($id, $data) {
        $query = "UPDATE products SET 
                    name = :name, brand = :brand, description = :description, 
                    price = :price, scent = :scent, notes = :notes,
                    season = :season, recommendation = :recommendation, image = :image, 
                    category_id = :category_id, stock_quantity = :stock_quantity,
                    is_bestseller = :is_bestseller, is_new_arrival = :is_new_arrival,
                    is_active = :is_active, updated_at = NOW()
                  WHERE id = :id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':name', $data['name']);
        $stmt->bindValue(':brand', $data['brand']);
        $stmt->bindValue(':description', $data['description']);
        $stmt->bindValue(':price', $data['price']);
        $stmt->bindValue(':scent', $data['scent']);
        $stmt->bindValue(':notes', $data['notes']);
        $stmt->bindValue(':season', $data['season']);
        $stmt->bindValue(':recommendation', $data['recommendation']);
        $stmt->bindValue(':image', $data['image']);
        $stmt->bindValue(':category_id', $data['category_id']);
        $stmt->bindValue(':stock_quantity', $data['stock_quantity']);
        $stmt->bindValue(':is_bestseller', $data['is_bestseller'] ?? 0);
        $stmt->bindValue(':is_new_arrival', $data['is_new_arrival'] ?? 0);
        $stmt->bindValue(':is_active', $data['is_active'] ?? 1);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        
        return $stmt->execute();
    }
    
    public function deleteProduct($id) {
        $query = "UPDATE products SET is_active = 0 WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        return $stmt->execute();
    }
    
    public function updateProductStock($id, $stock) {
        $query = "UPDATE products SET stock_quantity = :stock, updated_at = NOW() WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':stock', $stock, PDO::PARAM_INT);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        return $stmt->execute();
    }
    
    public function updateProductWithVariations($id, $data) {
        try {
            $this->conn->beginTransaction();
            
            // Update main product
            $query = "UPDATE products SET 
                        name = :name, brand = :brand, description = :description, 
                        sale_price = :sale_price, is_on_sale = :is_on_sale,
                        sale_start_date = :sale_start_date, sale_end_date = :sale_end_date,
                        is_flash_sale = :is_flash_sale, flash_sale_duration_hours = :flash_sale_duration_hours,
                        display_price_mode = :display_price_mode,
                        scent = :scent, notes = :notes, season = :season, recommendation = :recommendation, 
                        image = :image, image_2 = :image_2, image_3 = :image_3, image_4 = :image_4, image_5 = :image_5, image_6 = :image_6, category_id = :category_id, stock_quantity = :stock_quantity,
                        is_bestseller = :is_bestseller, is_new_arrival = :is_new_arrival,
                        is_active = :is_active, point_rewards = :point_rewards, updated_at = NOW()
                      WHERE id = :id";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindValue(':name', $data['name']);
            $stmt->bindValue(':brand', $data['brand']);
            $stmt->bindValue(':description', $data['description']);
            $stmt->bindValue(':sale_price', $data['sale_price']);
            $stmt->bindValue(':is_on_sale', $data['is_on_sale'] ?? 0);
            $stmt->bindValue(':sale_start_date', !empty($data['sale_start_date']) ? $data['sale_start_date'] : null);
            $stmt->bindValue(':sale_end_date', !empty($data['sale_end_date']) ? $data['sale_end_date'] : null);
            $stmt->bindValue(':is_flash_sale', $data['is_flash_sale'] ?? 0);
            $stmt->bindValue(':flash_sale_duration_hours', $data['flash_sale_duration_hours'] ?? 24);
            $stmt->bindValue(':display_price_mode', $data['display_price_mode'] ?? 'range');
            $stmt->bindValue(':scent', $data['scent']);
            $stmt->bindValue(':notes', $data['notes']);
            $stmt->bindValue(':season', $data['season']);
            $stmt->bindValue(':recommendation', $data['recommendation']);
            $stmt->bindValue(':image', $data['image']);
            $stmt->bindValue(':image_2', $data['image_2'] ?? '');
            $stmt->bindValue(':image_3', $data['image_3'] ?? '');
            $stmt->bindValue(':image_4', $data['image_4'] ?? '');
            $stmt->bindValue(':image_5', $data['image_5'] ?? '');
            $stmt->bindValue(':image_6', $data['image_6'] ?? '');
            $stmt->bindValue(':category_id', $data['category_id']);
            $stmt->bindValue(':stock_quantity', $data['stock_quantity']);
            $stmt->bindValue(':is_bestseller', $data['is_bestseller'] ?? 0);
            $stmt->bindValue(':is_new_arrival', $data['is_new_arrival'] ?? 0);
            $stmt->bindValue(':is_active', $data['is_active'] ?? 1);
            $stmt->bindValue(':point_rewards', $data['point_rewards'] ?? 0);
            $stmt->bindValue(':id', $id, PDO::PARAM_INT);
            
            $result = $stmt->execute();
            
            if ($result) {
                // Handle version variations (unified system)
                if (isset($data['version_variations'])) {
                    $this->addVersionVariations($id, $data['version_variations']);
                }
            }
            
            $this->conn->commit();
            return $result;
            
        } catch (Exception $e) {
            $this->conn->rollBack();
            return false;
        }
    }
    
    public function getAllProducts() {
        return $this->product->getAll();
    }
    
    public function getAllProductsWithVariations() {
        $query = "SELECT p.*, 
                         COALESCE(SUM(vv.stock_quantity), p.stock_quantity) as total_stock,
                         GROUP_CONCAT(CONCAT(vv.version_type, ':', vv.size, ':', vv.stock_quantity, ':', vv.price) SEPARATOR '|') as variations
                  FROM products p 
                  LEFT JOIN version_variations vv ON p.id = vv.product_id AND vv.is_active = 1
                  WHERE p.is_active = 1
                  GROUP BY p.id
                  ORDER BY p.created_at DESC";
        
        $stmt = $this->conn->query($query);
        $products = $stmt->fetchAll();
        
        // Process variations for display
        foreach ($products as &$product) {
            $product['variations_detail'] = [];
            if (!empty($product['variations'])) {
                $variations = explode('|', $product['variations']);
                foreach ($variations as $variation) {
                    if (!empty($variation)) {
                        list($version_type, $size, $stock, $price) = explode(':', $variation);
                        $product['variations_detail'][] = [
                            'version_type' => $version_type,
                            'size' => $size,
                            'stock' => (int)$stock,
                            'price' => (float)$price
                        ];
                    }
                }
            }
        }
        
        return $products;
    }
    
    public function getProductById($id) {
        return $this->product->getById($id);
    }
    
    public function getProductWithVariations($id) {
        $query = "SELECT p.*, 
                         COALESCE(SUM(vv.stock_quantity), p.stock_quantity) as total_stock
                  FROM products p 
                  LEFT JOIN version_variations vv ON p.id = vv.product_id AND vv.is_active = 1
                  WHERE p.id = :id
                  GROUP BY p.id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        $product = $stmt->fetch();
        
        if ($product) {
            // Get version variations
            $variationsQuery = "SELECT version_type, size, price, stock_quantity FROM version_variations WHERE product_id = :id AND is_active = 1 ORDER BY version_type, size";
            $variationsStmt = $this->conn->prepare($variationsQuery);
            $variationsStmt->bindValue(':id', $id, PDO::PARAM_INT);
            $variationsStmt->execute();
            $product['version_variations'] = $variationsStmt->fetchAll();
        }
        
        return $product;
    }
    
    // Banner management
    public function getBanners() {
        $banners = [];
        $banner_dir = __DIR__ . '/../../banner/';
        
        if (is_dir($banner_dir)) {
            $files = scandir($banner_dir);
            foreach ($files as $file) {
                $extension = strtolower(pathinfo($file, PATHINFO_EXTENSION));
                if (in_array($extension, ['jpg', 'jpeg', 'png', 'gif', 'mp4', 'webm', 'ogg', 'avi', 'mov'])) {
                    $banners[] = [
                        'filename' => $file,
                        'path' => 'banner/' . $file,
                        'size' => filesize($banner_dir . $file),
                        'modified' => filemtime($banner_dir . $file),
                        'type' => in_array($extension, ['mp4', 'webm', 'ogg', 'avi', 'mov']) ? 'video' : 'image'
                    ];
                }
            }
        }
        
        return $banners;
    }
    
    // Order management
    public function getAllOrders($search = '', $status_filter = '', $date_filter = '', $limit = 50, $offset = 0) {
        $query = "SELECT o.* 
                  FROM orders o 
                  WHERE 1=1";
        
        $params = [];
        
        // Add search filter
        if (!empty($search)) {
            $query .= " AND (o.order_number LIKE :search OR o.shipping_address LIKE :search)";
            $params[':search'] = '%' . $search . '%';
        }
        
        // Add status filter
        if (!empty($status_filter)) {
            $query .= " AND o.status = :status";
            $params[':status'] = $status_filter;
        }
        
        // Add date filter
        if (!empty($date_filter)) {
            $query .= " AND DATE(o.created_at) = :date";
            $params[':date'] = $date_filter;
        }
        
        $query .= " ORDER BY o.created_at DESC";
        
        // Add pagination
        if ($limit > 0) {
            $query .= " LIMIT :limit OFFSET :offset";
            $params[':limit'] = (int)$limit;
            $params[':offset'] = (int)$offset;
        }
        
        $stmt = $this->conn->prepare($query);
        
        // Bind parameters with proper types for LIMIT and OFFSET
        foreach ($params as $key => $value) {
            if ($key === ':limit' || $key === ':offset') {
                $stmt->bindValue($key, $value, PDO::PARAM_INT);
            } else {
                $stmt->bindValue($key, $value);
            }
        }
        
        $stmt->execute();
        return $stmt->fetchAll();
    }
    
    public function getOrdersCount($search = '', $status_filter = '', $date_filter = '') {
        $query = "SELECT COUNT(*) as total 
                  FROM orders o 
                  WHERE 1=1";
        
        $params = [];
        
        // Add search filter
        if (!empty($search)) {
            $query .= " AND (o.order_number LIKE :search OR o.shipping_address LIKE :search)";
            $params[':search'] = '%' . $search . '%';
        }
        
        // Add status filter
        if (!empty($status_filter)) {
            $query .= " AND o.status = :status";
            $params[':status'] = $status_filter;
        }
        
        // Add date filter
        if (!empty($date_filter)) {
            $query .= " AND DATE(o.created_at) = :date";
            $params[':date'] = $date_filter;
        }
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute($params);
        $result = $stmt->fetch();
        return $result['total'];
    }
    
    public function getTodayProcessingAndShippedOrders() {
        $today = date('Y-m-d');
        $query = "SELECT 
                    COUNT(CASE WHEN status = 'processing' THEN 1 END) as processing_count,
                    COUNT(CASE WHEN status = 'shipped' THEN 1 END) as shipped_count
                  FROM orders 
                  WHERE DATE(created_at) = :today 
                  AND status IN ('processing', 'shipped')";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':today', $today);
        $stmt->execute();
        
        return $stmt->fetch();
    }
    
    public function updateOrderStatus($order_id, $status, $tracking_number = null) {
        return $this->order->updateStatus($order_id, $status, $tracking_number);
    }
    
    // User management
    public function getAllUsers() {
        $query = "SELECT * FROM users WHERE is_guest = 0 ORDER BY created_at DESC";
        $stmt = $this->conn->query($query);
        return $stmt->fetchAll();
    }
    
    // Categories
    public function getCategories() {
        $query = "SELECT * FROM categories ORDER BY sort_order ASC, name ASC";
        $stmt = $this->conn->query($query);
        return $stmt->fetchAll();
    }
    
    // Get order by ID
    public function getOrderById($orderId) {
        $query = "SELECT * FROM orders WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':id', $orderId, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetch();
    }
    
    // Get order items
    public function getOrderItems($orderId) {
        $query = "SELECT * FROM order_items WHERE order_id = :order_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':order_id', $orderId, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll();
    }
    
    // Get customer by ID
    public function getCustomerById($customerId) {
        $query = "SELECT * FROM users WHERE id = :id AND is_guest = 0";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':id', $customerId, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetch();
    }
    
    // Get customer orders
    public function getCustomerOrders($customerId) {
        $query = "SELECT * FROM orders WHERE user_id = :user_id ORDER BY created_at DESC LIMIT 10";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':user_id', $customerId, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll();
    }
    
    // Get customer loyalty points
    public function getCustomerLoyaltyPoints($customerId) {
        $query = "SELECT * FROM user_loyalty_points WHERE user_id = :user_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':user_id', $customerId, PDO::PARAM_INT);
        $stmt->execute();
        
        $result = $stmt->fetch();
        return $result ? $result['points'] : 0;
    }
    
    // Feedback/Review Management Functions
    
    /**
     * Get all reviews with pagination and filters
     */
    public function getAllReviews($limit = 20, $offset = 0, $filters = []) {
        // Get delivery feedback instead of product reviews
        $query = "SELECT df.*, 
                         p.name as product_name, 
                         p.brand as product_brand, 
                         p.image as product_image,
                         CONCAT(u.first_name, ' ', u.last_name) as user_name,
                         u.email as user_email,
                         o.order_number,
                         df.feedback as comment,
                         df.rating,
                         df.created_at
                  FROM delivery_feedback df
                  LEFT JOIN orders o ON df.order_id = o.id
                  LEFT JOIN order_items oi ON o.id = oi.order_id
                  LEFT JOIN products p ON oi.product_id = p.id
                  LEFT JOIN users u ON df.user_id = u.id
                  WHERE 1=1";
        
        $params = [];
        
        // Apply filters
        if (!empty($filters['rating'])) {
            $query .= " AND df.rating = :rating";
            $params['rating'] = $filters['rating'];
        }
        
        if (!empty($filters['search'])) {
            $query .= " AND (p.name LIKE :search OR u.first_name LIKE :search OR u.last_name LIKE :search OR df.feedback LIKE :search)";
            $params['search'] = '%' . $filters['search'] . '%';
        }
        
        $query .= " ORDER BY df.created_at DESC LIMIT :limit OFFSET :offset";
        
        $stmt = $this->conn->prepare($query);
        
        foreach ($params as $key => $value) {
            $stmt->bindValue(':' . $key, $value);
        }
        
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        
        $stmt->execute();
        return $stmt->fetchAll();
    }
    
    /**
     * Get total count of reviews
     */
    public function getTotalReviewsCount($filters = []) {
        $query = "SELECT COUNT(*) as total
                  FROM delivery_feedback df
                  LEFT JOIN orders o ON df.order_id = o.id
                  LEFT JOIN order_items oi ON o.id = oi.order_id
                  LEFT JOIN products p ON oi.product_id = p.id
                  LEFT JOIN users u ON df.user_id = u.id
                  WHERE 1=1";
        
        $params = [];
        
        // Apply same filters as getAllReviews
        if (!empty($filters['rating'])) {
            $query .= " AND df.rating = :rating";
            $params['rating'] = $filters['rating'];
        }
        
        if (!empty($filters['search'])) {
            $query .= " AND (p.name LIKE :search OR u.first_name LIKE :search OR u.last_name LIKE :search OR df.feedback LIKE :search)";
            $params['search'] = '%' . $filters['search'] . '%';
        }
        
        $stmt = $this->conn->prepare($query);
        
        foreach ($params as $key => $value) {
            $stmt->bindValue(':' . $key, $value);
        }
        
        $stmt->execute();
        $result = $stmt->fetch();
        return $result['total'];
    }
    
    /**
     * Get review by ID
     */
    public function getReviewById($id) {
        $query = "SELECT r.*, 
                         p.name as product_name, 
                         p.brand as product_brand, 
                         p.image as product_image,
                         CONCAT(u.first_name, ' ', u.last_name) as user_name,
                         u.email as user_email
                  FROM reviews r
                  LEFT JOIN products p ON r.product_id = p.id
                  LEFT JOIN users u ON r.user_id = u.id
                  WHERE r.id = :id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetch();
    }
    
    /**
     * Update review verification status
     */
    public function updateReviewStatus($id, $isVerified) {
        $query = "UPDATE reviews SET is_verified = :is_verified, updated_at = CURRENT_TIMESTAMP WHERE id = :id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        $stmt->bindValue(':is_verified', $isVerified ? 1 : 0, PDO::PARAM_INT);
        
        return $stmt->execute();
    }
    
    /**
     * Delete review
     */
    public function deleteReview($id) {
        $query = "DELETE FROM delivery_feedback WHERE id = :id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        
        return $stmt->execute();
    }
    
    /**
     * Get review statistics
     */
    public function getReviewStatistics() {
        $stats = [];
        
        // Total reviews (delivery feedback)
        $stmt = $this->conn->query("SELECT COUNT(*) as total FROM delivery_feedback");
        $stats['total_reviews'] = $stmt->fetch()['total'];
        
        // Average rating
        $stmt = $this->conn->query("SELECT AVG(rating) as average FROM delivery_feedback");
        $result = $stmt->fetch();
        $stats['average_rating'] = $result['average'] ? round($result['average'], 1) : 0;
        
        // Reviews with feedback text
        $stmt = $this->conn->query("SELECT COUNT(*) as total FROM delivery_feedback WHERE feedback IS NOT NULL AND feedback != ''");
        $stats['verified_reviews'] = $stmt->fetch()['total'];
        
        // Reviews without feedback text
        $stmt = $this->conn->query("SELECT COUNT(*) as total FROM delivery_feedback WHERE feedback IS NULL OR feedback = ''");
        $stats['pending_reviews'] = $stmt->fetch()['total'];
        
        // Rating distribution
        $stmt = $this->conn->query("SELECT rating, COUNT(*) as count FROM delivery_feedback GROUP BY rating ORDER BY rating DESC");
        $stats['rating_distribution'] = $stmt->fetchAll();
        
        // Recent reviews (last 7 days)
        $stmt = $this->conn->query("SELECT COUNT(*) as total FROM delivery_feedback WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)");
        $stats['recent_reviews'] = $stmt->fetch()['total'];
        
        return $stats;
    }
    
    /**
     * Update product rating and review count
     */
    public function updateProductRating($productId) {
        $query = "UPDATE products SET 
                    rating = (SELECT AVG(rating) FROM reviews WHERE product_id = :product_id),
                    review_count = (SELECT COUNT(*) FROM reviews WHERE product_id = :product_id)
                  WHERE id = :product_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':product_id', $productId, PDO::PARAM_INT);
        
        return $stmt->execute();
    }
    
    // Shipping Fee Management Functions
    
    /**
     * Get all shipping fees
     */
    public function getAllShippingFees() {
        $query = "SELECT * FROM shipping_fees ORDER BY sort_order ASC, created_at ASC";
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt->fetchAll();
    }
    
    /**
     * Get shipping fee by ID
     */
    public function getShippingFeeById($id) {
        $query = "SELECT * FROM shipping_fees WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetch();
    }
    
    /**
     * Add shipping fee
     */
    public function addShippingFee($data) {
        $query = "INSERT INTO shipping_fees (name, description, fee_type, base_fee, per_kg_fee, min_weight, max_weight, min_order_amount, max_order_amount, delivery_days_min, delivery_days_max, is_active, sort_order) 
                  VALUES (:name, :description, :fee_type, :base_fee, :per_kg_fee, :min_weight, :max_weight, :min_order_amount, :max_order_amount, :delivery_days_min, :delivery_days_max, :is_active, :sort_order)";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':name', $data['name']);
        $stmt->bindValue(':description', $data['description']);
        $stmt->bindValue(':fee_type', $data['fee_type']);
        $stmt->bindValue(':base_fee', $data['base_fee']);
        $stmt->bindValue(':per_kg_fee', $data['per_kg_fee']);
        $stmt->bindValue(':min_weight', $data['min_weight']);
        $stmt->bindValue(':max_weight', $data['max_weight']);
        $stmt->bindValue(':min_order_amount', $data['min_order_amount']);
        $stmt->bindValue(':max_order_amount', $data['max_order_amount']);
        $stmt->bindValue(':delivery_days_min', $data['delivery_days_min'], PDO::PARAM_INT);
        $stmt->bindValue(':delivery_days_max', $data['delivery_days_max'], PDO::PARAM_INT);
        $stmt->bindValue(':is_active', $data['is_active'], PDO::PARAM_INT);
        $stmt->bindValue(':sort_order', $data['sort_order'], PDO::PARAM_INT);
        
        return $stmt->execute();
    }
    
    /**
     * Update shipping fee
     */
    public function updateShippingFee($id, $data) {
        $query = "UPDATE shipping_fees SET 
                    name = :name, 
                    description = :description, 
                    fee_type = :fee_type, 
                    base_fee = :base_fee, 
                    per_kg_fee = :per_kg_fee, 
                    min_weight = :min_weight, 
                    max_weight = :max_weight, 
                    min_order_amount = :min_order_amount, 
                    max_order_amount = :max_order_amount, 
                    delivery_days_min = :delivery_days_min, 
                    delivery_days_max = :delivery_days_max, 
                    is_active = :is_active, 
                    sort_order = :sort_order,
                    updated_at = CURRENT_TIMESTAMP
                  WHERE id = :id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        $stmt->bindValue(':name', $data['name']);
        $stmt->bindValue(':description', $data['description']);
        $stmt->bindValue(':fee_type', $data['fee_type']);
        $stmt->bindValue(':base_fee', $data['base_fee']);
        $stmt->bindValue(':per_kg_fee', $data['per_kg_fee']);
        $stmt->bindValue(':min_weight', $data['min_weight']);
        $stmt->bindValue(':max_weight', $data['max_weight']);
        $stmt->bindValue(':min_order_amount', $data['min_order_amount']);
        $stmt->bindValue(':max_order_amount', $data['max_order_amount']);
        $stmt->bindValue(':delivery_days_min', $data['delivery_days_min'], PDO::PARAM_INT);
        $stmt->bindValue(':delivery_days_max', $data['delivery_days_max'], PDO::PARAM_INT);
        $stmt->bindValue(':is_active', $data['is_active'], PDO::PARAM_INT);
        $stmt->bindValue(':sort_order', $data['sort_order'], PDO::PARAM_INT);
        
        return $stmt->execute();
    }
    
    /**
     * Delete shipping fee
     */
    public function deleteShippingFee($id) {
        $query = "DELETE FROM shipping_fees WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        return $stmt->execute();
    }
    
    /**
     * Update shipping fee status
     */
    public function updateShippingFeeStatus($id, $isActive) {
        $query = "UPDATE shipping_fees SET is_active = :is_active, updated_at = CURRENT_TIMESTAMP WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        $stmt->bindValue(':is_active', $isActive, PDO::PARAM_INT);
        return $stmt->execute();
    }
    
    /**
     * Update shipping fee order
     */
    public function updateShippingFeeOrder($orderData) {
        try {
            $this->conn->beginTransaction();
            
            $query = "UPDATE shipping_fees SET sort_order = :sort_order WHERE id = :id";
            $stmt = $this->conn->prepare($query);
            
            foreach ($orderData as $item) {
                $stmt->bindValue(':id', $item['id'], PDO::PARAM_INT);
                $stmt->bindValue(':sort_order', $item['sort_order'], PDO::PARAM_INT);
                $stmt->execute();
            }
            
            $this->conn->commit();
            return true;
        } catch (Exception $e) {
            $this->conn->rollBack();
            return false;
        }
    }
    
    /**
     * Get shipping fee for order amount
     */
    public function getShippingFeeForOrder($orderAmount) {
        $query = "SELECT * FROM shipping_fees 
                  WHERE is_active = 1 
                  AND (min_order_amount <= :order_amount OR min_order_amount = 0)
                  AND (max_order_amount IS NULL OR max_order_amount >= :order_amount OR max_order_amount = 0)
                  ORDER BY sort_order ASC, min_order_amount DESC
                  LIMIT 1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':order_amount', $orderAmount);
        $stmt->execute();
        return $stmt->fetch();
    }
    
    /**
     * Calculate shipping cost for order based on weight
     */
    public function calculateShippingCost($orderAmount, $weight = 0) {
        $shippingFee = $this->getShippingFeeForOrder($orderAmount);
        
        if (!$shippingFee) {
            return 0; // No shipping fee found
        }
        
        switch ($shippingFee['fee_type']) {
            case 'weight_based':
                // Calculate: base_fee + (weight * per_kg_fee)
                return $shippingFee['base_fee'] + ($weight * $shippingFee['per_kg_fee']);
            case 'free_shipping_threshold':
                return 0; // Free shipping
            default:
                return 0;
        }
    }
    
    /**
     * Get all delivered orders
     */
    public function getDeliveredOrders() {
        $query = "SELECT o.* 
                  FROM orders o 
                  WHERE o.status = 'delivered'
                  ORDER BY o.updated_at DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        return $stmt->fetchAll();
    }
    
    /**
     * Get delivered orders statistics
     */
    public function getDeliveredStats() {
        $stats = [];
        
        // Total delivered orders
        $stmt = $this->conn->query("SELECT COUNT(*) as total FROM orders WHERE status = 'delivered'");
        $stats['total_delivered'] = $stmt->fetch()['total'];
        
        // Total revenue from delivered orders
        $stmt = $this->conn->query("SELECT SUM(total_amount) as total FROM orders WHERE status = 'delivered'");
        $stats['total_revenue'] = $stmt->fetch()['total'] ?? 0;
        
        // Delivered orders this month
        $stmt = $this->conn->query("SELECT COUNT(*) as total FROM orders 
                                   WHERE status = 'delivered' 
                                   AND MONTH(updated_at) = MONTH(CURRENT_DATE()) 
                                   AND YEAR(updated_at) = YEAR(CURRENT_DATE())");
        $stats['this_month'] = $stmt->fetch()['total'];
        
        // Average order value for delivered orders
        $stmt = $this->conn->query("SELECT AVG(total_amount) as avg FROM orders WHERE status = 'delivered'");
        $stats['avg_order_value'] = $stmt->fetch()['avg'] ?? 0;
        
        return $stats;
    }
    
    /**
     * Get delivered orders by date range
     */
    public function getDeliveredOrdersByDateRange($startDate, $endDate) {
        $query = "SELECT o.* 
                  FROM orders o 
                  WHERE o.status = 'delivered'
                  AND DATE(o.updated_at) BETWEEN :start_date AND :end_date
                  ORDER BY o.updated_at DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':start_date', $startDate);
        $stmt->bindValue(':end_date', $endDate);
        $stmt->execute();
        
        return $stmt->fetchAll();
    }
    
    /**
     * Get delivery performance metrics
     */
    public function getDeliveryPerformance() {
        $metrics = [];
        
        // Average delivery time
        $stmt = $this->conn->query("SELECT AVG(TIMESTAMPDIFF(HOUR, created_at, updated_at)) as avg_hours 
                                   FROM orders WHERE status = 'delivered'");
        $avgHours = $stmt->fetch()['avg_hours'] ?? 0;
        $metrics['avg_delivery_hours'] = round($avgHours, 2);
        $metrics['avg_delivery_days'] = round($avgHours / 24, 2);
        
        // Fastest delivery (minimum time)
        $stmt = $this->conn->query("SELECT MIN(TIMESTAMPDIFF(HOUR, created_at, updated_at)) as min_hours 
                                   FROM orders WHERE status = 'delivered'");
        $minHours = $stmt->fetch()['min_hours'] ?? 0;
        $metrics['fastest_delivery_hours'] = $minHours;
        
        // Slowest delivery (maximum time)
        $stmt = $this->conn->query("SELECT MAX(TIMESTAMPDIFF(HOUR, created_at, updated_at)) as max_hours 
                                   FROM orders WHERE status = 'delivered'");
        $maxHours = $stmt->fetch()['max_hours'] ?? 0;
        $metrics['slowest_delivery_hours'] = $maxHours;
        
        return $metrics;
    }
    
    // Category Management Functions
    public function addCategory($data) {
        try {
            $query = "INSERT INTO categories (name, slug, description, image, is_active) 
                      VALUES (:name, :slug, :description, :image, :is_active)";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindValue(':name', $data['name']);
            $stmt->bindValue(':slug', $data['slug']);
            $stmt->bindValue(':description', $data['description'] ?? '');
            $stmt->bindValue(':image', $data['image'] ?? '');
            $stmt->bindValue(':is_active', $data['is_active'] ?? 1);
            
            return $stmt->execute();
        } catch (Exception $e) {
            return false;
        }
    }
    
    public function updateCategory($id, $data) {
        try {
            $query = "UPDATE categories SET 
                        name = :name, slug = :slug, description = :description, 
                        image = :image, is_active = :is_active, updated_at = NOW()
                      WHERE id = :id";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindValue(':name', $data['name']);
            $stmt->bindValue(':slug', $data['slug']);
            $stmt->bindValue(':description', $data['description'] ?? '');
            $stmt->bindValue(':image', $data['image'] ?? '');
            $stmt->bindValue(':is_active', $data['is_active'] ?? 1);
            $stmt->bindValue(':id', $id, PDO::PARAM_INT);
            
            return $stmt->execute();
        } catch (Exception $e) {
            return false;
        }
    }
    
    public function deleteCategory($id) {
        try {
            $this->conn->beginTransaction();
            
            // Move products to uncategorized (category_id = NULL)
            $updateQuery = "UPDATE products SET category_id = NULL WHERE category_id = :id";
            $updateStmt = $this->conn->prepare($updateQuery);
            $updateStmt->bindValue(':id', $id, PDO::PARAM_INT);
            $updateStmt->execute();
            
            // Delete the category
            $deleteQuery = "DELETE FROM categories WHERE id = :id";
            $deleteStmt = $this->conn->prepare($deleteQuery);
            $deleteStmt->bindValue(':id', $id, PDO::PARAM_INT);
            $result = $deleteStmt->execute();
            
            $this->conn->commit();
            return $result;
        } catch (Exception $e) {
            $this->conn->rollBack();
            return false;
        }
    }
    
    public function getCategoryById($id) {
        try {
            $query = "SELECT * FROM categories WHERE id = :id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindValue(':id', $id, PDO::PARAM_INT);
            $stmt->execute();
            
            return $stmt->fetch();
        } catch (Exception $e) {
            return false;
        }
    }
    
    public function categorySlugExists($slug, $excludeId = null) {
        try {
            $query = "SELECT COUNT(*) FROM categories WHERE slug = :slug";
            $params = [':slug' => $slug];
            
            if ($excludeId) {
                $query .= " AND id != :exclude_id";
                $params[':exclude_id'] = $excludeId;
            }
            
            $stmt = $this->conn->prepare($query);
            foreach ($params as $key => $value) {
                $stmt->bindValue($key, $value);
            }
            $stmt->execute();
            
            return $stmt->fetchColumn() > 0;
        } catch (Exception $e) {
            return false;
        }
    }
    
    public function updateCategoryOrder($categories) {
        try {
            $this->conn->beginTransaction();
            
            foreach ($categories as $category) {
                $query = "UPDATE categories SET sort_order = :order WHERE id = :id";
                $stmt = $this->conn->prepare($query);
                $stmt->bindValue(':order', $category['order'], PDO::PARAM_INT);
                $stmt->bindValue(':id', $category['id'], PDO::PARAM_INT);
                $stmt->execute();
            }
            
            $this->conn->commit();
            return true;
        } catch (Exception $e) {
            $this->conn->rollBack();
            return false;
        }
    }
    
}

$adminFunctions = new AdminFunctions();
?>