<?php
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';

$adminAuth->requireLogin();

header('Content-Type: application/json');

if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    echo json_encode(['success' => false, 'message' => 'Invalid order ID']);
    exit;
}

$orderId = (int)$_GET['id'];

try {
    $adminFunctions = new AdminFunctions();
    
    // Get order details
    $order = $adminFunctions->getOrderById($orderId);
    
    if (!$order) {
        echo json_encode(['success' => false, 'message' => 'Order not found']);
        exit;
    }
    
    // Get order items
    $orderItems = $adminFunctions->getOrderItems($orderId);
    
    // Decode shipping address
    $shipping_address = json_decode($order['shipping_address'], true);
    
    // Generate HTML content
    ob_start();
    // Use same logic as order-success page to determine display status
    $display_order_status = $order['status'];
    $payment_status = $order['payment_status'] ?? 'pending';
    
    // If payment is successful, show processing status
    if ($payment_status === 'paid' && $order['status'] === 'awaiting_payment') {
        $display_order_status = 'processing';
    }
    ?>
    <div class="order-details">
        <div class="order-header">
            <h3>Order Details - <?php echo htmlspecialchars($order['order_number']); ?></h3>
            <div class="order-status">
                <span class="status status-<?php echo $display_order_status; ?>">
                    <?php echo ucfirst(str_replace('_', ' ', $display_order_status)); ?>
                </span>
            </div>
        </div>
        
        <div class="order-info-grid">
            <div class="info-section">
                <h4>Order Information</h4>
                <div class="info-item">
                    <label>Order Number:</label>
                    <span><?php echo htmlspecialchars($order['order_number']); ?></span>
                </div>
                <div class="info-item">
                    <label>Order Date:</label>
                    <span><?php echo date('M j, Y g:i A', strtotime($order['created_at'])); ?></span>
                </div>
                <div class="info-item">
                    <label>Status:</label>
                    <span class="status status-<?php echo $display_order_status; ?>">
                        <?php echo ucfirst(str_replace('_', ' ', $display_order_status)); ?>
                    </span>
                </div>
                <div class="info-item">
                    <label>Payment Method:</label>
                    <span><?php echo htmlspecialchars($order['payment_method']); ?></span>
                </div>
                <div class="info-item">
                    <label>Payment Status:</label>
                    <span class="status status-<?php echo $payment_status; ?>">
                        <?php echo ucfirst($payment_status); ?>
                    </span>
                </div>
                <div class="info-item">
                    <label>Total Amount:</label>
                    <span class="amount">RM <?php echo number_format($order['total_amount'], 2); ?></span>
                </div>
            </div>
            
            <div class="info-section">
                <h4>Customer Information</h4>
                <?php if ($shipping_address): ?>
                    <div class="info-item">
                        <label>Name:</label>
                        <span><?php echo htmlspecialchars($shipping_address['first_name'] . ' ' . $shipping_address['last_name']); ?></span>
                    </div>
                    <div class="info-item">
                        <label>Email:</label>
                        <span><?php echo htmlspecialchars($shipping_address['email']); ?></span>
                    </div>
                    <div class="info-item">
                        <label>Phone:</label>
                        <span><?php echo htmlspecialchars($shipping_address['phone']); ?></span>
                    </div>
                <?php else: ?>
                    <p>Customer information not available</p>
                <?php endif; ?>
            </div>
            
            <div class="info-section">
                <h4>Shipping Address</h4>
                <?php if ($shipping_address): ?>
                    <div class="address-block">
                        <p><strong><?php echo htmlspecialchars($shipping_address['first_name'] . ' ' . $shipping_address['last_name']); ?></strong></p>
                        <p><?php echo htmlspecialchars($shipping_address['address']); ?></p>
                        <p><?php echo htmlspecialchars($shipping_address['city'] . ', ' . $shipping_address['state'] . ' ' . $shipping_address['zip_code']); ?></p>
                        <p><?php echo htmlspecialchars($shipping_address['country']); ?></p>
                    </div>
                <?php else: ?>
                    <p>Shipping address not available</p>
                <?php endif; ?>
            </div>
        </div>
        
        <div class="order-items-section">
            <h4>Order Items</h4>
            <div class="items-table">
                <table>
                    <thead>
                        <tr>
                            <th>Product</th>
                            <th>Size</th>
                            <th>Quantity</th>
                            <th>Price</th>
                            <th>Total</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($orderItems as $item): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($item['product_name']); ?></td>
                            <td><?php echo htmlspecialchars($item['size'] ?: 'N/A'); ?></td>
                            <td><?php echo $item['quantity']; ?></td>
                            <td>RM <?php echo number_format($item['product_price'], 2); ?></td>
                            <td>RM <?php echo number_format($item['total_price'], 2); ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                    <tfoot>
                        <tr class="total-row">
                            <td colspan="4"><strong>Total:</strong></td>
                            <td><strong>RM <?php echo number_format($order['total_amount'], 2); ?></strong></td>
                        </tr>
                    </tfoot>
                </table>
            </div>
        </div>
        
        <?php if ($order['notes']): ?>
        <div class="order-notes">
            <h4>Order Notes</h4>
            <p><?php echo htmlspecialchars($order['notes']); ?></p>
        </div>
        <?php endif; ?>
    </div>
    
    <style>
        .order-details {
            max-width: 100%;
            margin: 0;
        }
        
        .order-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
            padding-bottom: 1rem;
            border-bottom: 2px solid #e9ecef;
        }
        
        .order-header h3 {
            margin: 0;
            color: #333;
        }
        
        .order-info-grid {
            display: grid;
            grid-template-columns: 1fr 1fr 1fr;
            gap: 2rem;
            margin-bottom: 2rem;
        }
        
        .info-section {
            background: #f8f9fa;
            padding: 1.5rem;
            border-radius: 8px;
        }
        
        .info-section h4 {
            margin: 0 0 1rem 0;
            color: #8B4513;
            font-size: 1.1rem;
        }
        
        .info-item {
            display: flex;
            justify-content: space-between;
            margin-bottom: 0.5rem;
            padding-bottom: 0.5rem;
            border-bottom: 1px solid #e9ecef;
        }
        
        .info-item:last-child {
            border-bottom: none;
            margin-bottom: 0;
        }
        
        .info-item label {
            font-weight: 600;
            color: #666;
        }
        
        .info-item span {
            color: #333;
        }
        
        .amount {
            font-weight: 600;
            color: #8B4513;
            font-size: 1.1rem;
        }
        
        .address-block {
            background: white;
            padding: 1rem;
            border-radius: 6px;
            border: 1px solid #e9ecef;
        }
        
        .address-block p {
            margin: 0.25rem 0;
        }
        
        .order-items-section {
            margin-bottom: 2rem;
        }
        
        .order-items-section h4 {
            margin: 0 0 1rem 0;
            color: #8B4513;
        }
        
        .items-table {
            overflow-x: auto;
        }
        
        .items-table table {
            width: 100%;
            border-collapse: collapse;
            background: white;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .items-table th,
        .items-table td {
            padding: 1rem;
            text-align: left;
            border-bottom: 1px solid #e9ecef;
        }
        
        .items-table th {
            background: #8B4513;
            color: white;
            font-weight: 600;
        }
        
        .items-table tfoot .total-row {
            background: #f8f9fa;
            font-weight: 600;
        }
        
        .items-table tfoot .total-row td {
            border-top: 2px solid #8B4513;
            border-bottom: none;
        }
        
        .order-notes {
            background: #fff3cd;
            padding: 1rem;
            border-radius: 8px;
            border-left: 4px solid #ffc107;
        }
        
        .order-notes h4 {
            margin: 0 0 0.5rem 0;
            color: #856404;
        }
        
        .order-notes p {
            margin: 0;
            color: #856404;
        }
        
        @media (max-width: 768px) {
            .order-info-grid {
                grid-template-columns: 1fr;
                gap: 1rem;
            }
            
            .order-header {
                flex-direction: column;
                align-items: flex-start;
                gap: 1rem;
            }
        }
    </style>
    <?php
    $html = ob_get_clean();
    
    echo json_encode([
        'success' => true,
        'html' => $html
    ]);
    
} catch (Exception $e) {
    error_log("Error getting order details: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => 'Error loading order details']);
}
?>