<?php
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';

$adminAuth->requireLogin();

header('Content-Type: application/json');

if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    echo json_encode(['success' => false, 'message' => 'Invalid order ID']);
    exit;
}

$orderId = (int)$_GET['id'];

try {
    $adminFunctions = new AdminFunctions();
    
    // Get order details
    $order = $adminFunctions->getOrderById($orderId);
    
    if (!$order) {
        echo json_encode(['success' => false, 'message' => 'Order not found']);
        exit;
    }
    
    // Get order items
    $orderItems = $adminFunctions->getOrderItems($orderId);
    
    // Decode shipping address
    $shipping_address = json_decode($order['shipping_address'], true);
    
    // Generate simplified HTML content
    ob_start();
    ?>
    <div class="order-summary">
        <div class="summary-header">
            <h4>Order #<?php echo htmlspecialchars($order['order_number']); ?></h4>
        </div>
        
        <div class="summary-content">
            <div class="customer-info">
                <h5>Customer Details</h5>
                <?php if ($shipping_address): ?>
                    <p><strong>Name:</strong> <?php echo htmlspecialchars($shipping_address['first_name'] . ' ' . $shipping_address['last_name']); ?></p>
                    <p><strong>Phone:</strong> <?php echo htmlspecialchars($shipping_address['phone']); ?></p>
                    <p><strong>Address:</strong> <?php echo htmlspecialchars($shipping_address['address'] . ', ' . $shipping_address['city'] . ', ' . $shipping_address['state'] . ' ' . $shipping_address['zip_code']); ?></p>
                <?php else: ?>
                    <p>Customer information not available</p>
                <?php endif; ?>
            </div>
            
            <div class="items-info">
                <h5>Items Ordered</h5>
                <div class="items-list">
                    <?php foreach ($orderItems as $item): ?>
                        <div class="item-row">
                            <span class="item-name"><?php echo htmlspecialchars($item['product_name']); ?></span>
                            <span class="item-details">
                                <?php if ($item['size']): ?>
                                    Size: <?php echo htmlspecialchars($item['size']); ?> | 
                                <?php endif; ?>
                                Qty: <?php echo $item['quantity']; ?> | 
                                RM <?php echo number_format($item['total_price'], 2); ?>
                            </span>
                        </div>
                    <?php endforeach; ?>
                </div>
                <div class="order-total">
                    <strong>Total: RM <?php echo number_format($order['total_amount'], 2); ?></strong>
                </div>
            </div>
        </div>
    </div>
    
    <style>
        .order-summary {
            background: #f8f9fa;
            border-radius: 8px;
            padding: 1rem;
            margin-bottom: 1.5rem;
            border: 1px solid #e9ecef;
        }
        
        .summary-header h4 {
            margin: 0 0 1rem 0;
            color: #8B4513;
            font-size: 1.1rem;
            border-bottom: 2px solid #8B4513;
            padding-bottom: 0.5rem;
        }
        
        .summary-content {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 1.5rem;
        }
        
        .customer-info h5,
        .items-info h5 {
            margin: 0 0 0.75rem 0;
            color: #333;
            font-size: 0.95rem;
            font-weight: 600;
        }
        
        .customer-info p {
            margin: 0.25rem 0;
            font-size: 0.9rem;
            line-height: 1.4;
        }
        
        .items-list {
            max-height: 150px;
            overflow-y: auto;
            margin-bottom: 0.75rem;
        }
        
        .item-row {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            padding: 0.5rem 0;
            border-bottom: 1px solid #e9ecef;
            font-size: 0.85rem;
        }
        
        .item-row:last-child {
            border-bottom: none;
        }
        
        .item-name {
            font-weight: 500;
            color: #333;
            flex: 1;
            margin-right: 0.5rem;
        }
        
        .item-details {
            color: #666;
            font-size: 0.8rem;
            text-align: right;
            white-space: nowrap;
        }
        
        .order-total {
            padding-top: 0.5rem;
            border-top: 2px solid #8B4513;
            text-align: right;
            color: #8B4513;
            font-size: 0.9rem;
        }
        
        @media (max-width: 768px) {
            .summary-content {
                grid-template-columns: 1fr;
                gap: 1rem;
            }
            
            .item-row {
                flex-direction: column;
                align-items: flex-start;
            }
            
            .item-details {
                text-align: left;
                margin-top: 0.25rem;
            }
        }
    </style>
    <?php
    $html = ob_get_clean();
    
    echo json_encode([
        'success' => true,
        'html' => $html
    ]);
    
} catch (Exception $e) {
    error_log("Error getting order summary: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => 'Error loading order summary']);
}
?>