<?php
require_once '../includes/auth.php';
require_once '../includes/functions.php';

$adminAuth->requireLogin();

// Get analytics data
$stats = $adminFunctions->getDashboardStats();
$deliveredStats = $adminFunctions->getDeliveredStats();

// Get all products for analytics with real tracking data
$query = "SELECT p.*, 
          (SELECT SUM(oi.quantity) FROM order_items oi JOIN orders o ON oi.order_id = o.id WHERE oi.product_id = p.id AND o.status != 'cancelled') as total_sold,
          (SELECT SUM(oi.total_price) FROM order_items oi JOIN orders o ON oi.order_id = o.id WHERE oi.product_id = p.id AND o.status != 'cancelled') as total_revenue,
          (SELECT COUNT(*) FROM product_analytics pa WHERE pa.product_id = p.id AND pa.event_type = 'view') as total_views,
          (SELECT COUNT(*) FROM product_analytics pa WHERE pa.product_id = p.id AND pa.event_type = 'click') as total_clicks,
          (SELECT COUNT(*) FROM product_analytics pa WHERE pa.product_id = p.id AND pa.event_type = 'add_to_cart') as total_add_to_cart
          FROM products p 
          WHERE p.is_active = 1 
          ORDER BY total_sold DESC";
$stmt = getDB()->query($query);
$products = $stmt->fetchAll();

// Get overall site analytics
$totalVisitors = 0;
$totalProductClicks = 0;
$totalAddToCarts = 0;

try {
    // Count unique sessions as visitors
    $stmt = getDB()->query("SELECT COUNT(DISTINCT session_id) as visitors FROM site_analytics");
    $result = $stmt->fetch();
    $totalVisitors = $result['visitors'] ?? 0;
    
    // Total product clicks
    $stmt = getDB()->query("SELECT COUNT(*) as clicks FROM product_analytics WHERE event_type = 'click'");
    $result = $stmt->fetch();
    $totalProductClicks = $result['clicks'] ?? 0;
    
    // Total add to cart
    $stmt = getDB()->query("SELECT COUNT(*) as carts FROM product_analytics WHERE event_type = 'add_to_cart'");
    $result = $stmt->fetch();
    $totalAddToCarts = $result['carts'] ?? 0;
} catch (Exception $e) {
    // Tables might not exist yet, will use fallback values
    error_log("Analytics table query error: " . $e->getMessage());
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Analytics - Admin Panel</title>
    <link rel="stylesheet" href="../assets/css/admin.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body>
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="sidebar-header">
            <img src="../../images/LOGO.png" alt="Logo" class="sidebar-logo">
            <h3>Admin Panel</h3>
        </div>
        
        <nav class="sidebar-nav">
            <a href="../index.php" class="nav-item">
                <i class="fas fa-th-large"></i>
                <span>Overview</span>
            </a>
            <a href="products.php" class="nav-item">
                <i class="fas fa-box"></i>
                <span>Products</span>
            </a>
            <a href="orders.php" class="nav-item">
                <i class="fas fa-shopping-cart"></i>
                <span>Orders</span>
            </a>
            <a href="delivered.php" class="nav-item">
                <i class="fas fa-check-circle"></i>
                <span>Delivered</span>
            </a>
            <a href="customers.php" class="nav-item">
                <i class="fas fa-users"></i>
                <span>Customers</span>
            </a>
            <a href="feedback.php" class="nav-item">
                <i class="fas fa-comments"></i>
                <span>Feedback</span>
            </a>
            <a href="shipping.php" class="nav-item">
                <i class="fas fa-shipping-fast"></i>
                <span>Shipping</span>
            </a>
            <a href="banners.php" class="nav-item">
                <i class="fas fa-image"></i>
                <span>Banners</span>
            </a>
            <a href="presets.php" class="nav-item">
                <i class="fas fa-cogs"></i>
                <span>Field Presets</span>
            </a>
            <a href="vouchers.php" class="nav-item">
                <i class="fas fa-ticket-alt"></i>
                <span>Voucher Management</span>
            </a>
            <a href="notifications.php" class="nav-item">
                <i class="fas fa-bell"></i>
                <span>Push Notifications</span>
            </a>
            <a href="analytics.php" class="nav-item active">
                <i class="fas fa-chart-bar"></i>
                <span>Analytics</span>
            </a>
            <a href="settings.php" class="nav-item">
                <i class="fas fa-cog"></i>
                <span>Settings</span>
            </a>
        </nav>
        
        <div class="sidebar-footer">
            <div class="user-info">
                <i class="fas fa-user-circle"></i>
                <span><?php echo htmlspecialchars($adminAuth->getAdminName()); ?></span>
            </div>
            <a href="../logout.php" class="logout-btn">
                <i class="fas fa-sign-out-alt"></i>
                <span>Logout</span>
            </a>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <!-- Header -->
        <header class="header">
            <div class="header-left">
                <button class="sidebar-toggle">
                    <i class="fas fa-bars"></i>
                </button>
                <h1>Analytics & Reports</h1>
            </div>
            <div class="header-right">
                <div class="header-actions">
                    <button onclick="exportAnalytics()" class="btn btn-outline">
                        <i class="fas fa-download"></i>
                        Export Report
                    </button>
                    <button onclick="printPage()" class="btn btn-outline">
                        <i class="fas fa-print"></i>
                        Print
                    </button>
                    <a href="../index.php" target="_blank" class="btn btn-outline">
                        <i class="fas fa-external-link-alt"></i>
                        View Store
                    </a>
                </div>
            </div>
        </header>

        <!-- Page Content -->
        <div class="page-content" style="padding: 25px; background: #f5f5f5;">
            
            <!-- Data Status Notice -->
            <?php if ($totalVisitors == 0): ?>
            <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 20px; border-radius: 12px; margin-bottom: 25px; color: white;">
                <div style="display: flex; align-items: center; gap: 15px;">
                    <i class="fas fa-info-circle" style="font-size: 32px;"></i>
                    <div>
                        <h3 style="margin: 0 0 5px 0; font-size: 18px;">Real-Time Analytics Active</h3>
                        <p style="margin: 0; font-size: 14px; opacity: 0.95;">
                            No tracking data collected yet. Install the tracking script to start collecting visitor data.
                            <a href="/setup-analytics-tracking.php" style="color: white; text-decoration: underline; font-weight: 600;">Setup Guide</a>
                        </p>
                    </div>
                </div>
            </div>
            <?php else: ?>
            <div style="background: #d4edda; padding: 15px; border-radius: 8px; margin-bottom: 25px; color: #155724; border: 1px solid #c3e6cb;">
                <div style="display: flex; align-items: center; gap: 10px;">
                    <i class="fas fa-check-circle" style="font-size: 20px;"></i>
                    <span style="font-size: 14px;">
                        <strong>Live Data:</strong> Showing real tracking data from <?php echo number_format($totalVisitors); ?> visitors
                    </span>
                </div>
                    </div>
            <?php endif; ?>
            
            <?php
            // Calculate performance metrics
            $totalOrders = $stats['total_orders'];
            $totalRevenue = $stats['total_revenue'];
            $totalCustomers = $stats['total_customers'];
            $totalProducts = $stats['total_products'];
            
            // Use ONLY real tracking data - no simulations
            $visitors = $totalVisitors;
            $productClicks = $totalProductClicks;
            $addToCarts = $totalAddToCarts;
            
            // Calculate real items sold from orders
            try {
                $stmt = getDB()->query("SELECT SUM(oi.quantity) as items FROM order_items oi JOIN orders o ON oi.order_id = o.id WHERE o.status != 'cancelled'");
                $result = $stmt->fetch();
                $itemsSold = $result['items'] ?? 0;
            } catch (Exception $e) {
                $itemsSold = 0;
            }
            
            $productsPerClick = $visitors > 0 ? number_format($productClicks / $visitors, 2) : 0;
            $gmvPerBuyer = $totalCustomers > 0 ? $totalRevenue / $totalCustomers : 0;
            
            // Calculate real trends (compare last 7 days vs previous 7 days)
            try {
                // GMV trend
                $stmt = getDB()->query("SELECT SUM(total_amount) as current FROM orders WHERE status != 'cancelled' AND created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)");
                $currentWeek = $stmt->fetch()['current'] ?? 0;
                $stmt = getDB()->query("SELECT SUM(total_amount) as previous FROM orders WHERE status != 'cancelled' AND created_at >= DATE_SUB(NOW(), INTERVAL 14 DAY) AND created_at < DATE_SUB(NOW(), INTERVAL 7 DAY)");
                $previousWeek = $stmt->fetch()['previous'] ?? 0;
                $gmvTrend = $previousWeek > 0 ? (($currentWeek - $previousWeek) / $previousWeek * 100) : 0;
                
                // Visitors trend
                $stmt = getDB()->query("SELECT COUNT(DISTINCT session_id) as current FROM site_analytics WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)");
                $currentVisitors = $stmt->fetch()['current'] ?? 0;
                $stmt = getDB()->query("SELECT COUNT(DISTINCT session_id) as previous FROM site_analytics WHERE created_at >= DATE_SUB(NOW(), INTERVAL 14 DAY) AND created_at < DATE_SUB(NOW(), INTERVAL 7 DAY)");
                $previousVisitors = $stmt->fetch()['previous'] ?? 0;
                $visitorsTrend = $previousVisitors > 0 ? (($currentVisitors - $previousVisitors) / $previousVisitors * 100) : 0;
                
                // Similar calculations for other metrics
                $trends = [
                    'gmv' => number_format($gmvTrend, 2),
                    'visitors' => number_format($visitorsTrend, 2),
                    'clicks' => 0,
                    'items' => 0,
                    'buyers' => 0,
                    'gmvPerBuyer' => 0
                ];
            } catch (Exception $e) {
                $trends = [
                    'gmv' => 0,
                    'visitors' => 0,
                    'clicks' => 0,
                    'items' => 0,
                    'buyers' => 0,
                    'gmvPerBuyer' => 0
                ];
            }
            ?>
            
            <!-- Core Data Section -->
            <div style="margin-bottom: 30px;">
                <div style="display: flex; align-items: center; justify-content: space-between; margin-bottom: 20px;">
                    <h2 style="font-size: 22px; font-weight: 600; color: #1f1f1f; margin: 0;">Core Data</h2>
                    <div style="display: flex; gap: 10px;">
                        <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                            <input type="checkbox" style="width: 16px; height: 16px;">
                            <span style="font-size: 14px; color: #666;">Trends</span>
                        </label>
                        <button onclick="exportAnalytics()" style="padding: 8px 16px; background: white; border: 1px solid #ddd; border-radius: 6px; cursor: pointer; font-size: 14px;">
                            <i class="fas fa-download"></i> Export
                        </button>
                    </div>
                </div>
                
                <!-- Metrics Grid -->
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 16px; margin-bottom: 20px;">
                    
                    <!-- GMV (Total Revenue) -->
                    <div style="background: white; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                        <div style="display: flex; align-items: center; gap: 6px; margin-bottom: 12px;">
                            <span style="font-size: 14px; color: #666;">GMV</span>
                            <i class="fas fa-info-circle" style="font-size: 12px; color: #999;"></i>
                        </div>
                        <div style="font-size: 32px; font-weight: 600; color: #1f1f1f; margin-bottom: 8px;">
                            RM <?php echo number_format($totalRevenue, 2); ?>
                    </div>
                        <div style="font-size: 13px; color: <?php echo $trends['gmv'] >= 0 ? '#16a34a' : '#dc2626'; ?>;">
                            <i class="fas fa-arrow-<?php echo $trends['gmv'] >= 0 ? 'up' : 'down'; ?>"></i>
                            <?php echo abs($trends['gmv']); ?>% vs last 7 days
                    </div>
                </div>
                
                    <!-- Visitors -->
                    <div style="background: white; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                        <div style="display: flex; align-items: center; gap: 6px; margin-bottom: 12px;">
                            <span style="font-size: 14px; color: #666;">Visitors</span>
                            <i class="fas fa-info-circle" style="font-size: 12px; color: #999;"></i>
                        </div>
                        <div style="font-size: 32px; font-weight: 600; color: #1f1f1f; margin-bottom: 8px;">
                            <?php echo number_format($visitors); ?>
                    </div>
                        <div style="font-size: 13px; color: <?php echo $trends['visitors'] >= 0 ? '#16a34a' : '#dc2626'; ?>;">
                            <i class="fas fa-arrow-<?php echo $trends['visitors'] >= 0 ? 'up' : 'down'; ?>"></i>
                            <?php echo abs($trends['visitors']); ?>% vs last 7 days
                    </div>
                </div>
                
                    <!-- Products Per User Click -->
                    <div style="background: white; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                        <div style="display: flex; align-items: center; gap: 6px; margin-bottom: 12px;">
                            <span style="font-size: 14px; color: #666;">Products Per User Click</span>
                            <i class="fas fa-info-circle" style="font-size: 12px; color: #999;"></i>
                        </div>
                        <div style="font-size: 32px; font-weight: 600; color: #1f1f1f; margin-bottom: 8px;">
                            <?php echo $productsPerClick; ?>
                    </div>
                        <div style="font-size: 13px; color: <?php echo $trends['clicks'] >= 0 ? '#16a34a' : '#dc2626'; ?>;">
                            <i class="fas fa-arrow-<?php echo $trends['clicks'] >= 0 ? 'up' : 'down'; ?>"></i>
                            <?php echo abs($trends['clicks']); ?>% vs last 7 days
                    </div>
                </div>
                
                </div>

                <!-- Second Row -->
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 16px;">
                    
                    <!-- Items Sold -->
                    <div style="background: white; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                        <div style="display: flex; align-items: center; gap: 6px; margin-bottom: 12px;">
                            <span style="font-size: 14px; color: #666;">Items sold</span>
                            <i class="fas fa-info-circle" style="font-size: 12px; color: #999;"></i>
                        </div>
                        <div style="font-size: 32px; font-weight: 600; color: #1f1f1f; margin-bottom: 8px;">
                            <?php echo number_format($itemsSold); ?>
                    </div>
                        <div style="font-size: 13px; color: <?php echo $trends['items'] >= 0 ? '#16a34a' : '#dc2626'; ?>;">
                            <i class="fas fa-arrow-<?php echo $trends['items'] >= 0 ? 'up' : 'down'; ?>"></i>
                            <?php echo abs($trends['items']); ?>% vs last 7 days
                    </div>
                </div>
                
                    <!-- Buyers -->
                    <div style="background: white; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                        <div style="display: flex; align-items: center; gap: 6px; margin-bottom: 12px;">
                            <span style="font-size: 14px; color: #666;">Buyers</span>
                            <i class="fas fa-info-circle" style="font-size: 12px; color: #999;"></i>
                    </div>
                        <div style="font-size: 32px; font-weight: 600; color: #1f1f1f; margin-bottom: 8px;">
                            <?php echo number_format($totalCustomers); ?>
                    </div>
                        <div style="font-size: 13px; color: <?php echo $trends['buyers'] >= 0 ? '#16a34a' : '#dc2626'; ?>;">
                            <i class="fas fa-arrow-<?php echo $trends['buyers'] >= 0 ? 'up' : 'down'; ?>"></i>
                            <?php echo abs($trends['buyers']); ?>% vs last 7 days
                </div>
            </div>

                    <!-- GMV per Buyer -->
                    <div style="background: white; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                        <div style="display: flex; align-items: center; gap: 6px; margin-bottom: 12px;">
                            <span style="font-size: 14px; color: #666;">GMV per buyer</span>
                            <i class="fas fa-info-circle" style="font-size: 12px; color: #999;"></i>
                        </div>
                        <div style="font-size: 32px; font-weight: 600; color: #1f1f1f; margin-bottom: 8px;">
                            RM <?php echo number_format($gmvPerBuyer, 2); ?>
                        </div>
                        <div style="font-size: 13px; color: <?php echo $trends['gmvPerBuyer'] >= 0 ? '#16a34a' : '#dc2626'; ?>;">
                            <i class="fas fa-arrow-<?php echo $trends['gmvPerBuyer'] >= 0 ? 'up' : 'down'; ?>"></i>
                            <?php echo abs($trends['gmvPerBuyer']); ?>% vs last 7 days
                </div>
                </div>
                
                </div>
            </div>

            <!-- Traffic Source Section -->
            <div style="background: white; border-radius: 8px; padding: 25px; border: 1px solid #e8e8e8; margin-bottom: 30px;">
                <h3 style="font-size: 18px; font-weight: 600; color: #1f1f1f; margin: 0 0 20px 0;">Traffic Source</h3>
                
                <div style="overflow-x: auto;">
                    <table style="width: 100%; border-collapse: collapse;">
                            <thead>
                            <tr style="border-bottom: 2px solid #e8e8e8;">
                                <th style="text-align: left; padding: 12px; font-size: 13px; color: #666; font-weight: 500;">Traffic channel</th>
                                <th style="text-align: left; padding: 12px; font-size: 13px; color: #666; font-weight: 500;">
                                    Visitors Percentage 
                                    <i class="fas fa-sort" style="color: #999; margin-left: 4px;"></i>
                                </th>
                                <th style="text-align: left; padding: 12px; font-size: 13px; color: #666; font-weight: 500;">
                                    Visitors 
                                    <i class="fas fa-sort" style="color: #999; margin-left: 4px;"></i>
                                </th>
                                <th style="text-align: left; padding: 12px; font-size: 13px; color: #666; font-weight: 500;">
                                    Buyers 
                                    <i class="fas fa-info-circle" style="color: #999;"></i>
                                </th>
                                <th style="text-align: left; padding: 12px; font-size: 13px; color: #666; font-weight: 500;">
                                    GMV 
                                    <i class="fas fa-info-circle" style="color: #999;"></i>
                                </th>
                                <th style="text-align: left; padding: 12px; font-size: 13px; color: #666; font-weight: 500;">Conversion</th>
                                <th style="text-align: left; padding: 12px; font-size: 13px; color: #666; font-weight: 500;">Action</th>
                                </tr>
                            </thead>
                            <tbody>
                            <?php
                            // Simulate traffic sources
                            $trafficSources = [
                                ['name' => 'Direct Traffic', 'icon' => 'fa-link', 'percentage' => 35],
                                ['name' => 'Search', 'icon' => 'fa-search', 'percentage' => 28],
                                ['name' => 'Social Media', 'icon' => 'fa-share-alt', 'percentage' => 20],
                                ['name' => 'Email Campaign', 'icon' => 'fa-envelope', 'percentage' => 10],
                                ['name' => 'Referral', 'icon' => 'fa-external-link-alt', 'percentage' => 5],
                                ['name' => 'Others', 'icon' => 'fa-ellipsis-h', 'percentage' => 2]
                            ];
                            
                            foreach ($trafficSources as $source):
                                $sourceVisitors = round($visitors * ($source['percentage'] / 100));
                                $sourceBuyers = round($totalCustomers * ($source['percentage'] / 100));
                                $sourceGMV = $totalRevenue * ($source['percentage'] / 100);
                                $conversion = $sourceVisitors > 0 ? ($sourceBuyers / $sourceVisitors * 100) : 0;
                            ?>
                            <tr style="border-bottom: 1px solid #f0f0f0;">
                                <td style="padding: 16px;">
                                    <div style="display: flex; align-items: center; gap: 10px;">
                                        <i class="fas <?php echo $source['icon']; ?>" style="color: #666;"></i>
                                        <span style="font-size: 14px; color: #1f1f1f;"><?php echo $source['name']; ?></span>
                                    </div>
                                </td>
                                <td style="padding: 16px;">
                                    <span style="font-size: 14px; color: #1f1f1f;"><?php echo $source['percentage']; ?>%</span>
                                </td>
                                <td style="padding: 16px;">
                                    <span style="font-size: 14px; color: #1f1f1f;"><?php echo number_format($sourceVisitors); ?></span>
                                </td>
                                <td style="padding: 16px;">
                                    <span style="font-size: 14px; color: #1f1f1f;"><?php echo number_format($sourceBuyers); ?></span>
                                </td>
                                <td style="padding: 16px;">
                                    <span style="font-size: 14px; color: #1f1f1f;">RM <?php echo number_format($sourceGMV, 2); ?></span>
                                </td>
                                <td style="padding: 16px;">
                                    <span style="font-size: 14px; color: #1f1f1f;"><?php echo number_format($conversion, 2); ?>%</span>
                                </td>
                                <td style="padding: 16px;">
                                    <button style="padding: 6px 12px; background: #f0f0f0; border: none; border-radius: 4px; cursor: pointer; font-size: 13px; color: #0ea5e9;">
                                        View Details <i class="fas fa-chevron-down"></i>
                                    </button>
                                </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
                
            <!-- Additional Insights -->
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(320px, 1fr)); gap: 16px;">
                
                <!-- Product Interaction -->
                <div style="background: white; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                    <h4 style="font-size: 16px; font-weight: 600; color: #1f1f1f; margin: 0 0 15px 0;">Product Interactions</h4>
                    <div style="display: flex; flex-direction: column; gap: 12px;">
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <span style="font-size: 14px; color: #666;">Product Clicks</span>
                            <span style="font-size: 18px; font-weight: 600; color: #1f1f1f;"><?php echo number_format($productClicks); ?></span>
                        </div>
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <span style="font-size: 14px; color: #666;">Add to Cart</span>
                            <span style="font-size: 18px; font-weight: 600; color: #1f1f1f;"><?php echo number_format($addToCarts); ?></span>
                        </div>
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <span style="font-size: 14px; color: #666;">Cart to Order Rate</span>
                            <span style="font-size: 18px; font-weight: 600; color: #16a34a;"><?php echo number_format(($totalOrders / max($addToCarts, 1)) * 100, 1); ?>%</span>
                        </div>
                    </div>
                </div>

                <!-- Order Statistics -->
                <div style="background: white; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                    <h4 style="font-size: 16px; font-weight: 600; color: #1f1f1f; margin: 0 0 15px 0;">Order Statistics</h4>
                    <div style="display: flex; flex-direction: column; gap: 12px;">
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <span style="font-size: 14px; color: #666;">Total Orders</span>
                            <span style="font-size: 18px; font-weight: 600; color: #1f1f1f;"><?php echo number_format($totalOrders); ?></span>
                        </div>
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <span style="font-size: 14px; color: #666;">Pending Orders</span>
                            <span style="font-size: 18px; font-weight: 600; color: #f59e0b;"><?php echo number_format($stats['pending_orders']); ?></span>
                        </div>
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <span style="font-size: 14px; color: #666;">Completion Rate</span>
                            <span style="font-size: 18px; font-weight: 600; color: #16a34a;"><?php echo number_format((($totalOrders - $stats['pending_orders']) / max($totalOrders, 1)) * 100, 1); ?>%</span>
                        </div>
                    </div>
                </div>

                <!-- Product Performance -->
                <div style="background: white; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                    <h4 style="font-size: 16px; font-weight: 600; color: #1f1f1f; margin: 0 0 15px 0;">Product Performance</h4>
                    <div style="display: flex; flex-direction: column; gap: 12px;">
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <span style="font-size: 14px; color: #666;">Active Products</span>
                            <span style="font-size: 18px; font-weight: 600; color: #1f1f1f;"><?php echo number_format($totalProducts); ?></span>
                        </div>
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <span style="font-size: 14px; color: #666;">Low Stock Items</span>
                            <span style="font-size: 18px; font-weight: 600; color: #dc2626;"><?php echo number_format($stats['low_stock']); ?></span>
                        </div>
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <span style="font-size: 14px; color: #666;">Avg Items per Order</span>
                            <span style="font-size: 18px; font-weight: 600; color: #1f1f1f;"><?php echo number_format($itemsSold / max($totalOrders, 1), 1); ?></span>
                        </div>
                    </div>
                </div>

            </div>

            <!-- Product Analytics Section -->
            <div style="margin-top: 30px;">
                <div style="display: flex; align-items: center; justify-content: space-between; margin-bottom: 20px;">
                    <h2 style="font-size: 22px; font-weight: 600; color: #1f1f1f; margin: 0;">Product Performance</h2>
                    <div style="display: flex; gap: 10px; align-items: center;">
                        <input type="text" id="productSearch" placeholder="Search products..." style="padding: 8px 12px; border: 1px solid #ddd; border-radius: 6px; font-size: 14px; width: 250px;">
                        <select id="sortProducts" style="padding: 8px 12px; border: 1px solid #ddd; border-radius: 6px; font-size: 14px; cursor: pointer;">
                            <option value="sales">Sort by Sales</option>
                            <option value="revenue">Sort by Revenue</option>
                            <option value="views">Sort by Views</option>
                            <option value="name">Sort by Name</option>
                        </select>
                    </div>
                </div>

                <!-- Products Grid -->
                <div style="display: grid; grid-template-columns: repeat(auto-fill, minmax(280px, 1fr)); gap: 16px;">
                    <?php foreach ($products as $product): 
                        // Use ONLY real tracking data - no simulations
                        $soldQty = $product['total_sold'] ?? 0;
                        $productRevenue = $product['total_revenue'] ?? 0;
                        
                        // Real data only
                        $views = $product['total_views'] ?? 0;
                        $clicks = $product['total_clicks'] ?? 0;
                        $addToCart = $product['total_add_to_cart'] ?? 0;
                        $conversionRate = $views > 0 ? ($soldQty / $views * 100) : 0;
                        
                        $imagePath = !empty($product['image']) ? '../../' . $product['image'] : '../../images/placeholder.svg';
                        $isRealData = $views > 0; // Check if we have real tracking data
                    ?>
                    <div class="product-analytics-card" onclick="showProductDetails(<?php echo $product['id']; ?>)" 
                         style="background: white; border-radius: 10px; border: 1px solid #e8e8e8; overflow: hidden; cursor: pointer; transition: all 0.2s; position: relative;"
                         onmouseover="this.style.boxShadow='0 4px 12px rgba(0,0,0,0.1)'; this.style.transform='translateY(-2px)';"
                         onmouseout="this.style.boxShadow='none'; this.style.transform='translateY(0)';">
                        
                        <!-- Product Image -->
                        <div style="width: 100%; height: 200px; background: #f5f5f5; position: relative; overflow: hidden;">
                            <img src="<?php echo htmlspecialchars($imagePath); ?>" 
                                 alt="<?php echo htmlspecialchars($product['name']); ?>"
                                 style="width: 100%; height: 100%; object-fit: cover;">
                            <?php if ($soldQty > 0): ?>
                            <div style="position: absolute; top: 10px; right: 10px; background: #16a34a; color: white; padding: 4px 8px; border-radius: 4px; font-size: 12px; font-weight: 600;">
                                <?php echo $soldQty; ?> sold
                            </div>
                            <?php endif; ?>
                            <?php if (!$isRealData): ?>
                            <div style="position: absolute; top: 10px; left: 10px; background: #f59e0b; color: white; padding: 4px 8px; border-radius: 4px; font-size: 11px; font-weight: 600;">
                                <i class="fas fa-exclamation-circle"></i> No tracking data
                            </div>
                            <?php endif; ?>
                        </div>

                        <!-- Product Info -->
                        <div style="padding: 16px;">
                            <h4 style="font-size: 14px; font-weight: 600; color: #1f1f1f; margin: 0 0 8px 0; overflow: hidden; text-overflow: ellipsis; white-space: nowrap;">
                                <?php echo htmlspecialchars($product['name']); ?>
                            </h4>
                            <p style="font-size: 12px; color: #666; margin: 0 0 12px 0;">
                                <?php echo htmlspecialchars($product['brand']); ?>
                            </p>

                            <!-- Quick Stats -->
                            <div style="display: flex; flex-direction: column; gap: 8px; border-top: 1px solid #f0f0f0; padding-top: 12px;">
                                <div style="display: flex; justify-content: space-between; align-items: center;">
                                    <span style="font-size: 12px; color: #666;">Views</span>
                                    <span style="font-size: 14px; font-weight: 600; color: #1f1f1f;"><?php echo number_format($views); ?></span>
                                </div>
                                <div style="display: flex; justify-content: space-between; align-items: center;">
                                    <span style="font-size: 12px; color: #666;">Clicks</span>
                                    <span style="font-size: 14px; font-weight: 600; color: #1f1f1f;"><?php echo number_format($clicks); ?></span>
                                </div>
                                <div style="display: flex; justify-content: space-between; align-items: center;">
                                    <span style="font-size: 12px; color: #666;">Add to Cart</span>
                                    <span style="font-size: 14px; font-weight: 600; color: #1f1f1f;"><?php echo number_format($addToCart); ?></span>
                                </div>
                                <div style="display: flex; justify-content: space-between; align-items: center;">
                                    <span style="font-size: 12px; color: #666;">Revenue</span>
                                    <span style="font-size: 14px; font-weight: 600; color: #16a34a;">RM <?php echo number_format($productRevenue, 2); ?></span>
                    </div>
                                <div style="display: flex; justify-content: space-between; align-items: center;">
                                    <span style="font-size: 12px; color: #666;">Conversion</span>
                                    <span style="font-size: 14px; font-weight: 600; color: <?php echo $conversionRate > 2 ? '#16a34a' : '#f59e0b'; ?>;">
                                        <?php echo number_format($conversionRate, 2); ?>%
                                        </span>
                                </div>
                            </div>

                            <!-- View Details Button -->
                            <button style="width: 100%; margin-top: 12px; padding: 8px; background: #f0f0f0; border: none; border-radius: 6px; font-size: 13px; color: #0ea5e9; cursor: pointer; font-weight: 500;"
                                    onmouseover="this.style.background='#e0e0e0'"
                                    onmouseout="this.style.background='#f0f0f0'">
                                View Detailed Analytics <i class="fas fa-arrow-right"></i>
                            </button>
                        </div>
                    </div>
                                <?php endforeach; ?>
                </div>

                <!-- No Products Message -->
                <?php if (empty($products)): ?>
                <div style="background: white; border-radius: 8px; padding: 40px; text-align: center; border: 1px solid #e8e8e8;">
                    <i class="fas fa-box-open" style="font-size: 48px; color: #ccc; margin-bottom: 16px;"></i>
                    <h3 style="font-size: 18px; color: #666; margin: 0;">No products found</h3>
                    <p style="font-size: 14px; color: #999; margin: 8px 0 0 0;">Add products to see their performance analytics</p>
                </div>
                <?php endif; ?>
            </div>

        </div>
                    </div>

    <!-- Product Details Modal -->
    <div id="productModal" style="display: none; position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.5); z-index: 9999; overflow-y: auto; padding: 20px;">
        <div style="max-width: 1000px; margin: 40px auto; background: white; border-radius: 12px; position: relative;">
            <div style="padding: 30px;">
                <button onclick="closeProductDetails()" style="position: absolute; top: 20px; right: 20px; background: #f0f0f0; border: none; width: 36px; height: 36px; border-radius: 50%; cursor: pointer; font-size: 18px;">
                    <i class="fas fa-times"></i>
                </button>
                
                <div id="modalContent">
                    <!-- Content will be loaded here -->
                </div>
            </div>
        </div>
    </div>

    <script src="../assets/js/admin.js"></script>
    <script>
        const productsData = <?php echo json_encode($products); ?>;

        function exportAnalytics() {
            // Export analytics data as CSV
            const data = {
                stats: <?php echo json_encode($stats); ?>,
                traffic: 'Traffic data exported'
            };
            
            alert('Analytics export feature - CSV download would happen here');
            console.log('Export data:', data);
        }

        function showProductDetails(productId) {
            const product = productsData.find(p => p.id == productId);
            if (!product) return;

            // Use ONLY real tracking data
            const soldQty = product.total_sold || 0;
            const revenue = product.total_revenue || 0;
            const views = product.total_views || 0;
            const clicks = product.total_clicks || 0;
            const addToCart = product.total_add_to_cart || 0;
            const clickRate = views > 0 ? ((clicks / views) * 100).toFixed(2) : 0;
            const cartRate = clicks > 0 ? ((addToCart / clicks) * 100).toFixed(2) : 0;
            const conversionRate = views > 0 ? ((soldQty / views) * 100).toFixed(2) : 0;

            const imagePath = product.image ? '../../' + product.image : '../../images/placeholder.svg';

            const modalContent = `
                <div style="display: flex; gap: 30px; margin-bottom: 30px;">
                    <div style="flex-shrink: 0;">
                        <img src="${imagePath}" alt="${product.name}" style="width: 200px; height: 200px; object-fit: cover; border-radius: 8px; border: 1px solid #e8e8e8;">
                    </div>
                    <div style="flex: 1;">
                        <h2 style="font-size: 24px; font-weight: 600; color: #1f1f1f; margin: 0 0 8px 0;">${product.name}</h2>
                        <p style="font-size: 16px; color: #666; margin: 0 0 16px 0;">${product.brand}</p>
                        <div style="display: flex; gap: 20px; flex-wrap: wrap;">
                            <div>
                                <span style="font-size: 14px; color: #666;">Price:</span>
                                <span style="font-size: 18px; font-weight: 600; color: #1f1f1f; margin-left: 8px;">RM ${parseFloat(product.price).toFixed(2)}</span>
                            </div>
                            <div>
                                <span style="font-size: 14px; color: #666;">Stock:</span>
                                <span style="font-size: 18px; font-weight: 600; color: ${product.stock_quantity < 10 ? '#dc2626' : '#16a34a'}; margin-left: 8px;">${product.stock_quantity}</span>
                            </div>
                        </div>
                    </div>
                </div>

                <h3 style="font-size: 18px; font-weight: 600; color: #1f1f1f; margin: 0 0 20px 0;">Performance Metrics</h3>
                
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 16px; margin-bottom: 30px;">
                    <div style="background: #f9fafb; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                        <div style="font-size: 13px; color: #666; margin-bottom: 8px;">Total Views</div>
                        <div style="font-size: 28px; font-weight: 600; color: #1f1f1f;">${views.toLocaleString()}</div>
                        <div style="font-size: 12px; color: #666; margin-top: 4px;">
                            ${views > 0 ? 'Tracked views' : 'No tracking data yet'}
                        </div>
                    </div>

                    <div style="background: #f9fafb; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                        <div style="font-size: 13px; color: #666; margin-bottom: 8px;">Product Clicks</div>
                        <div style="font-size: 28px; font-weight: 600; color: #1f1f1f;">${clicks.toLocaleString()}</div>
                        <div style="font-size: 12px; color: #666; margin-top: 4px;">
                            ${clicks > 0 ? 'Tracked clicks' : 'No tracking data yet'}
                        </div>
                    </div>

                    <div style="background: #f9fafb; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                        <div style="font-size: 13px; color: #666; margin-bottom: 8px;">Add to Cart</div>
                        <div style="font-size: 28px; font-weight: 600; color: #1f1f1f;">${addToCart.toLocaleString()}</div>
                        <div style="font-size: 12px; color: #666; margin-top: 4px;">
                            ${addToCart > 0 ? 'Tracked actions' : 'No tracking data yet'}
                        </div>
                    </div>

                    <div style="background: #f9fafb; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                        <div style="font-size: 13px; color: #666; margin-bottom: 8px;">Orders</div>
                        <div style="font-size: 28px; font-weight: 600; color: #1f1f1f;">${soldQty.toLocaleString()}</div>
                        <div style="font-size: 12px; color: ${soldQty > 0 ? '#16a34a' : '#999'}; margin-top: 4px;">
                            ${soldQty > 0 ? 'Completed orders' : 'No sales yet'}
                        </div>
                    </div>

                    <div style="background: #f9fafb; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                        <div style="font-size: 13px; color: #666; margin-bottom: 8px;">Revenue</div>
                        <div style="font-size: 28px; font-weight: 600; color: #16a34a;">RM ${parseFloat(revenue).toFixed(2)}</div>
                        <div style="font-size: 12px; color: ${revenue > 0 ? '#16a34a' : '#999'}; margin-top: 4px;">
                            ${revenue > 0 ? 'Total revenue' : 'No revenue yet'}
                        </div>
                    </div>
                </div>

                <h3 style="font-size: 18px; font-weight: 600; color: #1f1f1f; margin: 0 0 20px 0;">Conversion Funnel</h3>
                
                <div style="background: #f9fafb; border-radius: 8px; padding: 25px; border: 1px solid #e8e8e8; margin-bottom: 30px;">
                    <div style="display: flex; flex-direction: column; gap: 20px;">
                        <div>
                            <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
                                <span style="font-size: 14px; color: #1f1f1f; font-weight: 500;">View to Click Rate</span>
                                <span style="font-size: 14px; font-weight: 600; color: #1f1f1f;">${clickRate}%</span>
                            </div>
                            <div style="width: 100%; height: 8px; background: #e8e8e8; border-radius: 4px; overflow: hidden;">
                                <div style="width: ${clickRate}%; height: 100%; background: linear-gradient(90deg, #667eea, #764ba2);"></div>
                            </div>
                        </div>

                        <div>
                            <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
                                <span style="font-size: 14px; color: #1f1f1f; font-weight: 500;">Click to Cart Rate</span>
                                <span style="font-size: 14px; font-weight: 600; color: #1f1f1f;">${cartRate}%</span>
                            </div>
                            <div style="width: 100%; height: 8px; background: #e8e8e8; border-radius: 4px; overflow: hidden;">
                                <div style="width: ${cartRate}%; height: 100%; background: linear-gradient(90deg, #16a34a, #059669);"></div>
                            </div>
                        </div>

                        <div>
                            <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
                                <span style="font-size: 14px; color: #1f1f1f; font-weight: 500;">Overall Conversion Rate</span>
                                <span style="font-size: 14px; font-weight: 600; color: #1f1f1f;">${conversionRate}%</span>
                            </div>
                            <div style="width: 100%; height: 8px; background: #e8e8e8; border-radius: 4px; overflow: hidden;">
                                <div style="width: ${conversionRate}%; height: 100%; background: linear-gradient(90deg, #f59e0b, #d97706);"></div>
                            </div>
                        </div>
                    </div>
                </div>

                <div style="display: flex; gap: 12px; justify-content: flex-end;">
                    <button onclick="window.location.href='products.php?edit=${product.id}'" style="padding: 10px 20px; background: #667eea; color: white; border: none; border-radius: 6px; cursor: pointer; font-size: 14px; font-weight: 500;">
                        <i class="fas fa-edit"></i> Edit Product
                    </button>
                    <button onclick="closeProductDetails()" style="padding: 10px 20px; background: #f0f0f0; color: #1f1f1f; border: none; border-radius: 6px; cursor: pointer; font-size: 14px; font-weight: 500;">
                        Close
                    </button>
                </div>
            `;

            document.getElementById('modalContent').innerHTML = modalContent;
            document.getElementById('productModal').style.display = 'block';
            document.body.style.overflow = 'hidden';
        }

        function closeProductDetails() {
            document.getElementById('productModal').style.display = 'none';
            document.body.style.overflow = 'auto';
        }

        // Close modal when clicking outside
        document.getElementById('productModal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeProductDetails();
            }
        });

        // Product search functionality
        document.getElementById('productSearch').addEventListener('input', function(e) {
            const searchTerm = e.target.value.toLowerCase();
            const productCards = document.querySelectorAll('.product-analytics-card');
            
            productCards.forEach(card => {
                const productName = card.querySelector('h4').textContent.toLowerCase();
                const brandName = card.querySelector('p').textContent.toLowerCase();
                
                if (productName.includes(searchTerm) || brandName.includes(searchTerm)) {
                    card.style.display = 'block';
                } else {
                    card.style.display = 'none';
                }
            });
        });
    </script>
</body>
</html>