<?php
require_once '../includes/auth.php';
require_once '../includes/functions.php';

$adminFunctions = new AdminFunctions();

// Handle actions
$action = $_GET['action'] ?? 'list';
$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    switch ($action) {
        case 'add':
            $data = [
                'name' => $_POST['name'],
                'description' => $_POST['description'],
                'fee_type' => $_POST['fee_type'],
                'base_fee' => $_POST['base_fee'],
                'per_kg_fee' => $_POST['per_kg_fee'],
                'min_weight' => $_POST['min_weight'],
                'max_weight' => !empty($_POST['max_weight']) ? $_POST['max_weight'] : null,
                'min_order_amount' => $_POST['min_order_amount'],
                'max_order_amount' => !empty($_POST['max_order_amount']) ? $_POST['max_order_amount'] : null,
                'delivery_days_min' => $_POST['delivery_days_min'],
                'delivery_days_max' => $_POST['delivery_days_max'],
                'is_active' => isset($_POST['is_active']) ? 1 : 0,
                'sort_order' => $_POST['sort_order']
            ];
            
            if ($adminFunctions->addShippingFee($data)) {
                $message = 'Shipping fee added successfully!';
            } else {
                $error = 'Failed to add shipping fee.';
            }
            break;
            
        case 'edit':
            $id = (int)$_POST['id'];
            $data = [
                'name' => $_POST['name'],
                'description' => $_POST['description'],
                'fee_type' => $_POST['fee_type'],
                'base_fee' => $_POST['base_fee'],
                'per_kg_fee' => $_POST['per_kg_fee'],
                'min_weight' => $_POST['min_weight'],
                'max_weight' => !empty($_POST['max_weight']) ? $_POST['max_weight'] : null,
                'min_order_amount' => $_POST['min_order_amount'],
                'max_order_amount' => !empty($_POST['max_order_amount']) ? $_POST['max_order_amount'] : null,
                'delivery_days_min' => $_POST['delivery_days_min'],
                'delivery_days_max' => $_POST['delivery_days_max'],
                'is_active' => isset($_POST['is_active']) ? 1 : 0,
                'sort_order' => $_POST['sort_order']
            ];
            
            if ($adminFunctions->updateShippingFee($id, $data)) {
                $message = 'Shipping fee updated successfully!';
            } else {
                $error = 'Failed to update shipping fee.';
            }
            break;
            
        case 'delete':
            $id = (int)$_POST['id'];
            
            if ($adminFunctions->deleteShippingFee($id)) {
                $message = 'Shipping fee deleted successfully!';
            } else {
                $error = 'Failed to delete shipping fee.';
            }
            break;
    }
}

// Get shipping fees
$shippingFees = $adminFunctions->getAllShippingFees();

// Get shipping fee for editing
$editShippingFee = null;
if (isset($_GET['edit']) && is_numeric($_GET['edit'])) {
    $editShippingFee = $adminFunctions->getShippingFeeById((int)$_GET['edit']);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Shipping Management - Admin Panel</title>
    <link rel="stylesheet" href="../assets/css/admin.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        /* Minimalist Table Design */
        .table-container {
            background: white;
            border-radius: 12px;
            overflow: hidden;
            box-shadow: 0 1px 3px rgba(0,0,0,0.08);
        }
        
        .admin-table {
            width: 100%;
            border-collapse: separate;
            border-spacing: 0;
        }
        
        .admin-table thead {
            background: #fafafa;
            border-bottom: 1px solid #f0f0f0;
        }
        
        .admin-table th {
            padding: 16px 20px;
            text-align: left;
            font-weight: 600;
            font-size: 13px;
            color: #666;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .admin-table tbody tr {
            border-bottom: 1px solid #f5f5f5;
            transition: background 0.2s ease;
        }
        
        .admin-table tbody tr:last-child {
            border-bottom: none;
        }
        
        .admin-table tbody tr:hover {
            background: #fafafa;
        }
        
        .admin-table td {
            padding: 20px;
            vertical-align: middle;
            font-size: 14px;
            color: #333;
        }
        
        /* Shipping Info */
        .shipping-info {
            display: flex;
            flex-direction: column;
            gap: 6px;
        }
        
        .shipping-info strong {
            font-size: 14px;
            font-weight: 600;
            color: #222;
            line-height: 1.4;
        }
        
        .shipping-info small {
            font-size: 12px;
            color: #999;
            font-weight: 400;
            line-height: 1.5;
        }
        
        /* Fee Type Badge */
        .fee-type-badge {
            display: inline-block;
            padding: 6px 12px;
            border-radius: 6px;
            font-size: 12px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .fee-type-weight_based {
            background: #e3f2fd;
            color: #1976d2;
        }
        
        .fee-type-free_shipping_threshold {
            background: #e8f5e9;
            color: #388e3c;
        }
        
        /* Pricing */
        .free-shipping {
            display: inline-block;
            padding: 6px 12px;
            background: #e8f5e9;
            color: #2e7d32;
            border-radius: 6px;
            font-weight: 700;
            font-size: 13px;
            letter-spacing: 0.5px;
        }
        
        .weight-pricing {
            display: flex;
            flex-direction: column;
            gap: 4px;
            align-items: center;
        }
        
        .base-fee {
            font-size: 14px;
            font-weight: 600;
            color: #333;
        }
        
        .per-kg-fee {
            font-size: 12px;
            color: #666;
        }
        
        /* Weight Range */
        .weight-range {
            font-size: 14px;
            color: #333;
            font-weight: 500;
        }
        
        .no-limit, .no-minimum {
            font-size: 13px;
            color: #999;
            font-style: italic;
        }
        
        /* Delivery Info */
        .delivery-info span {
            display: inline-block;
            padding: 6px 12px;
            background: #fff3e0;
            color: #e65100;
            border-radius: 6px;
            font-size: 13px;
            font-weight: 600;
        }
        
        /* Status Badge */
        .status-badge {
            display: inline-block;
            padding: 6px 14px;
            border-radius: 6px;
            font-size: 12px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .status-badge.active {
            background: #e8f5e9;
            color: #2e7d32;
        }
        
        .status-badge.inactive {
            background: #ffebee;
            color: #c62828;
        }
        
        /* Sort Order */
        .sort-order {
            display: inline-block;
            width: 32px;
            height: 32px;
            line-height: 32px;
            text-align: center;
            background: #f5f5f5;
            color: #666;
            border-radius: 6px;
            font-weight: 600;
            font-size: 14px;
        }
        
        /* Action Buttons */
        .action-buttons {
            display: flex;
            gap: 8px;
            justify-content: center;
            align-items: center;
        }
        
        .btn-sm {
            padding: 8px 12px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 14px;
            transition: all 0.2s ease;
            display: inline-flex;
            align-items: center;
            justify-content: center;
        }
        
        .btn-outline {
            background: #fff;
            color: #666;
            border: 1px solid #f0f0f0;
        }
        
        .btn-outline:hover {
            background: #f5f5f5;
            color: #333;
            border-color: #ddd;
            transform: translateY(-1px);
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        
        .btn-danger {
            background: #fff;
            color: #dc3545;
            border: 1px solid #f0f0f0;
        }
        
        .btn-danger:hover {
            background: #dc3545;
            color: white;
            border-color: #dc3545;
            transform: translateY(-1px);
            box-shadow: 0 2px 8px rgba(220, 53, 69, 0.2);
        }
        
        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 60px 20px;
        }
        
        .empty-state i {
            font-size: 48px;
            color: #e0e0e0;
            margin-bottom: 16px;
        }
        
        .empty-state h3 {
            font-size: 18px;
            color: #666;
            margin-bottom: 8px;
        }
        
        .empty-state p {
            font-size: 14px;
            color: #999;
        }
        
        /* Minimalist Alerts */
        .alert {
            padding: 16px 20px;
            border-radius: 10px;
            margin-bottom: 20px;
            font-size: 14px;
            display: flex;
            align-items: center;
            gap: 12px;
            border: 1px solid;
        }
        
        .alert-success {
            background: #f0f9f4;
            color: #0f5132;
            border-color: #d1e7dd;
        }
        
        .alert-success i {
            color: #198754;
        }
        
        .alert-error {
            background: #fff5f5;
            color: #842029;
            border-color: #f5c2c7;
        }
        
        .alert-error i {
            color: #dc3545;
        }
        
        /* Header Buttons */
        .btn-primary {
            background: #000;
            color: white;
            border: none;
            padding: 12px 24px;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.2s ease;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }
        
        .btn-primary:hover {
            background: #333;
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        }
        
        .header-actions .btn-outline {
            background: white;
            color: #666;
            border: 1px solid #e0e0e0;
            padding: 12px 20px;
            border-radius: 8px;
            text-decoration: none;
            font-size: 14px;
            font-weight: 500;
            transition: all 0.2s ease;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }
        
        .header-actions .btn-outline:hover {
            border-color: #333;
            color: #333;
            transform: translateY(-1px);
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="sidebar-header">
            <img src="../../images/LOGO.png" alt="Logo" class="sidebar-logo">
            <h3>Admin Panel</h3>
        </div>
        
        <nav class="sidebar-nav">
            <a href="../index.php" class="nav-item">
                <i class="fas fa-th-large"></i>
                <span>Overview</span>
            </a>
            <a href="products.php" class="nav-item">
                <i class="fas fa-box"></i>
                <span>Products</span>
            </a>
            <a href="orders.php" class="nav-item">
                <i class="fas fa-shopping-cart"></i>
                <span>Orders</span>
            </a>
            <a href="delivered.php" class="nav-item">
                <i class="fas fa-check-circle"></i>
                <span>Delivered</span>
            </a>
            <a href="customers.php" class="nav-item">
                <i class="fas fa-users"></i>
                <span>Customers</span>
            </a>
            <a href="feedback.php" class="nav-item">
                <i class="fas fa-comments"></i>
                <span>Feedback</span>
            </a>
            <a href="shipping.php" class="nav-item active">
                <i class="fas fa-shipping-fast"></i>
                <span>Shipping</span>
            </a>
            <a href="banners.php" class="nav-item">
                <i class="fas fa-image"></i>
                <span>Banners</span>
            </a>
            <a href="presets.php" class="nav-item">
                <i class="fas fa-cogs"></i>
                <span>Field Presets</span>
            </a>
            <a href="vouchers.php" class="nav-item">
                <i class="fas fa-ticket-alt"></i>
                <span>Voucher Management</span>
            </a>
            <a href="notifications.php" class="nav-item">
                <i class="fas fa-bell"></i>
                <span>Push Notifications</span>
            </a>
            <a href="analytics.php" class="nav-item">
                <i class="fas fa-chart-bar"></i>
                <span>Analytics</span>
            </a>
            <a href="settings.php" class="nav-item">
                <i class="fas fa-cog"></i>
                <span>Settings</span>
            </a>
        </nav>
        
        <div class="sidebar-footer">
            <div class="user-info">
                <i class="fas fa-user-circle"></i>
                <span><?php echo htmlspecialchars($adminAuth->getAdminName()); ?></span>
            </div>
            <a href="../logout.php" class="logout-btn">
                <i class="fas fa-sign-out-alt"></i>
                <span>Logout</span>
            </a>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <!-- Header -->
        <header class="header">
            <div class="header-left">
                <button class="sidebar-toggle">
                    <i class="fas fa-bars"></i>
                </button>
                <h1>Shipping Management</h1>
            </div>
            <div class="header-right">
                <div class="header-actions">
                    <button class="btn btn-primary" onclick="openAddModal()">
                        <i class="fas fa-plus"></i> Add Shipping Fee
                    </button>
                    <a href="../../index.php" target="_blank" class="btn btn-outline">
                        <i class="fas fa-external-link-alt"></i>
                        View Store
                    </a>
                </div>
            </div>
        </header>

        <!-- Content -->
        <div class="content">
            <!-- Messages -->
            <?php if ($message): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i> <?php echo $message; ?>
                </div>
            <?php endif; ?>

            <?php if ($error): ?>
                <div class="alert alert-error">
                    <i class="fas fa-exclamation-circle"></i> <?php echo $error; ?>
                </div>
            <?php endif; ?>

            <!-- Shipping Fees Table -->
            <div class="table-container">
                <table class="admin-table shipping-table">
                    <thead>
                        <tr>
                            <th width="20%">Name</th>
                            <th width="10%">Type</th>
                            <th width="15%">Pricing</th>
                            <th width="10%">Weight Range</th>
                            <th width="10%">Min Order</th>
                            <th width="10%">Delivery Days</th>
                            <th width="8%">Status</th>
                            <th width="7%">Sort</th>
                            <th width="10%">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($shippingFees)): ?>
                            <tr>
                                <td colspan="9" class="text-center">
                                    <div class="empty-state">
                                        <i class="fas fa-shipping-fast"></i>
                                        <h3>No shipping fees found</h3>
                                        <p>There are no shipping fees configured yet.</p>
                                    </div>
                                </td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($shippingFees as $fee): ?>
                                <tr data-shipping-id="<?php echo $fee['id'] ?? 0; ?>">
                                    <td>
                                        <div class="shipping-info">
                                            <strong><?php echo htmlspecialchars($fee['name'] ?? ''); ?></strong>
                                            <?php if (!empty($fee['description'])): ?>
                                                <small><?php echo htmlspecialchars($fee['description']); ?></small>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                    <td>
                                        <span class="fee-type-badge fee-type-<?php echo $fee['fee_type'] ?? 'weight_based'; ?>">
                                            <?php 
                                            $feeType = $fee['fee_type'] ?? 'weight_based';
                                            switch($feeType) {
                                                case 'weight_based': echo 'Weight Based'; break;
                                                case 'free_shipping_threshold': echo 'Free Threshold'; break;
                                                default: echo ucfirst($feeType);
                                            }
                                            ?>
                                        </span>
                                        <?php if (empty($fee['fee_type'])): ?>
                                            <small style="color: red;">(No type set)</small>
                                        <?php endif; ?>
                                    </td>
                                    <td class="text-center">
                                        <?php if (($fee['fee_type'] ?? 'weight_based') === 'free_shipping_threshold'): ?>
                                            <span class="free-shipping">FREE</span>
                                        <?php else: ?>
                                            <div class="weight-pricing">
                                                <div class="base-fee">Base: RM <?php echo number_format($fee['base_fee'] ?? 0, 2); ?></div>
                                                <div class="per-kg-fee">+ RM <?php echo number_format($fee['per_kg_fee'] ?? 0, 2); ?>/kg</div>
                                            </div>
                                        <?php endif; ?>
                                    </td>
                                    <td class="text-center">
                                        <?php if (($fee['fee_type'] ?? 'weight_based') === 'free_shipping_threshold'): ?>
                                            <span class="no-limit">No limit</span>
                                        <?php else: ?>
                                            <div class="weight-range">
                                                <?php if (($fee['min_weight'] ?? 0) > 0): ?>
                                                    <?php echo number_format($fee['min_weight'] ?? 0, 1); ?>kg
                                                <?php else: ?>
                                                    0kg
                                                <?php endif; ?>
                                                <?php if (isset($fee['max_weight']) && $fee['max_weight']): ?>
                                                    - <?php echo number_format($fee['max_weight'], 1); ?>kg
                                                <?php else: ?>
                                                    +
                                                <?php endif; ?>
                                            </div>
                                        <?php endif; ?>
                                    </td>
                                    <td class="text-center">
                                        <?php if (($fee['min_order_amount'] ?? 0) > 0): ?>
                                            RM <?php echo number_format($fee['min_order_amount'] ?? 0, 2); ?>
                                        <?php else: ?>
                                            <span class="no-minimum">No minimum</span>
                                        <?php endif; ?>
                                    </td>
                                    <td class="text-center">
                                        <div class="delivery-info">
                                            <?php if (($fee['delivery_days_min'] ?? 1) == ($fee['delivery_days_max'] ?? 3)): ?>
                                                <span><?php echo $fee['delivery_days_min'] ?? 1; ?> day<?php echo ($fee['delivery_days_min'] ?? 1) != 1 ? 's' : ''; ?></span>
                                            <?php else: ?>
                                                <span><?php echo $fee['delivery_days_min'] ?? 1; ?>-<?php echo $fee['delivery_days_max'] ?? 3; ?> days</span>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                    <td class="text-center">
                                        <span class="status-badge <?php echo ($fee['is_active'] ?? 1) ? 'active' : 'inactive'; ?>">
                                            <?php echo ($fee['is_active'] ?? 1) ? 'Active' : 'Inactive'; ?>
                                        </span>
                                    </td>
                                    <td class="text-center">
                                        <span class="sort-order"><?php echo $fee['sort_order'] ?? 0; ?></span>
                                    </td>
                                    <td class="text-center">
                                        <div class="action-buttons">
                                            <button class="btn btn-sm btn-outline" onclick="editShippingFee(<?php echo $fee['id'] ?? 0; ?>)" title="Edit">
                                                <i class="fas fa-edit"></i>
                                            </button>
                                            <button class="btn btn-sm btn-danger" onclick="deleteShippingFee(<?php echo $fee['id'] ?? 0; ?>)" title="Delete">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- Add/Edit Shipping Fee Modal -->
    <div id="shipping-modal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background-color: rgba(0, 0, 0, 0.5); z-index: 10000; align-items: center; justify-content: center;">
        <div style="background-color: white; border-radius: 8px; max-width: 600px; width: 90%; max-height: 90vh; overflow-y: auto; box-shadow: 0 10px 30px rgba(0, 0, 0, 0.3);">
            <div style="display: flex; justify-content: space-between; align-items: center; padding: 20px; border-bottom: 1px solid #eee;">
                <h3 id="modal-title" style="margin: 0; color: #000; font-size: 20px; font-weight: 600;">Add Shipping Fee</h3>
                <button onclick="closeShippingModal()" style="background: none; border: none; font-size: 24px; cursor: pointer; color: #666; padding: 5px;">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div style="padding: 30px;">
                <form id="shipping-form" method="POST">
                    <input type="hidden" name="id" id="shipping-id">
                    
                    <div class="form-group">
                        <label for="name">Name *</label>
                        <input type="text" id="name" name="name" class="form-control" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="description">Description</label>
                        <textarea id="description" name="description" class="form-control" rows="3"></textarea>
                    </div>
                    
                    <div class="form-group">
                        <label for="fee_type">Fee Type *</label>
                        <select id="fee_type" name="fee_type" class="form-control" required onchange="toggleFeeFields()">
                            <option value="weight_based">Weight Based (EZ Parcel)</option>
                            <option value="free_shipping_threshold">Free Shipping Threshold</option>
                        </select>
                    </div>
                    
                    <div class="form-group" id="weight-based-fields">
                        <div class="form-row">
                            <div class="form-group">
                                <label for="base_fee">Base Fee (RM) *</label>
                                <input type="number" id="base_fee" name="base_fee" class="form-control" step="0.01" min="0" placeholder="e.g., 5.00">
                            </div>
                            
                            <div class="form-group">
                                <label for="per_kg_fee">Per KG Fee (RM) *</label>
                                <input type="number" id="per_kg_fee" name="per_kg_fee" class="form-control" step="0.01" min="0" placeholder="e.g., 2.50">
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label for="min_weight">Min Weight (kg)</label>
                                <input type="number" id="min_weight" name="min_weight" class="form-control" step="0.1" min="0" placeholder="e.g., 0.0">
                            </div>
                            
                            <div class="form-group">
                                <label for="max_weight">Max Weight (kg)</label>
                                <input type="number" id="max_weight" name="max_weight" class="form-control" step="0.1" min="0" placeholder="e.g., 10.0 (leave empty for no limit)">
                            </div>
                        </div>
                    </div>
                    
                    <div class="form-group" id="free-shipping-fields" style="display: none;">
                        <label for="min_order_amount_free">Minimum Order Amount (RM) *</label>
                        <input type="number" id="min_order_amount_free" name="min_order_amount" class="form-control" step="0.01" min="0" placeholder="e.g., 150.00">
                    </div>
                    
                    <div class="form-row" id="order-amount-fields">
                        <div class="form-group">
                            <label for="min_order_amount">Minimum Order Amount</label>
                            <input type="number" id="min_order_amount" name="min_order_amount" class="form-control" step="0.01" min="0" value="0">
                        </div>
                        
                        <div class="form-group">
                            <label for="max_order_amount">Maximum Order Amount</label>
                            <input type="number" id="max_order_amount" name="max_order_amount" class="form-control" step="0.01" min="0">
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label for="delivery_days_min">Min Delivery Days *</label>
                            <input type="number" id="delivery_days_min" name="delivery_days_min" class="form-control" min="0" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="delivery_days_max">Max Delivery Days *</label>
                            <input type="number" id="delivery_days_max" name="delivery_days_max" class="form-control" min="0" required>
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label for="sort_order">Sort Order</label>
                            <input type="number" id="sort_order" name="sort_order" class="form-control" min="0" value="0">
                        </div>
                        
                        <div class="form-group">
                            <label class="checkbox-label">
                                <input type="checkbox" id="is_active" name="is_active" checked>
                                <span class="checkmark"></span>
                                Active
                            </label>
                        </div>
                    </div>
                    
                    <div style="display: flex; gap: 15px; justify-content: flex-end; margin-top: 30px; padding-top: 20px; border-top: 1px solid #eee;">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save"></i> Save
                        </button>
                        <button type="button" class="btn btn-outline" onclick="closeShippingModal()">
                            Cancel
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Delete Confirmation Modal -->
    <div id="delete-modal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background-color: rgba(0, 0, 0, 0.5); z-index: 10000; align-items: center; justify-content: center;">
        <div style="background-color: white; border-radius: 8px; max-width: 500px; width: 90%; box-shadow: 0 10px 30px rgba(0, 0, 0, 0.3);">
            <div style="display: flex; justify-content: space-between; align-items: center; padding: 20px; border-bottom: 1px solid #eee;">
                <h3 style="margin: 0; color: #000; font-size: 20px; font-weight: 600;">Confirm Delete</h3>
                <button onclick="closeDeleteModal()" style="background: none; border: none; font-size: 24px; cursor: pointer; color: #666; padding: 5px;">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div style="padding: 30px;">
                <p style="margin-bottom: 20px;">Are you sure you want to delete this shipping fee? This action cannot be undone.</p>
                <div style="display: flex; gap: 15px; justify-content: flex-end;">
                    <button class="btn btn-danger" onclick="confirmDelete()">
                        <i class="fas fa-trash"></i> Delete
                    </button>
                    <button class="btn btn-outline" onclick="closeDeleteModal()">
                        Cancel
                    </button>
                </div>
            </div>
        </div>
    </div>

    <script src="../assets/js/admin.js"></script>
    <script>
        let currentDeleteId = null;
        let isEditMode = false;
        
        // Define functions first

        function openAddModal() {
            const modal = document.getElementById('shipping-modal');
            if (!modal) {
                console.error('Modal element not found: shipping-modal');
                return;
            }
            
            isEditMode = false;
            document.getElementById('modal-title').textContent = 'Add Shipping Fee';
            document.getElementById('shipping-form').action = '?action=add';
            document.getElementById('shipping-form').reset();
            document.getElementById('shipping-id').value = '';
            document.getElementById('is_active').checked = true;
            toggleFeeFields();
            
            // Show modal
            modal.style.display = 'flex';
            document.body.style.overflow = 'hidden';
        }

        function editShippingFee(id) {
            // Fetch shipping fee data via AJAX
            fetch(`../ajax/shipping.php?action=get&id=${id}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        // Populate form with existing data
                        populateEditForm(data.data);
                        
                        // Set edit mode
                        isEditMode = true;
                        document.getElementById('modal-title').textContent = 'Edit Shipping Fee';
                        document.getElementById('shipping-form').action = '?action=edit';
                        document.getElementById('shipping-id').value = id;
                        
                        // Show modal
                        const modal = document.getElementById('shipping-modal');
                        if (modal) {
                            modal.style.display = 'flex';
                            document.body.style.overflow = 'hidden';
                        }
                    } else {
                        alert('Error loading shipping fee data: ' + (data.error || 'Unknown error'));
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Error loading shipping fee data');
                });
        }
        
        function populateEditForm(data) {
            // Populate form fields with existing data
            document.getElementById('name').value = data.name || '';
            document.getElementById('description').value = data.description || '';
            document.getElementById('fee_type').value = data.fee_type || 'weight_based';
            document.getElementById('base_fee').value = data.base_fee || '';
            document.getElementById('per_kg_fee').value = data.per_kg_fee || '';
            document.getElementById('min_weight').value = data.min_weight || '';
            document.getElementById('max_weight').value = data.max_weight || '';
            document.getElementById('min_order_amount').value = data.min_order_amount || '';
            document.getElementById('max_order_amount').value = data.max_order_amount || '';
            document.getElementById('delivery_days_min').value = data.delivery_days_min || '';
            document.getElementById('delivery_days_max').value = data.delivery_days_max || '';
            document.getElementById('sort_order').value = data.sort_order || '';
            document.getElementById('is_active').checked = data.is_active == 1;
            
            // Toggle fields based on fee type
            toggleFeeFields();
        }

        function toggleFeeFields() {
            const feeType = document.getElementById('fee_type').value;
            const weightBasedFields = document.getElementById('weight-based-fields');
            const freeShippingFields = document.getElementById('free-shipping-fields');
            const orderAmountFields = document.getElementById('order-amount-fields');
            
            if (feeType === 'free_shipping_threshold') {
                if (weightBasedFields) weightBasedFields.style.display = 'none';
                if (freeShippingFields) freeShippingFields.style.display = 'block';
                if (orderAmountFields) orderAmountFields.style.display = 'none';
            } else {
                if (weightBasedFields) weightBasedFields.style.display = 'block';
                if (freeShippingFields) freeShippingFields.style.display = 'none';
                if (orderAmountFields) orderAmountFields.style.display = 'block';
            }
        }

        // Initialize on page load
        document.addEventListener('DOMContentLoaded', function() {
            // Page loaded successfully
        });

        // Show notification function
        function showNotification(message, type = 'success') {
            const notification = document.createElement('div');
            notification.className = `alert alert-${type}`;
            notification.innerHTML = `<i class="fas fa-${type === 'success' ? 'check-circle' : 'exclamation-circle'}"></i> ${message}`;
            
            // Add to page
            const content = document.querySelector('.content');
            if (content) {
                content.insertBefore(notification, content.firstChild);
                
                // Auto remove after 5 seconds
                setTimeout(() => {
                    notification.style.opacity = '0';
                    setTimeout(() => {
                        if (notification.parentNode) {
                            notification.parentNode.removeChild(notification);
                        }
                    }, 300);
                }, 5000);
            }
        }

        // Edit shipping fee (simplified version)
        function editShippingFeeFull(id) {
            isEditMode = true;
            document.getElementById('modal-title').textContent = 'Edit Shipping Fee';
            document.getElementById('shipping-form').action = '?action=edit';
            document.getElementById('shipping-id').value = id;
            
            // Fetch shipping fee data and populate form
            fetch(`../ajax/shipping.php?action=get&id=${id}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        const fee = data.data;
                        document.getElementById('name').value = fee.name;
                        document.getElementById('description').value = fee.description || '';
                        document.getElementById('fee_type').value = fee.fee_type;
                        document.getElementById('base_fee').value = fee.base_fee || '';
                        document.getElementById('per_kg_fee').value = fee.per_kg_fee || '';
                        document.getElementById('min_weight').value = fee.min_weight || '';
                        document.getElementById('max_weight').value = fee.max_weight || '';
                        document.getElementById('min_order_amount').value = fee.min_order_amount;
                        document.getElementById('max_order_amount').value = fee.max_order_amount || '';
                        document.getElementById('delivery_days_min').value = fee.delivery_days_min;
                        document.getElementById('delivery_days_max').value = fee.delivery_days_max;
                        document.getElementById('sort_order').value = fee.sort_order;
                        document.getElementById('is_active').checked = fee.is_active == 1;
                        toggleFeeFields();
                        
                        const modal = document.getElementById('shipping-modal');
                        if (modal) {
                            modal.style.display = 'flex';
                            document.body.style.overflow = 'hidden';
                        }
                    } else {
                        showNotification(data.error || 'Failed to load shipping fee', 'error');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    showNotification('Error loading shipping fee', 'error');
                });
        }

        // Toggle fee fields based on fee type
        function toggleFeeFields() {
            const feeType = document.getElementById('fee_type').value;
            const weightBasedFields = document.getElementById('weight-based-fields');
            const freeShippingFields = document.getElementById('free-shipping-fields');
            const orderAmountFields = document.getElementById('order-amount-fields');
            
            if (feeType === 'free_shipping_threshold') {
                weightBasedFields.style.display = 'none';
                freeShippingFields.style.display = 'block';
                orderAmountFields.style.display = 'none';
                
                // Clear weight-based fields when switching to free shipping
                document.getElementById('base_fee').value = '0';
                document.getElementById('per_kg_fee').value = '0';
            } else {
                weightBasedFields.style.display = 'block';
                freeShippingFields.style.display = 'none';
                orderAmountFields.style.display = 'block';
                
                // Clear free shipping field when switching to weight-based
                document.getElementById('min_order_amount_free').value = '';
            }
        }

        // Close shipping modal
        function closeShippingModal() {
            const modal = document.getElementById('shipping-modal');
            if (modal) {
                modal.style.display = 'none';
                document.body.style.overflow = 'auto';
            }
        }

        // Delete shipping fee
        function deleteShippingFee(id) {
            currentDeleteId = id;
            const modal = document.getElementById('delete-modal');
            if (modal) {
                modal.style.display = 'flex';
                document.body.style.overflow = 'hidden';
            }
        }

        // Close delete modal
        function closeDeleteModal() {
            const modal = document.getElementById('delete-modal');
            if (modal) {
                modal.style.display = 'none';
                document.body.style.overflow = 'auto';
            }
            currentDeleteId = null;
        }

        // Confirm delete
        function confirmDelete() {
            if (!currentDeleteId) return;
            
            const form = document.createElement('form');
            form.method = 'POST';
            form.action = '?action=delete';
            
            const idInput = document.createElement('input');
            idInput.type = 'hidden';
            idInput.name = 'id';
            idInput.value = currentDeleteId;
            
            form.appendChild(idInput);
            document.body.appendChild(form);
            form.submit();
        }

        // Handle form submission
        document.getElementById('shipping-form').addEventListener('submit', function(e) {
            const feeType = document.getElementById('fee_type').value;
            
            // For free shipping threshold, copy the value from the free shipping field to the regular field
            if (feeType === 'free_shipping_threshold') {
                const freeShippingAmount = document.getElementById('min_order_amount_free').value;
                document.getElementById('min_order_amount').value = freeShippingAmount;
            }
        });

        // Close modals when clicking outside
        document.addEventListener('click', function(e) {
            if (e.target.id === 'shipping-modal' || e.target.id === 'delete-modal') {
                closeShippingModal();
                closeDeleteModal();
            }
        });
    </script>
</body>
</html>