<?php
require_once '../includes/auth.php';
require_once '../includes/functions.php';
require_once '../includes/Voucher.php';

$adminAuth->requireLogin();

$voucherModel = new Voucher();

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'create_voucher':
                // Handle image upload
                $imagePath = $_POST['voucher_image'] ?? ''; // Default to URL if provided
                
                if (isset($_FILES['voucher_image_file']) && $_FILES['voucher_image_file']['error'] === UPLOAD_ERR_OK) {
                    $uploadDir = __DIR__ . '/../../images/vouchers/';
                    $allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
                    $maxSize = 5 * 1024 * 1024; // 5MB
                    
                    // Create vouchers directory if it doesn't exist
                    if (!is_dir($uploadDir)) {
                        mkdir($uploadDir, 0755, true);
                    }
                    
                    $file = $_FILES['voucher_image_file'];
                    
                    if (in_array($file['type'], $allowedTypes) && $file['size'] <= $maxSize) {
                        $fileName = 'voucher_' . time() . '_' . preg_replace('/[^a-zA-Z0-9._-]/', '', $file['name']);
                        $filePath = $uploadDir . $fileName;
                        
                        if (move_uploaded_file($file['tmp_name'], $filePath)) {
                            $imagePath = 'images/vouchers/' . $fileName;
                        } else {
                            $error_message = 'Error uploading image. Please check directory permissions.';
                        }
                    } else {
                        $error_message = 'Invalid image type or file too large. Only JPG, PNG, GIF, and WebP files up to 5MB are allowed.';
                    }
                }
                
                $data = [
                    'voucher_name' => $_POST['voucher_name'],
                    'voucher_description' => $_POST['voucher_description'],
                    'voucher_image' => $imagePath,
                    'points_required' => (int)$_POST['points_required'],
                    'discount_type' => $_POST['discount_type'],
                    'discount_value' => (float)$_POST['discount_value'],
                    'max_redemptions' => !empty($_POST['max_redemptions']) ? (int)$_POST['max_redemptions'] : null,
                    'valid_from' => $_POST['valid_from'],
                    'valid_until' => $_POST['valid_until'],
                    'is_active' => isset($_POST['is_active']),
                    'created_by' => $_SESSION['admin_id'] ?? 1
                ];
                
                if ($voucherModel->createVoucher($data)) {
                    $success_message = "Voucher created successfully!";
                } else {
                    $error_message = "Failed to create voucher.";
                }
                break;
                
            case 'update_voucher':
                $id = (int)$_POST['voucher_id'];
                
                // Handle image upload for update
                $imagePath = $_POST['voucher_image'] ?? null;
                
                if (isset($_FILES['voucher_image_file']) && $_FILES['voucher_image_file']['error'] === UPLOAD_ERR_OK) {
                    $uploadDir = __DIR__ . '/../../images/vouchers/';
                    $allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
                    $maxSize = 5 * 1024 * 1024; // 5MB
                    
                    // Create vouchers directory if it doesn't exist
                    if (!is_dir($uploadDir)) {
                        mkdir($uploadDir, 0755, true);
                    }
                    
                    $file = $_FILES['voucher_image_file'];
                    
                    if (in_array($file['type'], $allowedTypes) && $file['size'] <= $maxSize) {
                        $fileName = 'voucher_' . time() . '_' . preg_replace('/[^a-zA-Z0-9._-]/', '', $file['name']);
                        $filePath = $uploadDir . $fileName;
                        
                        if (move_uploaded_file($file['tmp_name'], $filePath)) {
                            $imagePath = 'images/vouchers/' . $fileName;
                        } else {
                            $error_message = 'Error uploading image. Please check directory permissions.';
                        }
                    } else {
                        $error_message = 'Invalid image type or file too large. Only JPG, PNG, GIF, and WebP files up to 5MB are allowed.';
                    }
                }
                
                $data = [
                    'voucher_name' => $_POST['voucher_name'],
                    'voucher_description' => $_POST['voucher_description'],
                    'voucher_image' => $imagePath,
                    'points_required' => (int)$_POST['points_required'],
                    'discount_type' => $_POST['discount_type'],
                    'discount_value' => (float)$_POST['discount_value'],
                    'max_redemptions' => !empty($_POST['max_redemptions']) ? (int)$_POST['max_redemptions'] : null,
                    'valid_from' => $_POST['valid_from'],
                    'valid_until' => $_POST['valid_until'],
                    'is_active' => isset($_POST['is_active'])
                ];
                
                if ($voucherModel->updateVoucher($id, $data)) {
                    $success_message = "Voucher updated successfully!";
                } else {
                    $error_message = "Failed to update voucher.";
                }
                break;
                
            case 'delete_voucher':
                $id = (int)$_POST['voucher_id'];
                if ($voucherModel->deleteVoucher($id)) {
                    $success_message = "Voucher deleted successfully!";
                } else {
                    $error_message = "Failed to delete voucher.";
                }
                break;
                
            case 'create_tier':
                $data = [
                    'tier_name' => $_POST['tier_name'],
                    'min_points' => (int)$_POST['min_points'],
                    'max_points' => !empty($_POST['max_points']) ? (int)$_POST['max_points'] : null,
                    'tier_color' => $_POST['tier_color'],
                    'tier_icon' => $_POST['tier_icon'],
                    'benefits' => $_POST['benefits']
                ];
                
                if ($voucherModel->addRoyalTier($data)) {
                    $success_message = "Royal tier created successfully!";
                } else {
                    $error_message = "Failed to create royal tier.";
                }
                break;
                
            case 'update_tier':
                $id = (int)$_POST['tier_id'];
                $data = [
                    'tier_name' => $_POST['tier_name'],
                    'min_points' => (int)$_POST['min_points'],
                    'max_points' => !empty($_POST['max_points']) ? (int)$_POST['max_points'] : null,
                    'tier_color' => $_POST['tier_color'],
                    'tier_icon' => $_POST['tier_icon'],
                    'benefits' => $_POST['benefits']
                ];
                
                if ($voucherModel->updateRoyalTier($id, $data)) {
                    $success_message = "Royal tier updated successfully!";
                } else {
                    $error_message = "Failed to update royal tier.";
                }
                break;
                
            case 'delete_tier':
                $id = (int)$_POST['tier_id'];
                if ($voucherModel->deleteRoyalTier($id)) {
                    $success_message = "Royal tier deleted successfully!";
                } else {
                    $error_message = "Failed to delete royal tier.";
                }
                break;
        }
    }
}

// Get vouchers and stats
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 10;
$offset = ($page - 1) * $limit;
$search = $_GET['search'] ?? '';

$vouchers = $voucherModel->getAllVouchers($limit, $offset, $search);
$stats = $voucherModel->getVoucherStats();
$royalTiers = $voucherModel->getAllRoyalTiers();

// Get voucher for editing
$editVoucher = null;
if (isset($_GET['edit'])) {
    $editVoucher = $voucherModel->getVoucherById((int)$_GET['edit']);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Voucher Management - Admin Panel</title>
    <link rel="stylesheet" href="../assets/css/admin.css?v=<?php echo time(); ?>">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body>
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="sidebar-header">
            <img src="../../images/LOGO.png" alt="Logo" class="sidebar-logo">
            <h3>Admin Panel</h3>
        </div>
        
        <nav class="sidebar-nav">
            <a href="../index.php" class="nav-item">
                <i class="fas fa-th-large"></i>
                <span>Overview</span>
            </a>
            <a href="products.php" class="nav-item">
                <i class="fas fa-box"></i>
                <span>Products</span>
            </a>
            <a href="orders.php" class="nav-item">
                <i class="fas fa-shopping-cart"></i>
                <span>Orders</span>
            </a>
            <a href="delivered.php" class="nav-item">
                <i class="fas fa-check-circle"></i>
                <span>Delivered</span>
            </a>
            <a href="customers.php" class="nav-item">
                <i class="fas fa-users"></i>
                <span>Customers</span>
            </a>
            <a href="feedback.php" class="nav-item">
                <i class="fas fa-comments"></i>
                <span>Feedback</span>
            </a>
            <a href="shipping.php" class="nav-item">
                <i class="fas fa-shipping-fast"></i>
                <span>Shipping</span>
            </a>
            <a href="banners.php" class="nav-item">
                <i class="fas fa-image"></i>
                <span>Banners</span>
            </a>
            <a href="presets.php" class="nav-item">
                <i class="fas fa-cogs"></i>
                <span>Field Presets</span>
            </a>
            <a href="vouchers.php" class="nav-item active">
                <i class="fas fa-ticket-alt"></i>
                <span>Voucher Management</span>
            </a>
            <a href="notifications.php" class="nav-item">
                <i class="fas fa-bell"></i>
                <span>Push Notifications</span>
            </a>
            <a href="analytics.php" class="nav-item">
                <i class="fas fa-chart-bar"></i>
                <span>Analytics</span>
            </a>
            <a href="settings.php" class="nav-item">
                <i class="fas fa-cog"></i>
                <span>Settings</span>
            </a>
        </nav>
        
        <div class="sidebar-footer">
            <div class="user-info">
                <i class="fas fa-user-circle"></i>
                <span><?php echo htmlspecialchars($adminAuth->getAdminName()); ?></span>
            </div>
            <a href="../logout.php" class="logout-btn">
                <i class="fas fa-sign-out-alt"></i>
                <span>Logout</span>
            </a>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <!-- Header -->
        <header class="header">
            <div class="header-left">
                <button class="sidebar-toggle">
                    <i class="fas fa-bars"></i>
                </button>
                <h1>Voucher Management</h1>
            </div>
            <div class="header-right">
                <div class="header-actions">
                    <button onclick="openAddVoucherModal()" class="btn btn-primary">
                        <i class="fas fa-plus"></i>
                        Add Voucher
                    </button>
                    <a href="../index.php" target="_blank" class="btn btn-outline">
                        <i class="fas fa-external-link-alt"></i>
                        View Store
                    </a>
                </div>
            </div>
        </header>

        <!-- Content -->
        <div class="content">
            <?php if (isset($success_message)): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i> <?php echo $success_message; ?>
                </div>
            <?php endif; ?>

            <?php if (isset($error_message)): ?>
                <div class="alert alert-error">
                    <i class="fas fa-exclamation-circle"></i> <?php echo $error_message; ?>
                </div>
            <?php endif; ?>

            <!-- Statistics Cards -->
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-ticket-alt"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?php echo $stats['total_vouchers']; ?></h3>
                        <p>Total Vouchers</p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-check-circle"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?php echo $stats['active_vouchers']; ?></h3>
                        <p>Active Vouchers</p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-gift"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?php echo $stats['total_redemptions']; ?></h3>
                        <p>Total Redemptions</p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-star"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?php echo number_format($stats['avg_points_required'], 0); ?></h3>
                        <p>Avg Points Required</p>
                    </div>
                </div>
            </div>

            <!-- Search and Filter -->
            <div class="content-header">
                <div class="search-box">
                    <i class="fas fa-search"></i>
                    <input type="text" id="searchInput" placeholder="Search vouchers..." value="<?php echo htmlspecialchars($search); ?>">
                </div>
                <div class="filter-actions">
                    <button class="btn btn-outline" onclick="searchVouchers()">
                        <i class="fas fa-search"></i>
                        Search
                    </button>
                </div>
            </div>

            <!-- Vouchers Table -->
            <div class="table-container">
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Voucher</th>
                            <th>Points Required</th>
                            <th>Discount</th>
                            <th>Redemptions</th>
                            <th>Valid Until</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($vouchers as $voucher): ?>
                            <tr>
                                <td>
                                    <div class="voucher-info">
                                        <?php if ($voucher['voucher_image']): ?>
                                            <img src="<?php echo htmlspecialchars($voucher['voucher_image']); ?>" alt="Voucher" class="voucher-thumb">
                                        <?php endif; ?>
                                        <div>
                                            <h4><?php echo htmlspecialchars($voucher['voucher_name']); ?></h4>
                                            <p><?php echo htmlspecialchars($voucher['voucher_description']); ?></p>
                                        </div>
                                    </div>
                                </td>
                                <td>
                                    <span class="points-badge"><?php echo number_format($voucher['points_required']); ?> pts</span>
                                </td>
                                <td>
                                    <span class="discount-badge">
                                        <?php 
                                        if ($voucher['discount_type'] === 'percentage') {
                                            echo $voucher['discount_value'] . '%';
                                        } elseif ($voucher['discount_type'] === 'fixed_amount') {
                                            echo 'RM ' . $voucher['discount_value'];
                                        } else {
                                            echo ucfirst(str_replace('_', ' ', $voucher['discount_type']));
                                        }
                                        ?>
                                    </span>
                                </td>
                                <td>
                                    <?php echo $voucher['current_redemptions']; ?>
                                    <?php if ($voucher['max_redemptions']): ?>
                                        / <?php echo $voucher['max_redemptions']; ?>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo date('M j, Y', strtotime($voucher['valid_until'])); ?></td>
                                <td>
                                    <span class="status-badge <?php echo $voucher['is_active'] ? 'active' : 'inactive'; ?>">
                                        <?php echo $voucher['is_active'] ? 'Active' : 'Inactive'; ?>
                                    </span>
                                </td>
                                <td>
                                    <div class="action-buttons">
                                        <button onclick="editVoucher(<?php echo $voucher['id']; ?>)" class="btn btn-sm btn-outline">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <form method="POST" style="display: inline;" onsubmit="return confirm('Are you sure you want to delete this voucher?')">
                                            <input type="hidden" name="action" value="delete_voucher">
                                            <input type="hidden" name="voucher_id" value="<?php echo $voucher['id']; ?>">
                                            <button type="submit" class="btn btn-sm btn-danger">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </form>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <!-- Royal Tiers Section -->
            <div class="section">
                <div class="section-header">
                    <h2>Royal Tiers</h2>
                    <p>Manage customer loyalty tiers</p>
                    <button onclick="openTierModal()" class="btn btn-primary" style="margin-top: 15px;">
                        <i class="fas fa-plus"></i>
                        Add New Tier
                    </button>
                </div>
                <div class="tiers-grid">
                    <?php 
                    $tierIndex = 0;
                    foreach ($royalTiers as $tier): 
                        $tierIndex++;
                    ?>
                        <div class="tier-card tier-<?php echo strtolower($tier['tier_name']); ?>">
                            <div class="tier-icon">
                                <i class="<?php echo $tier['tier_icon']; ?>"></i>
                            </div>
                            <div class="tier-info">
                                <h3><?php echo $tier['tier_name']; ?></h3>
                                <p class="tier-points">
                                    <?php echo number_format($tier['min_points']); ?> - 
                                    <?php echo $tier['max_points'] ? number_format($tier['max_points']) : '∞'; ?> points
                                </p>
                                <p class="tier-benefits"><?php echo $tier['benefits']; ?></p>
                            </div>
                            <div class="tier-actions">
                                <button onclick="editTier(<?php echo $tier['id']; ?>)" class="btn btn-sm btn-outline">
                                    <i class="fas fa-edit"></i>
                                </button>
                                <button onclick="deleteTier(<?php echo $tier['id']; ?>)" class="btn btn-sm btn-danger">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
    </div>

    <!-- Add/Edit Voucher Modal -->
    <div id="voucherModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2 id="modalTitle">Add New Voucher</h2>
                <button class="modal-close" onclick="closeVoucherModal()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <form id="voucherForm" method="POST" enctype="multipart/form-data">
                <input type="hidden" name="action" value="create_voucher">
                <input type="hidden" name="voucher_id" id="voucherId">
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="voucher_name">Voucher Name</label>
                        <input type="text" id="voucher_name" name="voucher_name" required>
                    </div>
                    <div class="form-group">
                        <label for="points_required">Points Required</label>
                        <input type="number" id="points_required" name="points_required" min="1" required>
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="voucher_description">Description</label>
                    <textarea id="voucher_description" name="voucher_description" rows="3" required></textarea>
                </div>
                
                <div class="form-group">
                    <label for="voucher_image">Voucher Image</label>
                    <div class="image-upload-section">
                        <div class="image-upload-options">
                            <div class="upload-option">
                                <label class="upload-label">
                                    <input type="radio" name="image_source" value="file" checked>
                                    <span>Upload File</span>
                                </label>
                                <input type="file" id="voucher_image_file" name="voucher_image_file" accept="image/*" style="display: none;">
                                <button type="button" class="btn btn-outline" onclick="document.getElementById('voucher_image_file').click()">
                                    <i class="fas fa-upload"></i>
                                    Choose File
                                </button>
                            </div>
                            <div class="upload-option">
                                <label class="upload-label">
                                    <input type="radio" name="image_source" value="url">
                                    <span>Image URL</span>
                                </label>
                                <input type="url" id="voucher_image" name="voucher_image" placeholder="https://example.com/image.jpg" style="display: none;">
                            </div>
                        </div>
                        <div class="image-preview" id="imagePreview" style="display: none;">
                            <img id="previewImg" src="" alt="Preview">
                            <button type="button" class="remove-image" onclick="removeImage()">
                                <i class="fas fa-times"></i>
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="discount_type">Discount Type</label>
                        <select id="discount_type" name="discount_type" required>
                            <option value="percentage">Percentage</option>
                            <option value="fixed_amount">Fixed Amount</option>
                            <option value="free_shipping">Free Shipping</option>
                            <option value="free_product">Free Product</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="discount_value">Discount Value</label>
                        <input type="number" id="discount_value" name="discount_value" step="0.01" min="0" required>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="max_redemptions">Max Redemptions (optional)</label>
                        <input type="number" id="max_redemptions" name="max_redemptions" min="1">
                    </div>
                    <div class="form-group">
                        <label>
                            <input type="checkbox" id="is_active" name="is_active" checked>
                            Active
                        </label>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="valid_from">Valid From</label>
                        <input type="date" id="valid_from" name="valid_from" required>
                    </div>
                    <div class="form-group">
                        <label for="valid_until">Valid Until</label>
                        <input type="date" id="valid_until" name="valid_until" required>
                    </div>
                </div>
                
                <div class="modal-actions">
                    <button type="button" class="btn btn-outline" onclick="closeVoucherModal()">Cancel</button>
                    <button type="submit" class="btn btn-primary">Save Voucher</button>
                </div>
            </form>
        </div>
    </div>

    <!-- Add/Edit Tier Modal -->
    <div id="tierModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2 id="tierModalTitle">Add New Royal Tier</h2>
                <button class="modal-close" onclick="closeTierModal()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <form id="tierForm" method="POST">
                <input type="hidden" name="action" value="create_tier">
                <input type="hidden" name="tier_id" id="tierId">
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="tier_name">Tier Name</label>
                        <input type="text" id="tier_name" name="tier_name" required placeholder="e.g., Bronze, Silver, Gold">
                    </div>
                    <div class="form-group">
                        <label for="tier_color">Tier Color</label>
                        <input type="color" id="tier_color" name="tier_color" value="#CD7F32" required>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="min_points">Minimum Points</label>
                        <input type="number" id="min_points" name="min_points" min="0" required>
                    </div>
                    <div class="form-group">
                        <label for="max_points">Maximum Points (leave empty for unlimited)</label>
                        <input type="number" id="max_points" name="max_points" min="1">
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="tier_icon">Tier Icon</label>
                    <select id="tier_icon" name="tier_icon" required>
                        <option value="fas fa-medal">Medal</option>
                        <option value="fas fa-award">Award</option>
                        <option value="fas fa-trophy">Trophy</option>
                        <option value="fas fa-crown">Crown</option>
                        <option value="fas fa-gem">Gem</option>
                        <option value="fas fa-star">Star</option>
                        <option value="fas fa-diamond">Diamond</option>
                        <option value="fas fa-fire">Fire</option>
                        <option value="fas fa-bolt">Lightning</option>
                        <option value="fas fa-rocket">Rocket</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="benefits">Benefits Description</label>
                    <textarea id="benefits" name="benefits" rows="3" required placeholder="Describe the benefits for this tier..."></textarea>
                </div>
                
                <div class="modal-actions">
                    <button type="button" class="btn btn-outline" onclick="closeTierModal()">Cancel</button>
                    <button type="submit" class="btn btn-primary">Save Tier</button>
                </div>
            </form>
        </div>
    </div>

    <script>
        // Modal functions
        function openAddVoucherModal() {
            document.getElementById('modalTitle').textContent = 'Add New Voucher';
            document.querySelector('input[name="action"]').value = 'create_voucher';
            document.getElementById('voucherForm').reset();
            document.getElementById('voucherId').value = '';
            document.getElementById('valid_from').value = new Date().toISOString().split('T')[0];
            document.getElementById('valid_until').value = new Date(Date.now() + 365 * 24 * 60 * 60 * 1000).toISOString().split('T')[0];
            document.getElementById('voucherModal').style.display = 'flex';
        }

        function editVoucher(voucherId) {
            // This would typically fetch voucher data via AJAX
            // For now, redirect to edit mode
            window.location.href = 'vouchers.php?edit=' + voucherId;
        }

        function closeVoucherModal() {
            document.getElementById('voucherModal').style.display = 'none';
        }

        function searchVouchers() {
            const searchTerm = document.getElementById('searchInput').value;
            window.location.href = 'vouchers.php?search=' + encodeURIComponent(searchTerm);
        }

        // Handle edit mode
        <?php if ($editVoucher): ?>
        document.addEventListener('DOMContentLoaded', function() {
            document.getElementById('modalTitle').textContent = 'Edit Voucher';
            document.querySelector('input[name="action"]').value = 'update_voucher';
            document.getElementById('voucherId').value = '<?php echo $editVoucher['id']; ?>';
            document.getElementById('voucher_name').value = '<?php echo htmlspecialchars($editVoucher['voucher_name']); ?>';
            document.getElementById('voucher_description').value = '<?php echo htmlspecialchars($editVoucher['voucher_description']); ?>';
            document.getElementById('voucher_image').value = '<?php echo htmlspecialchars($editVoucher['voucher_image'] ?? ''); ?>';
            document.getElementById('points_required').value = '<?php echo $editVoucher['points_required']; ?>';
            document.getElementById('discount_type').value = '<?php echo $editVoucher['discount_type']; ?>';
            document.getElementById('discount_value').value = '<?php echo $editVoucher['discount_value']; ?>';
            document.getElementById('max_redemptions').value = '<?php echo $editVoucher['max_redemptions'] ?? ''; ?>';
            document.getElementById('valid_from').value = '<?php echo $editVoucher['valid_from']; ?>';
            document.getElementById('valid_until').value = '<?php echo $editVoucher['valid_until']; ?>';
            document.getElementById('is_active').checked = <?php echo $editVoucher['is_active'] ? 'true' : 'false'; ?>;
            document.getElementById('voucherModal').style.display = 'flex';
        });
        <?php endif; ?>

        // Search on Enter key
        document.getElementById('searchInput').addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                searchVouchers();
            }
        });

        // Image upload handling
        document.getElementById('voucher_image_file').addEventListener('change', function(e) {
            const file = e.target.files[0];
            if (file) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    document.getElementById('previewImg').src = e.target.result;
                    document.getElementById('imagePreview').style.display = 'block';
                };
                reader.readAsDataURL(file);
            }
        });

        // Image source radio button handling
        document.querySelectorAll('input[name="image_source"]').forEach(radio => {
            radio.addEventListener('change', function() {
                const fileInput = document.getElementById('voucher_image_file');
                const urlInput = document.getElementById('voucher_image');
                const preview = document.getElementById('imagePreview');
                
                if (this.value === 'file') {
                    fileInput.style.display = 'none';
                    urlInput.style.display = 'none';
                    preview.style.display = 'none';
                } else {
                    fileInput.style.display = 'none';
                    urlInput.style.display = 'block';
                    preview.style.display = 'none';
                }
            });
        });

        // URL input handling
        document.getElementById('voucher_image').addEventListener('input', function(e) {
            const url = e.target.value;
            if (url) {
                document.getElementById('previewImg').src = url;
                document.getElementById('imagePreview').style.display = 'block';
            } else {
                document.getElementById('imagePreview').style.display = 'none';
            }
        });

        function removeImage() {
            document.getElementById('voucher_image_file').value = '';
            document.getElementById('voucher_image').value = '';
            document.getElementById('imagePreview').style.display = 'none';
        }

        // Close modal when clicking outside
        window.addEventListener('click', function(e) {
            const modal = document.getElementById('voucherModal');
            if (e.target === modal) {
                closeVoucherModal();
            }
        });

        // Tier Management Functions
        function openTierModal() {
            document.getElementById('tierModalTitle').textContent = 'Add New Royal Tier';
            document.querySelector('#tierForm input[name="action"]').value = 'create_tier';
            document.getElementById('tierForm').reset();
            document.getElementById('tierId').value = '';
            document.getElementById('tier_color').value = '#CD7F32';
            document.getElementById('tierModal').style.display = 'flex';
        }

        function editTier(tierId) {
            // This would typically fetch tier data via AJAX
            // For now, redirect to edit mode
            window.location.href = 'vouchers.php?edit_tier=' + tierId;
        }

        function deleteTier(tierId) {
            if (confirm('Are you sure you want to delete this royal tier? This action cannot be undone.')) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.innerHTML = `
                    <input type="hidden" name="action" value="delete_tier">
                    <input type="hidden" name="tier_id" value="${tierId}">
                `;
                document.body.appendChild(form);
                form.submit();
            }
        }

        function closeTierModal() {
            document.getElementById('tierModal').style.display = 'none';
        }

        // Handle edit tier mode
        <?php if (isset($_GET['edit_tier'])): ?>
        document.addEventListener('DOMContentLoaded', function() {
            // This would typically fetch tier data via AJAX
            // For now, show the modal with edit mode
            document.getElementById('tierModalTitle').textContent = 'Edit Royal Tier';
            document.querySelector('#tierForm input[name="action"]').value = 'update_tier';
            document.getElementById('tierId').value = '<?php echo (int)$_GET['edit_tier']; ?>';
            document.getElementById('tierModal').style.display = 'flex';
        });
        <?php endif; ?>

        // Close tier modal when clicking outside
        window.addEventListener('click', function(e) {
            const modal = document.getElementById('tierModal');
            if (e.target === modal) {
                closeTierModal();
            }
        });
    </script>
</body>
</html>