# Admin Order Page - Database Structure

This document explains what database tables and columns the admin order page fetches and displays.

---

## Main Query Location

**File:** `apps/admin/includes/functions.php`  
**Function:** `getAllOrders()`  
**Line:** 472-529

---

## Database Tables Used

### 1. **`orders` Table** (Main Table)
This is the primary table that stores all order information.

#### Columns Fetched (All columns with `o.*`):

| Column Name | Data Type | Description |
|------------|-----------|-------------|
| `id` | INT | Primary key, unique order ID |
| `user_id` | INT | Foreign key to users table (can be NULL for guest orders) |
| `order_number` | VARCHAR(50) | Unique order number (e.g., ORD20241117001) |
| `total_amount` | DECIMAL(10,2) | Total order amount |
| `currency` | VARCHAR(3) | Currency code (default: 'RM') |
| `status` | ENUM | Order status: 'pending', 'awaiting_payment', 'processing', 'shipped', 'delivered', 'cancelled' |
| `shipping_address` | TEXT | JSON encoded shipping address details |
| `billing_address` | TEXT | JSON encoded billing address details |
| `payment_method` | VARCHAR(50) | Payment method used (e.g., 'toyyibpay_fpx', 'cod') |
| `payment_status` | ENUM | Payment status: 'pending', 'paid', 'failed', 'refunded' |
| `notes` | TEXT | Customer notes/remarks |
| `voucher_id` | INT | ID of voucher used (if any) |
| `voucher_discount_amount` | DECIMAL(10,2) | Discount amount from voucher |
| `voucher_discount_type` | VARCHAR(50) | Type of discount (percentage, fixed_amount, free_shipping) |
| `voucher_discount_value` | DECIMAL(10,2) | Voucher discount value |
| `shipping_method_id` | INT | Foreign key to shipping_fees table |
| `created_at` | TIMESTAMP | When order was created |
| `updated_at` | TIMESTAMP | When order was last updated |

---

### 2. **`shipping_fees` Table** (Joined Table)
This table is LEFT JOINed to get shipping method details.

**Join Condition:** `o.shipping_method_id = sf.id`

#### Columns Fetched:

| Column Name | Alias | Data Type | Description |
|------------|-------|-----------|-------------|
| `fee_type` | `shipping_fee_type` | VARCHAR(50) | Type: 'weight_based', 'free_shipping_threshold', 'pickup' |
| `name` | `shipping_method_name` | VARCHAR(191) | Shipping method name (e.g., 'Standard Shipping', 'Self Pickup') |

---

### 3. **`order_items` Table** (Related Table)
This table stores individual items in each order. It's queried separately when viewing order details.

#### Columns:

| Column Name | Data Type | Description |
|------------|-----------|-------------|
| `id` | INT | Primary key |
| `order_id` | INT | Foreign key to orders table |
| `product_id` | INT | Foreign key to products table |
| `product_name` | VARCHAR(191) | Product name at time of order |
| `product_price` | DECIMAL(10,2) | Product price at time of order |
| `quantity` | INT | Quantity ordered |
| `size` | VARCHAR(50) | Product size/variant |
| `version` | VARCHAR(50) | Product version (e.g., EDP, EDT) |
| `total_price` | DECIMAL(10,2) | Total price for this line item |
| `created_at` | TIMESTAMP | When item was added |

---

## SQL Query Structure

### Main Query:
```sql
SELECT o.*, 
       sf.fee_type as shipping_fee_type, 
       sf.name as shipping_method_name
FROM orders o 
LEFT JOIN shipping_fees sf ON o.shipping_method_id = sf.id
WHERE 1=1
  [AND search conditions]
  [AND status filter]
  [AND date filter]
  [AND delivery method filter]
ORDER BY o.created_at DESC
LIMIT :limit OFFSET :offset
```

---

## Filters Applied

### 1. **Search Filter**
- Searches in: `order_number` and `shipping_address`
- Type: LIKE search with wildcards
- Example: `%search_term%`

### 2. **Status Filter**
- Filters by: `o.status`
- Values: 'pending', 'awaiting_payment', 'processing', 'shipped', 'delivered', 'cancelled'

### 3. **Date Filter**
- Filters by: `DATE(o.created_at)`
- Format: YYYY-MM-DD

### 4. **Delivery Method Filter**
- **Pickup**: `sf.fee_type = 'pickup'`
- **Shipping**: `sf.fee_type != 'pickup' OR sf.fee_type IS NULL`

---

## Data Displayed in Admin Order Table

The admin order page (`apps/admin/pages/orders.php`) displays the following columns:

| Display Column | Data Source | Processing |
|---------------|-------------|------------|
| **Order ID** | `order_number` | Direct display |
| **Customer** | `shipping_address` | JSON decoded, extracts `first_name` + `last_name` |
| **Email** | `shipping_address` | JSON decoded, extracts `email` |
| **Total** | `total_amount` | Formatted as currency |
| **Delivery Method** | `shipping_method_name` | From joined `shipping_fees` table |
| **Status** | `status` | Displayed with color-coded badge |
| **Payment** | `payment_method` | Direct display |
| **Date** | `created_at` | Formatted as "Mon DD, YYYY" |
| **Actions** | - | View and Edit buttons |

---

## Shipping Address JSON Structure

The `shipping_address` column stores JSON data with this structure:

```json
{
  "first_name": "John",
  "last_name": "Doe",
  "email": "john@example.com",
  "phone": "0123456789",
  "address": "123 Main Street",
  "city": "Kuala Lumpur",
  "state": "Selangor",
  "zip_code": "50000",
  "country": "Malaysia"
}
```

This is decoded in PHP using:
```php
$shipping_address = json_decode($order['shipping_address'], true);
```

---

## Related Queries

### Count Query (for Pagination)
**Function:** `getOrdersCount()`

```sql
SELECT COUNT(*) as total 
FROM orders o 
LEFT JOIN shipping_fees sf ON o.shipping_method_id = sf.id
WHERE 1=1
  [AND same filters as main query]
```

### Order Details Query
**Function:** `getOrderById()`

```sql
SELECT * FROM orders WHERE id = :id
```

### Order Items Query
**Model:** `Order.php`

```sql
SELECT * FROM order_items WHERE order_id = :order_id
```

---

## Additional Tables Referenced

### 4. **`users` Table**
- Referenced by: `orders.user_id`
- Used for: Customer information (if not guest order)

### 5. **`products` Table**
- Referenced by: `order_items.product_id`
- Used for: Product details and stock management

### 6. **`vouchers` Table**
- Referenced by: `orders.voucher_id`
- Used for: Voucher/discount information

---

## Order Status Flow

```
pending → awaiting_payment → processing → shipped → delivered
                ↓
            cancelled (can happen at any stage)
```

### Status Descriptions:
- **pending**: Order created, awaiting payment (for COD/Bank Transfer)
- **awaiting_payment**: Order created, waiting for online payment
- **processing**: Payment received, order being prepared
- **shipped**: Order dispatched for delivery
- **delivered**: Order successfully delivered
- **cancelled**: Order cancelled

---

## Payment Status Flow

```
pending → paid
    ↓
  failed → refunded
```

### Payment Status Descriptions:
- **pending**: Payment not yet received
- **paid**: Payment successfully received
- **failed**: Payment attempt failed
- **refunded**: Payment refunded to customer

---

## Key Relationships

```
orders (1) ←→ (many) order_items
orders (many) ←→ (1) users
orders (many) ←→ (1) shipping_fees
orders (many) ←→ (1) vouchers
order_items (many) ←→ (1) products
```

---

## Performance Considerations

1. **Indexes** should be on:
   - `orders.id` (PRIMARY KEY)
   - `orders.order_number` (UNIQUE)
   - `orders.user_id` (FOREIGN KEY)
   - `orders.status` (for filtering)
   - `orders.created_at` (for sorting)
   - `orders.shipping_method_id` (FOREIGN KEY)

2. **Pagination** is used to limit results:
   - Default: 50 orders per page
   - Uses LIMIT and OFFSET

3. **LEFT JOIN** is used for shipping_fees:
   - Ensures orders without shipping method still appear
   - Returns NULL for shipping columns if no match

---

## Example Data Flow

1. Customer places order → Data inserted into `orders` table
2. Order items added → Data inserted into `order_items` table
3. Admin views orders page → Query fetches from `orders` + `shipping_fees`
4. Admin clicks "View" → Additional query fetches from `order_items`
5. Admin updates status → `orders.status` and `orders.updated_at` updated

---

## Summary

The admin order page primarily fetches data from:
- **Main Table**: `orders` (all columns)
- **Joined Table**: `shipping_fees` (fee_type, name)
- **Related Table**: `order_items` (fetched separately for details)

The data is filtered, paginated, and displayed in a table format with options to view details and update order status.

---

Last Updated: 2025-11-17

