# Forgot Password Setup Guide

This guide will help you set up the forgot password functionality for your 2 Days Parfume application.

## 📋 Overview

The forgot password feature allows users to reset their password via email. The system:
- Generates a secure token valid for 1 hour
- Sends a password reset email with a link
- Allows users to set a new password
- Marks tokens as used after successful reset

## 🔧 Installation Steps

### Step 1: Install PHPMailer (Recommended)

For better email delivery, install PHPMailer via Composer:

```bash
cd c:\Users\azfar\Downloads\apps\apps
composer require phpmailer/phpmailer
```

**Note:** If you don't have Composer installed, the system will fall back to PHP's built-in `mail()` function, but PHPMailer is highly recommended for better reliability.

### Step 2: Configure Email Settings

Edit `apps/config/email.php` and update the SMTP password:

```php
define('SMTP_PASSWORD', 'YOUR_ACTUAL_PASSWORD_HERE');
```

Replace `'YOUR_ACTUAL_PASSWORD_HERE'` with the actual password for `resetpassword@2dayparfum.com`.

**Email Configuration Details:**
- **SMTP Host:** mail.2dayparfum.com
- **SMTP Port:** 465 (SSL)
- **Username:** resetpassword@2dayparfum.com
- **Encryption:** SSL

### Step 3: Create Database Table

Run the SQL migration to create the password reset tokens table:

```sql
-- Connect to your database and run:
USE perfume_store;

CREATE TABLE IF NOT EXISTS password_reset_tokens (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    email VARCHAR(191) NOT NULL,
    token VARCHAR(255) NOT NULL,
    expires_at DATETIME NOT NULL,
    used BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_token (token),
    INDEX idx_email (email),
    INDEX idx_expires_at (expires_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
```

Or run the migration file:
```bash
mysql -u igoldvps_perfume2days -p igoldvps_perfume2days < apps/database/add_password_reset_tokens.sql
```

### Step 4: Test the Functionality

1. **Navigate to login page:**
   - Go to `https://2dayparfum.com/apps/login.php`
   - Click "Forgot Password?" link

2. **Request password reset:**
   - Enter your email address
   - Click "Send Reset Link"
   - Check your email for the reset link

3. **Reset password:**
   - Click the link in the email
   - Enter your new password
   - Confirm the password
   - Click "Reset Password"

4. **Login with new password:**
   - You'll be redirected to the login page
   - Login with your new password

## 📁 Files Created

1. **`apps/config/email.php`** - Email configuration and PHPMailer setup
2. **`apps/database/add_password_reset_tokens.sql`** - Database migration
3. **`apps/forgot-password.php`** - Password reset request page
4. **`apps/reset-password.php`** - Password reset form page
5. **`apps/login.php`** - Updated with forgot password link

## 🔒 Security Features

- **Secure Tokens:** 64-character random tokens using `random_bytes()`
- **Token Expiration:** Tokens expire after 1 hour
- **One-Time Use:** Tokens are marked as used after successful reset
- **Email Verification:** Only registered users can request password reset
- **Password Hashing:** Passwords are hashed using `password_hash()` with bcrypt
- **No Email Enumeration:** System doesn't reveal if email exists or not

## 🎨 User Experience

### Forgot Password Page
- Clean, modern design matching your auth pages
- Clear instructions for users
- Email validation
- Success state with helpful tips

### Reset Password Page
- Password visibility toggle
- Password requirements display
- Real-time password match validation
- Expired/invalid token handling
- Success confirmation with auto-redirect

### Email Template
- Professional HTML email design
- Clear call-to-action button
- Security warnings
- Expiration notice
- Alternative text link

## 🛠️ Troubleshooting

### Email Not Sending

**If using PHPMailer:**
1. Check SMTP credentials in `config/email.php`
2. Verify SMTP port (465 for SSL, 587 for TLS)
3. Check server firewall allows outbound SMTP connections
4. Review error logs: `apps/error_log`

**If using basic mail():**
1. Ensure your server has mail functionality enabled
2. Check PHP mail configuration in `php.ini`
3. Verify SPF/DKIM records for your domain

### Token Issues

**Token expired:**
- Tokens are valid for 1 hour only
- Request a new reset link

**Token already used:**
- Each token can only be used once
- Request a new reset link if needed

**Token not found:**
- Check if the link was copied correctly
- Ensure no extra characters in the URL

### Database Issues

**Table doesn't exist:**
```bash
# Run the migration again
mysql -u igoldvps_perfume2days -p igoldvps_perfume2days < apps/database/add_password_reset_tokens.sql
```

**Foreign key constraint fails:**
- Ensure the `users` table exists
- Check database user has proper permissions

## 📧 Email Configuration Options

### Option 1: PHPMailer with SMTP (Recommended)
- More reliable delivery
- Better error handling
- Works with most email providers
- Requires Composer installation

### Option 2: PHP mail() Function (Fallback)
- No additional dependencies
- May have delivery issues
- Depends on server configuration
- Automatically used if PHPMailer not available

## 🔄 Maintenance

### Clean Up Expired Tokens

Run this query periodically to clean up old tokens:

```sql
DELETE FROM password_reset_tokens 
WHERE expires_at < NOW() OR used = TRUE;
```

You can set up a cron job to run this daily:

```bash
# Add to crontab
0 2 * * * mysql -u igoldvps_perfume2days -p'Condition5594.' igoldvps_perfume2days -e "DELETE FROM password_reset_tokens WHERE expires_at < NOW() OR used = TRUE;"
```

## 📝 Customization

### Change Token Expiration Time

Edit `apps/forgot-password.php` line 35:

```php
// Change from 1 hour to 2 hours
$expires_at = date('Y-m-d H:i:s', strtotime('+2 hours'));
```

### Customize Email Template

Edit the `getPasswordResetEmailTemplate()` function in `apps/config/email.php` to customize:
- Colors and styling
- Logo (add your logo URL)
- Text content
- Footer information

### Change Password Requirements

Edit `apps/reset-password.php` to modify password validation:

```php
// Change minimum length from 8 to 10
elseif (strlen($new_password) < 10) {
    $error_message = 'Password must be at least 10 characters long.';
}
```

## 🎯 Next Steps

1. **Install PHPMailer** (if not already installed)
2. **Update SMTP password** in `config/email.php`
3. **Run database migration**
4. **Test the complete flow**
5. **Set up token cleanup cron job**
6. **Monitor email delivery**

## 📞 Support

If you encounter any issues:
1. Check the error logs: `apps/error_log`
2. Verify email configuration
3. Test with a different email address
4. Contact your hosting provider for SMTP issues

## ✅ Checklist

- [ ] PHPMailer installed (or using mail() fallback)
- [ ] Email configuration updated with correct password
- [ ] Database table created
- [ ] Tested forgot password flow
- [ ] Tested reset password flow
- [ ] Tested email delivery
- [ ] Set up token cleanup (optional)
- [ ] Customized email template (optional)

---

**Created:** 2025-11-20
**Version:** 1.0
**Author:** Augment Agent

