/**
 * Preset Manager JavaScript
 * Handles dynamic preset management and form integration
 */

class PresetManager {
    constructor() {
        this.apiUrl = '../ajax/presets.php';
        this.presets = {};
        this.init();
    }
    
    init() {
        this.loadAllPresets();
        this.setupEventListeners();
    }
    
    /**
     * Load all presets from API
     */
    async loadAllPresets() {
        try {
            const response = await fetch(this.apiUrl);
            const data = await response.json();
            
            if (data.success) {
                this.presets = data.data;
                this.updateFormDropdowns();
            } else {
                console.error('Failed to load presets:', data.error);
            }
        } catch (error) {
            console.error('Error loading presets:', error);
        }
    }
    
    /**
     * Get presets for specific field type
     */
    async getPresetsByFieldType(fieldType) {
        try {
            const response = await fetch(`${this.apiUrl}?field_type=${fieldType}`);
            const data = await response.json();
            
            if (data.success) {
                return data.data;
            } else {
                console.error('Failed to load presets for field type:', fieldType);
                return [];
            }
        } catch (error) {
            console.error('Error loading presets for field type:', error);
            return [];
        }
    }
    
    /**
     * Add new preset
     */
    async addPreset(presetData) {
        try {
            const response = await fetch(this.apiUrl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(presetData)
            });
            
            const data = await response.json();
            
            if (data.success) {
                // Reload presets
                await this.loadAllPresets();
                return { success: true, message: data.message };
            } else {
                return { success: false, error: data.error };
            }
        } catch (error) {
            return { success: false, error: error.message };
        }
    }
    
    /**
     * Update preset
     */
    async updatePreset(presetData) {
        try {
            const response = await fetch(this.apiUrl, {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(presetData)
            });
            
            const data = await response.json();
            
            if (data.success) {
                // Reload presets
                await this.loadAllPresets();
                return { success: true, message: data.message };
            } else {
                return { success: false, error: data.error };
            }
        } catch (error) {
            return { success: false, error: error.message };
        }
    }
    
    /**
     * Delete preset
     */
    async deletePreset(presetId) {
        try {
            const response = await fetch(`${this.apiUrl}?id=${presetId}`, {
                method: 'DELETE'
            });
            
            const data = await response.json();
            
            if (data.success) {
                // Reload presets
                await this.loadAllPresets();
                return { success: true, message: data.message };
            } else {
                return { success: false, error: data.error };
            }
        } catch (error) {
            return { success: false, error: error.message };
        }
    }
    
    /**
     * Update form dropdowns with current presets
     */
    updateFormDropdowns() {
        // Update brand dropdown
        this.updateDropdown('brand', this.presets.brand || []);
        
        // Update volume dropdown
        this.updateDropdown('volume', this.presets.volume || []);
        
        // Update size dropdown
        this.updateDropdown('size', this.presets.size || []);
        
        // Update season dropdown
        this.updateDropdown('season', this.presets.season || []);
        
        // Update notes dropdown
        this.updateDropdown('notes', this.presets.notes || []);
        
        // Update scent dropdown
        this.updateDropdown('scent', this.presets.scent || []);
        
        // Update recommendation dropdown
        this.updateDropdown('recommendation', this.presets.recommendation || []);
    }
    
    /**
     * Update specific dropdown with presets
     */
    updateDropdown(fieldType, presets) {
        const select = document.getElementById(fieldType);
        if (!select) return;
        
        // Store current value
        const currentValue = select.value;
        
        // Clear existing options (except first empty option)
        while (select.children.length > 1) {
            select.removeChild(select.lastChild);
        }
        
        // Add preset options
        presets.forEach(preset => {
            const option = document.createElement('option');
            option.value = preset.preset_value;
            option.textContent = preset.preset_value;
            select.appendChild(option);
        });
        
        // Restore current value if it still exists
        if (currentValue && Array.from(select.options).some(option => option.value === currentValue)) {
            select.value = currentValue;
        }
    }
    
    /**
     * Validate form field against presets
     */
    validateField(fieldType, value) {
        const presets = this.presets[fieldType] || [];
        return presets.some(preset => preset.preset_value === value);
    }
    
    /**
     * Get field types
     */
    getFieldTypes() {
        return {
            'brand': 'Brand',
            'volume': 'Volume',
            'size': 'Size',
            'season': 'Season',
            'notes': 'Notes',
            'scent': 'Scent',
            'recommendation': 'Recommendation'
        };
    }
    
    /**
     * Setup event listeners
     */
    setupEventListeners() {
        // Auto-refresh presets every 5 minutes
        setInterval(() => {
            this.loadAllPresets();
        }, 300000);
        
        // Listen for preset changes in admin forms
        document.addEventListener('presetAdded', () => {
            this.loadAllPresets();
        });
        
        document.addEventListener('presetUpdated', () => {
            this.loadAllPresets();
        });
        
        document.addEventListener('presetDeleted', () => {
            this.loadAllPresets();
        });
    }
    
    /**
     * Show notification
     */
    showNotification(message, type = 'success') {
        // Create notification element
        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        notification.innerHTML = `
            <div class="notification-content">
                <i class="fas fa-${type === 'success' ? 'check-circle' : 'exclamation-triangle'}"></i>
                <span>${message}</span>
            </div>
        `;
        
        // Add styles
        notification.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            background: ${type === 'success' ? '#d4edda' : '#f8d7da'};
            color: ${type === 'success' ? '#155724' : '#721c24'};
            padding: 15px 20px;
            border-radius: 6px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            z-index: 10000;
            animation: slideIn 0.3s ease;
        `;
        
        // Add animation styles
        const style = document.createElement('style');
        style.textContent = `
            @keyframes slideIn {
                from { transform: translateX(100%); opacity: 0; }
                to { transform: translateX(0); opacity: 1; }
            }
            @keyframes slideOut {
                from { transform: translateX(0); opacity: 1; }
                to { transform: translateX(100%); opacity: 0; }
            }
        `;
        document.head.appendChild(style);
        
        // Add to page
        document.body.appendChild(notification);
        
        // Remove after 3 seconds
        setTimeout(() => {
            notification.style.animation = 'slideOut 0.3s ease';
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.parentNode.removeChild(notification);
                }
            }, 300);
        }, 3000);
    }
}

// Initialize preset manager
const presetManager = new PresetManager();

// Export for use in other scripts
window.PresetManager = PresetManager;
window.presetManager = presetManager;