<?php
require_once '../../config/database.php';

class Preset {
    private $pdo;
    
    public function __construct() {
        $this->pdo = getDB();
        if (!$this->pdo) {
            throw new Exception("Database connection failed");
        }
    }
    
    /**
     * Get all presets for a specific field type
     */
    public function getPresetsByFieldType($fieldType) {
        $stmt = $this->pdo->prepare("
            SELECT * FROM product_field_presets 
            WHERE field_type = ? AND is_active = 1 
            ORDER BY sort_order, preset_value
        ");
        $stmt->execute([$fieldType]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * Get all presets grouped by field type
     */
    public function getAllPresetsGrouped() {
        $stmt = $this->pdo->query("
            SELECT * FROM product_field_presets 
            WHERE is_active = 1 
            ORDER BY field_type, sort_order, preset_value
        ");
        $presets = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $grouped = [];
        foreach ($presets as $preset) {
            $grouped[$preset['field_type']][] = $preset;
        }
        
        return $grouped;
    }
    
    /**
     * Add a new preset
     */
    public function addPreset($fieldType, $fieldName, $presetValue, $sortOrder = 0) {
        try {
            $stmt = $this->pdo->prepare("
                INSERT INTO product_field_presets (field_type, field_name, preset_value, sort_order) 
                VALUES (?, ?, ?, ?)
            ");
            return $stmt->execute([$fieldType, $fieldName, $presetValue, $sortOrder]);
        } catch (PDOException $e) {
            // Handle duplicate entry error
            if ($e->getCode() == 23000) {
                throw new Exception("This preset value already exists for this field type.");
            }
            throw $e;
        }
    }
    
    /**
     * Update a preset
     */
    public function updatePreset($id, $fieldType, $fieldName, $presetValue, $sortOrder, $isActive = true) {
        $stmt = $this->pdo->prepare("
            UPDATE product_field_presets 
            SET field_type = ?, field_name = ?, preset_value = ?, sort_order = ?, is_active = ? 
            WHERE id = ?
        ");
        return $stmt->execute([$fieldType, $fieldName, $presetValue, $sortOrder, $isActive ? 1 : 0, $id]);
    }
    
    /**
     * Delete a preset
     */
    public function deletePreset($id) {
        $stmt = $this->pdo->prepare("DELETE FROM product_field_presets WHERE id = ?");
        return $stmt->execute([$id]);
    }
    
    /**
     * Toggle preset status
     */
    public function togglePresetStatus($id) {
        $stmt = $this->pdo->prepare("
            UPDATE product_field_presets 
            SET is_active = NOT is_active 
            WHERE id = ?
        ");
        return $stmt->execute([$id]);
    }
    
    /**
     * Get preset by ID
     */
    public function getPresetById($id) {
        $stmt = $this->pdo->prepare("SELECT * FROM product_field_presets WHERE id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    /**
     * Get field types with their display names
     */
    public function getFieldTypes() {
        return [
            'brand' => 'Brand',
            'volume' => 'Volume',
            'size' => 'Size',
            'category' => 'Category',
            'season' => 'Season',
            'notes' => 'Notes',
            'scent' => 'Scent',
            'recommendation' => 'Recommendation'
        ];
    }
    
    /**
     * Get preset options for dropdown
     */
    public function getPresetOptions($fieldType) {
        $presets = $this->getPresetsByFieldType($fieldType);
        $options = [];
        
        foreach ($presets as $preset) {
            $options[] = [
                'value' => $preset['preset_value'],
                'label' => $preset['preset_value'],
                'id' => $preset['id']
            ];
        }
        
        return $options;
    }
    
    /**
     * Validate preset value
     */
    public function validatePresetValue($fieldType, $presetValue) {
        // Return false if presetValue is null or empty
        if (empty($presetValue)) {
            return false;
        }
        
        $presets = $this->getPresetsByFieldType($fieldType);
        
        foreach ($presets as $preset) {
            if (strtolower($preset['preset_value']) === strtolower($presetValue)) {
                return $preset['preset_value']; // Return the exact case from database
            }
        }
        
        return false; // Not found in presets
    }
    
    /**
     * Get statistics
     */
    public function getStatistics() {
        $stats = [];
        
        // Get count by field type
        $stmt = $this->pdo->query("
            SELECT field_type, COUNT(*) as count 
            FROM product_field_presets 
            WHERE is_active = 1 
            GROUP BY field_type
        ");
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($results as $result) {
            $stats[$result['field_type']] = $result['count'];
        }
        
        return $stats;
    }
}
?>