<?php
require_once '../includes/auth.php';
require_once '../includes/functions.php';

$adminAuth->requireLogin();

// Initialize AdminFunctions
$adminFunctions = new AdminFunctions();

// Define upload directory
$uploadDir = __DIR__ . '/../../banner/';

// Banner upload is now handled via AJAX in upload_banner.php

// Handle banner settings update
if ($_POST && isset($_POST['slide_timing'])) {
    $slideTiming = (int)$_POST['slide_timing'];
    if ($slideTiming >= 1 && $slideTiming <= 30) {
        // Save to a simple text file
        $settingsFile = __DIR__ . '/../../banner_settings.txt';
        file_put_contents($settingsFile, $slideTiming);
        $success = 'Banner settings updated successfully!';
    } else {
        $error = 'Invalid timing value. Please enter a number between 1 and 30.';
    }
}

// Load current banner settings
$settingsFile = __DIR__ . '/../../banner_settings.txt';
$currentTiming = 5; // default
if (file_exists($settingsFile)) {
    $currentTiming = (int)file_get_contents($settingsFile);
    if ($currentTiming < 1 || $currentTiming > 30) {
        $currentTiming = 5; // reset to default if invalid
    }
}

// Handle banner deletion
if ($_POST && isset($_POST['action']) && $_POST['action'] === 'delete_banner') {
    $bannerFile = $_POST['banner_file'];
    $filePath = __DIR__ . '/../../banner/' . $bannerFile;
    
    if (file_exists($filePath)) {
        if (unlink($filePath)) {
            $success = 'Banner deleted successfully!';
        } else {
            $error = 'Error deleting banner. Please try again.';
        }
    } else {
        $error = 'Banner file not found.';
    }
}

// Get banners
$banners = $adminFunctions->getBanners();

// Debug information
$debug_info = "Banner directory: " . $uploadDir;
$debug_info .= " | Directory exists: " . (is_dir($uploadDir) ? 'Yes' : 'No');
$debug_info .= " | Directory writable: " . (is_writable($uploadDir) ? 'Yes' : 'No');
$debug_info .= " | Banners found: " . count($banners);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Banner Management - Admin Panel</title>
    <link rel="stylesheet" href="../assets/css/admin.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
            color: #1A4D2A;
        }
        
        .form-group input[type="number"] {
            width: 100%;
            padding: 10px;
            border: 2px solid #ddd;
            border-radius: 5px;
            font-size: 16px;
            transition: border-color 0.3s ease;
        }
        
        .form-group input[type="number"]:focus {
            outline: none;
            border-color: #D4AF37;
        }
        
        .form-group small {
            display: block;
            margin-top: 5px;
            color: #666;
            font-size: 12px;
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="sidebar-header">
            <img src="../../images/LOGO.png" alt="Logo" class="sidebar-logo">
            <h3>Admin Panel</h3>
        </div>
        
        <nav class="sidebar-nav">
            <a href="../index.php" class="nav-item">
                <i class="fas fa-th-large"></i>
                <span>Overview</span>
            </a>
            <a href="products.php" class="nav-item">
                <i class="fas fa-box"></i>
                <span>Products</span>
            </a>
            <a href="orders.php" class="nav-item">
                <i class="fas fa-shopping-cart"></i>
                <span>Orders</span>
            </a>
            <a href="delivered.php" class="nav-item">
                <i class="fas fa-check-circle"></i>
                <span>Delivered</span>
            </a>
            <a href="customers.php" class="nav-item">
                <i class="fas fa-users"></i>
                <span>Customers</span>
            </a>
            <a href="feedback.php" class="nav-item">
                <i class="fas fa-comments"></i>
                <span>Feedback</span>
            </a>
            <a href="shipping.php" class="nav-item">
                <i class="fas fa-shipping-fast"></i>
                <span>Shipping</span>
            </a>
            <a href="banners.php" class="nav-item active">
                <i class="fas fa-image"></i>
                <span>Banners</span>
            </a>
            <a href="presets.php" class="nav-item">
                <i class="fas fa-cogs"></i>
                <span>Field Presets</span>
            </a>
            <a href="vouchers.php" class="nav-item">
                <i class="fas fa-ticket-alt"></i>
                <span>Voucher Management</span>
            </a>
            <a href="notifications.php" class="nav-item">
                <i class="fas fa-bell"></i>
                <span>Push Notifications</span>
            </a>
            <a href="analytics.php" class="nav-item">
                <i class="fas fa-chart-bar"></i>
                <span>Analytics</span>
            </a>
            <a href="settings.php" class="nav-item">
                <i class="fas fa-cog"></i>
                <span>Settings</span>
            </a>
        </nav>
        
        <div class="sidebar-footer">
            <div class="user-info">
                <i class="fas fa-user-circle"></i>
                <span><?php echo htmlspecialchars($adminAuth->getAdminName()); ?></span>
            </div>
            <a href="../logout.php" class="logout-btn">
                <i class="fas fa-sign-out-alt"></i>
                <span>Logout</span>
            </a>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <!-- Header -->
        <header class="header">
            <div class="header-left">
                <button class="sidebar-toggle">
                    <i class="fas fa-bars"></i>
                </button>
                <h1>Banner Management</h1>
            </div>
            <div class="header-right">
                <div class="header-actions">
                    <button onclick="showModal('uploadBannerModal')" class="btn btn-primary">
                        <i class="fas fa-upload"></i>
                        Upload Banner
                    </button>
                    <a href="../index.php" target="_blank" class="btn btn-outline">
                        <i class="fas fa-external-link-alt"></i>
                        View Store
                    </a>
                </div>
            </div>
        </header>

        <!-- Page Content -->
        <div class="page-content">
            <?php if (isset($success)): ?>
                <div class="alert" style="background-color: #d4edda; color: #155724; border: 1px solid #c3e6cb;">
                    <i class="fas fa-check-circle"></i>
                    <?php echo htmlspecialchars($success); ?>
                </div>
            <?php endif; ?>
            
             <?php if (isset($error)): ?>
                 <div class="alert alert-error">
                     <i class="fas fa-exclamation-circle"></i>
                     <?php echo htmlspecialchars($error); ?>
                 </div>
             <?php endif; ?>
             
             <!-- Debug Information -->
             <div class="alert" style="background-color: #e3f2fd; color: #1565c0; border: 1px solid #bbdefb; margin-bottom: 20px;">
                 <i class="fas fa-info-circle"></i>
                 <strong>Debug Info:</strong> <?php echo htmlspecialchars($debug_info); ?>
             </div>

            <!-- Banner Settings -->
            <div class="form-container">
                <h3>Banner Settings</h3>
                <form method="POST" id="bannerSettingsForm">
                    <div class="form-group">
                        <label for="slide_timing">Auto-slide Timing (seconds)</label>
                        <input type="number" id="slide_timing" name="slide_timing" value="<?php echo $currentTiming; ?>" min="1" max="30" required>
                        <small>How many seconds to wait before automatically sliding to the next banner (1-30 seconds)</small>
                    </div>
                    <div class="form-actions">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save"></i>
                            Save Settings
                        </button>
                    </div>
                </form>
            </div>

            <!-- Upload Banner Form -->
            <div class="form-container">
                <h3>Upload New Banner</h3>
                <form id="banner-form" enctype="multipart/form-data" style="display:none;">
                    <input type="file" id="banner_image" name="banner_image" accept="image/*,video/*">
                </form>
                <div class="image-upload">
                    <label for="banner_image" style="cursor: pointer; display: block; padding: 20px; text-align: center; border: 2px dashed #ddd; border-radius: 8px;">
                        <i class="fas fa-cloud-upload-alt" style="font-size: 48px; color: #ccc; margin-bottom: 10px;"></i>
                        <p>Click to select banner image or video</p>
                        <p style="font-size: 12px; color: #666;">Images: JPG, PNG, GIF | Videos: MP4, WebM, OGG, AVI, MOV (Max 50MB)</p>
                    </label>
                    <img id="bannerPreview" class="image-preview" style="display: none;">
                    <video id="videoPreview" class="video-preview" style="display: none; max-width: 100%; max-height: 200px; border-radius: 8px;"></video>
                </div>
            </div>

            <!-- Banners Grid -->
            <div class="banners-grid">
                <?php if (empty($banners)): ?>
                    <div class="empty-state">
                        <i class="fas fa-image"></i>
                        <h3>No Banners Found</h3>
                        <p>Upload your first banner to get started.</p>
                    </div>
                <?php else: ?>
                    <?php foreach ($banners as $banner): ?>
                        <div class="banner-card">
                            <div class="banner-image">
                                <?php 
                                $fileExtension = strtolower(pathinfo($banner['filename'], PATHINFO_EXTENSION));
                                $isVideo = in_array($fileExtension, ['mp4', 'webm', 'ogg', 'avi', 'mov']);
                                ?>
                                <?php if ($isVideo): ?>
                                    <video src="../../<?php echo htmlspecialchars($banner['path']); ?>" 
                                           alt="<?php echo htmlspecialchars($banner['filename']); ?>"
                                           onclick="viewBanner('<?php echo htmlspecialchars($banner['path']); ?>', 'video')"
                                           style="width: 100%; height: 150px; object-fit: cover; border-radius: 8px;">
                                        <i class="fas fa-play-circle" style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); font-size: 48px; color: white; text-shadow: 2px 2px 4px rgba(0,0,0,0.7);"></i>
                                    </video>
                                <?php else: ?>
                                    <img src="../../<?php echo htmlspecialchars($banner['path']); ?>" 
                                         alt="<?php echo htmlspecialchars($banner['filename']); ?>"
                                         onclick="viewBanner('<?php echo htmlspecialchars($banner['path']); ?>', 'image')">
                                <?php endif; ?>
                            </div>
                            <div class="banner-info">
                                <h4><?php echo htmlspecialchars($banner['filename']); ?></h4>
                                <p>Size: <?php echo formatFileSize($banner['size']); ?></p>
                                <p>Modified: <?php echo date('M j, Y H:i', $banner['modified']); ?></p>
                                <div class="banner-actions">
                                    <button onclick="viewBanner('<?php echo htmlspecialchars($banner['path']); ?>', '<?php echo $isVideo ? 'video' : 'image'; ?>')" 
                                            class="btn btn-sm btn-primary" title="View">
                                        <i class="fas fa-eye"></i>
                                    </button>
                                    <button onclick="deleteBanner('<?php echo htmlspecialchars($banner['filename']); ?>')" 
                                            class="btn btn-sm btn-danger" title="Delete">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Banner Preview Modal -->
    <div id="bannerPreviewModal" class="modal">
        <div class="modal-content" style="max-width: 90%; max-height: 90%;">
            <div class="modal-header">
                <h3 class="modal-title">Banner Preview</h3>
                <span class="close" onclick="hideModal('bannerPreviewModal')">&times;</span>
            </div>
            
            <div style="text-align: center;">
                <img id="previewImage" src="" alt="Banner Preview" style="max-width: 100%; max-height: 70vh; object-fit: contain; display: none;">
                <video id="previewVideo" src="" alt="Banner Preview" style="max-width: 100%; max-height: 70vh; object-fit: contain; display: none;" controls></video>
            </div>
        </div>
    </div>

    <!-- Delete Banner Modal -->
    <div id="deleteBannerModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title">Delete Banner</h3>
                <span class="close" onclick="hideModal('deleteBannerModal')">&times;</span>
            </div>
            
            <form method="POST" id="deleteBannerForm">
                <input type="hidden" name="action" value="delete_banner">
                <input type="hidden" name="banner_file" id="deleteBannerFile">
                
                <p>Are you sure you want to delete this banner? This action cannot be undone.</p>
                
                <div class="form-actions">
                    <button type="button" onclick="hideModal('deleteBannerModal')" class="btn btn-outline">Cancel</button>
                    <button type="submit" class="btn btn-danger">Delete Banner</button>
                </div>
            </form>
        </div>
    </div>

    <script src="../assets/js/admin.js"></script>
    <script>
        // Handle file selection and upload (same approach as profile upload)
        document.addEventListener('DOMContentLoaded', function() {
            const input = document.getElementById('banner_image');
            if (!input) return;
            
            input.addEventListener('change', function() {
                if (!this.files || !this.files[0]) return;
                
                const file = this.files[0];
                const imagePreview = document.getElementById('bannerPreview');
                const videoPreview = document.getElementById('videoPreview');
                
                // Show preview
                if (file.type.startsWith('image/')) {
                    const reader = new FileReader();
                    reader.onload = function(e) {
                        imagePreview.src = e.target.result;
                        imagePreview.style.display = 'block';
                        videoPreview.style.display = 'none';
                    };
                    reader.readAsDataURL(file);
                } else if (file.type.startsWith('video/')) {
                    const url = URL.createObjectURL(file);
                    videoPreview.src = url;
                    videoPreview.style.display = 'block';
                    imagePreview.style.display = 'none';
                }
                
                // Upload file
                const formData = new FormData();
                formData.append('banner_image', file);
                
                fetch('../ajax/upload_banner.php', { 
                    method: 'POST', 
                    body: formData 
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showNotification(data.message || 'Banner uploaded successfully!', 'success');
                        // Reload page to show new banner
                        setTimeout(() => {
                            window.location.reload();
                        }, 1000);
                    } else {
                        showNotification(data.message || 'Upload failed', 'error');
                    }
                    input.value = '';
                })
                .catch(error => {
                    console.error('Upload error:', error);
                    showNotification('Upload failed', 'error');
                    input.value = '';
                });
            });
        });
        
        function viewBanner(filePath, type) {
            const previewImage = document.getElementById('previewImage');
            const previewVideo = document.getElementById('previewVideo');
            
            // Hide both previews first
            previewImage.style.display = 'none';
            previewVideo.style.display = 'none';
            
            if (type === 'video') {
                previewVideo.src = '../../' + filePath;
                previewVideo.style.display = 'block';
            } else {
                previewImage.src = '../../' + filePath;
                previewImage.style.display = 'block';
            }
            
            showModal('bannerPreviewModal');
        }
        
        function deleteBanner(filename) {
            document.getElementById('deleteBannerFile').value = filename;
            showModal('deleteBannerModal');
        }
        
        // Simple notification function
        function showNotification(message, type = 'info') {
            const notification = document.createElement('div');
            notification.style.cssText = `
                position: fixed;
                top: 20px;
                right: 20px;
                padding: 15px 20px;
                border-radius: 5px;
                color: white;
                font-weight: bold;
                z-index: 10000;
                max-width: 300px;
                word-wrap: break-word;
            `;
            
            if (type === 'success') {
                notification.style.backgroundColor = '#28a745';
            } else if (type === 'error') {
                notification.style.backgroundColor = '#dc3545';
            } else {
                notification.style.backgroundColor = '#17a2b8';
            }
            
            notification.textContent = message;
            document.body.appendChild(notification);
            
            setTimeout(() => {
                notification.remove();
            }, 3000);
        }
    </script>
</body>
</html>

<?php
function formatFileSize($bytes) {
    if ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    } else {
        return $bytes . ' bytes';
    }
}
?>