<?php
require_once '../includes/auth.php';
require_once '../includes/functions.php';

$adminAuth->requireLogin();

// Get all cart activity
try {
    $conn = getDB();
    $query = "SELECT 
                c.id as cart_id,
                c.product_id,
                c.quantity,
                c.size,
                c.version,
                c.price,
                c.created_at,
                c.updated_at,
                u.id as user_id,
                u.first_name,
                u.last_name,
                u.email,
                u.avatar,
                p.name as product_name,
                p.brand as product_brand,
                p.image as product_image
              FROM cart c
              LEFT JOIN users u ON c.user_id = u.id
              LEFT JOIN products p ON c.product_id = p.id
              ORDER BY c.created_at DESC";
    
    $stmt = $conn->prepare($query);
    $stmt->execute();
    $cartItems = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    $cartItems = [];
    error_log("Error fetching cart activity: " . $e->getMessage());
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cart Activity - Admin Panel</title>
    <link rel="stylesheet" href="../assets/css/admin.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .product-preview {
            display: flex;
            align-items: center;
            gap: 10px;
        }
        .product-preview img {
            width: 50px;
            height: 50px;
            object-fit: cover;
            border-radius: 4px;
            border: 1px solid #e8e8e8;
        }
        .user-cell {
            display: flex;
            align-items: center;
            gap: 10px;
        }
        .user-avatar {
            width: 35px;
            height: 35px;
            border-radius: 50%;
            object-fit: cover;
        }
        .user-placeholder {
            width: 35px;
            height: 35px;
            border-radius: 50%;
            background-color: #f0f0f0;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        .guest-badge {
            background: #fbbf24;
            color: #78350f;
            padding: 2px 8px;
            border-radius: 12px;
            font-size: 11px;
            font-weight: 500;
        }
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 16px;
            margin-bottom: 24px;
        }
        .stat-card {
            background: white;
            padding: 20px;
            border-radius: 8px;
            border: 1px solid #e8e8e8;
        }
        .stat-value {
            font-size: 28px;
            font-weight: 600;
            color: #1f1f1f;
            margin-bottom: 4px;
        }
        .stat-label {
            font-size: 13px;
            color: #666;
        }
        .reminder-btn {
            background: #10b981;
            color: white;
            border: none;
            padding: 6px 12px;
            border-radius: 4px;
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: 5px;
            font-size: 12px;
            transition: all 0.2s;
        }
        .reminder-btn:hover {
            background: #059669;
            transform: translateY(-1px);
            box-shadow: 0 2px 8px rgba(16, 185, 129, 0.3);
        }
        .reminder-btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none;
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="sidebar-header">
            <img src="../../images/LOGO.png" alt="Logo" class="sidebar-logo">
            <h3>Admin Panel</h3>
        </div>
        
        <nav class="sidebar-nav">
            <a href="../index.php" class="nav-item">
                <i class="fas fa-th-large"></i>
                <span>Overview</span>
            </a>
            <a href="products.php" class="nav-item">
                <i class="fas fa-box"></i>
                <span>Products</span>
            </a>
            <a href="orders.php" class="nav-item">
                <i class="fas fa-shopping-cart"></i>
                <span>Orders</span>
            </a>
            <a href="delivered.php" class="nav-item">
                <i class="fas fa-check-circle"></i>
                <span>Delivered</span>
            </a>
            <a href="customers.php" class="nav-item">
                <i class="fas fa-users"></i>
                <span>Customers</span>
            </a>
            <a href="feedback.php" class="nav-item">
                <i class="fas fa-comments"></i>
                <span>Feedback</span>
            </a>
            <a href="shipping.php" class="nav-item">
                <i class="fas fa-shipping-fast"></i>
                <span>Shipping</span>
            </a>
            <a href="banners.php" class="nav-item">
                <i class="fas fa-image"></i>
                <span>Banners</span>
            </a>
            <a href="presets.php" class="nav-item">
                <i class="fas fa-cogs"></i>
                <span>Field Presets</span>
            </a>
            <a href="vouchers.php" class="nav-item">
                <i class="fas fa-ticket-alt"></i>
                <span>Voucher Management</span>
            </a>
            <a href="notifications.php" class="nav-item">
                <i class="fas fa-bell"></i>
                <span>Push Notifications</span>
            </a>
            <a href="analytics.php" class="nav-item">
                <i class="fas fa-chart-bar"></i>
                <span>Analytics</span>
            </a>
            <a href="settings.php" class="nav-item">
                <i class="fas fa-cog"></i>
                <span>Settings</span>
            </a>
        </nav>
        
        <div class="sidebar-footer">
            <div class="user-info">
                <i class="fas fa-user-circle"></i>
                <span><?php echo htmlspecialchars($adminAuth->getAdminName()); ?></span>
            </div>
            <a href="../logout.php" class="logout-btn">
                <i class="fas fa-sign-out-alt"></i>
                <span>Logout</span>
            </a>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <!-- Header -->
        <header class="header">
            <div class="header-left">
                <button class="sidebar-toggle">
                    <i class="fas fa-bars"></i>
                </button>
                <h1>Cart Activity</h1>
            </div>
            <div class="header-right">
                <div class="header-actions">
                    <button onclick="exportCartActivity()" class="btn btn-outline">
                        <i class="fas fa-download"></i>
                        Export Data
                    </button>
                    <a href="../index.php" target="_blank" class="btn btn-outline">
                        <i class="fas fa-external-link-alt"></i>
                        View Store
                    </a>
                </div>
            </div>
        </header>

        <!-- Page Content -->
        <div class="page-content">
            <!-- Statistics Cards -->
            <div class="stats-grid">
                <?php
                $totalItems = count($cartItems);
                $totalValue = array_sum(array_map(function($item) {
                    return $item['price'] * $item['quantity'];
                }, $cartItems));
                $uniqueUsers = count(array_unique(array_filter(array_column($cartItems, 'user_id'))));
                $guestCarts = count(array_filter($cartItems, function($item) {
                    return empty($item['user_id']);
                }));
                ?>
                
                <div class="stat-card">
                    <div class="stat-value"><?php echo number_format($totalItems); ?></div>
                    <div class="stat-label">Total Cart Items</div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-value">RM <?php echo number_format($totalValue, 2); ?></div>
                    <div class="stat-label">Potential Revenue</div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-value"><?php echo number_format($uniqueUsers); ?></div>
                    <div class="stat-label">Active Users</div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-value"><?php echo number_format($guestCarts); ?></div>
                    <div class="stat-label">Guest Carts</div>
                </div>
            </div>

            <!-- Search and Filter -->
            <div class="search-filter">
                <input type="text" class="search-input" placeholder="Search by user, product, or email..." id="cartSearch">
                <select class="filter-select" id="dateFilter">
                    <option value="">All Time</option>
                    <option value="today">Today</option>
                    <option value="week">This Week</option>
                    <option value="month">This Month</option>
                </select>
                <select class="filter-select" id="userFilter">
                    <option value="">All Users</option>
                    <option value="registered">Registered Users</option>
                    <option value="guest">Guest Users</option>
                </select>
            </div>

            <!-- Cart Activity Table -->
            <div class="table-container">
                <div class="table-content">
                    <table class="data-table" id="cartActivityTable">
                        <thead>
                            <tr>
                                <th>User</th>
                                <th>Product</th>
                                <th>Size</th>
                                <th>Quantity</th>
                                <th>Price</th>
                                <th>Total</th>
                                <th>Added Date</th>
                                <th>Last Updated</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($cartItems)): ?>
                            <tr>
                                <td colspan="9" style="text-align: center; padding: 40px; color: #999;">
                                    <i class="fas fa-shopping-cart" style="font-size: 48px; margin-bottom: 16px; opacity: 0.3;"></i>
                                    <p>No cart activity found</p>
                                </td>
                            </tr>
                            <?php else: ?>
                                <?php foreach ($cartItems as $item): ?>
                                <tr data-date="<?php echo $item['created_at']; ?>" data-user-type="<?php echo !empty($item['user_id']) ? 'registered' : 'guest'; ?>">
                                    <td>
                                        <div class="user-cell">
                                            <?php if (!empty($item['user_id'])): ?>
                                                <?php if ($item['avatar']): ?>
                                                    <img src="../../<?php echo htmlspecialchars($item['avatar']); ?>" 
                                                         alt="Avatar" class="user-avatar">
                                                <?php else: ?>
                                                    <div class="user-placeholder">
                                                        <i class="fas fa-user" style="color: #999; font-size: 14px;"></i>
                                                    </div>
                                                <?php endif; ?>
                                                <div>
                                                    <strong><?php echo htmlspecialchars($item['first_name'] . ' ' . $item['last_name']); ?></strong>
                                                    <br>
                                                    <small style="color: #666;"><?php echo htmlspecialchars($item['email']); ?></small>
                                                </div>
                                            <?php else: ?>
                                                <div class="user-placeholder">
                                                    <i class="fas fa-user-secret" style="color: #999; font-size: 14px;"></i>
                                                </div>
                                                <div>
                                                    <span class="guest-badge">Guest User</span>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="product-preview">
                                            <?php if ($item['product_image']): ?>
                                                <img src="../../<?php echo htmlspecialchars($item['product_image']); ?>" 
                                                     alt="Product">
                                            <?php else: ?>
                                                <div style="width: 50px; height: 50px; background: #f0f0f0; border-radius: 4px; display: flex; align-items: center; justify-content: center;">
                                                    <i class="fas fa-image" style="color: #ccc;"></i>
                                                </div>
                                            <?php endif; ?>
                                            <div>
                                                <strong><?php echo htmlspecialchars($item['product_name'] ?: 'Unknown Product'); ?></strong>
                                                <br>
                                                <small style="color: #666;"><?php echo htmlspecialchars($item['product_brand'] ?: ''); ?></small>
                                            </div>
                                        </div>
                                    </td>
                                    <td><?php echo htmlspecialchars($item['size'] ?: 'N/A'); ?></td>
                                    <td><?php echo number_format($item['quantity']); ?></td>
                                    <td>RM <?php echo number_format($item['price'], 2); ?></td>
                                    <td><strong>RM <?php echo number_format($item['price'] * $item['quantity'], 2); ?></strong></td>
                                    <td>
                                        <?php 
                                        $addedDate = new DateTime($item['created_at']);
                                        echo $addedDate->format('M j, Y');
                                        ?>
                                        <br>
                                        <small style="color: #666;"><?php echo $addedDate->format('g:i A'); ?></small>
                                    </td>
                                    <td>
                                        <?php 
                                        $updatedDate = new DateTime($item['updated_at']);
                                        $now = new DateTime();
                                        $diff = $now->diff($updatedDate);
                                        
                                        if ($diff->days == 0) {
                                            if ($diff->h == 0) {
                                                echo $diff->i . ' min ago';
                                            } else {
                                                echo $diff->h . ' hour' . ($diff->h > 1 ? 's' : '') . ' ago';
                                            }
                                        } elseif ($diff->days == 1) {
                                            echo 'Yesterday';
                                        } else {
                                            echo $updatedDate->format('M j, Y');
                                        }
                                        ?>
                                    </td>
                                    <td>
                                        <?php if (!empty($item['user_id'])): ?>
                                            <button 
                                                onclick="sendCartReminder(<?php echo $item['user_id']; ?>, '<?php echo htmlspecialchars(addslashes($item['first_name'])); ?>', '<?php echo htmlspecialchars(addslashes($item['product_name'])); ?>')"
                                                class="reminder-btn"
                                                title="Send cart reminder to user">
                                                <i class="fas fa-bell"></i>
                                                <span>Send Reminder</span>
                                            </button>
                                        <?php else: ?>
                                            <span style="color: #999; font-size: 12px;">
                                                <i class="fas fa-user-slash"></i> Guest
                                            </span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <script src="../assets/js/admin.js"></script>
    <script>
        // Send cart reminder notification
        function sendCartReminder(userId, userName, productName) {
            if (!confirm(`Send a cart reminder to ${userName}?`)) {
                return;
            }
            
            const button = event.target.closest('button');
            const originalContent = button.innerHTML;
            button.disabled = true;
            button.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Sending...';
            
            fetch('../ajax/send-cart-reminder.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams({
                    user_id: userId,
                    user_name: userName,
                    product_name: productName
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Show success message
                    showNotification('success', data.message);
                    
                    // Change button to success state
                    button.style.background = '#059669';
                    button.innerHTML = '<i class="fas fa-check"></i> Sent!';
                    
                    // Reset button after 3 seconds
                    setTimeout(() => {
                        button.disabled = false;
                        button.style.background = '#10b981';
                        button.innerHTML = originalContent;
                    }, 3000);
                } else {
                    showNotification('error', data.message);
                    button.disabled = false;
                    button.innerHTML = originalContent;
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showNotification('error', 'Failed to send reminder. Please try again.');
                button.disabled = false;
                button.innerHTML = originalContent;
            });
        }
        
        // Show notification helper
        function showNotification(type, message) {
            const notification = document.createElement('div');
            notification.style.cssText = `
                position: fixed;
                top: 20px;
                right: 20px;
                padding: 16px 24px;
                background: ${type === 'success' ? '#10b981' : '#ef4444'};
                color: white;
                border-radius: 8px;
                box-shadow: 0 4px 12px rgba(0,0,0,0.15);
                z-index: 10000;
                font-size: 14px;
                font-weight: 500;
                display: flex;
                align-items: center;
                gap: 10px;
                animation: slideIn 0.3s ease-out;
            `;
            
            notification.innerHTML = `
                <i class="fas fa-${type === 'success' ? 'check-circle' : 'exclamation-circle'}"></i>
                <span>${message}</span>
            `;
            
            document.body.appendChild(notification);
            
            setTimeout(() => {
                notification.style.animation = 'slideOut 0.3s ease-out';
                setTimeout(() => notification.remove(), 300);
            }, 4000);
        }
        
        // Add animation styles
        const style = document.createElement('style');
        style.textContent = `
            @keyframes slideIn {
                from { transform: translateX(400px); opacity: 0; }
                to { transform: translateX(0); opacity: 1; }
            }
            @keyframes slideOut {
                from { transform: translateX(0); opacity: 1; }
                to { transform: translateX(400px); opacity: 0; }
            }
        `;
        document.head.appendChild(style);
        
        // Search functionality
        document.getElementById('cartSearch').addEventListener('input', function() {
            const searchTerm = this.value.toLowerCase();
            const rows = document.querySelectorAll('#cartActivityTable tbody tr');
            
            rows.forEach(row => {
                const text = row.textContent.toLowerCase();
                if (text.includes(searchTerm)) {
                    row.style.display = '';
                } else {
                    row.style.display = 'none';
                }
            });
        });
        
        // Date filter
        document.getElementById('dateFilter').addEventListener('change', function() {
            const filterValue = this.value;
            const rows = document.querySelectorAll('#cartActivityTable tbody tr');
            
            rows.forEach(row => {
                const dateStr = row.getAttribute('data-date');
                if (!dateStr) {
                    row.style.display = '';
                    return;
                }
                
                const cellDate = new Date(dateStr);
                const now = new Date();
                let show = true;
                
                switch(filterValue) {
                    case 'today':
                        show = cellDate.toDateString() === now.toDateString();
                        break;
                    case 'week':
                        const weekAgo = new Date(now.getTime() - 7 * 24 * 60 * 60 * 1000);
                        show = cellDate >= weekAgo;
                        break;
                    case 'month':
                        const monthAgo = new Date(now.getTime() - 30 * 24 * 60 * 60 * 1000);
                        show = cellDate >= monthAgo;
                        break;
                }
                
                row.style.display = show ? '' : 'none';
            });
        });
        
        // User type filter
        document.getElementById('userFilter').addEventListener('change', function() {
            const filterValue = this.value;
            const rows = document.querySelectorAll('#cartActivityTable tbody tr');
            
            rows.forEach(row => {
                const userType = row.getAttribute('data-user-type');
                if (!userType) {
                    row.style.display = '';
                    return;
                }
                
                if (filterValue === '') {
                    row.style.display = '';
                } else if (filterValue === userType) {
                    row.style.display = '';
                } else {
                    row.style.display = 'none';
                }
            });
        });
        
        // Export functionality
        function exportCartActivity() {
            const table = document.getElementById('cartActivityTable');
            const rows = Array.from(table.querySelectorAll('tr:not([style*="display: none"])'));
            const csvContent = rows.map(row => 
                Array.from(row.querySelectorAll('th, td')).map(cell => {
                    // Clean the text content
                    const text = cell.textContent.trim().replace(/\s+/g, ' ');
                    return `"${text}"`;
                }).join(',')
            ).join('\n');
            
            const blob = new Blob([csvContent], { type: 'text/csv' });
            const url = window.URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = 'cart_activity_' + new Date().toISOString().split('T')[0] + '.csv';
            a.click();
            window.URL.revokeObjectURL(url);
        }
    </script>
</body>
</html>

