<?php
require_once '../includes/auth.php';
require_once '../includes/functions.php';
require_once '../../models/Loyalty.php';

$adminAuth->requireLogin();

// Get customers with loyalty points
$customers = $adminFunctions->getAllUsers();
$loyalty = new Loyalty();

// Add loyalty points to each customer - FIXED: Remove reference to prevent duplication
foreach ($customers as $index => $customer) {
    $loyaltyData = $loyalty->getUserLoyaltyData($customer['id']);
    $customers[$index]['loyalty_points'] = $loyaltyData ? $loyaltyData['points'] : 0;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Customers Management - Admin Panel</title>
    <link rel="stylesheet" href="../assets/css/admin.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body>
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="sidebar-header">
            <img src="../../images/LOGO.png" alt="Logo" class="sidebar-logo">
            <h3>Admin Panel</h3>
        </div>
        
        <nav class="sidebar-nav">
            <a href="../index.php" class="nav-item">
                <i class="fas fa-th-large"></i>
                <span>Overview</span>
            </a>
            <a href="products.php" class="nav-item">
                <i class="fas fa-box"></i>
                <span>Products</span>
            </a>
            <a href="orders.php" class="nav-item">
                <i class="fas fa-shopping-cart"></i>
                <span>Orders</span>
            </a>
            <a href="delivered.php" class="nav-item">
                <i class="fas fa-check-circle"></i>
                <span>Delivered</span>
            </a>
            <a href="customers.php" class="nav-item active">
                <i class="fas fa-users"></i>
                <span>Customers</span>
            </a>
            <a href="feedback.php" class="nav-item">
                <i class="fas fa-comments"></i>
                <span>Feedback</span>
            </a>
            <a href="shipping.php" class="nav-item">
                <i class="fas fa-shipping-fast"></i>
                <span>Shipping</span>
            </a>
            <a href="banners.php" class="nav-item">
                <i class="fas fa-image"></i>
                <span>Banners</span>
            </a>
            <a href="presets.php" class="nav-item">
                <i class="fas fa-cogs"></i>
                <span>Field Presets</span>
            </a>
            <a href="vouchers.php" class="nav-item">
                <i class="fas fa-ticket-alt"></i>
                <span>Voucher Management</span>
            </a>
            <a href="notifications.php" class="nav-item">
                <i class="fas fa-bell"></i>
                <span>Push Notifications</span>
            </a>
            <a href="analytics.php" class="nav-item">
                <i class="fas fa-chart-bar"></i>
                <span>Analytics</span>
            </a>
            <a href="settings.php" class="nav-item">
                <i class="fas fa-cog"></i>
                <span>Settings</span>
            </a>
        </nav>
        
        <div class="sidebar-footer">
            <div class="user-info">
                <i class="fas fa-user-circle"></i>
                <span><?php echo htmlspecialchars($adminAuth->getAdminName()); ?></span>
            </div>
            <a href="../logout.php" class="logout-btn">
                <i class="fas fa-sign-out-alt"></i>
                <span>Logout</span>
            </a>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <!-- Header -->
        <header class="header">
            <div class="header-left">
                <button class="sidebar-toggle">
                    <i class="fas fa-bars"></i>
                </button>
                <h1>Customers Management</h1>
            </div>
            <div class="header-right">
                <div class="header-actions">
                    <button onclick="exportCustomers()" class="btn btn-outline">
                        <i class="fas fa-download"></i>
                        Export Customers
                    </button>
                    <a href="../index.php" target="_blank" class="btn btn-outline">
                        <i class="fas fa-external-link-alt"></i>
                        View Store
                    </a>
                </div>
            </div>
        </header>

        <!-- Page Content -->
        <div class="page-content">
            <!-- Search and Filter -->
            <div class="search-filter">
                <input type="text" class="search-input" placeholder="Search customers..." id="customerSearch">
                <select class="filter-select" id="dateFilter">
                    <option value="">All Time</option>
                    <option value="today">Today</option>
                    <option value="week">This Week</option>
                    <option value="month">This Month</option>
                    <option value="year">This Year</option>
                </select>
            </div>

            <!-- Customers Table -->
            <div class="table-container">
                <div class="table-content">
                    <table class="data-table" id="customersTable">
                        <thead>
                            <tr>
                                <th>Avatar</th>
                                <th>Name</th>
                                <th>Email</th>
                                <th>Phone</th>
                                <th>Location</th>
                                <th>Points</th>
                                <th>Member Since</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($customers as $customer): ?>
                            <tr>
                                <td>
                                    <?php if ($customer['avatar']): ?>
                                        <img src="../../<?php echo htmlspecialchars($customer['avatar']); ?>" 
                                             alt="Avatar" style="width: 40px; height: 40px; border-radius: 50%; object-fit: cover;">
                                    <?php else: ?>
                                        <div style="width: 40px; height: 40px; border-radius: 50%; background-color: #f0f0f0; display: flex; align-items: center; justify-content: center;">
                                            <i class="fas fa-user" style="color: #999;"></i>
                                        </div>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <strong><?php echo htmlspecialchars($customer['first_name'] . ' ' . $customer['last_name']); ?></strong>
                                </td>
                                <td><?php echo htmlspecialchars($customer['email']); ?></td>
                                <td><?php echo htmlspecialchars($customer['phone'] ?: 'N/A'); ?></td>
                                <td>
                                    <?php if ($customer['city'] && $customer['state']): ?>
                                        <?php echo htmlspecialchars($customer['city'] . ', ' . $customer['state']); ?>
                                    <?php else: ?>
                                        N/A
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <span class="points-badge" style="
                                        background: linear-gradient(135deg, #FDD835, #FBC02D);
                                        color: #333;
                                        padding: 4px 8px;
                                        border-radius: 12px;
                                        font-size: 12px;
                                        font-weight: 600;
                                        display: inline-block;
                                    ">
                                        <?php echo number_format($customer['loyalty_points']); ?> pts
                                    </span>
                                </td>
                                <td><?php echo date('M j, Y', strtotime($customer['created_at'])); ?></td>
                                <td>
                                    <div style="display: flex; gap: 5px;">
                                        <button onclick="viewCustomer(<?php echo $customer['id']; ?>)" 
                                                class="btn btn-sm btn-primary" title="View Details">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                        <button onclick="viewCustomerOrders(<?php echo $customer['id']; ?>)" 
                                                class="btn btn-sm btn-warning" title="View Orders">
                                            <i class="fas fa-shopping-cart"></i>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Customer Details Modal -->
    <div id="customerDetailsModal" class="modal">
        <div class="modal-content" style="max-width: 600px;">
            <div class="modal-header">
                <h3 class="modal-title">Customer Details</h3>
                <span class="close" onclick="hideModal('customerDetailsModal')">&times;</span>
            </div>
            
            <div id="customerDetailsContent">
                <!-- Customer details will be loaded here -->
            </div>
        </div>
    </div>

    <!-- Customer Orders Modal -->
    <div id="customerOrdersModal" class="modal">
        <div class="modal-content" style="max-width: 800px;">
            <div class="modal-header">
                <h3 class="modal-title">Customer Orders</h3>
                <span class="close" onclick="hideModal('customerOrdersModal')">&times;</span>
            </div>
            
            <div id="customerOrdersContent">
                <!-- Customer orders will be loaded here -->
            </div>
        </div>
    </div>

    <script src="../assets/js/admin.js"></script>
    <script>
        function viewCustomer(customerId) {
            // Load customer details via AJAX
            fetch(`ajax/get_customer_details.php?id=${customerId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        document.getElementById('customerDetailsContent').innerHTML = data.html;
                        showModal('customerDetailsModal');
                    } else {
                        alert('Error loading customer details');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Error loading customer details');
                });
        }
        
        function viewCustomerOrders(customerId) {
            // Load customer orders via AJAX
            fetch(`ajax/get_customer_orders.php?id=${customerId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        document.getElementById('customerOrdersContent').innerHTML = data.html;
                        showModal('customerOrdersModal');
                    } else {
                        alert('Error loading customer orders');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Error loading customer orders');
                });
        }
        
        function exportCustomers() {
            const table = document.getElementById('customersTable');
            const rows = Array.from(table.querySelectorAll('tr'));
            const csvContent = rows.map(row => 
                Array.from(row.querySelectorAll('th, td')).map(cell => 
                    `"${cell.textContent.trim()}"`
                ).join(',')
            ).join('\n');
            
            const blob = new Blob([csvContent], { type: 'text/csv' });
            const url = window.URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = 'customers_export.csv';
            a.click();
            window.URL.revokeObjectURL(url);
        }
        
        // Search functionality
        document.getElementById('customerSearch').addEventListener('input', function() {
            const searchTerm = this.value.toLowerCase();
            const rows = document.querySelectorAll('#customersTable tbody tr');
            
            rows.forEach(row => {
                const text = row.textContent.toLowerCase();
                if (text.includes(searchTerm)) {
                    row.style.display = '';
                } else {
                    row.style.display = 'none';
                }
            });
        });
        
        // Date filter
        document.getElementById('dateFilter').addEventListener('change', function() {
            const filterValue = this.value;
            const rows = document.querySelectorAll('#customersTable tbody tr');
            
            rows.forEach(row => {
                const dateCell = row.cells[6]; // Member Since column
                if (dateCell) {
                    const cellDate = new Date(dateCell.textContent);
                    const now = new Date();
                    let show = true;
                    
                    switch(filterValue) {
                        case 'today':
                            show = cellDate.toDateString() === now.toDateString();
                            break;
                        case 'week':
                            const weekAgo = new Date(now.getTime() - 7 * 24 * 60 * 60 * 1000);
                            show = cellDate >= weekAgo;
                            break;
                        case 'month':
                            const monthAgo = new Date(now.getTime() - 30 * 24 * 60 * 60 * 1000);
                            show = cellDate >= monthAgo;
                            break;
                        case 'year':
                            const yearAgo = new Date(now.getTime() - 365 * 24 * 60 * 60 * 1000);
                            show = cellDate >= yearAgo;
                            break;
                    }
                    
                    row.style.display = show ? '' : 'none';
                }
            });
        });
    </script>
</body>
</html>