<?php
require_once '../includes/auth.php';
require_once '../includes/functions.php';

$adminFunctions = new AdminFunctions();

// Handle actions
$action = $_GET['action'] ?? 'list';
$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    switch ($action) {
        case 'update_status':
            $reviewId = (int)$_POST['review_id'];
            $status = $_POST['status'];
            
            if ($adminFunctions->updateReviewStatus($reviewId, $status)) {
                $message = 'Review status updated successfully!';
            } else {
                $error = 'Failed to update review status.';
            }
            break;
            
        case 'delete':
            $reviewId = (int)$_POST['review_id'];
            
            if ($adminFunctions->deleteReview($reviewId)) {
                $message = 'Review deleted successfully!';
            } else {
                $error = 'Failed to delete review.';
            }
            break;
    }
}

// Get all reviews with pagination
$page = (int)($_GET['page'] ?? 1);
$limit = 20;
$offset = ($page - 1) * $limit;

$reviews = $adminFunctions->getAllReviews($limit, $offset);
$totalReviews = $adminFunctions->getTotalReviewsCount();
$totalPages = ceil($totalReviews / $limit);

// Get review statistics
$stats = $adminFunctions->getReviewStatistics();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Feedback Management - Admin Panel</title>
    <link rel="stylesheet" href="../assets/css/admin.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        /* Minimalist Table Design */
        .table-container {
            background: white;
            border-radius: 12px;
            overflow: hidden;
            box-shadow: 0 1px 3px rgba(0,0,0,0.08);
        }
        
        .admin-table {
            width: 100%;
            border-collapse: separate;
            border-spacing: 0;
        }
        
        .admin-table thead {
            background: #fafafa;
            border-bottom: 1px solid #f0f0f0;
        }
        
        .admin-table th {
            padding: 16px 20px;
            text-align: left;
            font-weight: 600;
            font-size: 13px;
            color: #666;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .admin-table tbody tr {
            border-bottom: 1px solid #f5f5f5;
            transition: background 0.2s ease;
        }
        
        .admin-table tbody tr:last-child {
            border-bottom: none;
        }
        
        .admin-table tbody tr:hover {
            background: #fafafa;
        }
        
        .admin-table td {
            padding: 20px;
            vertical-align: middle;
            font-size: 14px;
            color: #333;
        }
        
        /* Product Info */
        .product-info {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .product-thumb {
            width: 48px;
            height: 48px;
            object-fit: cover;
            border-radius: 8px;
            border: 1px solid #f0f0f0;
        }
        
        .product-details {
            display: flex;
            flex-direction: column;
            gap: 4px;
        }
        
        .product-details strong {
            font-size: 14px;
            font-weight: 600;
            color: #222;
            line-height: 1.4;
        }
        
        .product-details small {
            font-size: 12px;
            color: #999;
            font-weight: 400;
        }
        
        /* Rating */
        .rating-display {
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .rating-display i {
            font-size: 14px;
            color: #ffd700;
        }
        
        .rating-number {
            font-size: 14px;
            font-weight: 600;
            color: #333;
            margin-left: 4px;
        }
        
        /* Comment */
        .comment-preview {
            max-width: 400px;
        }
        
        .comment-preview strong {
            display: block;
            font-size: 14px;
            color: #222;
            margin-bottom: 4px;
        }
        
        .comment-text {
            font-size: 13px;
            color: #666;
            line-height: 1.6;
        }
        
        .no-comment {
            font-size: 13px;
            color: #ccc;
            font-style: italic;
        }
        
        /* Date */
        .date-info {
            display: flex;
            flex-direction: column;
            gap: 4px;
        }
        
        .date-info .date {
            font-size: 14px;
            font-weight: 500;
            color: #333;
        }
        
        .date-info .time {
            font-size: 12px;
            color: #999;
        }
        
        /* Action Button */
        .btn-sm {
            padding: 8px 12px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 14px;
            transition: all 0.2s ease;
            display: inline-flex;
            align-items: center;
            justify-content: center;
        }
        
        .btn-danger {
            background: #fff;
            color: #dc3545;
            border: 1px solid #f0f0f0;
        }
        
        .btn-danger:hover {
            background: #dc3545;
            color: white;
            border-color: #dc3545;
            transform: translateY(-1px);
            box-shadow: 0 2px 8px rgba(220, 53, 69, 0.2);
        }
        
        .btn-danger i {
            font-size: 14px;
        }
        
        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 60px 20px;
        }
        
        .empty-state i {
            font-size: 48px;
            color: #e0e0e0;
            margin-bottom: 16px;
        }
        
        .empty-state h3 {
            font-size: 18px;
            color: #666;
            margin-bottom: 8px;
        }
        
        .empty-state p {
            font-size: 14px;
            color: #999;
        }
        
        /* Minimalist Pagination */
        .pagination {
            display: flex;
            justify-content: center;
            align-items: center;
            gap: 16px;
            margin-top: 24px;
            padding: 20px;
        }
        
        .pagination .btn-outline {
            padding: 10px 20px;
            border: 1px solid #e0e0e0;
            background: white;
            color: #666;
            border-radius: 8px;
            text-decoration: none;
            font-size: 14px;
            font-weight: 500;
            transition: all 0.2s ease;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }
        
        .pagination .btn-outline:hover {
            border-color: #333;
            color: #333;
            transform: translateY(-1px);
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        }
        
        .pagination-info {
            font-size: 14px;
            color: #666;
            padding: 0 16px;
        }
        
        /* Minimalist Alerts */
        .alert {
            padding: 16px 20px;
            border-radius: 10px;
            margin-bottom: 20px;
            font-size: 14px;
            display: flex;
            align-items: center;
            gap: 12px;
            border: 1px solid;
        }
        
        .alert-success {
            background: #f0f9f4;
            color: #0f5132;
            border-color: #d1e7dd;
        }
        
        .alert-success i {
            color: #198754;
        }
        
        .alert-error {
            background: #fff5f5;
            color: #842029;
            border-color: #f5c2c7;
        }
        
        .alert-error i {
            color: #dc3545;
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="sidebar-header">
            <img src="../../images/LOGO.png" alt="Logo" class="sidebar-logo">
            <h3>Admin Panel</h3>
        </div>
        
        <nav class="sidebar-nav">
            <a href="../index.php" class="nav-item">
                <i class="fas fa-th-large"></i>
                <span>Overview</span>
            </a>
            <a href="products.php" class="nav-item">
                <i class="fas fa-box"></i>
                <span>Products</span>
            </a>
            <a href="orders.php" class="nav-item">
                <i class="fas fa-shopping-cart"></i>
                <span>Orders</span>
            </a>
            <a href="delivered.php" class="nav-item">
                <i class="fas fa-check-circle"></i>
                <span>Delivered</span>
            </a>
            <a href="customers.php" class="nav-item">
                <i class="fas fa-users"></i>
                <span>Customers</span>
            </a>
            <a href="feedback.php" class="nav-item active">
                <i class="fas fa-comments"></i>
                <span>Feedback</span>
            </a>
            <a href="shipping.php" class="nav-item">
                <i class="fas fa-shipping-fast"></i>
                <span>Shipping</span>
            </a>
            <a href="banners.php" class="nav-item">
                <i class="fas fa-image"></i>
                <span>Banners</span>
            </a>
            <a href="presets.php" class="nav-item">
                <i class="fas fa-cogs"></i>
                <span>Field Presets</span>
            </a>
            <a href="vouchers.php" class="nav-item">
                <i class="fas fa-ticket-alt"></i>
                <span>Voucher Management</span>
            </a>
            <a href="notifications.php" class="nav-item">
                <i class="fas fa-bell"></i>
                <span>Push Notifications</span>
            </a>
            <a href="analytics.php" class="nav-item">
                <i class="fas fa-chart-bar"></i>
                <span>Analytics</span>
            </a>
            <a href="settings.php" class="nav-item">
                <i class="fas fa-cog"></i>
                <span>Settings</span>
            </a>
        </nav>
        
        <div class="sidebar-footer">
            <div class="user-info">
                <i class="fas fa-user-circle"></i>
                <span><?php echo htmlspecialchars($adminAuth->getAdminName()); ?></span>
            </div>
            <a href="../logout.php" class="logout-btn">
                <i class="fas fa-sign-out-alt"></i>
                <span>Logout</span>
            </a>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <!-- Header -->
        <header class="header">
            <div class="header-left">
                <button class="sidebar-toggle">
                    <i class="fas fa-bars"></i>
                </button>
                <h1>Feedback Management</h1>
            </div>
            <div class="header-right">
                <div class="header-actions">
                    <a href="../../index.php" target="_blank" class="btn btn-outline">
                        <i class="fas fa-external-link-alt"></i>
                        View Store
                    </a>
                </div>
            </div>
        </header>

        <!-- Content -->
        <div class="content">

        <!-- Messages -->
        <?php if ($message): ?>
            <div class="alert alert-success">
                <i class="fas fa-check-circle"></i> <?php echo $message; ?>
            </div>
        <?php endif; ?>

        <?php if ($error): ?>
            <div class="alert alert-error">
                <i class="fas fa-exclamation-circle"></i> <?php echo $error; ?>
            </div>
        <?php endif; ?>

            <!-- Reviews Table -->
            <div class="table-container">
                <table class="admin-table">
                    <thead>
                        <tr>
                            <th>Product</th>
                            <th>Rating</th>
                            <th>Comment</th>
                            <th>Date</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($reviews)): ?>
                            <tr>
                                <td colspan="5" class="text-center">
                                    <div class="empty-state">
                                        <i class="fas fa-comments"></i>
                                        <h3>No reviews found</h3>
                                        <p>There are no reviews to display at the moment.</p>
                                    </div>
                                </td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($reviews as $review): ?>
                                <tr data-review-id="<?php echo $review['id']; ?>">
                                    <td>
                                        <div class="product-info">
                                            <?php 
                                            // Handle both cases: with or without 'images/' prefix
                                            $imageFile = $review['product_image'] ?? '';
                                            if (!empty($imageFile)) {
                                                // If it already starts with 'images/', use as is
                                                if (strpos($imageFile, 'images/') === 0) {
                                                    $imagePath = '../../' . $imageFile;
                                                } else {
                                                    $imagePath = '../../images/' . $imageFile;
                                                }
                                            } else {
                                                $imagePath = '../../images/placeholder.svg';
                                            }
                                            ?>
                                            <img src="<?php echo $imagePath; ?>" 
                                                 alt="<?php echo htmlspecialchars($review['product_name'] ?? 'Product'); ?>" 
                                                 class="product-thumb"
                                                 onerror="this.onerror=null; this.style.display='none';">
                                            <div class="product-details">
                                                <strong><?php echo htmlspecialchars($review['product_name']); ?></strong>
                                                <small><?php echo htmlspecialchars($review['product_brand']); ?></small>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="rating-display">
                                            <?php for ($i = 1; $i <= 5; $i++): ?>
                                                <i class="fas fa-star <?php echo $i <= $review['rating'] ? 'star-filled' : 'star-empty'; ?>"></i>
                                            <?php endfor; ?>
                                            <span class="rating-number"><?php echo $review['rating']; ?></span>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="comment-preview">
                                            <?php if (!empty($review['title'])): ?>
                                                <strong><?php echo htmlspecialchars($review['title']); ?></strong><br>
                                            <?php endif; ?>
                                            <?php if (!empty($review['comment'])): ?>
                                                <span class="comment-text"><?php echo htmlspecialchars(substr($review['comment'], 0, 150)); ?><?php echo strlen($review['comment']) > 150 ? '...' : ''; ?></span>
                                            <?php else: ?>
                                                <span class="no-comment">No comment</span>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="date-info">
                                            <span class="date"><?php echo date('M j, Y', strtotime($review['created_at'])); ?></span>
                                            <small class="time"><?php echo date('g:i A', strtotime($review['created_at'])); ?></small>
                                        </div>
                                    </td>
                                    <td>
                                        <button class="btn btn-sm btn-danger" onclick="deleteReview(<?php echo $review['id']; ?>)" title="Delete Review">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>

        <!-- Pagination -->
        <?php if ($totalPages > 1): ?>
            <div class="pagination">
                <?php if ($page > 1): ?>
                    <a href="?page=<?php echo $page - 1; ?>" class="btn btn-outline">
                        <i class="fas fa-chevron-left"></i> Previous
                    </a>
                <?php endif; ?>
                
                <span class="pagination-info">
                    Page <?php echo $page; ?> of <?php echo $totalPages; ?>
                    (<?php echo number_format($totalReviews); ?> total reviews)
                </span>
                
                <?php if ($page < $totalPages): ?>
                    <a href="?page=<?php echo $page + 1; ?>" class="btn btn-outline">
                        Next <i class="fas fa-chevron-right"></i>
                    </a>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </div>

    <!-- Delete Confirmation Modal - Completely Inline Styled -->
    <div id="deleteModal" style="display:none;position:fixed;top:0;left:0;width:100%;height:100%;background:rgba(0,0,0,0.8);z-index:999999999;justify-content:center;align-items:center;">
        <div style="background:white;padding:30px;border-radius:10px;max-width:500px;width:90%;box-shadow:0 10px 50px rgba(0,0,0,0.5);">
            <div style="display:flex;justify-content:space-between;align-items:center;margin-bottom:20px;border-bottom:2px solid #eee;padding-bottom:15px;">
                <h3 style="margin:0;font-size:20px;color:#333;">Confirm Delete</h3>
                <button onclick="closeModal()" style="background:none;border:none;font-size:28px;cursor:pointer;color:#666;padding:0;width:30px;height:30px;">&times;</button>
            </div>
            <div style="margin-bottom:25px;">
                <p style="margin:0;color:#666;font-size:16px;line-height:1.5;">Are you sure you want to delete this review? This action cannot be undone.</p>
            </div>
            <div style="display:flex;gap:10px;justify-content:flex-end;">
                <button onclick="closeModal()" style="padding:12px 24px;border:1px solid #ddd;background:white;border-radius:6px;cursor:pointer;font-size:14px;font-weight:600;">Cancel</button>
                <button onclick="confirmDelete()" style="padding:12px 24px;border:none;background:#dc3545;color:white;border-radius:6px;cursor:pointer;font-size:14px;font-weight:600;">
                    <i class="fas fa-trash"></i> Delete
                </button>
            </div>
        </div>
    </div>

    <script src="../assets/js/admin.js"></script>
    <script>
        let currentDeleteId = null;

        // Simple delete review function
        function deleteReview(reviewId) {
            currentDeleteId = reviewId;
            const modal = document.getElementById('deleteModal');
            if (modal) {
                modal.style.display = 'flex';
                console.log('Modal opened for review ID:', reviewId);
            } else {
                alert('Error: Modal not found');
            }
        }

        // Close modal
        function closeModal() {
            const modal = document.getElementById('deleteModal');
            if (modal) {
                modal.style.display = 'none';
            }
            currentDeleteId = null;
        }

        // Confirm delete
        function confirmDelete() {
            if (!currentDeleteId) return;
            
            const formData = new FormData();
            formData.append('review_id', currentDeleteId);
            
            fetch('../ajax/feedback.php?action=delete', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('Review deleted successfully!');
                    closeModal();
                    location.reload();
                } else {
                    alert(data.error || 'Failed to delete review');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error deleting review');
            });
        }
        
        // Simple notification function (fallback)
        function showNotification(message, type) {
            alert(message);
        }

        // Close modal when clicking outside
        document.getElementById('deleteModal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeModal();
            }
        });
    </script>
</body>
</html>