<?php
require_once '../includes/auth.php';
require_once '../includes/functions.php';
require_once '../../models/FirebaseNotification.php';

$adminAuth->requireLogin();

$firebaseNotification = new FirebaseNotification();
$stats = $firebaseNotification->getNotificationStats();
$users = $firebaseNotification->getAllUsers();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Push Notifications - Admin Panel</title>
    <link rel="stylesheet" href="../assets/css/admin.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .notification-form {
            background: white;
            border-radius: 8px;
            padding: 24px;
            margin-bottom: 24px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #333;
        }
        
        .form-group input,
        .form-group textarea,
        .form-group select {
            width: 100%;
            padding: 12px;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 14px;
            transition: border-color 0.3s;
        }
        
        .form-group input:focus,
        .form-group textarea:focus,
        .form-group select:focus {
            outline: none;
            border-color: #000;
        }
        
        .form-group textarea {
            resize: vertical;
            min-height: 100px;
        }
        
        .target-options {
            display: flex;
            gap: 16px;
            margin-bottom: 20px;
        }
        
        .target-option {
            flex: 1;
            padding: 16px;
            border: 2px solid #ddd;
            border-radius: 8px;
            cursor: pointer;
            transition: all 0.3s;
            text-align: center;
        }
        
        .target-option:hover {
            border-color: #000;
        }
        
        .target-option.active {
            border-color: #000;
            background-color: #f8f9fa;
        }
        
        .target-option input[type="radio"] {
            display: none;
        }
        
        .target-option i {
            font-size: 24px;
            margin-bottom: 8px;
            display: block;
        }
        
        .user-select {
            display: none;
        }
        
        .user-select.active {
            display: block;
        }
        
        .user-search-container {
            position: relative;
        }
        
        .user-search-input {
            width: 100%;
            padding: 12px 40px 12px 12px;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 14px;
            transition: border-color 0.3s;
        }
        
        .user-search-input:focus {
            outline: none;
            border-color: #000;
        }
        
        .user-search-icon {
            position: absolute;
            right: 12px;
            top: 50%;
            transform: translateY(-50%);
            color: #666;
            pointer-events: none;
        }
        
        .user-options {
            position: absolute;
            top: 100%;
            left: 0;
            right: 0;
            background: white;
            border: 1px solid #ddd;
            border-top: none;
            border-radius: 0 0 6px 6px;
            max-height: 200px;
            overflow-y: auto;
            z-index: 1000;
            display: none;
        }
        
        .user-options.active {
            display: block;
        }
        
        .user-option {
            padding: 12px;
            cursor: pointer;
            border-bottom: 1px solid #eee;
            transition: background-color 0.2s;
        }
        
        .user-option:hover {
            background-color: #f8f9fa;
        }
        
        .user-option:last-child {
            border-bottom: none;
        }
        
        .user-option.hidden {
            display: none;
        }
        
        .user-name {
            font-weight: 600;
            color: #333;
        }
        
        .user-email {
            font-size: 12px;
            color: #666;
            margin-top: 2px;
        }
        
        .no-users-found {
            padding: 12px;
            text-align: center;
            color: #666;
            font-style: italic;
        }
        
        .btn-group {
            display: flex;
            gap: 12px;
            margin-top: 24px;
        }
        
        .btn {
            padding: 12px 24px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 14px;
            font-weight: 600;
            transition: all 0.3s;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }
        
        .btn-primary {
            background-color: #000;
            color: white;
        }
        
        .btn-primary:hover {
            background-color: #333;
        }
        
        .btn-secondary {
            background-color: #6c757d;
            color: white;
        }
        
        .btn-secondary:hover {
            background-color: #5a6268;
        }
        
        .btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 16px;
            margin-bottom: 24px;
        }
        
        .stat-card {
            background: white;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            text-align: center;
        }
        
        .stat-card h3 {
            font-size: 24px;
            margin: 0 0 8px 0;
            color: #000;
        }
        
        .stat-card p {
            margin: 0;
            color: #666;
            font-size: 14px;
        }
        
        .history-section {
            background: white;
            border-radius: 8px;
            padding: 24px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .history-header {
            display: flex;
            justify-content: between;
            align-items: center;
            margin-bottom: 20px;
        }
        
        .history-header h3 {
            margin: 0;
        }
        
        .table-container {
            overflow-x: auto;
        }
        
        .data-table {
            width: 100%;
            border-collapse: collapse;
        }
        
        .data-table th,
        .data-table td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #eee;
        }
        
        .data-table th {
            background-color: #f8f9fa;
            font-weight: 600;
            color: #333;
        }
        
        .status-badge {
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 12px;
            font-weight: 600;
        }
        
        .status-success {
            background-color: #d4edda;
            color: #155724;
        }
        
        .status-error {
            background-color: #f8d7da;
            color: #721c24;
        }
        
        .loading {
            display: none;
            text-align: center;
            padding: 20px;
        }
        
        .loading.active {
            display: block;
        }
        
        .alert {
            padding: 12px 16px;
            border-radius: 6px;
            margin-bottom: 20px;
            display: none;
        }
        
        .alert-success {
            background-color: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .alert-error {
            background-color: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .alert.active {
            display: block;
        }
        
        .pagination {
            display: flex;
            justify-content: center;
            align-items: center;
            gap: 8px;
            margin-top: 20px;
        }
        
        .pagination button {
            padding: 8px 12px;
            border: 1px solid #ddd;
            background: white;
            cursor: pointer;
            border-radius: 4px;
        }
        
        .pagination button:hover:not(:disabled) {
            background-color: #f8f9fa;
        }
        
        .pagination button:disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }
        
        .pagination .current-page {
            padding: 8px 12px;
            background-color: #000;
            color: white;
            border-radius: 4px;
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="sidebar-header">
            <img src="../../images/LOGO.png" alt="Logo" class="sidebar-logo">
            <h3>Admin Panel</h3>
        </div>
        
        <nav class="sidebar-nav">
            <a href="../index.php" class="nav-item">
                <i class="fas fa-th-large"></i>
                <span>Overview</span>
            </a>
            <a href="products.php" class="nav-item">
                <i class="fas fa-box"></i>
                <span>Products</span>
            </a>
            <a href="orders.php" class="nav-item">
                <i class="fas fa-shopping-cart"></i>
                <span>Orders</span>
            </a>
            <a href="delivered.php" class="nav-item">
                <i class="fas fa-check-circle"></i>
                <span>Delivered</span>
            </a>
            <a href="customers.php" class="nav-item">
                <i class="fas fa-users"></i>
                <span>Customers</span>
            </a>
            <a href="feedback.php" class="nav-item">
                <i class="fas fa-comments"></i>
                <span>Feedback</span>
            </a>
            <a href="shipping.php" class="nav-item">
                <i class="fas fa-shipping-fast"></i>
                <span>Shipping</span>
            </a>
            <a href="banners.php" class="nav-item">
                <i class="fas fa-image"></i>
                <span>Banners</span>
            </a>
            <a href="presets.php" class="nav-item">
                <i class="fas fa-cogs"></i>
                <span>Field Presets</span>
            </a>
            <a href="vouchers.php" class="nav-item">
                <i class="fas fa-ticket-alt"></i>
                <span>Voucher Management</span>
            </a>
            <a href="notifications.php" class="nav-item active">
                <i class="fas fa-bell"></i>
                <span>Push Notifications</span>
            </a>
            <a href="analytics.php" class="nav-item">
                <i class="fas fa-chart-bar"></i>
                <span>Analytics</span>
            </a>
            <a href="settings.php" class="nav-item">
                <i class="fas fa-cog"></i>
                <span>Settings</span>
            </a>
        </nav>
        
        <div class="sidebar-footer">
            <div class="user-info">
                <i class="fas fa-user-circle"></i>
                <span><?php echo htmlspecialchars($adminAuth->getAdminName()); ?></span>
            </div>
            <a href="../logout.php" class="logout-btn">
                <i class="fas fa-sign-out-alt"></i>
                <span>Logout</span>
            </a>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <!-- Header -->
        <header class="header">
            <div class="header-left">
                <button class="sidebar-toggle">
                    <i class="fas fa-bars"></i>
                </button>
                <h1>Push Notifications</h1>
            </div>
            <div class="header-right">
                <div class="header-actions">
                    <button id="testConnection" class="btn btn-secondary">
                        <i class="fas fa-wifi"></i>
                        Test Connection
                    </button>
                    <a href="../../index.php" target="_blank" class="btn btn-outline">
                        <i class="fas fa-external-link-alt"></i>
                        View Store
                    </a>
                </div>
            </div>
        </header>

        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <!-- Stats Cards -->
            <div class="stats-grid">
                <div class="stat-card">
                    <h3><?php echo number_format($stats['total']); ?></h3>
                    <p>Total Notifications</p>
                </div>
                <div class="stat-card">
                    <h3><?php echo number_format($stats['successful']); ?></h3>
                    <p>Successful</p>
                </div>
                <div class="stat-card">
                    <h3><?php echo number_format($stats['failed']); ?></h3>
                    <p>Failed</p>
                </div>
                <div class="stat-card">
                    <h3><?php echo number_format($stats['today']); ?></h3>
                    <p>Sent Today</p>
                </div>
                <div class="stat-card">
                    <h3><?php echo number_format($stats['this_week']); ?></h3>
                    <p>This Week</p>
                </div>
            </div>

            <!-- Notification Form -->
            <div class="notification-form">
                <h3>Send Push Notification</h3>
                
                <div class="alert" id="alert"></div>
                
                <form id="notificationForm">
                    <div class="form-group">
                        <label for="title">Notification Title *</label>
                        <input type="text" id="title" name="title" required maxlength="255" placeholder="Enter notification title">
                    </div>
                    
                    <div class="form-group">
                        <label for="body">Notification Body *</label>
                        <textarea id="body" name="body" required maxlength="1000" placeholder="Enter notification message"></textarea>
                    </div>
                    
                    <div class="target-options">
                        <label class="target-option active" for="target_all">
                            <input type="radio" id="target_all" name="target" value="all" checked>
                            <i class="fas fa-globe"></i>
                            <div>All Users</div>
                        </label>
                        <label class="target-option" for="target_user">
                            <input type="radio" id="target_user" name="target" value="user">
                            <i class="fas fa-user"></i>
                            <div>Specific User</div>
                        </label>
                    </div>
                    
                    <div class="form-group user-select" id="userSelect">
                        <label for="user_search">Select User</label>
                        <div class="user-search-container">
                            <input type="text" id="user_search" class="user-search-input" placeholder="Search users by name or email..." autocomplete="off">
                            <i class="fas fa-search user-search-icon"></i>
                            <div class="user-options" id="user_options">
                                <div class="user-option" data-value="" data-text="Choose a user...">
                                    <div class="user-name">Choose a user...</div>
                                </div>
                                <?php foreach ($users as $user): ?>
                                    <div class="user-option" data-value="<?php echo $user['id']; ?>" data-text="<?php echo htmlspecialchars($user['first_name'] . ' ' . $user['last_name'] . ' (' . $user['email'] . ')'); ?>">
                                        <div class="user-name"><?php echo htmlspecialchars($user['first_name'] . ' ' . $user['last_name']); ?></div>
                                        <div class="user-email"><?php echo htmlspecialchars($user['email']); ?></div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                        <input type="hidden" id="user_id" name="user_id" value="">
                    </div>
                    
                    <div class="btn-group">
                        <button type="submit" class="btn btn-primary" id="sendBtn">
                            <i class="fas fa-paper-plane"></i>
                            Send Notification
                        </button>
                        <button type="button" class="btn btn-secondary" id="clearBtn">
                            <i class="fas fa-times"></i>
                            Clear Form
                        </button>
                    </div>
                </form>
            </div>

            <!-- Notification History -->
            <div class="history-section">
                <div class="history-header">
                    <h3>Notification History</h3>
                </div>
                
                <div class="loading" id="historyLoading">
                    <i class="fas fa-spinner fa-spin"></i> Loading...
                </div>
                
                <div class="table-container" id="historyTable">
                    <!-- History table will be loaded here -->
                </div>
                
                <div class="pagination" id="pagination">
                    <!-- Pagination will be loaded here -->
                </div>
            </div>
        </div>
    </div>

    <script src="../assets/js/admin.js"></script>
    <script>
        let currentPage = 1;
        
        // Target selection
        document.querySelectorAll('input[name="target"]').forEach(radio => {
            radio.addEventListener('change', function() {
                document.querySelectorAll('.target-option').forEach(option => {
                    option.classList.remove('active');
                });
                this.closest('.target-option').classList.add('active');
                
                const userSelect = document.getElementById('userSelect');
                if (this.value === 'user') {
                    userSelect.classList.add('active');
                    document.getElementById('user_id').required = true;
                } else {
                    userSelect.classList.remove('active');
                    document.getElementById('user_id').required = false;
                }
            });
        });
        
        // User search and dropdown functionality
        const userSearch = document.getElementById('user_search');
        const userOptions = document.getElementById('user_options');
        const userIdInput = document.getElementById('user_id');
        const userOptionElements = document.querySelectorAll('.user-option');
        
        let isDropdownOpen = false;
        
        // Show dropdown when search input is focused
        userSearch.addEventListener('focus', function() {
            openDropdown();
        });
        
        // Search functionality
        userSearch.addEventListener('input', function() {
            const searchTerm = this.value.toLowerCase();
            let visibleCount = 0;
            
            userOptionElements.forEach(option => {
                const userName = option.querySelector('.user-name').textContent.toLowerCase();
                const userEmail = option.querySelector('.user-email') ? option.querySelector('.user-email').textContent.toLowerCase() : '';
                const searchText = option.getAttribute('data-text').toLowerCase();
                
                if (searchText.includes(searchTerm) || userName.includes(searchTerm) || userEmail.includes(searchTerm)) {
                    option.classList.remove('hidden');
                    visibleCount++;
                } else {
                    option.classList.add('hidden');
                }
            });
            
            // Show/hide "no users found" message
            const noUsersFound = userOptions.querySelector('.no-users-found');
            if (visibleCount === 0 && searchTerm.length > 0) {
                if (!noUsersFound) {
                    const noUsersDiv = document.createElement('div');
                    noUsersDiv.className = 'no-users-found';
                    noUsersDiv.textContent = 'No users found';
                    userOptions.appendChild(noUsersDiv);
                }
            } else if (noUsersFound) {
                noUsersFound.remove();
            }
            
            // Keep dropdown open while searching
            if (!isDropdownOpen) {
                openDropdown();
            }
        });
        
        // Close dropdown when clicking outside
        document.addEventListener('click', function(e) {
            if (!userSearch.contains(e.target) && !userOptions.contains(e.target)) {
                closeDropdown();
            }
        });
        
        // User option selection
        userOptions.addEventListener('click', function(e) {
            const option = e.target.closest('.user-option');
            if (option) {
                const value = option.getAttribute('data-value');
                const text = option.getAttribute('data-text');
                
                userIdInput.value = value;
                userSearch.value = text;
                
                // Clear search and show all options
                userOptionElements.forEach(opt => opt.classList.remove('hidden'));
                const noUsersFound = userOptions.querySelector('.no-users-found');
                if (noUsersFound) noUsersFound.remove();
                
                closeDropdown();
            }
        });
        
        function openDropdown() {
            userOptions.classList.add('active');
            isDropdownOpen = true;
        }
        
        function closeDropdown() {
            userOptions.classList.remove('active');
            isDropdownOpen = false;
        }
        
        // Form submission
        document.getElementById('notificationForm').addEventListener('submit', function(e) {
            e.preventDefault();
            
            const form = this; // Save form reference
            const formData = new FormData(form);
            const target = formData.get('target');
            const action = target === 'all' ? 'send_to_all' : 'send_to_user';
            
            if (target === 'user' && !userIdInput.value) {
                showAlert('Please select a user', 'error');
                return;
            }
            
            formData.append('action', action);
            
            const sendBtn = document.getElementById('sendBtn');
            sendBtn.disabled = true;
            sendBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Sending...';
            
            fetch('../ajax/notifications.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showAlert('Notification sent successfully!', 'success');
                    form.reset();
                    document.querySelectorAll('.target-option').forEach(option => {
                        option.classList.remove('active');
                    });
                    document.getElementById('target_all').closest('.target-option').classList.add('active');
                    document.getElementById('userSelect').classList.remove('active');
                    loadHistory(1);
                } else {
                    showAlert(data.message || 'Failed to send notification', 'error');
                }
            })
            .catch(error => {
                showAlert('An error occurred while sending notification', 'error');
                console.error('Error:', error);
            })
            .finally(() => {
                sendBtn.disabled = false;
                sendBtn.innerHTML = '<i class="fas fa-paper-plane"></i> Send Notification';
            });
        });
        
        // Clear form
        document.getElementById('clearBtn').addEventListener('click', function() {
            document.getElementById('notificationForm').reset();
            document.querySelectorAll('.target-option').forEach(option => {
                option.classList.remove('active');
            });
            document.getElementById('target_all').closest('.target-option').classList.add('active');
            document.getElementById('userSelect').classList.remove('active');
            document.getElementById('user_id').required = false;
            
            // Reset user selection
            userIdInput.value = '';
            userSearch.value = '';
            userOptionElements.forEach(opt => opt.classList.remove('hidden'));
            const noUsersFound = userOptions.querySelector('.no-users-found');
            if (noUsersFound) noUsersFound.remove();
            closeDropdown();
            
            hideAlert();
        });
        
        // Test connection
        document.getElementById('testConnection').addEventListener('click', function() {
            const btn = this;
            btn.disabled = true;
            btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Testing...';
            
            fetch('../ajax/notifications.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'action=test_connection'
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showAlert('Connection test successful!', 'success');
                } else {
                    showAlert('Connection test failed: ' + data.message, 'error');
                }
            })
            .catch(error => {
                showAlert('Connection test failed', 'error');
                console.error('Error:', error);
            })
            .finally(() => {
                btn.disabled = false;
                btn.innerHTML = '<i class="fas fa-wifi"></i> Test Connection';
            });
        });
        
        // Load notification history
        function loadHistory(page) {
            currentPage = page;
            const loading = document.getElementById('historyLoading');
            const table = document.getElementById('historyTable');
            const pagination = document.getElementById('pagination');
            
            loading.classList.add('active');
            table.innerHTML = '';
            pagination.innerHTML = '';
            
            fetch('../ajax/notifications.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `action=get_history&page=${page}`
            })
            .then(response => response.json())
            .then(data => {
                loading.classList.remove('active');
                
                if (data.success) {
                    renderHistoryTable(data.data);
                    renderPagination(data.pagination);
                } else {
                    table.innerHTML = '<p>Failed to load history</p>';
                }
            })
            .catch(error => {
                loading.classList.remove('active');
                table.innerHTML = '<p>Error loading history</p>';
                console.error('Error:', error);
            });
        }
        
        // Render history table
        function renderHistoryTable(history) {
            const table = document.getElementById('historyTable');
            
            if (history.length === 0) {
                table.innerHTML = '<p>No notifications sent yet.</p>';
                return;
            }
            
            let html = `
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Title</th>
                            <th>Body</th>
                            <th>Target</th>
                            <th>Admin</th>
                            <th>Status</th>
                            <th>Date</th>
                        </tr>
                    </thead>
                    <tbody>
            `;
            
            history.forEach(notification => {
                const statusClass = notification.success ? 'status-success' : 'status-error';
                const statusText = notification.success ? 'Success' : 'Failed';
                const date = new Date(notification.created_at).toLocaleString();
                
                html += `
                    <tr>
                        <td>${escapeHtml(notification.title)}</td>
                        <td>${escapeHtml(notification.body.substring(0, 100))}${notification.body.length > 100 ? '...' : ''}</td>
                        <td>${notification.target === 'all' ? 'All Users' : `User #${notification.target}`}</td>
                        <td>${notification.admin_name || 'System'}</td>
                        <td><span class="status-badge ${statusClass}">${statusText}</span></td>
                        <td>${date}</td>
                    </tr>
                `;
            });
            
            html += '</tbody></table>';
            table.innerHTML = html;
        }
        
        // Render pagination
        function renderPagination(pagination) {
            const container = document.getElementById('pagination');
            
            if (pagination.total_pages <= 1) {
                container.innerHTML = '';
                return;
            }
            
            let html = '';
            
            // Previous button
            html += `<button ${pagination.current_page <= 1 ? 'disabled' : ''} onclick="loadHistory(${pagination.current_page - 1})">
                        <i class="fas fa-chevron-left"></i>
                     </button>`;
            
            // Page numbers
            for (let i = 1; i <= pagination.total_pages; i++) {
                if (i === pagination.current_page) {
                    html += `<span class="current-page">${i}</span>`;
                } else {
                    html += `<button onclick="loadHistory(${i})">${i}</button>`;
                }
            }
            
            // Next button
            html += `<button ${pagination.current_page >= pagination.total_pages ? 'disabled' : ''} onclick="loadHistory(${pagination.current_page + 1})">
                        <i class="fas fa-chevron-right"></i>
                     </button>`;
            
            container.innerHTML = html;
        }
        
        // Utility functions
        function showAlert(message, type) {
            const alert = document.getElementById('alert');
            if (!alert) {
                console.error('Alert element not found');
                return;
            }
            alert.textContent = message;
            alert.className = `alert alert-${type} active`;
            
            setTimeout(() => {
                hideAlert();
            }, 5000);
        }
        
        function hideAlert() {
            const alert = document.getElementById('alert');
            if (!alert) return;
            alert.classList.remove('active');
        }
        
        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
        
        // Load initial history
        loadHistory(1);
    </script>
</body>
</html>