<?php
require_once '../includes/auth.php';
require_once '../includes/functions.php';

$adminAuth->requireLogin();

// Handle status updates
if ($_POST && isset($_POST['action']) && $_POST['action'] === 'update_status') {
    $orderId = $_POST['order_id'];
    $status = $_POST['status'];
    $trackingNumber = $_POST['tracking_number'] ?? '';
    
    if ($adminFunctions->updateOrderStatus($orderId, $status, $trackingNumber)) {
        // Send notification to customer about order status update
        try {
            require_once '../../models/FirebaseNotification.php';
            $firebaseNotification = new FirebaseNotification();
            
            // Get order details for notification
            $orderDetails = $adminFunctions->getOrderById($orderId);
            if ($orderDetails) {
                $shippingAddress = json_decode($orderDetails['shipping_address'], true);
                $customerName = $shippingAddress['first_name'] ?? 'Customer';
                
                // Create notification message based on status
                $notificationTitle = 'Order Status Update';
                $notificationBody = '';
                
                switch ($status) {
                    case 'processing':
                        $notificationBody = "Hi {$customerName}, your order #{$orderDetails['order_number']} is now being processed. We'll update you once it's ready for shipping.";
                        break;
                    case 'shipped':
                        if (!empty($trackingNumber)) {
                            $notificationBody = "Hi {$customerName}, great news! Your order #{$orderDetails['order_number']} has been shipped. Tracking number: {$trackingNumber}";
                        } else {
                            $notificationBody = "Hi {$customerName}, your order #{$orderDetails['order_number']} has been shipped and is on its way to you.";
                        }
                        break;
                    case 'delivered':
                        $notificationBody = "Hi {$customerName}, your order #{$orderDetails['order_number']} has been delivered. Thank you for your purchase!";
                        break;
                    case 'cancelled':
                        $notificationBody = "Hi {$customerName}, we're sorry to inform you that your order #{$orderDetails['order_number']} has been cancelled. Please contact us if you have any questions.";
                        break;
                    default:
                        $notificationBody = "Hi {$customerName}, your order #{$orderDetails['order_number']} status has been updated to " . ucfirst($status) . ".";
                }
                
                // Send notification to the specific customer
                $result = $firebaseNotification->sendNotificationToUser(
                    $notificationTitle,
                    $notificationBody,
                    $orderDetails['user_id'],
                    $adminAuth->getAdminName()
                );
                
                if ($result['success']) {
                    $success = 'Order status updated and customer notified successfully!';
                } else {
                    $success = 'Order status updated successfully, but failed to notify customer.';
                }
            } else {
                $success = 'Order status updated successfully!';
            }
        } catch (Exception $e) {
            error_log("Notification error: " . $e->getMessage());
            $success = 'Order status updated successfully, but failed to send notification.';
        }
    } else {
        $error = 'Error updating order status. Please try again.';
    }
}

// Get search and filter parameters
$search = $_GET['search'] ?? '';
$status_filter = $_GET['status'] ?? '';
$date_filter = $_GET['date'] ?? '';
$delivery_method_filter = $_GET['delivery_method'] ?? '';

// Get pagination parameters
$entries_per_page = (int)($_GET['entries'] ?? 50);
$current_page = (int)($_GET['page'] ?? 1);

// Validate entries per page
$allowed_entries = [50, 100, 250, 500];
if (!in_array($entries_per_page, $allowed_entries)) {
    $entries_per_page = 50;
}

// Calculate offset
$offset = ($current_page - 1) * $entries_per_page;

// Get total count for pagination
$total_orders = $adminFunctions->getOrdersCount($search, $status_filter, $date_filter, $delivery_method_filter);
$total_pages = ceil($total_orders / $entries_per_page);

// Get orders with filters and pagination
$orders = $adminFunctions->getAllOrders($search, $status_filter, $date_filter, $entries_per_page, $offset, $delivery_method_filter);

// Get today's processing and shipped orders
$todayStats = $adminFunctions->getTodayProcessingAndShippedOrders();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Orders Management - Admin Panel</title>
    <link rel="stylesheet" href="../assets/css/admin.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body>
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="sidebar-header">
            <img src="../../images/LOGO.png" alt="Logo" class="sidebar-logo">
            <h3>Admin Panel</h3>
        </div>
        
        <nav class="sidebar-nav">
            <a href="../index.php" class="nav-item">
                <i class="fas fa-th-large"></i>
                <span>Overview</span>
            </a>
            <a href="products.php" class="nav-item">
                <i class="fas fa-box"></i>
                <span>Products</span>
            </a>
            <a href="orders.php" class="nav-item active">
                <i class="fas fa-shopping-cart"></i>
                <span>Orders</span>
            </a>
            <a href="delivered.php" class="nav-item">
                <i class="fas fa-check-circle"></i>
                <span>Delivered</span>
            </a>
            <a href="customers.php" class="nav-item">
                <i class="fas fa-users"></i>
                <span>Customers</span>
            </a>
            <a href="feedback.php" class="nav-item">
                <i class="fas fa-comments"></i>
                <span>Feedback</span>
            </a>
            <a href="shipping.php" class="nav-item">
                <i class="fas fa-shipping-fast"></i>
                <span>Shipping</span>
            </a>
            <a href="banners.php" class="nav-item">
                <i class="fas fa-image"></i>
                <span>Banners</span>
            </a>
            <a href="presets.php" class="nav-item">
                <i class="fas fa-cogs"></i>
                <span>Field Presets</span>
            </a>
            <a href="vouchers.php" class="nav-item">
                <i class="fas fa-ticket-alt"></i>
                <span>Voucher Management</span>
            </a>
            <a href="notifications.php" class="nav-item">
                <i class="fas fa-bell"></i>
                <span>Push Notifications</span>
            </a>
            <a href="analytics.php" class="nav-item">
                <i class="fas fa-chart-bar"></i>
                <span>Analytics</span>
            </a>
            <a href="settings.php" class="nav-item">
                <i class="fas fa-cog"></i>
                <span>Settings</span>
            </a>
        </nav>
        
        <div class="sidebar-footer">
            <div class="user-info">
                <i class="fas fa-user-circle"></i>
                <span><?php echo htmlspecialchars($adminAuth->getAdminName()); ?></span>
            </div>
            <a href="../logout.php" class="logout-btn">
                <i class="fas fa-sign-out-alt"></i>
                <span>Logout</span>
            </a>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <!-- Header -->
        <header class="header">
            <div class="header-left">
                <button class="sidebar-toggle">
                    <i class="fas fa-bars"></i>
                </button>
                <h1>Orders Management</h1>
            </div>
            <div class="header-right">
                <div class="header-actions">
                    <button onclick="location.reload()" class="btn btn-outline" title="Refresh Orders">
                        <i class="fas fa-sync-alt"></i>
                        Refresh
                    </button>
                    <button onclick="exportOrders()" class="btn btn-outline">
                        <i class="fas fa-download"></i>
                        Export Orders
                    </button>
                    <a href="../index.php" target="_blank" class="btn btn-outline">
                        <i class="fas fa-external-link-alt"></i>
                        View Store
                    </a>
                </div>
            </div>
        </header>

        <!-- Page Content -->
        <div class="page-content">
            <?php if (isset($success)): ?>
                <div class="alert" style="background-color: #d4edda; color: #155724; border: 1px solid #c3e6cb;">
                    <i class="fas fa-check-circle"></i>
                    <?php echo htmlspecialchars($success); ?>
                </div>
            <?php endif; ?>
            
            <?php if (isset($error)): ?>
                <div class="alert alert-error">
                    <i class="fas fa-exclamation-circle"></i>
                    <?php echo htmlspecialchars($error); ?>
                </div>
            <?php endif; ?>

            <!-- Today's Stats Card -->
            <div class="stats-card">
                <div class="stats-header">
                    <h3>Today's Orders</h3>
                    <span class="stats-date"><?php echo date('M j, Y'); ?></span>
                </div>
                <div class="stats-content">
                    <div class="stat-item">
                        <div class="stat-number"><?php echo $todayStats['processing_count']; ?></div>
                        <div class="stat-label">Processing</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-number"><?php echo $todayStats['shipped_count']; ?></div>
                        <div class="stat-label">Shipped</div>
                    </div>
                </div>
            </div>

            <!-- Debug Info (remove in production) -->
            <?php if (isset($_GET['debug']) && $_GET['debug'] === '1'): ?>
            <div class="debug-info" style="background: #f8f9fa; padding: 15px; border-radius: 8px; margin-bottom: 20px; font-family: monospace; font-size: 12px;">
                <h4>Debug Information:</h4>
                <p><strong>Search:</strong> <?php echo htmlspecialchars($search); ?></p>
                <p><strong>Status Filter:</strong> <?php echo htmlspecialchars($status_filter); ?></p>
                <p><strong>Date Filter:</strong> <?php echo htmlspecialchars($date_filter); ?></p>
                <p><strong>Total Orders Found:</strong> <?php echo count($orders); ?></p>
                <p><strong>Recent Order Statuses:</strong></p>
                <ul>
                    <?php foreach (array_slice($orders, 0, 3) as $order): ?>
                        <li>Order #<?php echo $order['order_number']; ?>: <?php echo $order['status']; ?> (Updated: <?php echo $order['updated_at']; ?>)</li>
                    <?php endforeach; ?>
                </ul>
            </div>
            <?php endif; ?>

            <!-- Search and Filter -->
            <div class="search-filter">
                <form method="GET" id="filterForm" class="filter-form">
                    <input type="text" class="search-input" placeholder="Search orders..." 
                           name="search" value="<?php echo htmlspecialchars($search); ?>" id="orderSearch">
                    <select class="filter-select" name="status" id="statusFilter">
                        <option value="">All Status</option>
                        <option value="pending" <?php echo $status_filter === 'pending' ? 'selected' : ''; ?>>Pending</option>
                        <option value="processing" <?php echo $status_filter === 'processing' ? 'selected' : ''; ?>>Processing</option>
                        <option value="shipped" <?php echo $status_filter === 'shipped' ? 'selected' : ''; ?>>Shipped</option>
                        <option value="delivered" <?php echo $status_filter === 'delivered' ? 'selected' : ''; ?>>Delivered</option>
                        <option value="cancelled" <?php echo $status_filter === 'cancelled' ? 'selected' : ''; ?>>Cancelled</option>
                    </select>
                    <select class="filter-select" name="delivery_method" id="deliveryMethodFilter">
                        <option value="">All Delivery Methods</option>
                        <option value="pickup" <?php echo ($_GET['delivery_method'] ?? '') === 'pickup' ? 'selected' : ''; ?>>Pickup Orders</option>
                        <option value="shipping" <?php echo ($_GET['delivery_method'] ?? '') === 'shipping' ? 'selected' : ''; ?>>Shipping Orders</option>
                    </select>
                    <input type="date" class="filter-select" name="date" 
                           value="<?php echo htmlspecialchars($date_filter); ?>" id="dateFilter" placeholder="Filter by date">
                    <select class="filter-select" name="entries" id="entriesFilter">
                        <option value="50" <?php echo $entries_per_page === 50 ? 'selected' : ''; ?>>50 entries</option>
                        <option value="100" <?php echo $entries_per_page === 100 ? 'selected' : ''; ?>>100 entries</option>
                        <option value="250" <?php echo $entries_per_page === 250 ? 'selected' : ''; ?>>250 entries</option>
                        <option value="500" <?php echo $entries_per_page === 500 ? 'selected' : ''; ?>>500 entries</option>
                    </select>
                    <input type="hidden" name="page" value="<?php echo $current_page; ?>" id="pageInput">
                    <button type="submit" class="btn btn-primary filter-btn">
                        <i class="fas fa-search"></i> Filter
                    </button>
                    <?php if ($search || $status_filter || $date_filter || $delivery_method_filter): ?>
                        <a href="orders.php?entries=<?php echo $entries_per_page; ?>" class="btn btn-outline clear-btn">
                            <i class="fas fa-times"></i> Clear
                        </a>
                    <?php endif; ?>
                </form>
            </div>

            <!-- Pagination Info and Controls -->
            <div class="pagination-container">
                <div class="pagination-info">
                    <span>Showing <?php echo min($offset + 1, $total_orders); ?> to <?php echo min($offset + $entries_per_page, $total_orders); ?> of <?php echo $total_orders; ?> orders</span>
                </div>
                
                <?php if ($total_pages > 1): ?>
                <div class="pagination-controls">
                    <?php if ($current_page > 1): ?>
                        <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $current_page - 1])); ?>" 
                           class="btn btn-outline btn-sm">
                            <i class="fas fa-chevron-left"></i> Previous
                        </a>
                    <?php endif; ?>
                    
                    <?php
                    // Show page numbers
                    $start_page = max(1, $current_page - 2);
                    $end_page = min($total_pages, $current_page + 2);
                    
                    if ($start_page > 1): ?>
                        <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => 1])); ?>" 
                           class="btn btn-outline btn-sm">1</a>
                        <?php if ($start_page > 2): ?>
                            <span class="pagination-dots">...</span>
                        <?php endif; ?>
                    <?php endif; ?>
                    
                    <?php for ($i = $start_page; $i <= $end_page; $i++): ?>
                        <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>" 
                           class="btn btn-sm <?php echo $i === $current_page ? 'btn-primary' : 'btn-outline'; ?>">
                            <?php echo $i; ?>
                        </a>
                    <?php endfor; ?>
                    
                    <?php if ($end_page < $total_pages): ?>
                        <?php if ($end_page < $total_pages - 1): ?>
                            <span class="pagination-dots">...</span>
                        <?php endif; ?>
                        <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $total_pages])); ?>" 
                           class="btn btn-outline btn-sm"><?php echo $total_pages; ?></a>
                    <?php endif; ?>
                    
                    <?php if ($current_page < $total_pages): ?>
                        <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $current_page + 1])); ?>" 
                           class="btn btn-outline btn-sm">
                            Next <i class="fas fa-chevron-right"></i>
                        </a>
                    <?php endif; ?>
                </div>
                <?php endif; ?>
            </div>

            <!-- Orders Table -->
            <div class="table-container">
                <div class="table-content">
                    <table class="data-table" id="ordersTable">
                        <thead>
                            <tr>
                                <th>Order ID</th>
                                <th>Customer</th>
                                <th>Email</th>
                                <th>Total</th>
                                <th>Delivery Method</th>
                                <th>Status</th>
                                <th>Payment</th>
                                <th>Date</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($orders as $order): ?>
                            <tr>
                                <td>
                                    <strong><?php echo $order['order_number']; ?></strong>
                                </td>
                                <td>
                                    <?php 
                                    $shipping_address = json_decode($order['shipping_address'], true);
                                    if ($shipping_address) {
                                        echo htmlspecialchars($shipping_address['first_name'] . ' ' . $shipping_address['last_name']);
                                    } else {
                                        echo 'N/A';
                                    }
                                    ?>
                                </td>
                                <td>
                                    <?php 
                                    if ($shipping_address) {
                                        echo htmlspecialchars($shipping_address['email']);
                                    } else {
                                        echo 'N/A';
                                    }
                                    ?>
                                </td>
                                <td>RM <?php echo number_format($order['total_amount'], 2); ?></td>
                                <td>
                                    <?php
                                    // Get shipping method info from joined data
                                    $shipping_fee_type = $order['shipping_fee_type'] ?? null;
                                    $shipping_method_name = $order['shipping_method_name'] ?? 'Shipping';
                                    
                                    if ($shipping_fee_type === 'pickup') {
                                        $delivery_method = 'Pickup';
                                        $delivery_badge_class = 'delivery-pickup';
                                    } else {
                                        $delivery_method = $shipping_method_name;
                                        $delivery_badge_class = 'delivery-shipping';
                                    }
                                    ?>
                                    <span class="delivery-badge <?php echo $delivery_badge_class; ?>">
                                        <?php echo htmlspecialchars($delivery_method); ?>
                                    </span>
                                </td>
                                <td>
                                    <?php
                                    // Use consistent status display logic
                                    $display_order_status = $order['status'];
                                    $payment_status = $order['payment_status'] ?? 'pending';
                                    
                                    // If payment is successful, show processing status
                                    if ($payment_status === 'paid' && $order['status'] === 'awaiting_payment') {
                                        $display_order_status = 'processing';
                                    }
                                    
                                    // Ensure we show the actual database status, not a derived one
                                    $actual_status = $order['status'];
                                    ?>
                                    <span class="status status-<?php echo $actual_status; ?>">
                                        <?php 
                                        switch($actual_status) {
                                            case 'awaiting_payment': echo 'Awaiting Payment'; break;
                                            case 'pending': echo 'Pending'; break;
                                            case 'processing': echo 'Processing'; break;
                                            case 'shipped': echo 'Shipped'; break;
                                            case 'delivered': echo 'Delivered'; break;
                                            case 'cancelled': echo 'Cancelled'; break;
                                            default: echo ucfirst(str_replace('_', ' ', $actual_status));
                                        }
                                        ?>
                                    </span>
                                    <?php if ($actual_status !== $display_order_status): ?>
                                        <br><small style="color: #666; font-style: italic;">
                                            (Display: <?php echo ucfirst($display_order_status); ?>)
                                        </small>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo htmlspecialchars($order['payment_method']); ?></td>
                                <td><?php echo date('M j, Y', strtotime($order['created_at'])); ?></td>
                                <td>
                                    <div style="display: flex; gap: 5px;">
                                        <button onclick="viewOrder(<?php echo $order['id']; ?>)" 
                                                class="btn btn-sm btn-primary" title="View Details">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                        <button onclick="updateOrderStatus(<?php echo $order['id']; ?>)" 
                                                class="btn btn-sm btn-warning" title="Update Status">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Order Details Modal -->
    <div id="orderDetailsModal" class="modal">
        <div class="modal-content" style="max-width: 800px;">
            <div class="modal-header">
                <h3 class="modal-title">Order Details</h3>
                <span class="close" onclick="hideModal('orderDetailsModal')">&times;</span>
            </div>
            
            <div id="orderDetailsContent">
                <!-- Order details will be loaded here -->
            </div>
        </div>
    </div>

    <!-- Update Status Modal -->
    <div id="updateStatusModal" class="modal">
        <div class="modal-content" style="max-width: 700px;">
            <div class="modal-header">
                <h3 class="modal-title">Update Order Status</h3>
                <span class="close" onclick="hideModal('updateStatusModal')">&times;</span>
            </div>
            
            <!-- Order Summary Section -->
            <div id="orderSummaryContent">
                <!-- Order summary will be loaded here -->
            </div>
            
            <form method="POST" id="statusUpdateForm">
                <input type="hidden" name="action" value="update_status">
                <input type="hidden" name="order_id" id="updateOrderId">
                
                <div class="form-group-full">
                    <label for="status">New Status</label>
                    <select id="status" name="status" required>
                        <option value="pending">Pending</option>
                        <option value="processing">Processing</option>
                        <option value="shipped">Shipped</option>
                        <option value="delivered">Delivered</option>
                        <option value="cancelled">Cancelled</option>
                    </select>
                </div>
                
                <div class="form-group-full">
                    <label for="tracking_number">Tracking Number (Optional)</label>
                    <input type="text" id="tracking_number" name="tracking_number" 
                           placeholder="Enter tracking number - will auto-set status to 'Shipped'">
                    <small class="form-help">Entering a tracking number will automatically set status to 'Shipped'</small>
                    <div id="tracking-info" style="display: none; background: #e8f5e8; border: 1px solid #4caf50; border-radius: 4px; padding: 8px; margin-top: 8px; font-size: 12px; color: #2e7d32;">
                        <i class="fas fa-info-circle"></i>
                        <span>Tracking number already exists for this order</span>
                    </div>
                </div>
                
                <div class="form-actions">
                    <div class="notification-info" style="background: #e3f2fd; border: 1px solid #2196f3; border-radius: 6px; padding: 12px; margin-bottom: 16px; display: flex; align-items: center; gap: 8px;">
                        <i class="fas fa-bell" style="color: #1976d2;"></i>
                        <span style="color: #1976d2; font-size: 14px; font-weight: 500;">
                            Customer will be automatically notified of this status update
                        </span>
                    </div>
                    <button type="button" onclick="hideModal('updateStatusModal')" class="btn btn-outline">Cancel</button>
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-bell"></i>
                        Update Status & Notify
                    </button>
                </div>
            </form>
        </div>
    </div>

    <script src="../assets/js/admin.js"></script>
    <script>
        function viewOrder(orderId) {
            // Load order details via AJAX
            fetch(`ajax/get_order_details.php?id=${orderId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        document.getElementById('orderDetailsContent').innerHTML = data.html;
                        showModal('orderDetailsModal');
                    } else {
                        alert('Error loading order details');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Error loading order details');
                });
        }
        
        function updateOrderStatus(orderId) {
            document.getElementById('updateOrderId').value = orderId;
            
            // Load order summary first to get current tracking number
            fetch(`ajax/get_order_summary.php?id=${orderId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        document.getElementById('orderSummaryContent').innerHTML = data.html;
                        
                        // Check if order already has a tracking number and preserve it
                        if (data.tracking_number && data.tracking_number.trim() !== '') {
                            document.getElementById('tracking_number').value = data.tracking_number;
                            // Auto-set status to 'shipped' if tracking number exists
                            document.getElementById('status').value = 'shipped';
                            // Show tracking info
                            document.getElementById('tracking-info').style.display = 'block';
                        } else {
                            // Reset tracking number field only if no tracking number exists
                            document.getElementById('tracking_number').value = '';
                            // Hide tracking info
                            document.getElementById('tracking-info').style.display = 'none';
                        }
                    } else {
                        document.getElementById('orderSummaryContent').innerHTML = '<p>Error loading order details</p>';
                        // Reset tracking number field on error
                        document.getElementById('tracking_number').value = '';
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    document.getElementById('orderSummaryContent').innerHTML = '<p>Error loading order details</p>';
                    // Reset tracking number field on error
                    document.getElementById('tracking_number').value = '';
                });
            
            showModal('updateStatusModal');
        }
        
        // Update tracking number field requirements based on status
        document.getElementById('status').addEventListener('change', function() {
            const status = this.value;
            const trackingInput = document.getElementById('tracking_number');
            
            if (status === 'shipped' || status === 'delivered') {
                trackingInput.required = true;
            } else {
                trackingInput.required = false;
                // Don't clear the tracking number, let staff keep it if they want
            }
        });
        
        // Auto-update status to 'shipped' when tracking number is entered
        document.getElementById('tracking_number').addEventListener('input', function() {
            const trackingNumber = this.value.trim();
            const statusSelect = document.getElementById('status');
            const trackingInfo = document.getElementById('tracking-info');
            
            if (trackingNumber.length > 0) {
                // If tracking number is entered, automatically set status to 'shipped'
                if (statusSelect.value !== 'shipped' && statusSelect.value !== 'delivered') {
                    statusSelect.value = 'shipped';
                    // Trigger the change event to update requirements
                    statusSelect.dispatchEvent(new Event('change'));
                    
                    // Add visual feedback
                    statusSelect.style.backgroundColor = '#e8f5e8';
                    statusSelect.style.borderColor = '#28a745';
                    
                    // Show notification
                    const notification = document.createElement('div');
                    notification.style.cssText = `
                        position: fixed;
                        top: 20px;
                        right: 20px;
                        background: #28a745;
                        color: white;
                        padding: 10px 15px;
                        border-radius: 4px;
                        font-size: 14px;
                        z-index: 10000;
                        box-shadow: 0 2px 10px rgba(0,0,0,0.2);
                    `;
                    notification.textContent = 'Status automatically set to "Shipped"';
                    document.body.appendChild(notification);
                    
                    setTimeout(() => {
                        statusSelect.style.backgroundColor = '';
                        statusSelect.style.borderColor = '';
                        if (notification.parentNode) {
                            notification.parentNode.removeChild(notification);
                        }
                    }, 3000);
                }
                // Show tracking info when tracking number is entered
                trackingInfo.style.display = 'block';
            } else {
                // Hide tracking info when tracking number is cleared
                trackingInfo.style.display = 'none';
            }
        });
        
        // Add notification info to the form
        document.getElementById('statusUpdateForm').addEventListener('submit', function(e) {
            const status = document.getElementById('status').value;
            const trackingNumber = document.getElementById('tracking_number').value;
            
            // Show notification
            const notification = document.createElement('div');
            notification.style.cssText = `
                position: fixed;
                top: 20px;
                right: 20px;
                background: #007bff;
                color: white;
                padding: 12px 18px;
                border-radius: 6px;
                font-size: 14px;
                z-index: 10000;
                box-shadow: 0 4px 12px rgba(0,0,0,0.2);
                display: flex;
                align-items: center;
                gap: 8px;
            `;
            notification.innerHTML = `
                <i class="fas fa-bell"></i>
                <span>Updating order status and notifying customer...</span>
            `;
            document.body.appendChild(notification);
            
            // Remove notification after 3 seconds
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.parentNode.removeChild(notification);
                }
            }, 3000);
        });
        
        function exportOrders() {
            const table = document.getElementById('ordersTable');
            const rows = Array.from(table.querySelectorAll('tr'));
            const csvContent = rows.map(row => 
                Array.from(row.querySelectorAll('th, td')).map(cell => 
                    `"${cell.textContent.trim()}"`
                ).join(',')
            ).join('\n');
            
            const blob = new Blob([csvContent], { type: 'text/csv' });
            const url = window.URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = 'orders_export.csv';
            a.click();
            window.URL.revokeObjectURL(url);
        }
        
        // Auto-submit form when filters change
        document.getElementById('statusFilter').addEventListener('change', function() {
            // Reset to page 1 when filter changes
            document.getElementById('pageInput').value = 1;
            document.getElementById('filterForm').submit();
        });
        
        document.getElementById('dateFilter').addEventListener('change', function() {
            // Reset to page 1 when filter changes
            document.getElementById('pageInput').value = 1;
            document.getElementById('filterForm').submit();
        });
        
        document.getElementById('entriesFilter').addEventListener('change', function() {
            // Reset to page 1 when entries per page changes
            document.getElementById('pageInput').value = 1;
            document.getElementById('filterForm').submit();
        });
        
        document.getElementById('deliveryMethodFilter').addEventListener('change', function() {
            // Reset to page 1 when delivery method filter changes
            document.getElementById('pageInput').value = 1;
            document.getElementById('filterForm').submit();
        });
    </script>
    
    <style>
        .form-help {
            display: block;
            margin-top: 5px;
            font-size: 12px;
            color: #666;
            font-style: italic;
        }
        
        .form-group-full {
            margin-bottom: 20px;
        }
        
        /* Stats Card Styles */
        .stats-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 12px;
            padding: 1.5rem;
            margin-bottom: 2rem;
            color: white;
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
        }
        
        .stats-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1rem;
        }
        
        .stats-header h3 {
            margin: 0;
            font-size: 1.2rem;
            font-weight: 600;
        }
        
        .stats-date {
            font-size: 0.9rem;
            opacity: 0.9;
        }
        
        .stats-content {
            display: flex;
            gap: 2rem;
        }
        
        .stat-item {
            text-align: center;
            flex: 1;
        }
        
        .stat-number {
            font-size: 2.5rem;
            font-weight: bold;
            margin-bottom: 0.5rem;
            text-shadow: 0 2px 4px rgba(0,0,0,0.2);
        }
        
        .stat-label {
            font-size: 0.9rem;
            opacity: 0.9;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        /* Delivery Method Badge Styles */
        .delivery-badge {
            display: inline-block;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .delivery-pickup {
            background-color: #fff3e0;
            color: #e65100;
            border: 1px solid #ffcc02;
        }
        
        .delivery-shipping {
            background-color: #e3f2fd;
            color: #1976d2;
            border: 1px solid #2196f3;
        }
        
        /* Filter Form Styles */
        .filter-form {
            display: flex;
            gap: 1rem;
            align-items: center;
            flex-wrap: wrap;
        }
        
        .filter-btn {
            padding: 0.5rem 1rem;
            font-size: 0.9rem;
        }
        
        .clear-btn {
            padding: 0.5rem 1rem;
            font-size: 0.9rem;
        }
        
        .search-input, .filter-select {
            min-width: 150px;
        }
        
        /* Pagination Styles */
        .pagination-container {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin: 1.5rem 0;
            padding: 1rem;
            background: #f8f9fa;
            border-radius: 8px;
            border: 1px solid #e9ecef;
        }
        
        .pagination-info {
            color: #6c757d;
            font-size: 0.9rem;
        }
        
        .pagination-controls {
            display: flex;
            gap: 0.5rem;
            align-items: center;
        }
        
        .pagination-controls .btn-sm {
            padding: 0.375rem 0.75rem;
            font-size: 0.875rem;
            border-radius: 4px;
            text-decoration: none;
            transition: all 0.2s ease;
        }
        
        .pagination-controls .btn-primary {
            background-color: #007bff;
            border-color: #007bff;
            color: white;
        }
        
        .pagination-controls .btn-outline {
            background-color: white;
            border: 1px solid #dee2e6;
            color: #6c757d;
        }
        
        .pagination-controls .btn-outline:hover {
            background-color: #e9ecef;
            border-color: #adb5bd;
            color: #495057;
        }
        
        .pagination-dots {
            padding: 0.375rem 0.5rem;
            color: #6c757d;
            font-size: 0.875rem;
        }
        
        @media (max-width: 768px) {
            .stats-content {
                flex-direction: column;
                gap: 1rem;
            }
            
            .filter-form {
                flex-direction: column;
                align-items: stretch;
            }
            
            .search-input, .filter-select, .filter-btn, .clear-btn {
                width: 100%;
                min-width: auto;
            }
            
            .pagination-container {
                flex-direction: column;
                gap: 1rem;
                text-align: center;
            }
            
            .pagination-controls {
                flex-wrap: wrap;
                justify-content: center;
            }
        }
    </style>
</body>
</html>