<?php
session_start();
require_once '../../config/database.php';
require_once '../includes/Preset.php';
require_once '../../models/Product.php';

// Simple admin authentication
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: ../login.php');
    exit;
}

// Get database connection
$pdo = getDB();
if (!$pdo) {
    die("Database connection failed");
}

$presetModel = new Preset();
$productModel = new Product();

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Handle custom brand
    $brand = $_POST['brand'];
    if ($brand === 'custom' && !empty($_POST['custom_brand'])) {
        $brand = $_POST['custom_brand'];
        
        // Add custom brand to presets if it doesn't exist
        try {
            $presetModel->addPreset('brand', 'Brand', $brand, 999);
        } catch (Exception $e) {
            // Brand already exists, that's fine
        }
    }
    
    $productData = [
        'name' => $_POST['name'],
        'brand' => $brand,
        'description' => $_POST['description'],
        'price' => $_POST['price'],
        'volume' => $_POST['volume'],
        'category_id' => $_POST['category_id'],
        'season' => $_POST['season'],
        'notes' => $_POST['notes'],
        'scent' => $_POST['scent'],
        'recommendation' => $_POST['recommendation'],
        'stock_quantity' => $_POST['stock_quantity'],
        'is_bestseller' => isset($_POST['is_bestseller']) ? 1 : 0,
        'is_new_arrival' => isset($_POST['is_new_arrival']) ? 1 : 0,
        'is_active' => isset($_POST['is_active']) ? 1 : 0
    ];
    
    // Handle image upload
    if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
        $uploadDir = 'images/';
        $uploadFile = $uploadDir . basename($_FILES['image']['name']);
        
        if (move_uploaded_file($_FILES['image']['tmp_name'], $uploadFile)) {
            $productData['image'] = $uploadFile;
        }
    }
    
    // Add product sizes
    $sizes = [];
    if (isset($_POST['sizes']) && is_array($_POST['sizes'])) {
        foreach ($_POST['sizes'] as $size) {
            if (!empty($size['size']) && !empty($size['price_adjustment'])) {
                $sizes[] = [
                    'size' => $size['size'],
                    'price_adjustment' => $size['price_adjustment'],
                    'stock_quantity' => $size['stock_quantity'] ?? 0
                ];
            }
        }
    }
    
    try {
        $productId = $productModel->addProduct($productData, $sizes);
        $success_message = "Product added successfully!";
        
        // Reset form
        $_POST = [];
    } catch (Exception $e) {
        $error_message = "Error adding product: " . $e->getMessage();
    }
}

// Get presets for dropdowns
$brandPresets = $presetModel->getPresetsByFieldType('brand');
$volumePresets = $presetModel->getPresetsByFieldType('volume');
$sizePresets = $presetModel->getPresetsByFieldType('size');
$seasonPresets = $presetModel->getPresetsByFieldType('season');
$notesPresets = $presetModel->getPresetsByFieldType('notes');
$scentPresets = $presetModel->getPresetsByFieldType('scent');
$recommendationPresets = $presetModel->getPresetsByFieldType('recommendation');

// Get categories
$stmt = $pdo->query("SELECT * FROM categories WHERE is_active = 1 ORDER BY name");
$categories = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin - Add Product</title>
    <link rel="stylesheet" href="../../styles.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        .admin-container {
            max-width: 1000px;
            margin: 0 auto;
            padding: 20px;
            background: #f8f9fa;
            min-height: 100vh;
        }
        
        .admin-header {
            background: linear-gradient(135deg, #1A4D2A, #2d5a3d);
            color: white;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 30px;
            box-shadow: 0 4px 15px rgba(26, 77, 42, 0.3);
        }
        
        .admin-header h1 {
            margin: 0;
            font-size: 2rem;
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .admin-header p {
            margin: 10px 0 0 0;
            opacity: 0.9;
        }
        
        .form-container {
            background: white;
            border-radius: 10px;
            padding: 30px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        
        .form-section {
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 1px solid #e9ecef;
        }
        
        .form-section:last-child {
            border-bottom: none;
        }
        
        .form-section h3 {
            color: #1A4D2A;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .form-row {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-bottom: 20px;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #1A4D2A;
        }
        
        .form-group input,
        .form-group select,
        .form-group textarea {
            width: 100%;
            padding: 12px;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 14px;
            box-sizing: border-box;
        }
        
        .form-group input:focus,
        .form-group select:focus,
        .form-group textarea:focus {
            outline: none;
            border-color: #1A4D2A;
            box-shadow: 0 0 0 2px rgba(26, 77, 42, 0.2);
        }
        
        .form-group textarea {
            height: 100px;
            resize: vertical;
        }
        
        .form-checkbox {
            display: flex;
            align-items: center;
            gap: 8px;
            margin-bottom: 15px;
        }
        
        .form-checkbox input {
            width: auto;
        }
        
        .size-row {
            display: grid;
            grid-template-columns: 1fr 1fr 1fr auto;
            gap: 15px;
            align-items: end;
            margin-bottom: 15px;
            padding: 15px;
            background: #f8f9fa;
            border-radius: 6px;
        }
        
        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 14px;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #1A4D2A, #2d5a3d);
            color: white;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(26, 77, 42, 0.3);
        }
        
        .btn-secondary {
            background: #6c757d;
            color: white;
        }
        
        .btn-secondary:hover {
            background: #5a6268;
        }
        
        .btn-danger {
            background: #dc3545;
            color: white;
        }
        
        .btn-danger:hover {
            background: #c82333;
        }
        
        .btn-success {
            background: #28a745;
            color: white;
        }
        
        .btn-success:hover {
            background: #218838;
        }
        
        .alert {
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 6px;
            font-weight: 500;
        }
        
        .alert-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .alert-danger {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .preset-info {
            background: #e3f2fd;
            border: 1px solid #bbdefb;
            border-radius: 6px;
            padding: 15px;
            margin-bottom: 20px;
        }
        
        .preset-info h4 {
            margin: 0 0 10px 0;
            color: #1976d2;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .preset-info p {
            margin: 0;
            color: #1565c0;
            font-size: 0.9rem;
        }
        
        .back-link {
            display: inline-flex;
            align-items: center;
            gap: 8px;
            color: #1A4D2A;
            text-decoration: none;
            margin-bottom: 20px;
            font-weight: 500;
        }
        
        .back-link:hover {
            color: #2d5a3d;
        }
        
        @media (max-width: 768px) {
            .admin-container {
                padding: 10px;
            }
            
            .form-row {
                grid-template-columns: 1fr;
            }
            
            .size-row {
                grid-template-columns: 1fr;
                gap: 10px;
            }
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <a href="presets.php" class="back-link">
            <i class="fas fa-arrow-left"></i>
            Back to Presets Management
        </a>
        
        <div class="admin-header">
            <h1>
                <i class="fas fa-plus-circle"></i>
                Add New Product
            </h1>
            <p>Create a new product using standardized field presets</p>
        </div>

        <?php if (isset($success_message)): ?>
            <div class="alert alert-success">
                <i class="fas fa-check-circle"></i>
                <?php echo $success_message; ?>
            </div>
        <?php endif; ?>

        <?php if (isset($error_message)): ?>
            <div class="alert alert-danger">
                <i class="fas fa-exclamation-triangle"></i>
                <?php echo $error_message; ?>
            </div>
        <?php endif; ?>

        <div class="preset-info">
            <h4><i class="fas fa-info-circle"></i> Using Standardized Presets</h4>
            <p>This form uses predefined field options to ensure consistency across all products. All dropdown values are managed through the Presets Management system.</p>
        </div>

        <div class="form-container">
            <form method="POST" enctype="multipart/form-data">
                <!-- Basic Information -->
                <div class="form-section">
                    <h3><i class="fas fa-info"></i> Basic Information</h3>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label for="name">Product Name *</label>
                            <input type="text" id="name" name="name" required value="<?php echo isset($_POST['name']) ? htmlspecialchars($_POST['name']) : ''; ?>">
                        </div>
                        
                        <div class="form-group">
                            <label for="brand">Brand *</label>
                            <select id="brand" name="brand" required onchange="toggleCustomBrand()">
                                <option value="">Select Brand</option>
                                <?php foreach ($brandPresets as $preset): ?>
                                    <option value="<?php echo htmlspecialchars($preset['preset_value']); ?>" 
                                            <?php echo (isset($_POST['brand']) && $_POST['brand'] === $preset['preset_value']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($preset['preset_value']); ?>
                                    </option>
                                <?php endforeach; ?>
                                <option value="custom" <?php echo (isset($_POST['brand']) && $_POST['brand'] === 'custom') ? 'selected' : ''; ?>>+ Add Custom Brand</option>
                            </select>
                        </div>
                        
                        <div class="form-group" id="custom-brand-group" style="display: none;">
                            <label for="custom_brand">Custom Brand Name *</label>
                            <input type="text" id="custom_brand" name="custom_brand" placeholder="Enter custom brand name">
                            <small style="color: #6c757d; font-size: 0.85rem;">This will be added to the brand presets for future use</small>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label for="description">Description *</label>
                        <textarea id="description" name="description" required><?php echo isset($_POST['description']) ? htmlspecialchars($_POST['description']) : ''; ?></textarea>
                    </div>
                </div>

                <!-- Pricing & Volume -->
                <div class="form-section">
                    <h3><i class="fas fa-dollar-sign"></i> Pricing & Volume</h3>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label for="price">Base Price (RM) *</label>
                            <input type="number" id="price" name="price" step="0.01" min="0" required value="<?php echo isset($_POST['price']) ? htmlspecialchars($_POST['price']) : ''; ?>">
                        </div>
                        
                        <div class="form-group">
                            <label for="volume">Volume *</label>
                            <select id="volume" name="volume" required>
                                <option value="">Select Volume</option>
                                <?php foreach ($volumePresets as $preset): ?>
                                    <option value="<?php echo htmlspecialchars($preset['preset_value']); ?>" 
                                            <?php echo (isset($_POST['volume']) && $_POST['volume'] === $preset['preset_value']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($preset['preset_value']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                </div>

                <!-- Product Details -->
                <div class="form-section">
                    <h3><i class="fas fa-tags"></i> Product Details</h3>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label for="category_id">Category *</label>
                            <select id="category_id" name="category_id" required>
                                <option value="">Select Category</option>
                                <?php foreach ($categories as $category): ?>
                                    <option value="<?php echo $category['id']; ?>" 
                                            <?php echo (isset($_POST['category_id']) && $_POST['category_id'] == $category['id']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($category['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="season">Season *</label>
                            <select id="season" name="season" required>
                                <option value="">Select Season</option>
                                <?php foreach ($seasonPresets as $preset): ?>
                                    <option value="<?php echo htmlspecialchars($preset['preset_value']); ?>" 
                                            <?php echo (isset($_POST['season']) && $_POST['season'] === $preset['preset_value']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($preset['preset_value']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label for="notes">Notes *</label>
                            <select id="notes" name="notes" required>
                                <option value="">Select Notes</option>
                                <?php foreach ($notesPresets as $preset): ?>
                                    <option value="<?php echo htmlspecialchars($preset['preset_value']); ?>" 
                                            <?php echo (isset($_POST['notes']) && $_POST['notes'] === $preset['preset_value']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($preset['preset_value']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="scent">Scent *</label>
                            <select id="scent" name="scent" required>
                                <option value="">Select Scent</option>
                                <?php foreach ($scentPresets as $preset): ?>
                                    <option value="<?php echo htmlspecialchars($preset['preset_value']); ?>" 
                                            <?php echo (isset($_POST['scent']) && $_POST['scent'] === $preset['preset_value']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($preset['preset_value']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label for="recommendation">Recommendation *</label>
                        <select id="recommendation" name="recommendation" required>
                            <option value="">Select Recommendation</option>
                            <?php foreach ($recommendationPresets as $preset): ?>
                                <option value="<?php echo htmlspecialchars($preset['preset_value']); ?>" 
                                        <?php echo (isset($_POST['recommendation']) && $_POST['recommendation'] === $preset['preset_value']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($preset['preset_value']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>

                <!-- Product Sizes -->
                <div class="form-section">
                    <h3><i class="fas fa-ruler"></i> Product Sizes</h3>
                    <div id="sizes-container">
                        <div class="size-row">
                            <div class="form-group">
                                <label>Size</label>
                                <select name="sizes[0][size]" required>
                                    <option value="">Select Size</option>
                                    <?php foreach ($sizePresets as $preset): ?>
                                        <option value="<?php echo htmlspecialchars($preset['preset_value']); ?>">
                                            <?php echo htmlspecialchars($preset['preset_value']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="form-group">
                                <label>Price Adjustment (RM)</label>
                                <input type="number" name="sizes[0][price_adjustment]" step="0.01" value="0" required>
                            </div>
                            <div class="form-group">
                                <label>Stock Quantity</label>
                                <input type="number" name="sizes[0][stock_quantity]" min="0" value="0">
                            </div>
                            <div class="form-group">
                                <button type="button" class="btn btn-danger" onclick="removeSize(this)" style="display: none;">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </div>
                        </div>
                    </div>
                    <button type="button" class="btn btn-success" onclick="addSize()">
                        <i class="fas fa-plus"></i>
                        Add Size
                    </button>
                </div>

                <!-- Inventory & Status -->
                <div class="form-section">
                    <h3><i class="fas fa-warehouse"></i> Inventory & Status</h3>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label for="stock_quantity">Stock Quantity *</label>
                            <input type="number" id="stock_quantity" name="stock_quantity" min="0" required value="<?php echo isset($_POST['stock_quantity']) ? htmlspecialchars($_POST['stock_quantity']) : ''; ?>">
                        </div>
                        
                        <div class="form-group">
                            <label for="image">Product Image</label>
                            <input type="file" id="image" name="image" accept="image/*">
                        </div>
                    </div>
                    
                    <div class="form-checkbox">
                        <input type="checkbox" id="is_bestseller" name="is_bestseller" <?php echo (isset($_POST['is_bestseller'])) ? 'checked' : ''; ?>>
                        <label for="is_bestseller">Best Seller</label>
                    </div>
                    
                    <div class="form-checkbox">
                        <input type="checkbox" id="is_new_arrival" name="is_new_arrival" <?php echo (isset($_POST['is_new_arrival'])) ? 'checked' : ''; ?>>
                        <label for="is_new_arrival">New Arrival</label>
                    </div>
                    
                    <div class="form-checkbox">
                        <input type="checkbox" id="is_active" name="is_active" checked>
                        <label for="is_active">Active</label>
                    </div>
                </div>

                <div style="text-align: center; margin-top: 30px;">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-save"></i>
                        Add Product
                    </button>
                </div>
            </form>
        </div>
    </div>

    <script>
        let sizeIndex = 1;
        
        function addSize() {
            const container = document.getElementById('sizes-container');
            const sizeRow = document.createElement('div');
            sizeRow.className = 'size-row';
            
            sizeRow.innerHTML = `
                <div class="form-group">
                    <label>Size</label>
                    <select name="sizes[${sizeIndex}][size]" required>
                        <option value="">Select Size</option>
                        <?php foreach ($sizePresets as $preset): ?>
                            <option value="<?php echo htmlspecialchars($preset['preset_value']); ?>">
                                <?php echo htmlspecialchars($preset['preset_value']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group">
                    <label>Price Adjustment (RM)</label>
                    <input type="number" name="sizes[${sizeIndex}][price_adjustment]" step="0.01" value="0" required>
                </div>
                <div class="form-group">
                    <label>Stock Quantity</label>
                    <input type="number" name="sizes[${sizeIndex}][stock_quantity]" min="0" value="0">
                </div>
                <div class="form-group">
                    <button type="button" class="btn btn-danger" onclick="removeSize(this)">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            `;
            
            container.appendChild(sizeRow);
            sizeIndex++;
            
            // Show remove buttons for all size rows
            const removeButtons = container.querySelectorAll('.btn-danger');
            removeButtons.forEach(btn => btn.style.display = 'block');
        }
        
        function removeSize(button) {
            button.closest('.size-row').remove();
            
            // Hide remove buttons if only one size row remains
            const container = document.getElementById('sizes-container');
            const sizeRows = container.querySelectorAll('.size-row');
            if (sizeRows.length === 1) {
                const removeButton = sizeRows[0].querySelector('.btn-danger');
                if (removeButton) removeButton.style.display = 'none';
            }
        }
        
        function toggleCustomBrand() {
            const brandSelect = document.getElementById('brand');
            const customBrandGroup = document.getElementById('custom-brand-group');
            const customBrandInput = document.getElementById('custom_brand');
            
            if (brandSelect.value === 'custom') {
                customBrandGroup.style.display = 'block';
                customBrandInput.required = true;
            } else {
                customBrandGroup.style.display = 'none';
                customBrandInput.required = false;
                customBrandInput.value = '';
            }
        }
        
        // Initialize on page load
        document.addEventListener('DOMContentLoaded', function() {
            toggleCustomBrand();
        });
    </script>
</body>
</html>