<?php
// Suppress Xdebug output to prevent HTML errors in form fields
ini_set('display_errors', 0);
error_reporting(E_ERROR | E_PARSE);

require_once '../includes/auth.php';
require_once '../includes/functions.php';

$adminAuth->requireLogin();

// Handle form submissions
if ($_POST) {
    $action = $_POST['action'] ?? '';
    
    // Handle image uploads
    $imagePath = $_POST['image'] ?? ''; // Default to URL if provided
    $image2Path = $_POST['image_2'] ?? ''; // Second image URL if provided
    $image3Path = $_POST['image_3'] ?? ''; // Third image URL if provided
    $image4Path = $_POST['image_4'] ?? ''; // Fourth image URL if provided
    $image5Path = $_POST['image_5'] ?? ''; // Fifth image URL if provided
    $image6Path = $_POST['image_6'] ?? ''; // Sixth image URL if provided
    
    // Function to handle single image upload
    function handleImageUpload($file, $uploadDir, $allowedTypes, $maxSize) {
        if ($file['error'] !== UPLOAD_ERR_OK) {
            return ['success' => false, 'error' => 'No file uploaded'];
        }
        
        // Check if upload directory exists and is writable
        if (!is_dir($uploadDir)) {
            return ['success' => false, 'error' => 'Upload directory does not exist: ' . $uploadDir];
        } elseif (!is_writable($uploadDir)) {
            return ['success' => false, 'error' => 'Upload directory is not writable. Please check permissions.'];
        } elseif (in_array($file['type'], $allowedTypes)) {
            if ($file['size'] <= $maxSize) {
                $fileName = 'product_' . time() . '_' . preg_replace('/[^a-zA-Z0-9._-]/', '', $file['name']);
                $filePath = $uploadDir . $fileName;
                
                if (move_uploaded_file($file['tmp_name'], $filePath)) {
                    return ['success' => true, 'path' => 'images/' . $fileName];
                } else {
                    return ['success' => false, 'error' => 'Error uploading image. Please check directory permissions.'];
                }
            } else {
                return ['success' => false, 'error' => 'Image file too large. Maximum size is 2MB.'];
            }
        } else {
            return ['success' => false, 'error' => 'Invalid image type. Only JPG, PNG, and GIF files are allowed.'];
        }
    }
    
    // Handle first image upload
    if (isset($_FILES['image_file']) && $_FILES['image_file']['error'] === UPLOAD_ERR_OK) {
        $uploadDir = __DIR__ . '/../../images/';
        $allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif'];
        $maxSize = 2 * 1024 * 1024; // 2MB
        
        $result = handleImageUpload($_FILES['image_file'], $uploadDir, $allowedTypes, $maxSize);
        if ($result['success']) {
            $imagePath = $result['path'];
        } else {
            $error = $result['error'];
        }
    }
    
    // Handle second image upload
    if (isset($_FILES['image_2_file']) && $_FILES['image_2_file']['error'] === UPLOAD_ERR_OK) {
        $uploadDir = __DIR__ . '/../../images/';
        $allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif'];
        $maxSize = 2 * 1024 * 1024; // 2MB
        
        $result = handleImageUpload($_FILES['image_2_file'], $uploadDir, $allowedTypes, $maxSize);
        if ($result['success']) {
            $image2Path = $result['path'];
        } else {
            $error = $result['error'];
        }
    }
    
    // Handle third image upload
    if (isset($_FILES['image_3_file']) && $_FILES['image_3_file']['error'] === UPLOAD_ERR_OK) {
        $uploadDir = __DIR__ . '/../../images/';
        $allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif'];
        $maxSize = 2 * 1024 * 1024; // 2MB
        
        $result = handleImageUpload($_FILES['image_3_file'], $uploadDir, $allowedTypes, $maxSize);
        if ($result['success']) {
            $image3Path = $result['path'];
        } else {
            $error = $result['error'];
        }
    }
    
    // Handle fourth image upload
    if (isset($_FILES['image_4_file']) && $_FILES['image_4_file']['error'] === UPLOAD_ERR_OK) {
        $uploadDir = __DIR__ . '/../../images/';
        $allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif'];
        $maxSize = 2 * 1024 * 1024; // 2MB
        
        $result = handleImageUpload($_FILES['image_4_file'], $uploadDir, $allowedTypes, $maxSize);
        if ($result['success']) {
            $image4Path = $result['path'];
        } else {
            $error = $result['error'];
        }
    }
    
    // Handle fifth image upload
    if (isset($_FILES['image_5_file']) && $_FILES['image_5_file']['error'] === UPLOAD_ERR_OK) {
        $uploadDir = __DIR__ . '/../../images/';
        $allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif'];
        $maxSize = 2 * 1024 * 1024; // 2MB
        
        $result = handleImageUpload($_FILES['image_5_file'], $uploadDir, $allowedTypes, $maxSize);
        if ($result['success']) {
            $image5Path = $result['path'];
        } else {
            $error = $result['error'];
        }
    }
    
    // Handle sixth image upload
    if (isset($_FILES['image_6_file']) && $_FILES['image_6_file']['error'] === UPLOAD_ERR_OK) {
        $uploadDir = __DIR__ . '/../../images/';
        $allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif'];
        $maxSize = 2 * 1024 * 1024; // 2MB
        
        $result = handleImageUpload($_FILES['image_6_file'], $uploadDir, $allowedTypes, $maxSize);
        if ($result['success']) {
            $image6Path = $result['path'];
        } else {
            $error = $result['error'];
        }
    }
    
    if ($action === 'add') {
        // Check if product name already exists
        $productName = trim($_POST['name'] ?? '');
        if ($adminFunctions->productNameExists($productName)) {
            $error = 'This product name already exists. Please use a different name.';
        } else {
        // Validate preset-driven fields
        require_once '../includes/Preset.php';
        $presetModel = new Preset();
        $brand = $_POST['brand'];
        $scent = $_POST['scent'];
        $notes = $_POST['notes'];
        $season = $_POST['season'];
        $recommendation = $_POST['recommendation'];
        
        // Validate version variations
        $versionTypes = $_POST['version_types'] ?? [];
        $versionSizes = $_POST['version_sizes'] ?? [];
        $versionPrices = $_POST['version_prices'] ?? [];
        $versionStock = $_POST['version_stock'] ?? [];
        
        if (empty($versionTypes) || count($versionTypes) === 0) {
            $error = 'Please add at least one version variation.';
        } else {
            // Validate all version types and sizes
            $allValid = true;
            foreach ($versionTypes as $index => $versionType) {
                if (empty($versionType) || empty($versionSizes[$index]) || empty($versionPrices[$index])) {
                    $allValid = false;
                    break;
                }
            }
            
            if (!$allValid) {
                $error = 'Please fill in all version variation fields.';
            } else if (!$presetModel->validatePresetValue('brand', $brand) || 
                !$presetModel->validatePresetValue('scent', $scent) ||
                !$presetModel->validatePresetValue('notes', $notes) ||
                !$presetModel->validatePresetValue('season', $season) ||
                !$presetModel->validatePresetValue('recommendation', $recommendation)) {
                $error = 'Please select valid values from presets for all required fields.';
            } else {
        $productData = [
            'name' => $_POST['name'],
            'brand' => $_POST['brand'],
            'description' => $_POST['description'],
            'currency' => $_POST['currency'] ?? 'RM',
                'display_price_mode' => $_POST['display_price_mode'] ?? 'range',
                'scent' => $_POST['scent'],
                'notes' => $_POST['notes'],
                'season' => $_POST['season'],
                'recommendation' => $_POST['recommendation'],
            'image' => $imagePath,
            'image_2' => $image2Path,
            'image_3' => $image3Path,
            'image_4' => $image4Path,
            'image_5' => $image5Path,
            'image_6' => $image6Path,
            'category_id' => $_POST['category_id'],
                'stock_quantity' => array_sum($versionStock), // Total stock from all version variations
            'is_bestseller' => isset($_POST['is_bestseller']) ? 1 : 0,
            'is_new_arrival' => isset($_POST['is_new_arrival']) ? 1 : 0,
                'is_active' => isset($_POST['is_active']) ? 1 : 0,
                'is_out_of_stock' => isset($_POST['is_out_of_stock']) ? 1 : 0,
                'restock_date' => !empty($_POST['restock_date']) ? $_POST['restock_date'] : null,
                'restock_message' => !empty($_POST['restock_message']) ? $_POST['restock_message'] : null,
                'point_rewards' => 0, // Points are now auto-calculated based on price
                'version_variations' => array_map(function($type, $size, $price, $stock) {
                    return [
                        'version_type' => $type,
                        'size' => $size,
                        'price' => $price,
                        'stock_quantity' => $stock,
                        'is_active' => 1
                    ];
                }, $versionTypes, $versionSizes, $versionPrices, $versionStock)
        ];
        
        if ($adminFunctions->addProduct($productData)) {
            header('Location: products.php?success=Product added successfully!');
            exit();
        } else {
            $error = 'Error adding product. Please try again.';
            }
            }
        }
        }
    } elseif ($action === 'edit') {
        $productId = $_POST['product_id'];
        // Validate preset-driven fields
        require_once '../includes/Preset.php';
        $presetModel = new Preset();
        $brand = $_POST['brand'];
        $scent = $_POST['scent'];
        $notes = $_POST['notes'];
        $season = $_POST['season'];
        $recommendation = $_POST['recommendation'];
        
        if (!$presetModel->validatePresetValue('brand', $brand) || 
            !$presetModel->validatePresetValue('scent', $scent) ||
            !$presetModel->validatePresetValue('notes', $notes) ||
            !$presetModel->validatePresetValue('season', $season) ||
            !$presetModel->validatePresetValue('recommendation', $recommendation)) {
            $error = 'Please select valid values from presets for all required fields.';
        } else {
        // Handle version variations for edit
        $versionTypes = $_POST['version_types'] ?? [];
        $versionSizes = $_POST['version_sizes'] ?? [];
        $versionPrices = $_POST['version_prices'] ?? [];
        $versionStock = $_POST['version_stock'] ?? [];
        
        if (empty($versionTypes) || count($versionTypes) === 0) {
            $error = 'Please add at least one version variation.';
        } else {
            // Validate all version types and sizes
            $allValid = true;
            foreach ($versionTypes as $index => $versionType) {
                if (empty($versionType) || empty($versionSizes[$index]) || empty($versionPrices[$index])) {
                    $allValid = false;
                    break;
                }
            }
            
            if (!$allValid) {
                $error = 'Please fill in all version variation fields.';
            } else {
        $productData = [
            'name' => $_POST['name'],
            'brand' => $_POST['brand'],
            'description' => $_POST['description'],
            'currency' => $_POST['currency'] ?? 'RM',
                'display_price_mode' => $_POST['display_price_mode'] ?? 'range',
                'scent' => $_POST['scent'],
                'notes' => $_POST['notes'],
                'season' => $_POST['season'],
                'recommendation' => $_POST['recommendation'],
            'image' => $imagePath,
            'image_2' => $image2Path,
            'image_3' => $image3Path,
            'image_4' => $image4Path,
            'image_5' => $image5Path,
            'image_6' => $image6Path,
            'category_id' => $_POST['category_id'],
                'stock_quantity' => array_sum($versionStock), // Total stock from all version variations
            'is_bestseller' => isset($_POST['is_bestseller']) ? 1 : 0,
            'is_new_arrival' => isset($_POST['is_new_arrival']) ? 1 : 0,
                'is_active' => isset($_POST['is_active']) ? 1 : 0,
                'is_out_of_stock' => isset($_POST['is_out_of_stock']) ? 1 : 0,
                'restock_date' => !empty($_POST['restock_date']) ? $_POST['restock_date'] : null,
                'restock_message' => !empty($_POST['restock_message']) ? $_POST['restock_message'] : null,
                'point_rewards' => 0, // Points are now auto-calculated based on price
                'version_variations' => array_map(function($type, $size, $price, $stock) {
                    return [
                        'version_type' => $type,
                        'size' => $size,
                        'price' => $price,
                        'stock_quantity' => $stock,
                        'is_active' => 1
                    ];
                }, $versionTypes, $versionSizes, $versionPrices, $versionStock)
        ];
        
            if ($adminFunctions->updateProductWithVariations($productId, $productData)) {
            $success = 'Product updated successfully!';
        } else {
            $error = 'Error updating product. Please try again.';
            }
            }
        }
        }
    } elseif ($action === 'delete') {
        $productId = $_POST['product_id'];
        if ($adminFunctions->deleteProduct($productId)) {
            $success = 'Product deleted successfully!';
        } else {
            $error = 'Error deleting product. Please try again.';
        }
    }
}

// Get products with volume variations and total stock
$products = $adminFunctions->getAllProductsWithVariations();
$categories = $adminFunctions->getCategories();

// Get presets for dynamic dropdowns
require_once '../includes/Preset.php';
$presetModel = new Preset();
$brandPresets = $presetModel->getPresetsByFieldType('brand');
$scentPresets = $presetModel->getPresetsByFieldType('scent');
$notesPresets = $presetModel->getPresetsByFieldType('notes');
$seasonPresets = $presetModel->getPresetsByFieldType('season');
$recommendationPresets = $presetModel->getPresetsByFieldType('recommendation');
$sizePresets = $presetModel->getPresetsByFieldType('size');
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Products Management - Admin Panel</title>
    <link rel="stylesheet" href="../assets/css/admin.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body>
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="sidebar-header">
            <img src="../../images/LOGO.png" alt="Logo" class="sidebar-logo">
            <h3>Admin Panel</h3>
        </div>
        
        <nav class="sidebar-nav">
            <a href="../index.php" class="nav-item">
                <i class="fas fa-th-large"></i>
                <span>Overview</span>
            </a>
            <a href="products.php" class="nav-item active">
                <i class="fas fa-box"></i>
                <span>Products</span>
            </a>
            <a href="orders.php" class="nav-item">
                <i class="fas fa-shopping-cart"></i>
                <span>Orders</span>
            </a>
            <a href="delivered.php" class="nav-item">
                <i class="fas fa-check-circle"></i>
                <span>Delivered</span>
            </a>
            <a href="customers.php" class="nav-item">
                <i class="fas fa-users"></i>
                <span>Customers</span>
            </a>
            <a href="feedback.php" class="nav-item">
                <i class="fas fa-comments"></i>
                <span>Feedback</span>
            </a>
            <a href="shipping.php" class="nav-item">
                <i class="fas fa-shipping-fast"></i>
                <span>Shipping</span>
            </a>
            <a href="banners.php" class="nav-item">
                <i class="fas fa-image"></i>
                <span>Banners</span>
            </a>
            <a href="presets.php" class="nav-item">
                <i class="fas fa-cogs"></i>
                <span>Field Presets</span>
            </a>
            <a href="vouchers.php" class="nav-item">
                <i class="fas fa-ticket-alt"></i>
                <span>Voucher Management</span>
            </a>
            <a href="notifications.php" class="nav-item">
                <i class="fas fa-bell"></i>
                <span>Push Notifications</span>
            </a>
            <a href="analytics.php" class="nav-item">
                <i class="fas fa-chart-bar"></i>
                <span>Analytics</span>
            </a>
            <a href="settings.php" class="nav-item">
                <i class="fas fa-cog"></i>
                <span>Settings</span>
            </a>
        </nav>
        
        <div class="sidebar-footer">
            <div class="user-info">
                <i class="fas fa-user-circle"></i>
                <span><?php echo htmlspecialchars($adminAuth->getAdminName()); ?></span>
            </div>
            <a href="../logout.php" class="logout-btn">
                <i class="fas fa-sign-out-alt"></i>
                <span>Logout</span>
            </a>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <!-- Header -->
        <header class="header">
            <div class="header-left">
                <button class="sidebar-toggle">
                    <i class="fas fa-bars"></i>
                </button>
                <h1>Products Management</h1>
            </div>
            <div class="header-right">
                <div class="header-actions">
                    <button onclick="openAddProductModal()" class="btn btn-primary">
                        <i class="fas fa-plus"></i>
                        Add Product
                    </button>
                    <a href="cart-activity.php" class="btn btn-outline" style="background: #f59e0b; color: white; border-color: #f59e0b;">
                        <i class="fas fa-cart-plus"></i>
                        Cart Activity
                    </a>
                    <a href="presets.php" class="btn btn-outline">
                        <i class="fas fa-cogs"></i>
                        Add Preset
                    </a>
                    <a href="flash-sale.php" class="btn btn-outline" style="background: #dc2626; color: white; border-color: #dc2626;">
                        <i class="fas fa-bolt"></i>
                        Flash Sale
                    </a>
                    <a href="../index.php" target="_blank" class="btn btn-outline">
                        <i class="fas fa-external-link-alt"></i>
                        View Store
                    </a>
                </div>
            </div>
        </header>

        <!-- Page Content -->
        <div class="page-content">
            <?php if (isset($success)): ?>
                <div class="alert" style="background-color: #d4edda; color: #155724; border: 1px solid #c3e6cb;">
                    <i class="fas fa-check-circle"></i>
                    <?php echo htmlspecialchars($success); ?>
                </div>
            <?php endif; ?>
            
            <?php if (isset($_GET['success'])): ?>
                <div class="alert" style="background-color: #d4edda; color: #155724; border: 1px solid #c3e6cb;">
                    <i class="fas fa-check-circle"></i>
                    <?php echo htmlspecialchars($_GET['success']); ?>
                </div>
            <?php endif; ?>
            
            <?php if (isset($error)): ?>
                <div class="alert alert-error">
                    <i class="fas fa-exclamation-circle"></i>
                    <?php echo htmlspecialchars($error); ?>
                </div>
            <?php endif; ?>

            <!-- Search and Filter -->
            <div class="search-filter">
                <input type="text" class="search-input" placeholder="Search products..." id="productSearch">
                <select class="filter-select" id="categoryFilter">
                    <option value="">All Categories</option>
                    <?php foreach ($categories as $category): ?>
                        <option value="<?php echo htmlspecialchars($category['name']); ?>">
                            <?php echo htmlspecialchars($category['name']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
                <select class="filter-select" id="statusFilter">
                    <option value="">All Status</option>
                    <option value="active">Active</option>
                    <option value="inactive">Inactive</option>
                    <option value="bestseller">Best Sellers</option>
                    <option value="new">New Arrivals</option>
                    <option value="out_of_stock">Out of Stock</option>
                </select>
                <button onclick="openCategoryManager()" class="btn btn-outline">
                    <i class="fas fa-layer-group"></i>
                    Manage Categories
                </button>
            </div>

            <!-- Products Table -->
            <div class="table-container">
                <div class="table-content">
                    <table class="data-table" id="productsTable">
                        <thead>
                            <tr>
                                <th>Image</th>
                                <th>Product</th>
                                <th>Brand</th>
                                <th>Price</th>
                                <th>Total Stock</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($products as $product): ?>
                            <tr>
                                <td>
                                    <img src="../../<?php echo htmlspecialchars($product['image']); ?>" 
                                         alt="<?php echo htmlspecialchars($product['name']); ?>"
                                         style="width: 50px; height: 50px; object-fit: cover; border-radius: 4px;"
                                         onerror="this.src='data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNTAiIGhlaWdodD0iNTAiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+PHJlY3Qgd2lkdGg9IjUwIiBoZWlnaHQ9IjUwIiBmaWxsPSIjZjRmNGY0Ii8+PHRleHQgeD0iMjUiIHk9IjI1IiBmb250LWZhbWlseT0iQXJpYWwiIGZvbnQtc2l6ZT0iMTIiIGZpbGw9IiM5OTkiIHRleHQtYW5jaG9yPSJtaWRkbGUiPkltYWdlPC90ZXh0Pjwvc3ZnPg=='">
                                </td>
                                <td>
                                    <div>
                                        <strong><?php echo htmlspecialchars($product['name']); ?></strong>
                                        <br>
                                        <?php if (!empty($product['variations_detail'])): ?>
                                            <small class="text-muted">
                                                <?php foreach ($product['variations_detail'] as $variation): ?>
                                                    <?php echo htmlspecialchars($variation['size']); ?> (<?php echo $variation['stock']; ?>)<br>
                                                <?php endforeach; ?>
                                            </small>
                                        <?php else: ?>
                                            <small class="text-muted">No variations</small>
                                        <?php endif; ?>
                                    </div>
                                </td>
                                <td><?php echo htmlspecialchars($product['brand']); ?></td>
                                <td>
                                    <div>
                                        <?php if (!empty($product['variations_detail'])): ?>
                                            <?php foreach ($product['variations_detail'] as $variation): ?>
                                                <div style="margin-bottom: 5px;">
                                                    <strong><?php echo htmlspecialchars($variation['size']); ?>: RM <?php echo number_format($variation['price'], 2); ?></strong>
                                                </div>
                                            <?php endforeach; ?>
                                        <?php else: ?>
                                            <span style="color: #999;">No variations</span>
                                        <?php endif; ?>
                                    </div>
                                </td>
                                <td>
                                    <div style="text-align: center;">
                                        <strong style="font-size: 16px; color: #333;"><?php echo $product['total_stock']; ?></strong>
                                        <br>
                                        <small style="color: #666;">Total Stock</small>
                                    </div>
                                </td>
                                <td class="status-column">
                                    <div>
                                        <?php if ($product['is_active']): ?>
                                            <span class="status status-delivered">Active</span>
                                        <?php else: ?>
                                            <span class="status status-cancelled">Inactive</span>
                                        <?php endif; ?>
                                        <br>
                                        <?php if ($product['is_bestseller']): ?>
                                            <small class="status status-processing">Best Seller</small>
                                        <?php endif; ?>
                                        <?php if ($product['is_new_arrival']): ?>
                                            <small class="status status-pending">New</small>
                                        <?php endif; ?>
                                        <?php if ($product['is_out_of_stock']): ?>
                                            <small class="status" style="background: #dc3545; color: white;">
                                                <i class="fas fa-exclamation-triangle"></i> Out of Stock
                                            </small>
                                            <?php if (!empty($product['restock_date'])): ?>
                                                <br>
                                                <small style="color: #666; font-size: 11px;">
                                                    <i class="fas fa-clock"></i> Restock: <?php echo date('M j, Y', strtotime($product['restock_date'])); ?>
                                                </small>
                                            <?php endif; ?>
                                            <?php if (!empty($product['restock_message'])): ?>
                                                <br>
                                                <small style="color: #666; font-size: 11px; font-style: italic;">
                                                    "<?php echo htmlspecialchars($product['restock_message']); ?>"
                                                </small>
                                            <?php endif; ?>
                                        <?php endif; ?>
                                    </div>
                                </td>
                                <td>
                                    <div style="display: flex; gap: 5px;">
                                        <button onclick="editProduct(<?php echo $product['id']; ?>)" 
                                                class="btn btn-sm btn-warning" title="Edit">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <button onclick="deleteProduct(<?php echo $product['id']; ?>)" 
                                                class="btn btn-sm btn-danger" title="Delete">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Add/Edit Product Modal -->
    <div id="addProductModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title">Add New Product</h3>
                <span class="close" onclick="hideModal('addProductModal')">&times;</span>
            </div>

            <form method="POST" enctype="multipart/form-data" data-validate>
                <input type="hidden" name="action" value="add">

                <div class="form-row">
                    <div class="form-group-full">
                        <label for="name">Product Name *</label>
                        <input type="text" id="name" name="name" required>
                    </div>
                    <div class="form-group-full">
                        <label for="brand">Brand *</label>
                        <select id="brand" name="brand" required>
                            <option value="">Select Brand (manage in Presets)</option>
                            <?php foreach ($brandPresets as $preset): ?>
                                <option value="<?php echo htmlspecialchars($preset['preset_value']); ?>">
                                    <?php echo htmlspecialchars($preset['preset_value']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group-full">
                        <label for="currency">Currency</label>
                        <select id="currency" name="currency">
                            <option value="RM" selected>RM (Malaysian Ringgit)</option>
                            <option value="USD">USD (US Dollar)</option>
                            <option value="EUR">EUR (Euro)</option>
                            <option value="SGD">SGD (Singapore Dollar)</option>
                        </select>
                    </div>
                </div>

                <!-- Sale Settings -->

                <!-- Display Price Settings -->
                <div class="form-group-full">
                    <label style="font-size: 18px; font-weight: bold; color: #27ae60; margin-bottom: 15px;">
                        <i class="fas fa-tag"></i> Display Price Settings
                    </label>
                </div>

                <div class="form-row">
                    <div class="form-group-full">
                        <label for="display_price_mode">Price Display Mode</label>
                        <select id="display_price_mode" name="display_price_mode">
                            <option value="range" selected>
                                Range Price (e.g., RM 66.00 - RM 100.00)
                            </option>
                            <option value="single">
                                Single Price (e.g., RM 66.00)
                            </option>
                        </select>
                        <small class="form-help">Choose how prices are displayed on the frontend. Range shows min-max prices, Single shows only the minimum price.</small>
                    </div>
                </div>
                
                <div class="form-group-full">
                    <label style="font-size: 18px; font-weight: bold; color: #2c3e50; margin-bottom: 15px;">
                        <i class="fas fa-cubes"></i> Version Variations & Pricing *
                    </label>
                    <div id="versionVariations" class="variations-container">
                        <div class="version-variation-item">
                            <div class="form-row">
                                <div class="form-group-full">
                                    <select name="version_types[]" class="version-select" required>
                                        <option value="">Select Version</option>
                                        <option value="EDP / LIGHT">EDP / LIGHT</option>
                                        <option value="PARFUM / MEDIUM">PARFUM / MEDIUM</option>
                                        <option value="EXTRAIT / STRONG">EXTRAIT / STRONG</option>
                                    </select>
                                </div>
                                <div class="form-group-full">
                                    <select name="version_sizes[]" class="size-select" required>
                                        <option value="">Select Size</option>
                                        <?php foreach ($sizePresets as $preset): ?>
                                            <option value="<?php echo htmlspecialchars($preset['preset_value']); ?>">
                                                <?php echo htmlspecialchars($preset['preset_value']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="form-group-full">
                                    <input type="number" name="version_prices[]" step="0.01" placeholder="Price" class="price-input" required>
                                </div>
                                <div class="form-group-full">
                                    <input type="number" name="version_stock[]" placeholder="Stock" class="stock-input" required>
                                </div>
                                <div class="form-group-full">
                                    <button type="button" class="btn btn-outline btn-sm remove-version-variation" style="display:none;">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                    <button type="button" id="addVersionVariation" class="btn btn-outline btn-sm">
                        <i class="fas fa-plus"></i> Add Version Variation
                    </button>
                </div>
                
                
                <div class="form-row">
                    <div class="form-group-full">
                        <label for="scent">Scent *</label>
                        <select id="scent" name="scent" required>
                            <option value="">Select Scent (manage in Presets)</option>
                            <?php foreach ($scentPresets as $preset): ?>
                                <option value="<?php echo htmlspecialchars($preset['preset_value']); ?>">
                                    <?php echo htmlspecialchars($preset['preset_value']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group-full">
                        <label for="notes">Notes *</label>
                        <select id="notes" name="notes" required>
                            <option value="">Select Notes (manage in Presets)</option>
                            <?php foreach ($notesPresets as $preset): ?>
                                <option value="<?php echo htmlspecialchars($preset['preset_value']); ?>">
                                    <?php echo htmlspecialchars($preset['preset_value']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group-full">
                        <label for="season">Season *</label>
                        <select id="season" name="season" required>
                            <option value="">Select Season (manage in Presets)</option>
                            <?php foreach ($seasonPresets as $preset): ?>
                                <option value="<?php echo htmlspecialchars($preset['preset_value']); ?>">
                                    <?php echo htmlspecialchars($preset['preset_value']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group-full">
                        <label for="recommendation">Recommendation *</label>
                        <select id="recommendation" name="recommendation" required>
                            <option value="">Select Recommendation (manage in Presets)</option>
                            <?php foreach ($recommendationPresets as $preset): ?>
                                <option value="<?php echo htmlspecialchars($preset['preset_value']); ?>">
                                    <?php echo htmlspecialchars($preset['preset_value']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>

                <div class="form-group-full">
                    <label for="description">Description</label>
                    <textarea id="description" name="description" rows="3" placeholder="Additional product details..."></textarea>
                </div>

                <div class="form-row">
                    <div class="form-group-full">
                        <label for="category_id">Category *</label>
                        <select id="category_id" name="category_id" required>
                            <option value="">Select Category</option>
                            <?php foreach ($categories as $category): ?>
                                <option value="<?php echo $category['id']; ?>">
                                    <?php echo htmlspecialchars($category['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group-full">
                        <label for="stock_quantity">Total Stock Quantity (Auto-calculated) *</label>
                        <input type="number" id="stock_quantity" name="stock_quantity" required readonly
                               value="0"
                               style="background-color: #f0f0f0; cursor: not-allowed;">
                        <small class="form-help" style="color: #666; font-size: 12px; display: block; margin-top: 5px;">
                            <i class="fas fa-info-circle"></i> This field is automatically calculated from the version variations stock quantities below.
                        </small>
                    </div>
                </div>
                
                <div class="form-group-full">
                    <label for="image">Product Image</label>
                    <div class="image-upload-container">
                        <div class="image-upload">
                            <input type="file" id="image_file" name="image_file" accept="image/*" onchange="previewImage(this)">
                            <label for="image_file" class="upload-label">
                                <i class="fas fa-cloud-upload-alt"></i>
                                <span>Click to upload image</span>
                                <small>JPG, PNG, GIF (Max 2MB)</small>
                            </label>
                            <img id="image_preview" class="image-preview" style="display: none;">
                        </div>
                        <div class="image-url-section">
                            <label for="image">Or enter image URL:</label>
                            <input type="text" id="image" name="image"
                                   placeholder="images/product-name.jpg">
                        </div>
                    </div>
                </div>

                <div class="form-group-full">
                    <label for="image_2">Product Image 2 (Optional)</label>
                    <div class="image-upload-container">
                        <div class="image-upload">
                            <input type="file" id="image_2_file" name="image_2_file" accept="image/*" onchange="previewImage2(this)">
                            <label for="image_2_file" class="upload-label">
                                <i class="fas fa-cloud-upload-alt"></i>
                                <span>Click to upload second image</span>
                                <small>JPG, PNG, GIF (Max 2MB)</small>
                            </label>
                            <img id="image_2_preview" class="image-preview" style="display: none;">
                        </div>
                        <div class="image-url-section">
                            <label for="image_2">Or enter second image URL:</label>
                            <input type="text" id="image_2" name="image_2"
                                   placeholder="images/product-name-2.jpg">
                        </div>
                    </div>
                </div>

                <div class="form-group-full">
                    <label for="image_3">Product Image 3 (Optional)</label>
                    <div class="image-upload-container">
                        <div class="image-upload">
                            <input type="file" id="image_3_file" name="image_3_file" accept="image/*" onchange="previewImage3(this)">
                            <label for="image_3_file" class="upload-label">
                                <i class="fas fa-cloud-upload-alt"></i>
                                <span>Click to upload third image</span>
                                <small>JPG, PNG, GIF (Max 2MB)</small>
                            </label>
                            <img id="image_3_preview" class="image-preview" style="display: none;">
                        </div>
                        <div class="image-url-section">
                            <label for="image_3">Or enter third image URL:</label>
                            <input type="text" id="image_3" name="image_3"
                                   placeholder="images/product-name-3.jpg">
                        </div>
                    </div>
                </div>

                <div class="form-group-full">
                    <label for="image_4">Product Image 4 (Optional)</label>
                    <div class="image-upload-container">
                        <div class="image-upload">
                            <input type="file" id="image_4_file" name="image_4_file" accept="image/*" onchange="previewImage4(this)">
                            <label for="image_4_file" class="upload-label">
                                <i class="fas fa-cloud-upload-alt"></i>
                                <span>Click to upload fourth image</span>
                                <small>JPG, PNG, GIF (Max 2MB)</small>
                            </label>
                            <img id="image_4_preview" class="image-preview" style="display: none;">
                        </div>
                        <div class="image-url-section">
                            <label for="image_4">Or enter fourth image URL:</label>
                            <input type="text" id="image_4" name="image_4"
                                   placeholder="images/product-name-4.jpg">
                        </div>
                    </div>
                </div>

                <div class="form-group-full">
                    <label for="image_5">Product Image 5 (Optional)</label>
                    <div class="image-upload-container">
                        <div class="image-upload">
                            <input type="file" id="image_5_file" name="image_5_file" accept="image/*" onchange="previewImage5(this)">
                            <label for="image_5_file" class="upload-label">
                                <i class="fas fa-cloud-upload-alt"></i>
                                <span>Click to upload fifth image</span>
                                <small>JPG, PNG, GIF (Max 2MB)</small>
                            </label>
                            <img id="image_5_preview" class="image-preview" style="display: none;">
                        </div>
                        <div class="image-url-section">
                            <label for="image_5">Or enter fifth image URL:</label>
                            <input type="text" id="image_5" name="image_5"
                                   placeholder="images/product-name-5.jpg">
                        </div>
                    </div>
                </div>

                <div class="form-group-full">
                    <label for="image_6">Product Image 6 (Optional)</label>
                    <div class="image-upload-container">
                        <div class="image-upload">
                            <input type="file" id="image_6_file" name="image_6_file" accept="image/*" onchange="previewImage6(this)">
                            <label for="image_6_file" class="upload-label">
                                <i class="fas fa-cloud-upload-alt"></i>
                                <span>Click to upload sixth image</span>
                                <small>JPG, PNG, GIF (Max 2MB)</small>
                            </label>
                            <img id="image_6_preview" class="image-preview" style="display: none;">
                        </div>
                        <div class="image-url-section">
                            <label for="image_6">Or enter sixth image URL:</label>
                            <input type="text" id="image_6" name="image_6"
                                   placeholder="images/product-name-6.jpg">
                        </div>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group-full">
                        <label>Point Rewards (Auto-calculated)</label>
                        <div class="calculated-points-display">
                            <div class="points-info">
                                <i class="fas fa-info-circle"></i>
                                <span>Points are automatically calculated based on product price (RM1 = 1 point)</span>
                            </div>
                            <div class="points-preview">
                                <strong>Points will be calculated when variations are added</strong>
                            </div>
                        </div>
                        <small class="form-help">Points are automatically calculated based on product price. No manual input required.</small>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group-full">
                        <label>
                            <input type="checkbox" id="is_bestseller" name="is_bestseller" value="1">
                            Best Seller
                        </label>
                    </div>
                    <div class="form-group-full">
                        <label>
                            <input type="checkbox" id="is_new_arrival" name="is_new_arrival" value="1">
                            New Arrival
                        </label>
                    </div>
                    <div class="form-group-full">
                        <label>
                            <input type="checkbox" id="is_active" name="is_active" value="1" checked>
                            Active
                        </label>
                    </div>
                </div>

                <!-- Out of Stock Settings -->
                <div class="form-group-full">
                    <label style="font-size: 18px; font-weight: bold; color: #e74c3c; margin-bottom: 15px;">
                        <i class="fas fa-exclamation-triangle"></i> Stock Status
                    </label>
                </div>

                <div class="form-row">
                    <div class="form-group-full">
                        <label>
                            <input type="checkbox" name="is_out_of_stock" value="1" id="is_out_of_stock"
                                   onchange="toggleRestockFields()">
                            Mark as Out of Stock
                        </label>
                        <small class="form-help">When checked, this product will be marked as out of stock and customers will see the restock information.</small>
                    </div>
                </div>

                <div id="restock_fields" style="display: none;">
                    <div class="form-row">
                        <div class="form-group-full">
                            <label for="restock_date">Expected Restock Date</label>
                            <input type="datetime-local" id="restock_date" name="restock_date">
                            <small class="form-help">When do you expect this product to be back in stock?</small>
                        </div>
                        <div class="form-group-full">
                            <label for="restock_message">Restock Message</label>
                            <input type="text" id="restock_message" name="restock_message"
                                   placeholder="e.g., 'Back in stock soon!' or 'Expected by end of month'">
                            <small class="form-help">Custom message to show customers about restocking (optional).</small>
                        </div>
                    </div>
                </div>

                <div class="form-actions">
                    <button type="button" onclick="hideModal('addProductModal')" class="btn btn-outline">Cancel</button>
                    <button type="submit" class="btn btn-primary">Add Product</button>
                </div>
            </form>
        </div>
    </div>

    <!-- Delete Product Modal -->
    <div id="deleteProductModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title">Delete Product</h3>
                <span class="close" onclick="hideModal('deleteProductModal')">&times;</span>
            </div>
            
            <form method="POST" id="deleteForm">
                <input type="hidden" name="action" value="delete">
                <input type="hidden" name="product_id" id="deleteProductId">
                
                <p>Are you sure you want to delete this product? This action cannot be undone.</p>
                
                <div class="form-actions">
                    <button type="button" onclick="hideModal('deleteProductModal')" class="btn btn-outline">Cancel</button>
                    <button type="submit" class="btn btn-danger">Delete Product</button>
                </div>
            </form>
        </div>
    </div>

    <!-- Category Management Modal -->
    <div id="categoryManagerModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title">Category Management</h3>
                <span class="close" onclick="hideModal('categoryManagerModal')">&times;</span>
            </div>
            
            <div class="category-manager-content">
                <!-- Add New Category Section -->
                <div class="add-category-section">
                    <h4><i class="fas fa-plus-circle"></i> Add New Category</h4>
                    <form id="addCategoryForm" class="category-form">
                        <div class="form-row">
                            <div class="form-group-full">
                                <label for="categoryName">Category Name *</label>
                                <input type="text" id="categoryName" name="name" required placeholder="Enter category name">
                            </div>
                            <div class="form-group-full">
                                <label for="categorySlug">Slug *</label>
                                <input type="text" id="categorySlug" name="slug" required placeholder="category-slug">
                            </div>
                        </div>
                        <div class="form-group-full">
                            <label for="categoryDescription">Description</label>
                            <textarea id="categoryDescription" name="description" rows="3" placeholder="Category description (optional)"></textarea>
                        </div>
                        <div class="form-group-full">
                            <label for="categoryImage">Image URL</label>
                            <input type="text" id="categoryImage" name="image" placeholder="images/category-image.jpg">
                        </div>
                        <div class="form-actions">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-plus"></i> Add Category
                            </button>
                        </div>
                    </form>
                </div>

                <!-- Categories List Section -->
                <div class="categories-list-section">
                    <h4><i class="fas fa-list"></i> Manage Categories <small>(Drag to reorder)</small></h4>
                    <div id="categoriesList" class="categories-list">
                        <?php foreach ($categories as $index => $category): ?>
                            <div class="category-item" data-id="<?php echo $category['id']; ?>" data-order="<?php echo $index + 1; ?>">
                                <div class="category-drag-handle">
                                    <i class="fas fa-grip-vertical"></i>
                                </div>
                                <div class="category-info">
                                    <div class="category-name"><?php echo htmlspecialchars($category['name']); ?></div>
                                    <div class="category-details">
                                        <span class="category-slug"><?php echo htmlspecialchars($category['slug']); ?></span>
                                        <?php if (!empty($category['description'])): ?>
                                            <span class="category-description"><?php echo htmlspecialchars($category['description']); ?></span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <div class="category-actions">
                                    <button onclick="editCategory(<?php echo $category['id']; ?>)" class="btn btn-sm btn-warning" title="Edit">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                    <button onclick="deleteCategory(<?php echo $category['id']; ?>)" class="btn btn-sm btn-danger" title="Delete">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Edit Category Modal -->
    <div id="editCategoryModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title">Edit Category</h3>
                <span class="close" onclick="hideModal('editCategoryModal')">&times;</span>
            </div>
            
            <form id="editCategoryForm" class="category-form">
                <input type="hidden" id="editCategoryId" name="id">
                
                <div class="form-row">
                    <div class="form-group-full">
                        <label for="editCategoryName">Category Name *</label>
                        <input type="text" id="editCategoryName" name="name" required>
                    </div>
                    <div class="form-group-full">
                        <label for="editCategorySlug">Slug *</label>
                        <input type="text" id="editCategorySlug" name="slug" required>
                    </div>
                </div>
                <div class="form-group-full">
                    <label for="editCategoryDescription">Description</label>
                    <textarea id="editCategoryDescription" name="description" rows="3"></textarea>
                </div>
                <div class="form-group-full">
                    <label for="editCategoryImage">Image URL</label>
                    <input type="text" id="editCategoryImage" name="image">
                </div>
                <div class="form-group-full">
                    <label>
                        <input type="checkbox" id="editCategoryActive" name="is_active" value="1">
                        Active
                    </label>
                </div>
                <div class="form-actions">
                    <button type="button" onclick="hideModal('editCategoryModal')" class="btn btn-outline">Cancel</button>
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-save"></i> Update Category
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Delete Category Modal -->
    <div id="deleteCategoryModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title">Delete Category</h3>
                <span class="close" onclick="hideModal('deleteCategoryModal')">&times;</span>
            </div>
            
            <form id="deleteCategoryForm">
                <input type="hidden" id="deleteCategoryId" name="id">
                
                <p>Are you sure you want to delete this category? This action cannot be undone.</p>
                <p><strong>Note:</strong> Products in this category will be moved to "Uncategorized".</p>
                
                <div class="form-actions">
                    <button type="button" onclick="hideModal('deleteCategoryModal')" class="btn btn-outline">Cancel</button>
                    <button type="submit" class="btn btn-danger">Delete Category</button>
                </div>
            </form>
        </div>
    </div>

    <script src="../assets/js/admin.js"></script>
    <script src="../assets/js/preset-manager.js"></script>
    <script>
        // Store size presets for JavaScript use
        const sizePresets = <?php echo json_encode(array_map(function($p) { return $p['preset_value']; }, $sizePresets)); ?>;

        // Global function to calculate total stock from all version variations
        function updateTotalStock() {
            const versionVariationsContainer = document.getElementById('versionVariations');
            const stockInputs = versionVariationsContainer.querySelectorAll('.stock-input');
            let totalStock = 0;

            stockInputs.forEach(input => {
                const value = parseInt(input.value) || 0;
                totalStock += value;
            });

            document.getElementById('stock_quantity').value = totalStock;
        }

        (function(){
            const addBtn = document.querySelector('.header .btn.btn-primary');
            async function ensurePresets(){
                try {
                    const res = await fetch('../ajax/presets.php');
                    if (!res.ok) {
                        return;
                    }
                    const json = await res.json();
                    const groups = json && json.data ? json.data : {};
                    const hasBrand = Array.isArray(groups.brand) && groups.brand.length > 0;
                    const hasVolume = Array.isArray(groups.volume) && groups.volume.length > 0;
                    const hasScent = Array.isArray(groups.scent) && groups.scent.length > 0;
                    const hasNotes = Array.isArray(groups.notes) && groups.notes.length > 0;
                    const hasSeason = Array.isArray(groups.season) && groups.season.length > 0;
                    const hasRecommendation = Array.isArray(groups.recommendation) && groups.recommendation.length > 0;
                    
                    if (!hasBrand || !hasVolume || !hasScent || !hasNotes || !hasSeason || !hasRecommendation) {
                        const bar = document.createElement('div');
                        bar.className = 'alert';
                        bar.style.backgroundColor = '#fff3cd';
                        bar.style.color = '#856404';
                        bar.style.border = '1px solid #ffeeba';
                        bar.style.marginBottom = '15px';
                        bar.innerHTML = '<i class="fas fa-info-circle"></i> Please configure presets for <strong>Brand</strong>, <strong>Volume</strong>, <strong>Scent</strong>, <strong>Notes</strong>, <strong>Season</strong>, and <strong>Recommendation</strong> before adding products. <a href="presets.php" style="text-decoration:underline">Open Presets</a>';
                        const container = document.querySelector('.page-content');
                        if (container && !container.querySelector('.alert')) { container.prepend(bar); }
                    }
                } catch (e) {
                    // Preset check failed, but allow button to work
                }
            }
            ensurePresets();
            
            // Version variations management
            const versionVariationsContainer = document.getElementById('versionVariations');
            const addVersionVariationBtn = document.getElementById('addVersionVariation');
            
            function addVersionVariation() {
                const newVariation = document.createElement('div');
                newVariation.className = 'version-variation-item';
                newVariation.innerHTML = `
                    <div class="form-row">
                        <div class="form-group-full">
                            <select name="version_types[]" class="version-select" required>
                                <option value="">Select Version</option>
                                <option value="EDP / LIGHT">EDP / LIGHT</option>
                                <option value="PARFUM / MEDIUM">PARFUM / MEDIUM</option>
                                <option value="EXTRAIT / STRONG">EXTRAIT / STRONG</option>
                            </select>
                        </div>
                        <div class="form-group-full">
                            <select name="version_sizes[]" class="size-select" required>
                                <option value="">Select Size</option>
                                ${sizePresets.map(size => `<option value="${size}">${size}</option>`).join('')}
                            </select>
                        </div>
                        <div class="form-group-full">
                            <input type="number" name="version_prices[]" step="0.01" placeholder="Price" class="price-input" required>
                        </div>
                        <div class="form-group-full">
                            <input type="number" name="version_stock[]" placeholder="Stock" class="stock-input" required>
                        </div>
                        <div class="form-group-full">
                            <button type="button" class="btn btn-outline btn-sm remove-version-variation">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                    </div>
                `;
                versionVariationsContainer.appendChild(newVariation);
                updateVersionRemoveButtons();
            }
            
            function updateVersionRemoveButtons() {
                const items = versionVariationsContainer.querySelectorAll('.version-variation-item');
                items.forEach((item, index) => {
                    const removeBtn = item.querySelector('.remove-version-variation');
                    if (items.length > 1) {
                        removeBtn.style.display = 'inline-block';
                    } else {
                        removeBtn.style.display = 'none';
                    }
                });
            }
            
            function removeVersionVariation(btn) {
                const item = btn.closest('.version-variation-item');
                item.remove();
                updateVersionRemoveButtons();
            }
            
            // Event listeners for version variations
            addVersionVariationBtn.addEventListener('click', function() {
                addVersionVariation();
                updateTotalStock();
            });
            
            versionVariationsContainer.addEventListener('click', function(e) {
                if (e.target.closest('.remove-version-variation')) {
                    removeVersionVariation(e.target.closest('.remove-version-variation'));
                    updateTotalStock();
                }
            });
            
            // Listen for changes in stock inputs to auto-calculate total
            versionVariationsContainer.addEventListener('input', function(e) {
                if (e.target.classList.contains('stock-input')) {
                    updateTotalStock();
                }
            });

            // Initialize version variations
            updateVersionRemoveButtons();
            
            // Calculate initial total stock on page load (for edit mode)
            updateTotalStock();
        })();
    </script>
    <style>
        .image-upload-container {
            display: flex;
            flex-direction: column;
            gap: 15px;
        }
        
        .image-upload {
            border: 2px dashed #ddd;
            border-radius: 8px;
            padding: 20px;
            text-align: center;
            transition: border-color 0.3s;
            position: relative;
        }
        
        .image-upload:hover {
            border-color: #000;
        }
        
        .image-upload input[type="file"] {
            display: none;
        }
        
        .upload-label {
            cursor: pointer;
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 10px;
        }
        
        .upload-label i {
            font-size: 48px;
            color: #ccc;
        }
        
        .upload-label span {
            font-size: 16px;
            color: #000;
            font-weight: 600;
        }
        
        .upload-label small {
            font-size: 12px;
            color: #666;
        }
        
        .image-preview {
            max-width: 200px;
            max-height: 200px;
            border-radius: 8px;
            margin: 10px auto;
            display: block;
        }
        
        .image-url-section {
            border-top: 1px solid #eee;
            padding-top: 15px;
        }
        
        .image-url-section label {
            font-size: 14px;
            color: #666;
            margin-bottom: 8px;
            display: block;
        }
        
        .image-url-section input {
            width: 100%;
            padding: 8px 12px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 14px;
        }
        
        /* Volume Variations Styles */
        .variations-container {
            border: 1px solid #ddd;
            border-radius: 8px;
            padding: 15px;
            margin-bottom: 15px;
            background-color: #f9f9f9;
        }
        
        .variation-item {
            margin-bottom: 15px;
            padding: 10px;
            border: 1px solid #e0e0e0;
            border-radius: 6px;
            background-color: white;
        }
        
        .variation-item:last-child {
            margin-bottom: 0;
        }
        
        .variation-item .form-row {
            display: grid;
            grid-template-columns: 2fr 1fr 1fr auto;
            gap: 10px;
            align-items: end;
        }
        
        .variation-item .form-group-full {
            margin-bottom: 0;
        }
        
        .variation-item .volume-select,
        .variation-item .price-input,
        .variation-item .stock-input {
            width: 100%;
            padding: 8px 12px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 14px;
        }
        
        .variation-item .remove-variation {
            padding: 8px 12px;
            height: 38px;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        #addVolumeVariation {
            margin-top: 10px;
        }
        
        /* Version Variations Styles */
        .version-variation-item {
            margin-bottom: 15px;
            padding: 10px;
            border: 1px solid #e0e0e0;
            border-radius: 6px;
            background-color: white;
        }
        
        .version-variation-item:last-child {
            margin-bottom: 0;
        }
        
        .version-variation-item .form-row {
            display: grid;
            grid-template-columns: 2fr 1fr 1fr 1fr auto;
            gap: 10px;
            align-items: end;
        }
        
        .version-variation-item .form-group-full {
            margin-bottom: 0;
        }
        
        .version-variation-item .version-select,
        .version-variation-item .size-select,
        .version-variation-item .price-input,
        .version-variation-item .stock-input {
            width: 100%;
            padding: 8px 12px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 14px;
        }
        
        .version-variation-item .remove-version-variation {
            padding: 8px 12px;
            height: 38px;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        #addVersionVariation {
            margin-top: 10px;
        }
        
        /* Responsive Modal Styles */
        .modal-content {
            width: 95%;
            max-width: 800px;
            margin: 2% auto;
            max-height: 95vh;
            overflow-y: auto;
        }
        
        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
            margin-bottom: 20px;
        }
        
        .form-group-full {
            margin-bottom: 20px;
        }
        
        .form-group-full label {
            display: block;
            margin-bottom: 8px;
            color: #000;
            font-weight: 600;
            font-size: 14px;
        }
        
        .form-group-full input,
        .form-group-full textarea,
        .form-group-full select {
            width: 100%;
            padding: 12px 16px;
            border: 2px solid #ddd;
            border-radius: 6px;
            font-size: 16px;
            transition: border-color 0.3s;
            box-sizing: border-box;
        }
        
        .form-group-full input:focus,
        .form-group-full textarea:focus,
        .form-group-full select:focus {
            outline: none;
            border-color: #000;
        }
        
        .form-group-full textarea {
            min-height: 100px;
            resize: vertical;
        }
        
        .form-actions {
            display: flex;
            gap: 15px;
            justify-content: flex-end;
            margin-top: 30px;
            flex-wrap: wrap;
        }
        
        .btn {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            padding: 12px 24px;
            border: none;
            border-radius: 6px;
            font-size: 16px;
            font-weight: 600;
            text-decoration: none;
            cursor: pointer;
            transition: all 0.3s;
            white-space: nowrap;
        }
        
        .btn-primary {
            background-color: #000;
            color: white;
        }
        
        .btn-primary:hover {
            background-color: #333;
        }
        
        .btn-outline {
            background-color: transparent;
            color: #000;
            border: 2px solid #000;
        }
        
        .btn-outline:hover {
            background-color: #000;
            color: white;
        }
        
        /* Mobile Responsive Styles */
        @media (max-width: 768px) {
            .modal-content {
                width: 98%;
                margin: 1% auto;
                padding: 20px;
                max-height: 98vh;
            }
            
            .form-row {
                grid-template-columns: 1fr;
                gap: 15px;
            }
            
            .form-group-full input,
            .form-group-full textarea,
            .form-group-full select {
                font-size: 16px; /* Prevents zoom on iOS */
                padding: 14px 16px;
            }
            
            .form-actions {
                flex-direction: column;
                gap: 10px;
            }
            
            .btn {
                width: 100%;
                padding: 14px 24px;
            }
            
            .upload-label i {
                font-size: 36px;
            }
            
            .upload-label span {
                font-size: 14px;
            }
            
            .image-preview {
                max-width: 150px;
                max-height: 150px;
            }
            
            .image-upload {
                padding: 15px;
            }
        }
        
        @media (max-width: 480px) {
            .modal-content {
                padding: 15px;
            }
            
            .form-group-full input,
            .form-group-full textarea,
            .form-group-full select {
                padding: 12px 14px;
            }
            
            .upload-label i {
                font-size: 32px;
            }
            
            .upload-label span {
                font-size: 13px;
            }
            
            .upload-label small {
                font-size: 11px;
            }
            
            .image-preview {
                max-width: 120px;
                max-height: 120px;
            }
        }
        
        /* Tablet Styles */
        @media (min-width: 769px) and (max-width: 1024px) {
            .modal-content {
                width: 90%;
                max-width: 700px;
            }
            
            .form-row {
                gap: 18px;
            }
        }
        
        /* Large Desktop Styles */
        @media (min-width: 1200px) {
            .modal-content {
                max-width: 900px;
            }
            
            .form-row {
                gap: 25px;
            }
        }

        /* Category Management Styles */
        .category-manager-content {
            max-height: 70vh;
            overflow-y: auto;
        }

        .add-category-section {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 20px;
            border: 1px solid #e9ecef;
        }

        .add-category-section h4 {
            color: #1A4D2A;
            margin-bottom: 15px;
            font-size: 16px;
        }

        .categories-list-section h4 {
            color: #1A4D2A;
            margin-bottom: 15px;
            font-size: 16px;
        }

        .categories-list-section h4 small {
            color: #666;
            font-weight: normal;
            font-size: 12px;
        }

        .categories-list {
            border: 1px solid #e9ecef;
            border-radius: 8px;
            background: white;
        }

        .category-item {
            display: flex;
            align-items: center;
            padding: 15px;
            border-bottom: 1px solid #f0f0f0;
            background: white;
            transition: all 0.3s ease;
            cursor: move;
        }

        .category-item:last-child {
            border-bottom: none;
        }

        .category-item:hover {
            background: #f8f9fa;
        }

        .category-item.dragging {
            opacity: 0.5;
            transform: rotate(5deg);
        }

        .category-drag-handle {
            margin-right: 15px;
            color: #666;
            cursor: grab;
            padding: 5px;
        }

        .category-drag-handle:active {
            cursor: grabbing;
        }

        .category-info {
            flex: 1;
        }

        .category-name {
            font-weight: 600;
            color: #1A4D2A;
            margin-bottom: 5px;
        }

        .category-details {
            display: flex;
            gap: 15px;
            font-size: 12px;
            color: #666;
        }

        .category-slug {
            background: #e9ecef;
            padding: 2px 6px;
            border-radius: 3px;
            font-family: monospace;
        }

        .category-description {
            font-style: italic;
        }

        .category-actions {
            display: flex;
            gap: 5px;
        }

        .category-form {
            background: white;
            padding: 0;
        }

        .category-form .form-row {
            margin-bottom: 15px;
        }

        .category-form .form-group-full {
            margin-bottom: 15px;
        }

        .category-form .form-actions {
            margin-top: 20px;
        }

        /* Drag and Drop Visual Feedback */
        .categories-list.drag-over {
            background: #f0f8ff;
            border-color: #007bff;
        }

        .category-item.drag-over-top {
            border-top: 3px solid #007bff;
        }

        .category-item.drag-over-bottom {
            border-bottom: 3px solid #007bff;
        }

        /* Mobile Responsive for Category Management */
        @media (max-width: 768px) {
            .category-manager-content {
                max-height: 80vh;
            }

            .add-category-section {
                padding: 15px;
            }

            .category-item {
                padding: 12px;
                flex-direction: column;
                align-items: flex-start;
            }

            .category-drag-handle {
                margin-right: 0;
                margin-bottom: 10px;
                align-self: flex-end;
            }

            .category-info {
                width: 100%;
                margin-bottom: 10px;
            }

            .category-details {
                flex-direction: column;
                gap: 5px;
            }

            .category-actions {
                width: 100%;
                justify-content: flex-end;
            }
        }
    </style>
    <script>
        function openAddProductModal() {
            // Reset the form for adding a new product
            resetProductForm();

            // Show the modal
            showModal('addProductModal');
        }

        function resetProductForm() {
            // Update modal title
            document.querySelector('#addProductModal .modal-title').textContent = 'Add New Product';

            // Reset form
            const form = document.querySelector('#addProductModal form');
            form.reset();

            // Set action to add
            const actionInput = form.querySelector('input[name="action"]');
            actionInput.value = 'add';

            // Remove product_id if it exists
            const productIdInput = form.querySelector('input[name="product_id"]');
            if (productIdInput) {
                productIdInput.remove();
            }

            // Reset version variations to one empty variation
            const variationsContainer = document.getElementById('versionVariations');
            variationsContainer.innerHTML = '';
            addEmptyVersionVariation();
            attachVersionVariationListeners();

            // Reset submit button text
            const submitBtn = document.querySelector('#addProductModal .btn-primary[type="submit"]');
            if (submitBtn) {
                submitBtn.textContent = 'Add Product';
            }
        }
        
        function editProduct(productId) {
            // Fetch product data via AJAX instead of reloading the page
            fetch('get-product.php?id=' + productId)
                .then(response => {
                    if (!response.ok) {
                        throw new Error('Failed to fetch product data');
                    }
                    return response.json();
                })
                .then(product => {
                    populateEditForm(product);
                    showModal('addProductModal');
                })
                .catch(error => {
                    console.error('Error:', error);
                    showNotification('Error loading product data. Please try again.', 'error');
                });
        }

        function populateEditForm(product) {
            // Update modal title
            document.querySelector('#addProductModal .modal-title').textContent = 'Edit Product';

            // Set form action to edit
            const form = document.querySelector('#addProductModal form');
            const actionInput = form.querySelector('input[name="action"]');
            actionInput.value = 'edit';

            // Add or update product_id hidden field
            let productIdInput = form.querySelector('input[name="product_id"]');
            if (!productIdInput) {
                productIdInput = document.createElement('input');
                productIdInput.type = 'hidden';
                productIdInput.name = 'product_id';
                actionInput.parentNode.insertBefore(productIdInput, actionInput.nextSibling);
            }
            productIdInput.value = product.id;

            // Populate basic fields
            document.getElementById('name').value = product.name || '';
            document.getElementById('brand').value = product.brand || '';
            document.getElementById('description').value = product.description || '';
            document.getElementById('currency').value = product.currency || 'RM';
            document.getElementById('display_price_mode').value = product.display_price_mode || 'range';
            document.getElementById('scent').value = product.scent || '';
            document.getElementById('notes').value = product.notes || '';
            document.getElementById('season').value = product.season || '';
            document.getElementById('recommendation').value = product.recommendation || '';
            document.getElementById('category_id').value = product.category_id || '';
            document.getElementById('stock_quantity').value = product.stock_quantity || '0';

            // Populate image fields
            document.getElementById('image').value = product.image || '';
            document.getElementById('image_2').value = product.image_2 || '';
            document.getElementById('image_3').value = product.image_3 || '';
            document.getElementById('image_4').value = product.image_4 || '';
            document.getElementById('image_5').value = product.image_5 || '';
            document.getElementById('image_6').value = product.image_6 || '';

            // Populate checkboxes
            document.getElementById('is_bestseller').checked = product.is_bestseller == 1;
            document.getElementById('is_new_arrival').checked = product.is_new_arrival == 1;
            document.getElementById('is_active').checked = product.is_active == 1;
            document.getElementById('is_out_of_stock').checked = product.is_out_of_stock == 1;

            // Populate restock fields
            document.getElementById('restock_date').value = product.restock_date || '';
            document.getElementById('restock_message').value = product.restock_message || '';

            // Populate version variations
            const variationsContainer = document.getElementById('versionVariations');
            variationsContainer.innerHTML = ''; // Clear existing variations

            if (product.version_variations && product.version_variations.length > 0) {
                product.version_variations.forEach((variation, index) => {
                    // Build size options dynamically
                    const sizeOptions = sizePresets.map(size =>
                        `<option value="${size}" ${variation.size == size ? 'selected' : ''}>${size}</option>`
                    ).join('');

                    const variationHtml = `
                        <div class="version-variation-item">
                            <div class="form-row">
                                <div class="form-group-full">
                                    <select name="version_types[]" class="version-select" required>
                                        <option value="">Select Version</option>
                                        <option value="EDP / LIGHT" ${variation.version_type == 'EDP / LIGHT' ? 'selected' : ''}>EDP / LIGHT</option>
                                        <option value="PARFUM / MEDIUM" ${variation.version_type == 'PARFUM / MEDIUM' ? 'selected' : ''}>PARFUM / MEDIUM</option>
                                        <option value="EXTRAIT / STRONG" ${variation.version_type == 'EXTRAIT / STRONG' ? 'selected' : ''}>EXTRAIT / STRONG</option>
                                    </select>
                                </div>
                                <div class="form-group-full">
                                    <select name="version_sizes[]" class="size-select" required>
                                        <option value="">Select Size</option>
                                        ${sizeOptions}
                                    </select>
                                </div>
                                <div class="form-group-full">
                                    <input type="number" name="version_prices[]" step="0.01" placeholder="Price" class="price-input" value="${variation.price}" required>
                                </div>
                                <div class="form-group-full">
                                    <input type="number" name="version_stock[]" placeholder="Stock" class="stock-input" value="${variation.stock_quantity}" required>
                                </div>
                                <div class="form-group-full">
                                    <button type="button" class="btn btn-outline btn-sm remove-version-variation">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            </div>
                        </div>
                    `;
                    variationsContainer.insertAdjacentHTML('beforeend', variationHtml);
                });
            } else {
                // Add one empty variation if none exist
                addEmptyVersionVariation();
            }

            // Re-attach event listeners for remove buttons
            attachVersionVariationListeners();
            updateTotalStock();

            // Update restock fields visibility
            const restockFields = document.getElementById('restock_fields');
            if (product.is_out_of_stock == 1) {
                restockFields.style.display = 'block';
            } else {
                restockFields.style.display = 'none';
            }

            // Update submit button text
            const submitBtn = document.querySelector('#addProductModal .btn-primary[type="submit"]');
            if (submitBtn) {
                submitBtn.textContent = 'Update Product';
            }
        }

        function attachVersionVariationListeners() {
            // Update remove button visibility
            const items = document.querySelectorAll('#versionVariations .version-variation-item');
            items.forEach((item, index) => {
                const removeBtn = item.querySelector('.remove-version-variation');
                if (items.length > 1) {
                    removeBtn.style.display = 'inline-block';
                } else {
                    removeBtn.style.display = 'none';
                }
            });
        }

        function addEmptyVersionVariation() {
            const variationsContainer = document.getElementById('versionVariations');

            // Build size options dynamically
            const sizeOptions = sizePresets.map(size =>
                `<option value="${size}">${size}</option>`
            ).join('');

            const variationHtml = `
                <div class="version-variation-item">
                    <div class="form-row">
                        <div class="form-group-full">
                            <select name="version_types[]" class="version-select" required>
                                <option value="">Select Version</option>
                                <option value="EDP / LIGHT">EDP / LIGHT</option>
                                <option value="PARFUM / MEDIUM">PARFUM / MEDIUM</option>
                                <option value="EXTRAIT / STRONG">EXTRAIT / STRONG</option>
                            </select>
                        </div>
                        <div class="form-group-full">
                            <select name="version_sizes[]" class="size-select" required>
                                <option value="">Select Size</option>
                                ${sizeOptions}
                            </select>
                        </div>
                        <div class="form-group-full">
                            <input type="number" name="version_prices[]" step="0.01" placeholder="Price" class="price-input" required>
                        </div>
                        <div class="form-group-full">
                            <input type="number" name="version_stock[]" placeholder="Stock" class="stock-input" required>
                        </div>
                        <div class="form-group-full">
                            <button type="button" class="btn btn-outline btn-sm remove-version-variation">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                    </div>
                </div>
            `;
            variationsContainer.insertAdjacentHTML('beforeend', variationHtml);
        }
        
        function deleteProduct(productId) {
            document.getElementById('deleteProductId').value = productId;
            showModal('deleteProductModal');
        }
        
        // Search functionality
        document.getElementById('productSearch').addEventListener('input', function() {
            const searchTerm = this.value.toLowerCase();
            const rows = document.querySelectorAll('#productsTable tbody tr');
            
            rows.forEach(row => {
                const text = row.textContent.toLowerCase();
                if (text.includes(searchTerm)) {
                    row.style.display = '';
                } else {
                    row.style.display = 'none';
                }
            });
        });
        
        // Category filter
        document.getElementById('categoryFilter').addEventListener('change', function() {
            const filterValue = this.value;
            const rows = document.querySelectorAll('#productsTable tbody tr');
            
            rows.forEach(row => {
                if (filterValue === '' || row.textContent.includes(filterValue)) {
                    row.style.display = '';
                } else {
                    row.style.display = 'none';
                }
            });
        });
        
        // Status filter
        document.getElementById('statusFilter').addEventListener('change', function() {
            const filterValue = this.value;
            const rows = document.querySelectorAll('#productsTable tbody tr');
            
            rows.forEach(row => {
                if (filterValue === '') {
                    row.style.display = '';
                } else if (filterValue === 'active' && row.textContent.includes('Active')) {
                    row.style.display = '';
                } else if (filterValue === 'inactive' && row.textContent.includes('Inactive')) {
                    row.style.display = '';
                } else if (filterValue === 'bestseller' && row.textContent.includes('Best Seller')) {
                    row.style.display = '';
                } else if (filterValue === 'new' && row.textContent.includes('New')) {
                    row.style.display = '';
                } else if (filterValue === 'out_of_stock' && row.textContent.includes('Out of Stock')) {
                    row.style.display = '';
                } else {
                    row.style.display = 'none';
                }
            });
        });
        
        // Image preview function
        function previewImage(input) {
            if (input.files && input.files[0]) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    const preview = document.getElementById('image_preview');
                    preview.src = e.target.result;
                    preview.style.display = 'block';
                    document.querySelector('.upload-label').style.display = 'none';
                };
                reader.readAsDataURL(input.files[0]);
            }
        }
        
        // Second image preview function
        function previewImage2(input) {
            if (input.files && input.files[0]) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    const preview = document.getElementById('image_2_preview');
                    preview.src = e.target.result;
                    preview.style.display = 'block';
                    input.closest('.image-upload').querySelector('.upload-label').style.display = 'none';
                };
                reader.readAsDataURL(input.files[0]);
            }
        }
        
        // Third image preview function
        function previewImage3(input) {
            if (input.files && input.files[0]) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    const preview = document.getElementById('image_3_preview');
                    preview.src = e.target.result;
                    preview.style.display = 'block';
                    input.closest('.image-upload').querySelector('.upload-label').style.display = 'none';
                };
                reader.readAsDataURL(input.files[0]);
            }
        }
        
        // Fourth image preview function
        function previewImage4(input) {
            if (input.files && input.files[0]) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    const preview = document.getElementById('image_4_preview');
                    preview.src = e.target.result;
                    preview.style.display = 'block';
                    input.closest('.image-upload').querySelector('.upload-label').style.display = 'none';
                };
                reader.readAsDataURL(input.files[0]);
            }
        }
        
        // Fifth image preview function
        function previewImage5(input) {
            if (input.files && input.files[0]) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    const preview = document.getElementById('image_5_preview');
                    preview.src = e.target.result;
                    preview.style.display = 'block';
                    input.closest('.image-upload').querySelector('.upload-label').style.display = 'none';
                };
                reader.readAsDataURL(input.files[0]);
            }
        }
        
        // Sixth image preview function
        function previewImage6(input) {
            if (input.files && input.files[0]) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    const preview = document.getElementById('image_6_preview');
                    preview.src = e.target.result;
                    preview.style.display = 'block';
                    input.closest('.image-upload').querySelector('.upload-label').style.display = 'none';
                };
                reader.readAsDataURL(input.files[0]);
            }
        }
        
        // Check for addProduct hash on page load
        document.addEventListener('DOMContentLoaded', function() {
            if (window.location.hash === '#addProduct') {
                // Clear the hash from URL
                window.history.replaceState('', document.title, window.location.pathname);
                // Show the add product modal
                showModal('addProductModal');
            }
        });

        // Category Management Functions
        function openCategoryManager() {
            showModal('categoryManagerModal');
            initializeDragAndDrop();
        }

        function editCategory(categoryId) {
            // Fetch category data and populate edit form
            fetch(`../ajax/get_category.php?id=${categoryId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        const category = data.category;
                        document.getElementById('editCategoryId').value = category.id;
                        document.getElementById('editCategoryName').value = category.name;
                        document.getElementById('editCategorySlug').value = category.slug;
                        document.getElementById('editCategoryDescription').value = category.description || '';
                        document.getElementById('editCategoryImage').value = category.image || '';
                        document.getElementById('editCategoryActive').checked = category.is_active == 1;
                        
                        hideModal('categoryManagerModal');
                        showModal('editCategoryModal');
                    } else {
                        showNotification('Error loading category data', 'error');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    showNotification('Error loading category data', 'error');
                });
        }

        function deleteCategory(categoryId) {
            document.getElementById('deleteCategoryId').value = categoryId;
            hideModal('categoryManagerModal');
            showModal('deleteCategoryModal');
        }

        // Auto-generate slug from category name
        document.getElementById('categoryName').addEventListener('input', function() {
            const slug = this.value.toLowerCase()
                .replace(/[^a-z0-9\s-]/g, '')
                .replace(/\s+/g, '-')
                .replace(/-+/g, '-')
                .trim('-');
            document.getElementById('categorySlug').value = slug;
        });

        // Category form submissions
        document.getElementById('addCategoryForm').addEventListener('submit', function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            formData.append('action', 'add_category');
            
            fetch('../ajax/category_manager.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showNotification('Category added successfully!', 'success');
                    this.reset();
                    refreshCategoriesList();
                } else {
                    showNotification(data.message || 'Error adding category', 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showNotification('Error adding category', 'error');
            });
        });

        document.getElementById('editCategoryForm').addEventListener('submit', function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            formData.append('action', 'edit_category');
            
            fetch('../ajax/category_manager.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showNotification('Category updated successfully!', 'success');
                    hideModal('editCategoryModal');
                    refreshCategoriesList();
                } else {
                    showNotification(data.message || 'Error updating category', 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showNotification('Error updating category', 'error');
            });
        });

        document.getElementById('deleteCategoryForm').addEventListener('submit', function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            formData.append('action', 'delete_category');
            
            fetch('../ajax/category_manager.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showNotification('Category deleted successfully!', 'success');
                    hideModal('deleteCategoryModal');
                    refreshCategoriesList();
                    // Refresh the page to update category filter
                    setTimeout(() => {
                        location.reload();
                    }, 1000);
                } else {
                    showNotification(data.message || 'Error deleting category', 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showNotification('Error deleting category', 'error');
            });
        });

        function refreshCategoriesList() {
            fetch('../ajax/get_categories.php')
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        const categoriesList = document.getElementById('categoriesList');
                        categoriesList.innerHTML = '';
                        
                        data.categories.forEach((category, index) => {
                            const categoryItem = document.createElement('div');
                            categoryItem.className = 'category-item';
                            categoryItem.setAttribute('data-id', category.id);
                            categoryItem.setAttribute('data-order', index + 1);
                            categoryItem.innerHTML = `
                                <div class="category-drag-handle">
                                    <i class="fas fa-grip-vertical"></i>
                                </div>
                                <div class="category-info">
                                    <div class="category-name">${category.name}</div>
                                    <div class="category-details">
                                        <span class="category-slug">${category.slug}</span>
                                        ${category.description ? `<span class="category-description">${category.description}</span>` : ''}
                                    </div>
                                </div>
                                <div class="category-actions">
                                    <button onclick="editCategory(${category.id})" class="btn btn-sm btn-warning" title="Edit">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                    <button onclick="deleteCategory(${category.id})" class="btn btn-sm btn-danger" title="Delete">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            `;
                            categoriesList.appendChild(categoryItem);
                        });
                        
                        initializeDragAndDrop();
                    }
                })
                .catch(error => {
                    console.error('Error refreshing categories:', error);
                });
        }

        // Drag and Drop functionality
        function initializeDragAndDrop() {
            const categoriesList = document.getElementById('categoriesList');
            const categoryItems = categoriesList.querySelectorAll('.category-item');
            
            categoryItems.forEach(item => {
                item.draggable = true;
                
                item.addEventListener('dragstart', function(e) {
                    this.classList.add('dragging');
                    e.dataTransfer.effectAllowed = 'move';
                    e.dataTransfer.setData('text/html', this.outerHTML);
                });
                
                item.addEventListener('dragend', function(e) {
                    this.classList.remove('dragging');
                });
                
                item.addEventListener('dragover', function(e) {
                    e.preventDefault();
                    e.dataTransfer.dropEffect = 'move';
                    
                    const dragging = document.querySelector('.dragging');
                    if (dragging && dragging !== this) {
                        const rect = this.getBoundingClientRect();
                        const midY = rect.top + rect.height / 2;
                        
                        if (e.clientY < midY) {
                            this.classList.add('drag-over-top');
                            this.classList.remove('drag-over-bottom');
                        } else {
                            this.classList.add('drag-over-bottom');
                            this.classList.remove('drag-over-top');
                        }
                    }
                });
                
                item.addEventListener('dragleave', function(e) {
                    this.classList.remove('drag-over-top', 'drag-over-bottom');
                });
                
                item.addEventListener('drop', function(e) {
                    e.preventDefault();
                    this.classList.remove('drag-over-top', 'drag-over-bottom');
                    
                    const dragging = document.querySelector('.dragging');
                    if (dragging && dragging !== this) {
                        const rect = this.getBoundingClientRect();
                        const midY = rect.top + rect.height / 2;
                        
                        if (e.clientY < midY) {
                            this.parentNode.insertBefore(dragging, this);
                        } else {
                            this.parentNode.insertBefore(dragging, this.nextSibling);
                        }
                        
                        updateCategoryOrder();
                    }
                });
            });
        }

        function updateCategoryOrder() {
            const categoryItems = document.querySelectorAll('.category-item');
            const orderData = [];
            
            categoryItems.forEach((item, index) => {
                orderData.push({
                    id: item.getAttribute('data-id'),
                    order: index + 1
                });
            });
            
            fetch('../ajax/category_manager.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    action: 'update_order',
                    categories: orderData
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showNotification('Category order updated!', 'success');
                } else {
                    showNotification('Error updating category order', 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showNotification('Error updating category order', 'error');
            });
        }

        // Toggle restock fields based on out of stock checkbox
        function toggleRestockFields() {
            const outOfStockCheckbox = document.getElementById('is_out_of_stock');
            const restockFields = document.getElementById('restock_fields');
            
            if (outOfStockCheckbox.checked) {
                restockFields.style.display = 'block';
            } else {
                restockFields.style.display = 'none';
                // Clear restock fields when unchecked
                document.getElementById('restock_date').value = '';
                document.getElementById('restock_message').value = '';
            }
        }

        // Form validation for out of stock
        document.addEventListener('DOMContentLoaded', function() {
            const form = document.querySelector('form[data-validate]');
            if (form) {
                form.addEventListener('submit', function(e) {
                    const outOfStockCheckbox = document.getElementById('is_out_of_stock');
                    const restockDate = document.getElementById('restock_date');
                    
                    if (outOfStockCheckbox && outOfStockCheckbox.checked) {
                        if (!restockDate.value) {
                            e.preventDefault();
                            alert('Please select an expected restock date when marking a product as out of stock.');
                            restockDate.focus();
                            return false;
                        }
                        
                        // Validate that restock date is in the future
                        const restockDateTime = new Date(restockDate.value);
                        const now = new Date();
                        
                        if (restockDateTime <= now) {
                            e.preventDefault();
                            alert('Restock date must be in the future.');
                            restockDate.focus();
                            return false;
                        }
                    }
                });
            }
        });

        // Notification function
        function showNotification(message, type = 'info') {
            const notification = document.createElement('div');
            notification.className = `notification notification-${type}`;
            notification.style.cssText = `
                position: fixed;
                top: 20px;
                right: 20px;
                padding: 15px 20px;
                border-radius: 5px;
                color: white;
                font-weight: 600;
                z-index: 10000;
                max-width: 300px;
                box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            `;
            
            if (type === 'success') {
                notification.style.backgroundColor = '#28a745';
            } else if (type === 'error') {
                notification.style.backgroundColor = '#dc3545';
            } else {
                notification.style.backgroundColor = '#007bff';
            }
            
            notification.textContent = message;
            document.body.appendChild(notification);
            
            setTimeout(() => {
                notification.remove();
            }, 3000);
        }
    </script>
</body>
</html>
</html>