<?php
// Suppress all output except JSON
ob_start();
error_reporting(0);
ini_set('display_errors', 0);

session_start();
require_once '../includes/auth-check.php';
require_once '../models/Loyalty.php';

// Clear any output buffer
ob_clean();

header('Content-Type: application/json');

// Check if user is logged in
if (!isLoggedIn()) {
    echo json_encode(['success' => false, 'message' => 'Please log in to use point codes']);
    exit;
}

$userId = getUserId();
$input = json_decode(file_get_contents('php://input'), true);

// Ensure userId is properly formatted as integer
$userId = (int) $userId;

// Debug logging
error_log("Point code attempt - User ID: " . ($userId ?: 'NULL') . ", Code: " . ($input['code'] ?? 'NULL'));

if (!isset($input['code']) || empty(trim($input['code']))) {
    echo json_encode(['success' => false, 'message' => 'Please enter a code']);
    exit;
}

// Validate user exists in database
if (!$userId) {
    echo json_encode(['success' => false, 'message' => 'Please log in to use point codes']);
    exit;
}

try {
    $loyalty = new Loyalty();
    $conn = $loyalty->getConnection();
    
    // Check if user exists in database with more detailed validation
    $userCheckQuery = "SELECT id, is_guest, first_name, last_name, email FROM users WHERE id = :user_id";
    $stmt = $conn->prepare($userCheckQuery);
    $stmt->bindValue(':user_id', $userId);
    $stmt->execute();
    $userData = $stmt->fetch();
    
    if (!$userData) {
        // User doesn't exist in database, clear session and redirect
        error_log("Point code error: User ID $userId not found in database. Session data: " . print_r($_SESSION, true));
        session_destroy();
        echo json_encode(['success' => false, 'message' => 'Invalid user session. Please log in again.']);
        exit;
    }
    
    // Additional validation - ensure user ID is valid integer
    if (!is_numeric($userId) || $userId <= 0) {
        error_log("Point code error: Invalid user ID format: $userId");
        session_destroy();
        echo json_encode(['success' => false, 'message' => 'Invalid user session. Please log in again.']);
        exit;
    }
    
    // Check if user is a guest (guests shouldn't be able to redeem point codes)
    if ($userData['is_guest']) {
        echo json_encode(['success' => false, 'message' => 'Please register an account to use point codes']);
        exit;
    }
    
    // Log successful user validation
    error_log("Point code user validation successful - User ID: $userId, Name: " . $userData['first_name'] . " " . $userData['last_name']);
    
} catch (Exception $e) {
    error_log("Point code user validation error: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => 'Error validating user: ' . $e->getMessage()]);
    exit;
}

$code = strtoupper(trim($input['code']));

try {
    $loyalty = new Loyalty();
    $conn = $loyalty->getConnection();
    
    // Check if code exists and is valid
    $query = "SELECT * FROM point_codes 
              WHERE code = :code 
              AND is_active = 1 
              AND valid_from <= CURDATE() 
              AND valid_until >= CURDATE()";
    
    $stmt = $conn->prepare($query);
    $stmt->bindValue(':code', $code);
    $stmt->execute();
    $pointCode = $stmt->fetch();
    
    if (!$pointCode) {
        echo json_encode(['success' => false, 'message' => 'Invalid or expired code']);
        exit;
    }
    
    // Check if code has reached max uses
    if ($pointCode['max_uses'] && $pointCode['current_uses'] >= $pointCode['max_uses']) {
        echo json_encode(['success' => false, 'message' => 'This code has reached its usage limit']);
        exit;
    }
    
    // Check if user has already used this code
    $usageQuery = "SELECT id FROM user_point_code_usage 
                   WHERE user_id = :user_id AND code_id = :code_id";
    
    $stmt = $conn->prepare($usageQuery);
    $stmt->bindValue(':user_id', $userId);
    $stmt->bindValue(':code_id', $pointCode['id']);
    $stmt->execute();
    
    if ($stmt->fetch()) {
        echo json_encode(['success' => false, 'message' => 'You have already used this code']);
        exit;
    }
    
    // Ensure user has a loyalty account first
    $loyaltyAccountQuery = "SELECT id FROM user_loyalty_points WHERE user_id = :user_id";
    $stmt = $conn->prepare($loyaltyAccountQuery);
    $stmt->bindValue(':user_id', $userId);
    $stmt->execute();
    
    if (!$stmt->fetch()) {
        // Initialize loyalty account for user
        $initQuery = "INSERT INTO user_loyalty_points (user_id, points, tier, total_spent) 
                      VALUES (:user_id, 0, 'bronze', 0.00)";
        $stmt = $conn->prepare($initQuery);
        $stmt->bindValue(':user_id', $userId);
        $stmt->execute();
    }
    
    // Award points to user first (without transaction)
    $points = $pointCode['points'];
    $description = 'Point code: ' . $code . ($pointCode['description'] ? ' - ' . $pointCode['description'] : '');
    
    // Double-check user exists before proceeding (production safety check)
    $finalUserCheck = "SELECT id FROM users WHERE id = :user_id";
    $stmt = $conn->prepare($finalUserCheck);
    $stmt->bindValue(':user_id', $userId);
    $stmt->execute();
    $finalUserData = $stmt->fetch();
    
    if (!$finalUserData) {
        error_log("Production error: User ID $userId not found in final check. Session destroyed.");
        session_destroy();
        echo json_encode(['success' => false, 'message' => 'Invalid user session. Please log in again.']);
        exit;
    }
    
    if ($loyalty->addPoints($userId, $points, 'bonus', $description, null, $pointCode['id'])) {
        // Record code usage (works with MyISAM - no foreign key constraints)
        $usageInsertQuery = "INSERT INTO user_point_code_usage 
                            (user_id, code_id, code, points_awarded) 
                            VALUES (:user_id, :code_id, :code, :points_awarded)";
        
        $stmt = $conn->prepare($usageInsertQuery);
        $stmt->bindValue(':user_id', $userId);
        $stmt->bindValue(':code_id', $pointCode['id']);
        $stmt->bindValue(':code', $code);
        $stmt->bindValue(':points_awarded', $points);
        
        if ($stmt->execute()) {
            // Update code usage count
            $updateUsageQuery = "UPDATE point_codes 
                                SET current_uses = current_uses + 1 
                                WHERE id = :code_id";
            
            $stmt = $conn->prepare($updateUsageQuery);
            $stmt->bindValue(':code_id', $pointCode['id']);
            $stmt->execute();
            
            echo json_encode([
                'success' => true, 
                'message' => 'Code redeemed successfully!',
                'points' => $points,
                'code' => $code
            ]);
        } else {
            error_log("Failed to insert into user_point_code_usage for user $userId");
            echo json_encode(['success' => false, 'message' => 'Failed to record code usage. Please try again.']);
        }
    } else {
        echo json_encode(['success' => false, 'message' => 'Failed to award points. Please try again.']);
    }
    
} catch (Exception $e) {
    // Log the error for debugging
    error_log("Point code redemption error for user ID: " . $userId . " - " . $e->getMessage());
    error_log("Stack trace: " . $e->getTraceAsString());
    echo json_encode(['success' => false, 'message' => 'Error processing code: ' . $e->getMessage()]);
}
?>