<?php
session_start();
require_once '../config/database.php';
require_once '../models/Order.php';
require_once '../includes/auth-check.php';

header('Content-Type: application/json');

// Check if user is logged in
if (!isset($_SESSION['user_id']) || $_SESSION['user_id'] === 'guest') {
    echo json_encode(['success' => false, 'message' => 'Not authorized']);
    exit;
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

if (!$input || !isset($input['order_id']) || !isset($input['rating'])) {
    echo json_encode(['success' => false, 'message' => 'Invalid input']);
    exit;
}

$order_id = (int)$input['order_id'];
$rating = (int)$input['rating'];
$feedback = $input['feedback'] ?? '';

// Validate rating
if ($rating < 1 || $rating > 5) {
    echo json_encode(['success' => false, 'message' => 'Invalid rating']);
    exit;
}

try {
    $order = new Order();
    
    // Verify the order belongs to the current user
    $order_data = $order->getById($order_id);
    if (!$order_data || $order_data['user_id'] != $_SESSION['user_id']) {
        echo json_encode(['success' => false, 'message' => 'Order not found or not authorized']);
        exit;
    }
    
    // Get database connection
    $pdo = getDB();
    
    // Create delivery_feedback table if it doesn't exist
    $createTableQuery = "
        CREATE TABLE IF NOT EXISTS delivery_feedback (
            id INT AUTO_INCREMENT PRIMARY KEY,
            order_id INT NOT NULL,
            user_id INT NOT NULL,
            rating INT NOT NULL,
            feedback TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            UNIQUE KEY unique_order_feedback (order_id)
        )
    ";
    $pdo->exec($createTableQuery);
    
    // Check if feedback already exists for this order
    $checkQuery = "SELECT id FROM delivery_feedback WHERE order_id = :order_id";
    $stmt = $pdo->prepare($checkQuery);
    $stmt->bindValue(':order_id', $order_id);
    $stmt->execute();
    
    if ($stmt->rowCount() > 0) {
        echo json_encode(['success' => false, 'message' => 'Feedback already submitted for this order']);
        exit;
    }
    
    // Insert feedback
    $insertQuery = "
        INSERT INTO delivery_feedback (order_id, user_id, rating, feedback) 
        VALUES (:order_id, :user_id, :rating, :feedback)
    ";
    $stmt = $pdo->prepare($insertQuery);
    $stmt->bindValue(':order_id', $order_id);
    $stmt->bindValue(':user_id', $_SESSION['user_id']);
    $stmt->bindValue(':rating', $rating);
    $stmt->bindValue(':feedback', $feedback);
    
    $result = $stmt->execute();
    
    if ($result) {
        echo json_encode(['success' => true, 'message' => 'Feedback submitted successfully']);
    } else {
        echo json_encode(['success' => false, 'message' => 'Failed to submit feedback']);
    }
    
} catch (Exception $e) {
    error_log("Error submitting feedback: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
}
?>