<?php
/**
 * Simple Analytics Tracking - Counter Based
 * Just increments counters instead of creating new rows
 */

session_start();
require_once '../config/database.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Get tracking data
$data = json_decode(file_get_contents('php://input'), true);

if (!$data || !isset($data['event_type'])) {
    echo json_encode(['success' => false, 'error' => 'Invalid data']);
    exit;
}

try {
    $conn = getDB();
    $eventType = $data['event_type'];
    
    // Handle different event types
    switch ($eventType) {
        case 'product_view':
        case 'add_favorite':
        case 'add_to_cart':
            if (!isset($data['product_id'])) {
                throw new Exception('Product ID is required');
            }
            
            $productId = (int)$data['product_id'];
            
            // Ensure product row exists
            $stmt = $conn->prepare("
                INSERT INTO product_analytics_simple (product_id, views, favorites, add_to_cart)
                VALUES (?, 0, 0, 0)
                ON DUPLICATE KEY UPDATE product_id = product_id
            ");
            $stmt->execute([$productId]);
            
            // Increment the appropriate counter
            if ($eventType === 'product_view') {
                $stmt = $conn->prepare("
                    UPDATE product_analytics_simple 
                    SET views = views + 1, 
                        last_viewed = NOW(),
                        conversion_rate = CASE 
                            WHEN views > 0 THEN ROUND((add_to_cart / views) * 100, 2)
                            ELSE 0 
                        END
                    WHERE product_id = ?
                ");
            } elseif ($eventType === 'add_favorite') {
                $stmt = $conn->prepare("
                    UPDATE product_analytics_simple 
                    SET favorites = favorites + 1,
                        last_favorited = NOW()
                    WHERE product_id = ?
                ");
            } elseif ($eventType === 'add_to_cart') {
                $stmt = $conn->prepare("
                    UPDATE product_analytics_simple 
                    SET add_to_cart = add_to_cart + 1,
                        last_added_to_cart = NOW(),
                        conversion_rate = CASE 
                            WHEN views > 0 THEN ROUND((add_to_cart / views) * 100, 2)
                            ELSE 0 
                        END
                    WHERE product_id = ?
                ");
            }
            
            $stmt->execute([$productId]);
            
            // Get updated counts
            $stmt = $conn->prepare("
                SELECT views, favorites, add_to_cart, conversion_rate 
                FROM product_analytics_simple 
                WHERE product_id = ?
            ");
            $stmt->execute([$productId]);
            $stats = $stmt->fetch(PDO::FETCH_ASSOC);
            
            echo json_encode([
                'success' => true,
                'tracked' => true,
                'event_type' => $eventType,
                'product_id' => $productId,
                'updated_stats' => $stats
            ]);
            break;
            
        default:
            throw new Exception('Invalid event type');
    }
    
} catch (Exception $e) {
    error_log("Analytics tracking error: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
?>
