<?php
session_start();
require_once '../models/Order.php';
require_once '../includes/auth-check.php';

header('Content-Type: application/json');

// Check if user is logged in
if (!isset($_SESSION['user_id']) || $_SESSION['user_id'] === 'guest') {
    echo json_encode(['success' => false, 'message' => 'Not authorized']);
    exit;
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

if (!$input || !isset($input['order_id']) || !isset($input['status'])) {
    echo json_encode(['success' => false, 'message' => 'Invalid input']);
    exit;
}

$order_id = (int)$input['order_id'];
$status = $input['status'];

try {
    $order = new Order();
    
    // Verify the order belongs to the current user
    $order_data = $order->getById($order_id);
    if (!$order_data || $order_data['user_id'] != $_SESSION['user_id']) {
        echo json_encode(['success' => false, 'message' => 'Order not found or not authorized']);
        exit;
    }
    
    // Update the order status
    $result = $order->updateStatus($order_id, $status);
    
    if ($result) {
        // Award loyalty points when order is marked as delivered
        if ($status === 'delivered') {
            try {
                require_once '../models/Loyalty.php';
                $loyalty = new Loyalty();
                
                // Get order items with product details for point calculation
                $order_items = $order->getOrderItemsForPoints($order_id);
                if (!empty($order_items)) {
                    $total_points = 0;
                    $points_awarded = [];
                    
                    foreach ($order_items as $item) {
                        if (!empty($item['point_rewards']) && $item['point_rewards'] > 0) {
                            $points_per_item = $item['point_rewards'] * $item['quantity'];
                            $total_points += $points_per_item;
                            $points_awarded[] = [
                                'product_name' => $item['product_name'],
                                'quantity' => $item['quantity'],
                                'points_per_item' => $item['point_rewards'],
                                'total_points' => $points_per_item
                            ];
                        }
                    }
                    
                    if ($total_points > 0) {
                        // Award points to user
                        $user_id = $_SESSION['user_id'];
                        $description = "Delivery confirmation reward for Order #{$order_id}";
                        
                        $points_result = $loyalty->addPoints($user_id, $total_points, 'earned', $description, $order_id, 'DELIVERY' . $order_id);
                        
                        if ($points_result) {
                            error_log("Loyalty points awarded for delivery confirmation - Order {$order_id}: {$total_points} points to user {$user_id}");
                            
                            // Log detailed points breakdown
                            foreach ($points_awarded as $award) {
                                error_log("Delivery points breakdown: {$award['product_name']} x{$award['quantity']} = {$award['total_points']} points ({$award['points_per_item']} per item)");
                            }
                            
                            // Add points info to response
                            echo json_encode([
                                'success' => true, 
                                'message' => 'Order status updated successfully',
                                'points_awarded' => $total_points,
                                'points_breakdown' => $points_awarded
                            ]);
                        } else {
                            error_log("Failed to award loyalty points for delivery confirmation - Order {$order_id}");
                            echo json_encode(['success' => true, 'message' => 'Order status updated successfully (points award failed)']);
                        }
                    } else {
                        error_log("No loyalty points to award for delivery confirmation - Order {$order_id} (no products with point rewards)");
                        echo json_encode(['success' => true, 'message' => 'Order status updated successfully']);
                    }
                } else {
                    error_log("No order items found for point calculation for delivery confirmation - Order {$order_id}");
                    echo json_encode(['success' => true, 'message' => 'Order status updated successfully']);
                }
            } catch (Exception $e) {
                error_log("Error awarding loyalty points for delivery confirmation - Order {$order_id}: " . $e->getMessage());
                echo json_encode(['success' => true, 'message' => 'Order status updated successfully (points award error)']);
            }
        } else {
            echo json_encode(['success' => true, 'message' => 'Order status updated successfully']);
        }
    } else {
        echo json_encode(['success' => false, 'message' => 'Failed to update order status']);
    }
    
} catch (Exception $e) {
    error_log("Error updating order status: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => 'An error occurred']);
}
?>