<?php
/**
 * Check Order Status Diagnostic Tool
 * This helps identify why orders are showing as "cancelled"
 */

require_once 'config/database.php';

// Enable error display
ini_set('display_errors', 1);
error_reporting(E_ALL);

echo "<h1>Order Status Diagnostic</h1>";
echo "<style>
    body { font-family: Arial, sans-serif; padding: 20px; background: #f5f5f5; }
    .section { background: white; padding: 20px; margin: 10px 0; border-radius: 5px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
    table { width: 100%; border-collapse: collapse; margin: 10px 0; }
    th, td { padding: 10px; text-align: left; border-bottom: 1px solid #ddd; }
    th { background: #f0f0f0; font-weight: bold; }
    .status-cancelled { color: red; font-weight: bold; }
    .status-pending { color: orange; }
    .status-awaiting_payment { color: blue; }
    .status-processing { color: green; }
    .status-shipped { color: purple; }
    .status-delivered { color: darkgreen; }
    .error { color: red; }
    .success { color: green; }
    .warning { color: orange; }
</style>";

try {
    $conn = getDB();
    
    // Section 1: Check order status enum values
    echo "<div class='section'>";
    echo "<h2>1. Order Status Column Configuration</h2>";
    
    $stmt = $conn->query("SHOW COLUMNS FROM orders LIKE 'status'");
    $column_info = $stmt->fetch(PDO::FETCH_ASSOC);
    
    echo "<p><strong>Column Type:</strong> " . htmlspecialchars($column_info['Type']) . "</p>";
    echo "<p><strong>Default Value:</strong> " . htmlspecialchars($column_info['Default'] ?? 'NULL') . "</p>";
    
    // Extract enum values
    preg_match("/^enum\(\'(.*)\'\)$/", $column_info['Type'], $matches);
    $enum_values = explode("','", $matches[1]);
    
    echo "<p><strong>Allowed Status Values:</strong></p>";
    echo "<ul>";
    foreach ($enum_values as $value) {
        echo "<li>" . htmlspecialchars($value) . "</li>";
    }
    echo "</ul>";
    echo "</div>";
    
    // Section 2: Recent orders and their statuses
    echo "<div class='section'>";
    echo "<h2>2. Recent Orders (Last 20)</h2>";
    
    $stmt = $conn->query("
        SELECT 
            id,
            order_number,
            status,
            payment_status,
            payment_method,
            total_amount,
            created_at,
            updated_at
        FROM orders 
        ORDER BY created_at DESC 
        LIMIT 20
    ");
    
    $orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if (empty($orders)) {
        echo "<p class='warning'>No orders found in the database.</p>";
    } else {
        echo "<table>";
        echo "<tr>
                <th>ID</th>
                <th>Order Number</th>
                <th>Status</th>
                <th>Payment Status</th>
                <th>Payment Method</th>
                <th>Total</th>
                <th>Created</th>
                <th>Updated</th>
              </tr>";
        
        foreach ($orders as $order) {
            $status_class = 'status-' . $order['status'];
            echo "<tr>";
            echo "<td>" . htmlspecialchars($order['id']) . "</td>";
            echo "<td>" . htmlspecialchars($order['order_number']) . "</td>";
            echo "<td class='{$status_class}'>" . htmlspecialchars($order['status']) . "</td>";
            echo "<td>" . htmlspecialchars($order['payment_status']) . "</td>";
            echo "<td>" . htmlspecialchars($order['payment_method']) . "</td>";
            echo "<td>RM " . number_format($order['total_amount'], 2) . "</td>";
            echo "<td>" . htmlspecialchars($order['created_at']) . "</td>";
            echo "<td>" . htmlspecialchars($order['updated_at']) . "</td>";
            echo "</tr>";
        }
        
        echo "</table>";
    }
    echo "</div>";
    
    // Section 3: Status distribution
    echo "<div class='section'>";
    echo "<h2>3. Order Status Distribution</h2>";
    
    $stmt = $conn->query("
        SELECT 
            status,
            COUNT(*) as count,
            ROUND(COUNT(*) * 100.0 / (SELECT COUNT(*) FROM orders), 2) as percentage
        FROM orders 
        GROUP BY status
        ORDER BY count DESC
    ");
    
    $distribution = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo "<table>";
    echo "<tr><th>Status</th><th>Count</th><th>Percentage</th></tr>";
    
    foreach ($distribution as $row) {
        $status_class = 'status-' . $row['status'];
        echo "<tr>";
        echo "<td class='{$status_class}'>" . htmlspecialchars($row['status']) . "</td>";
        echo "<td>" . htmlspecialchars($row['count']) . "</td>";
        echo "<td>" . htmlspecialchars($row['percentage']) . "%</td>";
        echo "</tr>";
    }
    
    echo "</table>";
    echo "</div>";
    
    // Section 4: Check for cancelled orders
    echo "<div class='section'>";
    echo "<h2>4. Cancelled Orders Analysis</h2>";
    
    $stmt = $conn->query("
        SELECT COUNT(*) as total FROM orders WHERE status = 'cancelled'
    ");
    $cancelled_count = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    if ($cancelled_count > 0) {
        echo "<p class='error'>⚠️ Found {$cancelled_count} cancelled orders</p>";
        
        // Get details of cancelled orders
        $stmt = $conn->query("
            SELECT 
                id,
                order_number,
                payment_method,
                payment_status,
                total_amount,
                created_at,
                updated_at,
                TIMESTAMPDIFF(SECOND, created_at, updated_at) as seconds_to_cancel
            FROM orders 
            WHERE status = 'cancelled'
            ORDER BY created_at DESC
            LIMIT 10
        ");
        
        $cancelled_orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        echo "<p><strong>Recent Cancelled Orders:</strong></p>";
        echo "<table>";
        echo "<tr>
                <th>Order Number</th>
                <th>Payment Method</th>
                <th>Payment Status</th>
                <th>Total</th>
                <th>Created</th>
                <th>Cancelled</th>
                <th>Time to Cancel</th>
              </tr>";
        
        foreach ($cancelled_orders as $order) {
            echo "<tr>";
            echo "<td>" . htmlspecialchars($order['order_number']) . "</td>";
            echo "<td>" . htmlspecialchars($order['payment_method']) . "</td>";
            echo "<td>" . htmlspecialchars($order['payment_status']) . "</td>";
            echo "<td>RM " . number_format($order['total_amount'], 2) . "</td>";
            echo "<td>" . htmlspecialchars($order['created_at']) . "</td>";
            echo "<td>" . htmlspecialchars($order['updated_at']) . "</td>";
            echo "<td>" . ($order['seconds_to_cancel'] > 0 ? $order['seconds_to_cancel'] . " seconds" : "Immediately") . "</td>";
            echo "</tr>";
        }
        
        echo "</table>";
        
        echo "<div style='background: #fff3cd; padding: 15px; margin-top: 15px; border-left: 4px solid #ffc107;'>";
        echo "<h3>Possible Reasons for Cancelled Status:</h3>";
        echo "<ol>";
        echo "<li><strong>User Cancelled:</strong> Customer manually cancelled the order</li>";
        echo "<li><strong>Payment Failed:</strong> Payment gateway returned failure and order was auto-cancelled</li>";
        echo "<li><strong>Admin Action:</strong> Admin manually changed status to cancelled</li>";
        echo "<li><strong>System Error:</strong> Bug in payment callback or order processing</li>";
        echo "</ol>";
        echo "</div>";
        
    } else {
        echo "<p class='success'>✓ No cancelled orders found</p>";
    }
    echo "</div>";
    
    // Section 5: Check payment callback logs
    echo "<div class='section'>";
    echo "<h2>5. Recommendations</h2>";
    
    echo "<div style='background: #d1ecf1; padding: 15px; border-left: 4px solid #0c5460;'>";
    echo "<h3>To Fix Cancelled Orders:</h3>";
    echo "<ol>";
    echo "<li><strong>Check PHP Error Logs:</strong> Look for 'toyyibPay Callback' or 'Order' related errors</li>";
    echo "<li><strong>Verify Payment Callback:</strong> Make sure ToyyibPay can reach your callback URL</li>";
    echo "<li><strong>Check Order Creation:</strong> Verify initial_status is set correctly in checkout.php</li>";
    echo "<li><strong>Manual Fix:</strong> You can manually update order status in admin panel</li>";
    echo "</ol>";
    
    echo "<h3>To Manually Update Order Status:</h3>";
    echo "<p>Go to Admin Panel → Orders → Click on the order → Update Status</p>";
    echo "<p>Or run this SQL query (replace ORDER_ID with actual order ID):</p>";
    echo "<pre>UPDATE orders SET status = 'processing', payment_status = 'paid' WHERE id = ORDER_ID;</pre>";
    echo "</div>";
    
    echo "</div>";
    
    // Section 6: Quick fix tool
    echo "<div class='section'>";
    echo "<h2>6. Quick Fix Tool</h2>";
    
    if (isset($_POST['fix_order_id'])) {
        $order_id = (int)$_POST['fix_order_id'];
        $new_status = $_POST['new_status'];
        $new_payment_status = $_POST['new_payment_status'];
        
        $stmt = $conn->prepare("UPDATE orders SET status = ?, payment_status = ? WHERE id = ?");
        $result = $stmt->execute([$new_status, $new_payment_status, $order_id]);
        
        if ($result) {
            echo "<p class='success'>✓ Order #{$order_id} updated successfully to status: {$new_status}, payment: {$new_payment_status}</p>";
            echo "<p><a href='check-order-status.php'>Refresh page</a></p>";
        } else {
            echo "<p class='error'>✗ Failed to update order</p>";
        }
    }
    
    echo "<form method='POST'>";
    echo "<p><strong>Update Order Status:</strong></p>";
    echo "<label>Order ID: <input type='number' name='fix_order_id' required></label><br><br>";
    echo "<label>New Status: 
            <select name='new_status' required>
                <option value='pending'>Pending</option>
                <option value='awaiting_payment'>Awaiting Payment</option>
                <option value='processing' selected>Processing</option>
                <option value='shipped'>Shipped</option>
                <option value='delivered'>Delivered</option>
                <option value='cancelled'>Cancelled</option>
            </select>
          </label><br><br>";
    echo "<label>Payment Status: 
            <select name='new_payment_status' required>
                <option value='pending'>Pending</option>
                <option value='paid' selected>Paid</option>
                <option value='failed'>Failed</option>
                <option value='refunded'>Refunded</option>
            </select>
          </label><br><br>";
    echo "<button type='submit' style='padding: 10px 20px; background: #28a745; color: white; border: none; border-radius: 5px; cursor: pointer;'>Update Order</button>";
    echo "</form>";
    echo "</div>";
    
} catch (Exception $e) {
    echo "<div class='section'>";
    echo "<p class='error'>Error: " . htmlspecialchars($e->getMessage()) . "</p>";
    echo "</div>";
}

echo "<div class='section'>";
echo "<p><a href='admin/pages/orders.php'>← Back to Admin Orders</a></p>";
echo "</div>";
?>

