<?php
session_start();
require_once 'models/Cart.php';
require_once 'models/User.php';
require_once 'models/Order.php';
require_once 'models/ToyyibPay.php';
require_once 'config/payment.php';

// Check if user is logged in - redirect to signup if not
if (!isset($_SESSION['user_id']) || empty($_SESSION['user_id'])) {
    // Store the current URL to redirect back after signup
    $_SESSION['redirect_after_signup'] = $_SERVER['REQUEST_URI'];
    header('Location: signup.php?message=signup_required');
    exit();
}

// Get cart items from database
$cart = new Cart();
$user = new User();
$order = new Order();
$toyyibpay = new ToyyibPay();
$user_id = $_SESSION['user_id'] ?? null;
$session_id = session_id();

$cart_items = $cart->getCartItems($user_id, $session_id);

// Check if specific items are selected for checkout
$selected_cart_items = [];
if (isset($_GET['selected_items'])) {
    $selected_item_ids = json_decode($_GET['selected_items'], true);
    if (is_array($selected_item_ids)) {
        $selected_cart_items = array_filter($cart_items, function($item) use ($selected_item_ids) {
            return in_array($item['id'], $selected_item_ids);
        });
    }
}

// Check for direct purchase (Buy Now) - bypass cart
$direct_purchase_item = null;
if (isset($_GET['direct_purchase'])) {
    $direct_purchase_data = json_decode($_GET['direct_purchase'], true);
    if ($direct_purchase_data && isset($direct_purchase_data['product_id'])) {
        // Create a temporary cart item structure for direct purchase
        require_once 'models/Product.php';
        $product = new Product();
        $product_data = $product->getById($direct_purchase_data['product_id']);
        
        if ($product_data) {
            $direct_purchase_item = [
                'id' => 'direct_' . time(), // Temporary ID
                'product_id' => $direct_purchase_data['product_id'],
                'quantity' => $direct_purchase_data['quantity'] ?? 1,
                'size' => $direct_purchase_data['size'] ?? '',
                'version' => $direct_purchase_data['version'] ?? '',
                'price' => $direct_purchase_data['price'] ?? $product_data['price'],
                'name' => $product_data['name'],
                'brand' => $product_data['brand'],
                'image' => $product_data['image'],
                'currency' => $product_data['currency'],
                'is_direct_purchase' => true
            ];
        }
    }
}

// Use direct purchase item if available, otherwise selected items, otherwise all cart items
if ($direct_purchase_item) {
    $checkout_items = [$direct_purchase_item];
} elseif (!empty($selected_cart_items)) {
    $checkout_items = $selected_cart_items;
} else {
    $checkout_items = $cart_items;
}

// Calculate totals for selected items only
$subtotal = 0;
$total_quantity = 0;
foreach ($checkout_items as $item) {
    $subtotal += $item['price'] * $item['quantity'];
    $total_quantity += $item['quantity'];
}

// Calculate shipping for selected items
$weight = $total_quantity * 0.1;
$shipping_cost = $cart->calculateShippingCost($subtotal, $weight);

$cart_total = [
    'subtotal' => $subtotal,
    'shipping' => $shipping_cost,
    'total' => $subtotal + $shipping_cost,
    'item_count' => count($checkout_items),
    'total_quantity' => $total_quantity
];

// Get shipping information (force fresh calculation)
$shipping_info = null;
if (!empty($checkout_items) && $cart_total['subtotal'] > 0) {
    // Force fresh shipping calculation by clearing any potential cache
    $shipping_info = $cart->getShippingInfo($cart_total['subtotal'], $cart_total['total_quantity'] * 0.1);
    
    // Update cart total with fresh shipping cost
    $cart_total['shipping'] = $shipping_info['cost'];
    $cart_total['total'] = $cart_total['subtotal'] + $shipping_info['cost'];
}

// Get user data if logged in
$user_data = null;
$saved_addresses = [];
if ($user_id) {
    $user_data = $user->getById($user_id);
    $saved_addresses = $user->getShippingAddresses($user_id);
}

// Redirect if no items to checkout
if (empty($checkout_items)) {
    header('Location: cart.php');
    exit;
}

// Get user's claimed vouchers for selection
$claimedVouchers = [];
$appliedVoucher = null;
if ($user_id) {
    require_once 'admin/includes/Voucher.php';
    $voucherModel = new Voucher();
    $claimedVouchers = $voucherModel->getUserRedeemedVouchers($user_id, 'active');
    
    // Voucher application is now handled dynamically with JavaScript
    // No need for form submission and page refresh
}

// Check for applied voucher in session
if (isset($_SESSION['applied_voucher']) && !$appliedVoucher) {
    $appliedVoucher = $_SESSION['applied_voucher'];
    
    // Apply discount to cart total
    $discountAmount = 0;
    if ($appliedVoucher['discount_type'] === 'percentage') {
        $discountAmount = ($cart_total['subtotal'] * $appliedVoucher['discount_value']) / 100;
    } elseif ($appliedVoucher['discount_type'] === 'fixed_amount') {
        $discountAmount = min($appliedVoucher['discount_value'], $cart_total['subtotal']);
    } elseif ($appliedVoucher['discount_type'] === 'free_shipping') {
        $discountAmount = $cart_total['shipping'];
    }
    
    $cart_total['total'] = max(0, $cart_total['total'] - $discountAmount);
}

// Restore form data from session if available
$form_data = [];
if (isset($_SESSION['checkout_form_data'])) {
    $form_data = $_SESSION['checkout_form_data'];
    // Clear the session data after using it
    unset($_SESSION['checkout_form_data']);
}

$success_message = $success_message ?? '';
$error_message = '';

// Function to validate pickup time (server-side)
function validatePickupTime($notes) {
    // More flexible time pattern that matches various formats
    $timePattern = '/(\d{1,2})(?::(\d{1,2}))?\s*(am|pm|a\.m\.|p\.m\.|a\.m|p\.m)/i';
    preg_match_all($timePattern, $notes, $matches, PREG_SET_ORDER);
    
    if (empty($matches)) {
        return false; // No time mentioned
    }
    
    foreach ($matches as $match) {
        $hour = (int)$match[1];
        $minute = isset($match[2]) ? (int)$match[2] : 0;
        
        // Convert to 24-hour format
        $hour24 = $hour;
        if (stripos($match[0], 'pm') !== false && $hour !== 12) {
            $hour24 += 12;
        } elseif (stripos($match[0], 'am') !== false && $hour === 12) {
            $hour24 = 0;
        }
        
        // Check if time is between 10 AM (10:00) and 5 PM (17:00)
        if ($hour24 >= 10 && $hour24 < 17) {
            return true; // Found at least one valid time
        }
    }
    
    return false; // No valid time found
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Debug: Log that we received a POST request
    error_log("Checkout: Received POST request");
    error_log("Checkout: POST data: " . print_r($_POST, true));
    
    $saved_address_id = $_POST['saved_address_id'] ?? '';
    
    // If a saved address is selected, use that data
    if (!empty($saved_address_id)) {
        $selected_address = null;
        foreach ($saved_addresses as $addr) {
            if ($addr['id'] == $saved_address_id) {
                $selected_address = $addr;
                break;
            }
        }
        
        if ($selected_address) {
            $first_name = $selected_address['first_name'];
            $last_name = $selected_address['last_name'];
            $phone = $selected_address['phone'];
            $address = $selected_address['address'];
            $city = $selected_address['city'];
            $state = $selected_address['state'];
            $zip_code = $selected_address['zip_code'];
            $country = $selected_address['country'];
        } else {
            $error_message = 'Selected address not found. Please try again.';
        }
    } else {
        // Use form data for new address
        $first_name = trim($_POST['first_name'] ?? '');
        $last_name = trim($_POST['last_name'] ?? '');
        $phone = trim($_POST['phone'] ?? '');
        $address = trim($_POST['address'] ?? '');
        $city = trim($_POST['city'] ?? '');
        $state = trim($_POST['state'] ?? '');
        $zip_code = trim($_POST['zip_code'] ?? '');
        $country = trim($_POST['country'] ?? 'Malaysia');
    }
    
    $email = trim($_POST['email'] ?? '');
    $payment_method = $_POST['payment_method'] ?? $_POST['payment_method_fallback'] ?? 'toyyibpay_fpx';
    $notes = trim($_POST['notes'] ?? '');
    
    // Debug: Log payment method
    error_log("Checkout: Payment method: " . $payment_method);
    
    // Basic validation
    if (empty($first_name) || empty($last_name) || empty($email) || empty($phone) || 
        empty($address) || empty($city) || empty($state) || empty($zip_code) || empty($payment_method)) {
        $error_message = 'Please fill in all required fields.';
        error_log("Checkout: Validation failed - missing required fields");
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error_message = 'Please enter a valid email address.';
    } else {
        // Check if pickup order requires notes
        $shipping_method_id = $_POST['shipping_method_id'] ?? '';
        if (!empty($shipping_method_id)) {
            // Get shipping method name from database
            require_once 'config/database.php';
            $conn = getDB();
            $shipping_query = "SELECT name FROM shipping_fees WHERE id = :id AND is_active = 1";
            $shipping_stmt = $conn->prepare($shipping_query);
            $shipping_stmt->bindValue(':id', $shipping_method_id);
            $shipping_stmt->execute();
            $shipping_method = $shipping_stmt->fetch();
            
            if ($shipping_method && stripos($shipping_method['name'], 'pickup') !== false) {
                // This is a pickup order, notes are required
                if (empty($notes)) {
                    $error_message = 'Please specify your preferred pickup time in the order remarks section.';
                } else {
                    // Only validate pickup time if a time is actually mentioned
                    $timePattern = '/(\d{1,2})(?::(\d{1,2}))?\s*(am|pm|a\.m\.|p\.m\.|a\.m|p\.m)/i';
                    if (preg_match($timePattern, $notes)) {
                        // Time is mentioned, validate it
                        if (!validatePickupTime($notes)) {
                            $error_message = 'Please specify a pickup time between 10 AM - 5 PM in the order remarks.';
                        }
                    }
                    // If no time is mentioned but notes are provided, that's okay
                    // The user might be providing other pickup instructions
                }
            }
        }
        
        if (empty($error_message)) {
        // Get discounted total from form (if voucher was applied via JavaScript)
        $final_total = $cart_total['total'] ?? 0;
        if (isset($_POST['discounted_total']) && !empty($_POST['discounted_total'])) {
            $final_total = (float)$_POST['discounted_total'];
        }
        
        // Process order
        $order_data = [
            'user_id' => $user_id,
            'session_id' => $session_id,
            'first_name' => $first_name,
            'last_name' => $last_name,
            'email' => $email,
            'phone' => $phone,
            'address' => $address,
            'city' => $city,
            'state' => $state,
            'zip_code' => $zip_code,
            'country' => $country,
            'payment_method' => $payment_method,
            'notes' => $notes,
            'subtotal' => $cart_total['subtotal'] ?? 0,
            'shipping' => $cart_total['shipping'] ?? 0,
            'shipping_method_id' => $_POST['shipping_method_id'] ?? null,
            'total' => $final_total, // Use discounted total
            'items' => $checkout_items,
            'initial_status' => isToyyibPayMethod($payment_method) ? 'awaiting_payment' : 'pending',
            'voucher_id' => $appliedVoucher['voucher_id'] ?? null,
            'voucher_discount_amount' => $appliedVoucher ? $discountAmount : 0.00,
            'voucher_discount_type' => $appliedVoucher['discount_type'] ?? null,
            'voucher_discount_value' => $appliedVoucher['discount_value'] ?? null
        ];
        
        // Debug: Log voucher data being sent to order
        error_log("Checkout Debug - Applied Voucher: " . ($appliedVoucher ? 'Yes' : 'No'));
        if ($appliedVoucher) {
            error_log("Checkout Debug - Voucher ID: " . $appliedVoucher['voucher_id']);
            error_log("Checkout Debug - Discount Amount: " . $discountAmount);
            error_log("Checkout Debug - Discount Type: " . $appliedVoucher['discount_type']);
            error_log("Checkout Debug - Discount Value: " . $appliedVoucher['discount_value']);
        }
        
        try {
            $order_id = $order->create($order_data);
            
            if ($order_id) {
                // Mark applied voucher as used if one was applied (for ALL successful orders)
                if ($appliedVoucher && $user_id) {
                    $voucherModel->markVoucherAsUsed($user_id, $appliedVoucher['voucher_id']);
                    // Clear applied voucher and form data from session
                    unset($_SESSION['applied_voucher']);
                    unset($_SESSION['checkout_form_data']);
                }
                
                // Check if payment method is toyyibPay
                if (isToyyibPayMethod($payment_method)) {
                    // Validate toyyibPay configuration
                    if (empty(TOYYIBPAY_CATEGORY_CODE)) {
                        $error_message = 'Payment system not configured. Please contact support.';
                        error_log("toyyibPay Error: Category code not set");
                    } else {
                        // Create toyyibPay bill
                        $bill_data = [
                            'bill_name' => 'Order #' . $order_id,
                            'bill_description' => 'Payment for Order #' . $order_id,
                            'amount' => $final_total, // Use discounted total
                            'order_id' => $order_id,
                            'customer_name' => $first_name . ' ' . $last_name,
                            'customer_email' => $email,
                            'customer_phone' => $phone,
                            'payment_method' => $payment_method
                        ];
                        
                        $payment_result = $toyyibpay->createBill($bill_data);
                        
                        if ($payment_result && !isset($payment_result['error'])) {
                            // Remove only selected items from cart after successful order creation
                            if (!empty($selected_cart_items)) {
                                foreach ($selected_cart_items as $item) {
                                    $cart->removeItem($item['id']);
                                }
                            } else {
                                $cart->clearCart($user_id, $session_id);
                            }
                            
                            // Redirect to toyyibPay payment page
                            header('Location: ' . $payment_result['payment_url']);
                            exit;
                        } else {
                            // Handle specific error cases
                            if (isset($payment_result['error'])) {
                                if ($payment_result['error'] === 'insufficient_prepaid') {
                                    $error_message = 'Payment system temporarily unavailable. Please contact support or try again later.';
                                } else {
                                    $error_message = $payment_result['message'] ?? 'Failed to create payment. Please try again or contact support.';
                                }
                        } else {
                            $error_message = 'Failed to create payment. Please try again or contact support.';
                            }
                            error_log("toyyibPay Error: Failed to create bill for order ID: " . $order_id . " - " . ($payment_result['message'] ?? 'Unknown error'));
                        }
                    }
                } else {
                    // For non-toyyibPay methods (COD, Bank Transfer)
                    // Remove only selected items from cart after successful order
                    if (!empty($selected_cart_items)) {
                        foreach ($selected_cart_items as $item) {
                            $cart->removeItem($item['id']);
                        }
                    } else {
                        $cart->clearCart($user_id, $session_id);
                    }
                    
                    // Redirect to success page
                    header('Location: order-success.php?id=' . $order_id);
                    exit;
                }
            } else {
                // Check if it's a stock issue by looking at recent error logs
                $error_message = 'Failed to create order. This may be due to insufficient stock or other issues. Please check your cart and try again.';
                error_log("Order Creation Error: Failed to create order for user: " . ($user_id ?: $session_id));
                
                // Redirect to cart with error message
                $_SESSION['error_message'] = $error_message;
                header('Location: cart.php');
                exit;
            }
        } catch (Exception $e) {
            $error_message = 'An error occurred while processing your order. Please try again.';
            error_log("Checkout Error: " . $e->getMessage());
            error_log("Checkout Error Trace: " . $e->getTraceAsString());
        }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Checkout - 2 Days Parfume</title>
    <link rel="stylesheet" href="styles.css">
    <link rel="stylesheet" href="checkout.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="script.js" defer></script>
</head>
<body>
    <!-- Top Navigation -->
    <nav class="top-nav">
        <div class="nav-left">
            <i class="fas fa-arrow-left" onclick="goBack()"></i>
        </div>
        <div class="nav-center">
            <div class="logo">
                <img src="images/LOGO.png" alt="2 Days Parfume Logo" class="logo-img">
            </div>
        </div>
        <div class="nav-right">
            <div class="cart-icon">
                <i class="fas fa-shopping-cart"></i>
                <span class="cart-count" style="<?php echo count($checkout_items) > 0 ? 'display: inline-block;' : 'display: none;'; ?>"><?php echo count($checkout_items); ?></span>
            </div>
        </div>
    </nav>

    <!-- Checkout Header -->
    <div class="checkout-header">
        <h1>Checkout</h1>
        <p class="checkout-subtitle">Complete your order</p>
    </div>

    <!-- Success/Error Messages -->
    <?php if ($success_message): ?>
        <div class="alert alert-success">
            <i class="fas fa-check-circle"></i>
            <?php echo htmlspecialchars($success_message); ?>
        </div>
    <?php endif; ?>

    <?php if ($error_message): ?>
        <div class="alert alert-error">
            <i class="fas fa-exclamation-circle"></i>
            <?php echo htmlspecialchars($error_message); ?>
            <?php if (PAYMENT_ENVIRONMENT === 'development'): ?>
                <br><small>Debug: Check PHP error logs for more details</small>
            <?php endif; ?>
        </div>
    <?php endif; ?>

    <main class="checkout-main">
        <div class="checkout-container">
            <!-- Checkout Form -->
            <div class="checkout-form-section">
                <form method="POST" class="checkout-form" id="checkout-form">
                    <!-- Hidden input for discounted total -->
                    <input type="hidden" id="discounted_total" name="discounted_total" value="<?php echo $cart_total['total']; ?>">
                    
                    <!-- Contact Information -->
                    <div class="form-section">
                        <h2 class="section-title">Contact information</h2>
                        <div class="form-group">
                            <input type="email" id="email" name="email" 
                                   value="<?php echo htmlspecialchars($user_data['email'] ?? ''); ?>" 
                                   placeholder="Email address" required>
                        </div>
                        <div class="checkbox-group">
                            <input type="checkbox" id="newsletter" name="newsletter">
                            <label for="newsletter">Email me with news and offers</label>
                        </div>
                    </div>

                    <!-- Shipping Address -->
                    <div class="form-section">
                        <h2 class="section-title">Shipping address</h2>
                        
                        <?php if (!empty($saved_addresses)): ?>
                        <div class="saved-addresses-section">
                            <div class="address-selection-header">
                                <h3>Choose from saved addresses</h3>
                                <button type="button" class="btn-manage-addresses" onclick="window.open('shipping-address.php', '_blank')">
                                    <i class="fas fa-cog"></i>
                                    Manage Addresses
                                </button>
                            </div>
                            
                            <div class="saved-addresses-grid">
                                <?php foreach ($saved_addresses as $address): ?>
                                <div class="saved-address-card" onclick="selectSavedAddress(<?php echo htmlspecialchars(json_encode($address)); ?>)">
                                    <div class="address-card-header">
                                        <div class="address-name">
                                            <i class="fas fa-map-marker-alt"></i>
                                            <?php echo htmlspecialchars($address['address_name']); ?>
                                            <?php if ($address['is_default']): ?>
                                                <span class="default-badge">Default</span>
                                            <?php endif; ?>
                                        </div>
                                        <div class="address-radio">
                                            <input type="radio" name="saved_address_id" value="<?php echo $address['id']; ?>" 
                                                   id="address_<?php echo $address['id']; ?>"
                                                   <?php echo ($form_data['saved_address_id'] ?? '') == $address['id'] ? 'checked' : ''; ?>>
                                        </div>
                                    </div>
                                    <div class="address-details">
                                        <div class="address-full-name"><?php echo htmlspecialchars($address['first_name'] . ' ' . $address['last_name']); ?></div>
                                        <div class="address-phone"><?php echo htmlspecialchars($address['phone']); ?></div>
                                        <div class="address-text"><?php echo htmlspecialchars($address['address']); ?></div>
                                        <div class="address-location"><?php echo htmlspecialchars($address['city'] . ', ' . $address['state'] . ' ' . $address['zip_code'] . ', ' . $address['country']); ?></div>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </div>
                            
                            <div class="address-divider">
                                <span>Or enter a new address</span>
                            </div>
                        </div>
                        <?php endif; ?>
                        <div class="form-row">
                            <div class="form-group">
                                <input type="text" id="first_name" name="first_name" 
                                       value="<?php echo htmlspecialchars($form_data['first_name'] ?? $user_data['first_name'] ?? ''); ?>" 
                                       placeholder="First name" required>
                            </div>
                            <div class="form-group">
                                <input type="text" id="last_name" name="last_name" 
                                       value="<?php echo htmlspecialchars($form_data['last_name'] ?? $user_data['last_name'] ?? ''); ?>" 
                                       placeholder="Last name" required>
                            </div>
                        </div>
                        <div class="form-group">
                            <input type="text" id="address" name="address" 
                                   value="<?php echo htmlspecialchars($form_data['address'] ?? $user_data['address'] ?? ''); ?>" 
                                   placeholder="Address" required>
                        </div>
                        <div class="form-group">
                            <input type="text" id="apartment" name="apartment" 
                                   placeholder="Apartment, suite, etc. (optional)">
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <input type="text" id="city" name="city" 
                                       value="<?php echo htmlspecialchars($form_data['city'] ?? $user_data['city'] ?? ''); ?>" 
                                       placeholder="City" required>
                            </div>
                            <div class="form-group">
                                <select id="country" name="country" required>
                                    <option value="">Country/Region</option>
                                    <?php 
                                    $selected_country = $form_data['country'] ?? $user_data['country'] ?? 'Malaysia';
                                    ?>
                                    <option value="Malaysia" <?php echo $selected_country === 'Malaysia' ? 'selected' : ''; ?>>Malaysia</option>
                                    <option value="Singapore" <?php echo $selected_country === 'Singapore' ? 'selected' : ''; ?>>Singapore</option>
                                    <option value="Thailand" <?php echo $selected_country === 'Thailand' ? 'selected' : ''; ?>>Thailand</option>
                                    <option value="Indonesia" <?php echo $selected_country === 'Indonesia' ? 'selected' : ''; ?>>Indonesia</option>
                                </select>
                            </div>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <input type="text" id="state" name="state" 
                                       value="<?php echo htmlspecialchars($form_data['state'] ?? $user_data['state'] ?? ''); ?>" 
                                       placeholder="State" required>
                            </div>
                            <div class="form-group">
                                <input type="text" id="zip_code" name="zip_code" 
                                       value="<?php echo htmlspecialchars($form_data['postcode'] ?? $user_data['zip_code'] ?? ''); ?>" 
                                       placeholder="ZIP code" required>
                            </div>
                        </div>
                        <div class="form-group">
                            <input type="tel" id="phone" name="phone" 
                                   value="<?php echo htmlspecialchars($form_data['phone'] ?? $user_data['phone'] ?? ''); ?>" 
                                   placeholder="Phone number" required>
                        </div>
                    </div>

                    <!-- Shipping Method -->
                    <div class="form-section">
                        <h2 class="section-title">Shipping method</h2>
                        <p class="section-subtitle" style="color: #666; margin-bottom: 15px;">Please select your preferred shipping method</p>
                        
                        <div class="form-group">
                            <select id="shipping-method-select" name="shipping_method_id" class="form-control" required>
                                <option value="">Select shipping method</option>
                                <?php
                                // Get database connection
                                require_once 'config/database.php';
                                $conn = getDB();
                                
                                // Get all active shipping options
                                $shipping_options_query = "SELECT * FROM shipping_fees 
                                                          WHERE is_active = 1 
                                                          ORDER BY sort_order ASC, min_order_amount ASC";
                                $shipping_options_stmt = $conn->prepare($shipping_options_query);
                                $shipping_options_stmt->execute();
                                $shipping_options = $shipping_options_stmt->fetchAll();
                                
                                // Debug: Log shipping options found
                                error_log("Checkout Debug - Found " . count($shipping_options) . " shipping options");
                                foreach ($shipping_options as $opt) {
                                    error_log("Shipping Option: " . $opt['name'] . " (ID: " . $opt['id'] . ", Type: " . $opt['fee_type'] . ", Active: " . $opt['is_active'] . ")");
                                }
                                
                                if ($shipping_options):
                                    foreach ($shipping_options as $index => $option):
                                        // Calculate cost for this option
                                        $option_cost = 0;
                                        if ($option['fee_type'] === 'weight_based') {
                                            $option_cost = $option['base_fee'];
                                        } elseif ($option['fee_type'] === 'free_shipping_threshold') {
                                            $option_cost = ($subtotal >= $option['min_order_amount']) ? 0 : $option['base_fee'];
                                        } elseif ($option['fee_type'] === 'pickup') {
                                            $option_cost = 0;
                                        }
                                        
                                        // Check if this option is available for current order amount
                                        $is_available = true;
                                        if ($option['min_order_amount'] > 0 && $subtotal < $option['min_order_amount']) {
                                            $is_available = false;
                                            error_log("Shipping option '{$option['name']}' filtered out: subtotal ($subtotal) < min_order_amount ({$option['min_order_amount']})");
                                        }
                                        if ($option['max_order_amount'] > 0 && $subtotal > $option['max_order_amount']) {
                                            $is_available = false;
                                            error_log("Shipping option '{$option['name']}' filtered out: subtotal ($subtotal) > max_order_amount ({$option['max_order_amount']})");
                                        }
                                        
                                        error_log("Shipping option '{$option['name']}' availability: " . ($is_available ? 'YES' : 'NO') . " (subtotal: $subtotal)");
                                        
                                        // Build option text
                                        $option_text = $option['name'];
                                        if ($option['delivery_days_min'] > 0 || $option['delivery_days_max'] > 0) {
                                            if ($option['delivery_days_min'] == $option['delivery_days_max']) {
                                                $option_text .= " ({$option['delivery_days_min']} day" . ($option['delivery_days_min'] != 1 ? 's' : '') . ")";
                                            } else {
                                                $option_text .= " ({$option['delivery_days_min']}-{$option['delivery_days_max']} days)";
                                            }
                                        }
                                        
                                        // Add cost to option text
                                        if ($option_cost > 0) {
                                            $option_text .= " - RM " . number_format($option_cost, 2);
                                        } else {
                                            $option_text .= " - Free";
                                        }
                                        
                                        // Add availability note if not available
                                        if (!$is_available) {
                                            if ($option['min_order_amount'] > 0 && $subtotal < $option['min_order_amount']) {
                                                $option_text .= " (Min order: RM " . number_format($option['min_order_amount'], 2) . ")";
                                            } elseif ($option['max_order_amount'] > 0 && $subtotal > $option['max_order_amount']) {
                                                $option_text .= " (Max order: RM " . number_format($option['max_order_amount'], 2) . ")";
                                            }
                                        }
                                ?>
                                    <option value="<?php echo $option['id']; ?>" 
                                            data-cost="<?php echo $option_cost; ?>"
                                            <?php echo !$is_available ? 'disabled' : ''; ?>>
                                        <?php echo htmlspecialchars($option_text); ?>
                                    </option>
                                <?php 
                                    endforeach;
                                else:
                                ?>
                                    <option value="default" data-cost="<?php echo $cart_total['shipping']; ?>">
                                        Standard Shipping (3-5 days) - <?php echo $cart_total['shipping'] > 0 ? 'RM ' . number_format($cart_total['shipping'], 2) : 'Free'; ?>
                                    </option>
                                <?php endif; ?>
                            </select>
                        </div>
                    </div>

                    <!-- Voucher Section -->
                    <?php if ($user_id): ?>
                    <div class="form-section">
                        <h2 class="section-title">Apply Voucher</h2>
                        <p class="voucher-subtitle">Select a voucher from your claimed vouchers</p>
                        
                        <div class="voucher-section">
                            <?php 
                            // Debug: Log voucher information
                            error_log("Checkout Debug - User ID: " . ($user_id ?: 'Not logged in'));
                            error_log("Checkout Debug - Claimed vouchers count: " . count($claimedVouchers));
                            foreach ($claimedVouchers as $voucher) {
                                error_log("Checkout Debug - Voucher: " . $voucher['voucher_name'] . " (ID: " . $voucher['voucher_id'] . ", Status: " . $voucher['status'] . ")");
                            }
                            ?>
                            <?php if (!empty($claimedVouchers)): ?>
                                <?php if ($appliedVoucher): ?>
                                    <!-- Applied Voucher Display -->
                                    <div class="applied-voucher">
                                        <div class="voucher-info">
                                            <i class="fas fa-check-circle"></i>
                                            <span class="voucher-name"><?php echo htmlspecialchars($appliedVoucher['voucher_name']); ?></span>
                                            <span class="voucher-discount">
                                                <?php 
                                                if ($appliedVoucher['discount_type'] === 'percentage') {
                                                    echo $appliedVoucher['discount_value'] . '% OFF';
                                                } elseif ($appliedVoucher['discount_type'] === 'fixed_amount') {
                                                    echo 'RM ' . $appliedVoucher['discount_value'] . ' OFF';
                                                } else {
                                                    echo ucfirst(str_replace('_', ' ', $appliedVoucher['discount_type']));
                                                }
                                                ?>
                                            </span>
                                        </div>
                                        <button type="button" class="remove-voucher-btn" onclick="removeAppliedVoucher()">
                                            <i class="fas fa-times"></i>
                                        </button>
                                    </div>
                                <?php else: ?>
                                    <!-- Voucher Selection -->
                                    <div class="voucher-selection">
                                        <div class="voucher-options">
                                            <?php foreach ($claimedVouchers as $voucher): ?>
                                                <div class="voucher-option">
                                                    <input type="radio" id="voucher_<?php echo $voucher['voucher_id']; ?>" name="selected_voucher" value="<?php echo $voucher['voucher_id']; ?>" class="voucher-radio" 
                                                           data-voucher-id="<?php echo $voucher['voucher_id']; ?>"
                                                           data-voucher-name="<?php echo htmlspecialchars($voucher['voucher_name']); ?>"
                                                           data-discount-type="<?php echo $voucher['discount_type']; ?>"
                                                           data-discount-value="<?php echo $voucher['discount_value']; ?>">
                                                    <label for="voucher_<?php echo $voucher['voucher_id']; ?>" class="voucher-label">
                                                        <div class="voucher-card">
                                                            <div class="voucher-card-header">
                                                                <span class="voucher-name"><?php echo htmlspecialchars($voucher['voucher_name']); ?></span>
                                                                <span class="voucher-discount">
                                                                    <?php 
                                                                    if ($voucher['discount_type'] === 'percentage') {
                                                                        echo $voucher['discount_value'] . '% OFF';
                                                                    } elseif ($voucher['discount_type'] === 'fixed_amount') {
                                                                        echo 'RM ' . $voucher['discount_value'] . ' OFF';
                                                                    } else {
                                                                        echo ucfirst(str_replace('_', ' ', $voucher['discount_type']));
                                                                    }
                                                                    ?>
                                                                </span>
                                                            </div>
                                                            <div class="voucher-card-description">
                                                                <?php echo htmlspecialchars($voucher['voucher_description']); ?>
                                                            </div>
                                                        </div>
                                                    </label>
                                                </div>
                                            <?php endforeach; ?>
                                        </div>
                                        <button type="button" class="apply-voucher-btn" onclick="applyVoucher()">
                                            <i class="fas fa-ticket-alt"></i>
                                            Apply Selected Voucher
                                        </button>
                                    </div>
                                <?php endif; ?>
                            <?php else: ?>
                                <div class="no-voucher-message">
                                    <i class="fas fa-ticket-alt"></i>
                                    <p>No vouchers available. <a href="royalty-rewards.php" style="color: #1A4D2A; text-decoration: underline;">Claim vouchers</a> to get discounts!</p>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                    <?php endif; ?>

                    <!-- Payment Method -->
                    <div class="form-section">
                        <h2 class="section-title">Payment</h2>
                        <p class="payment-subtitle">All transactions are secure and encrypted.</p>
                        
                        <div class="payment-methods">
                            <?php 
                            $selected_payment = $form_data['payment_method'] ?? 'toyyibpay_fpx';
                            ?>
                            <div class="payment-option">
                                <input type="radio" id="toyyibpay_fpx" name="payment_method" value="toyyibpay_fpx" 
                                       <?php echo $selected_payment === 'toyyibpay_fpx' ? 'checked' : ''; ?> required>
                                <label for="toyyibpay_fpx" class="payment-label">
                                    <span class="payment-text">FPX Online Banking</span>
                                    <span class="payment-icon">🏦</span>
                                </label>
                            </div>
                            <div class="payment-option">
                                <input type="radio" id="toyyibpay_card" name="payment_method" value="toyyibpay_card"
                                       <?php echo $selected_payment === 'toyyibpay_card' ? 'checked' : ''; ?>>
                                <label for="toyyibpay_card" class="payment-label">
                                    <span class="payment-text">Credit/Debit Card</span>
                                    <span class="payment-icon">💳</span>
                                </label>
                            </div>
                            
                            <!-- Hidden input to ensure payment_method is always submitted -->
                            <input type="hidden" name="payment_method_fallback" value="<?php echo htmlspecialchars($selected_payment); ?>">
                        </div>
                    </div>

                    <!-- Order Remarks -->
                    <div class="form-section">
                        <h2 class="section-title">Order Remarks <span id="remarks-required" style="display: none; color: #dc3545;">*</span></h2>
                        <p class="remarks-subtitle" id="remarks-subtitle">Add any special requests or notes about your order</p>
                        <div class="form-group">
                            <textarea id="notes" name="notes" rows="4" 
                                      placeholder="Example: Please gift wrap this item, or I prefer delivery in the afternoon..."
                                      maxlength="500"><?php echo htmlspecialchars($form_data['notes'] ?? ''); ?></textarea>
                            <small class="form-hint">
                                <i class="fas fa-info-circle"></i>
                                <span id="remarks-hint">Maximum 500 characters</span>
                            </small>
                        </div>
                    </div>

                    <!-- Billing Address -->
                    <div class="form-section">
                        <div class="checkbox-group">
                            <input type="checkbox" id="billing_same" name="billing_same" 
                                   <?php echo isset($form_data['billing_same']) ? ($form_data['billing_same'] ? 'checked' : '') : 'checked'; ?>>
                            <label for="billing_same">Use shipping address as billing address</label>
                        </div>
                    </div>

                </form>
            </div>

            <!-- Order Summary -->
            <div class="order-summary-section">
                <div class="order-summary">
                    <h2 class="summary-title">
                        <i class="fas fa-shopping-bag"></i>
                        Order Summary
                        <?php if ($direct_purchase_item): ?>
                            <span class="direct-purchase-badge">Direct Purchase</span>
                        <?php endif; ?>
                    </h2>
                    
                    <!-- Order Items -->
                    <div class="order-items">
                        <?php foreach ($checkout_items as $item): ?>
                        <div class="order-item">
                            <div class="item-image">
                                <img src="<?php echo htmlspecialchars($item['image']); ?>" alt="<?php echo htmlspecialchars($item['name']); ?>" 
                                     onerror="this.src='data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMTAwIiBoZWlnaHQ9IjEwMCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cmVjdCB3aWR0aD0iMTAwIiBoZWlnaHQ9IjEwMCIgZmlsbD0iI2Y0ZjRmNCIvPjx0ZXh0IHg9IjUwIiB5PSI1MCIgZm9udC1mYW1pbHk9IkFyaWFsIiBmb250LXNpemU9IjE0IiBmaWxsPSIjOTk5IiB0ZXh0LWFuY2hvcj0ibWlkZGxlIj5JbWFnZTwvdGV4dD48L3N2Zz4='">
                            </div>
                            <div class="item-details">
                                <h3 class="item-name"><?php echo htmlspecialchars($item['name']); ?></h3>
                                <?php if ($item['size']): ?>
                                    <p class="item-size">
                            <?php if (!empty($item['version'])): ?>
                                Version: <?php echo htmlspecialchars($item['version']); ?> | 
                            <?php endif; ?>
                            Size: <?php echo htmlspecialchars($item['size']); ?>
                        </p>
                                <?php endif; ?>
                                <p class="item-quantity">Qty: <?php echo $item['quantity']; ?></p>
                            </div>
                            <div class="item-price">
                                RM <?php echo number_format($item['price'] * $item['quantity'], 2); ?>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>

                    <!-- Order Totals -->
                    <div class="order-totals">
                        <div class="total-row">
                            <span>Subtotal</span>
                            <span>RM <?php echo number_format($cart_total['subtotal'] ?? 0, 2); ?></span>
                        </div>
                        <div class="total-row">
                            <div class="shipping-info">
                                <span>Shipping</span>
                                <?php if ($shipping_info): ?>
                                    <small class="shipping-method"><?php echo htmlspecialchars($shipping_info['name']); ?></small>
                                <?php endif; ?>
                            </div>
                            <span>
                                <?php if ($cart_total['shipping'] > 0): ?>
                                    RM <?php echo number_format($cart_total['shipping'], 2); ?>
                                <?php else: ?>
                                    Free
                                <?php endif; ?>
                            </span>
                        </div>
                        <?php if ($appliedVoucher): ?>
                        <div class="total-row voucher-discount">
                            <div class="voucher-info">
                                <span>Voucher Discount</span>
                                <div class="voucher-details">
                                    <small class="voucher-name"><?php echo htmlspecialchars($appliedVoucher['voucher_name']); ?></small>
                                    <span class="voucher-type">
                                        <?php 
                                        if ($appliedVoucher['discount_type'] === 'percentage') {
                                            echo $appliedVoucher['discount_value'] . '% OFF';
                                        } elseif ($appliedVoucher['discount_type'] === 'fixed_amount') {
                                            echo 'RM ' . $appliedVoucher['discount_value'] . ' OFF';
                                        } else {
                                            echo ucfirst(str_replace('_', ' ', $appliedVoucher['discount_type']));
                                        }
                                        ?>
                                    </span>
                                </div>
                            </div>
                            <span class="discount-amount">
                                -RM <?php 
                                $discountAmount = 0;
                                if ($appliedVoucher['discount_type'] === 'percentage') {
                                    $discountAmount = ($cart_total['subtotal'] * $appliedVoucher['discount_value']) / 100;
                                } elseif ($appliedVoucher['discount_type'] === 'fixed_amount') {
                                    $discountAmount = min($appliedVoucher['discount_value'], $cart_total['subtotal']);
                                } elseif ($appliedVoucher['discount_type'] === 'free_shipping') {
                                    $discountAmount = $cart_total['shipping'];
                                }
                                echo number_format($discountAmount, 2); 
                                ?>
                            </span>
                        </div>
                        <?php endif; ?>
                        <div class="total-row total">
                            <span>Total</span>
                            <span>RM <?php echo number_format($cart_total['total'] ?? 0, 2); ?></span>
                        </div>
                    </div>

                    <!-- Security Notice -->
                    <div class="security-notice">
                        <i class="fas fa-shield-alt"></i>
                        <span>Your payment information is secure and encrypted</span>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Complete Order Button at Bottom -->
        <div class="complete-order-section">
            <button type="submit" form="checkout-form" class="place-order-btn">
                Complete order
            </button>
        </div>
    </main>

    <!-- Bottom Navigation -->
    <nav class="bottom-nav">
        <div class="nav-item" onclick="window.location.href='index.php'">
            <i class="fas fa-home"></i>
        </div>
        <div class="nav-item" onclick="window.location.href='products.php'">
            <i class="fas fa-box"></i>
        </div>
        <div class="nav-item" onclick="window.location.href='cart.php'">
            <i class="fas fa-shopping-cart"></i>
        </div>
        <div class="nav-item" onclick="checkRegisteredUserAndRedirect('royalty-rewards.php')">
            <i class="fas fa-crown"></i>
        </div>
    </nav>

    <script>
        function goBack() {
            window.history.back();
        }

        // Reset order totals to correct state
        function resetOrderTotals() {
            console.log('Resetting order totals to correct state...');
            
            // Only clear phantom voucher discounts (invisible ones)
            const phantomVoucherDiscounts = document.querySelectorAll('.voucher-discount');
            phantomVoucherDiscounts.forEach(row => {
                if (row.offsetParent === null) {
                    console.log('Removing phantom voucher discount:', row);
                    row.remove();
                }
            });
            
            // Get current subtotal and shipping
            const subtotalElement = document.querySelector('.order-totals .total-row:first-child span:last-child');
            const subtotal = parseFloat(subtotalElement.textContent.replace('RM ', '').replace(',', '')) || 0;
            
            const shippingElement = document.querySelector('.order-totals .total-row:nth-child(2) span:last-child');
            const shippingCost = shippingElement && shippingElement.textContent !== 'Free' ? 
                parseFloat(shippingElement.textContent.replace('RM ', '').replace(',', '')) : 0;
            
            // Calculate correct total
            const correctTotal = subtotal + shippingCost;
            
            // Update total display
            const totalElement = document.querySelector('.order-totals .total-row.total span:last-child');
            if (totalElement) {
                totalElement.textContent = 'RM ' + correctTotal.toFixed(2);
                console.log('Reset total to:', correctTotal);
            }
            
            // Update hidden input
            const discountedTotalInput = document.getElementById('discounted_total');
            if (discountedTotalInput) {
                discountedTotalInput.value = correctTotal.toFixed(2);
            }
        }

        // Handle shipping option selection
        document.addEventListener('DOMContentLoaded', function() {
            // Debug: Check voucher section
            const voucherSection = document.querySelector('.voucher-section');
            console.log('Voucher section found:', voucherSection);
            if (voucherSection) {
                console.log('Voucher section content:', voucherSection.innerHTML);
                const voucherOptions = voucherSection.querySelectorAll('.voucher-option');
                console.log('Voucher options found:', voucherOptions.length);
            }
            
            // Reset order totals on page load
            resetOrderTotals();
            
            // Initialize shipping method name on page load
            const shippingSelect = document.getElementById('shipping-method-select');
            if (shippingSelect && shippingSelect.value) {
                const selectedOption = shippingSelect.options[shippingSelect.selectedIndex];
                const shippingMethodName = selectedOption.textContent.split(' - ')[0];
                updateShippingMethodName(shippingMethodName);
                handlePickupOrderRequirements(shippingMethodName);
            }
            
            // Handle shipping method dropdown selection
            
            if (shippingSelect) {
                shippingSelect.addEventListener('change', function() {
                    const selectedOption = this.options[this.selectedIndex];
                    const shippingCost = parseFloat(selectedOption.dataset.cost) || 0;
                    const shippingMethodName = selectedOption.textContent.split(' - ')[0]; // Extract method name before cost
                    
                    // Update shipping method name display
                    updateShippingMethodName(shippingMethodName);
                    
                    // Handle pickup order requirements
                    handlePickupOrderRequirements(shippingMethodName);
                    
                    // Update shipping cost and total
                    updateTotals(shippingCost);
                });
            }
        });

        // Update shipping method name display
        function updateShippingMethodName(methodName) {
            const shippingMethodElement = document.querySelector('.shipping-method');
            if (shippingMethodElement) {
                shippingMethodElement.textContent = methodName;
                console.log('Updated shipping method name to:', methodName);
            } else {
                console.log('Shipping method element not found');
            }
        }

        // Handle pickup order requirements
        function handlePickupOrderRequirements(methodName) {
            const isPickupOrder = methodName.toLowerCase().includes('pickup');
            const notesTextarea = document.getElementById('notes');
            const remarksRequired = document.getElementById('remarks-required');
            const remarksSubtitle = document.getElementById('remarks-subtitle');
            const remarksHint = document.getElementById('remarks-hint');
            
            if (isPickupOrder) {
                // Make remarks required for pickup orders
                notesTextarea.required = true;
                remarksRequired.style.display = 'inline';
                remarksSubtitle.textContent = 'Please specify your preferred pickup time (10 AM - 5 PM)';
                remarksHint.textContent = 'Please specify your preferred pickup time between 10 AM - 5 PM. Maximum 500 characters';
                notesTextarea.placeholder = 'Example: I will pickup at 2pm, or Please call me before pickup at 1:30pm, or Pickup at 3pm';
                
                // Add validation for pickup time
                notesTextarea.addEventListener('input', validatePickupTime);
            } else {
                // Reset to optional for other shipping methods
                notesTextarea.required = false;
                remarksRequired.style.display = 'none';
                remarksSubtitle.textContent = 'Add any special requests or notes about your order';
                remarksHint.textContent = 'Maximum 500 characters';
                notesTextarea.placeholder = 'Example: Please gift wrap this item, or I prefer delivery in the afternoon...';
                
                // Remove pickup time validation
                notesTextarea.removeEventListener('input', validatePickupTime);
            }
        }

        // Validate pickup time
        function validatePickupTime() {
            const notesTextarea = document.getElementById('notes');
            const value = notesTextarea.value.toLowerCase();
            
            // More flexible time pattern that matches various formats
            const timePattern = /(\d{1,2})(?::(\d{0,2}))?\s*(am|pm|a\.m\.|p\.m\.|a\.m|p\.m)/gi;
            const matches = value.match(timePattern);
            
            if (matches) {
                let isValidTime = false;
                let hasInvalidTime = false;
                
                matches.forEach(match => {
                    // Extract hour and minute from the match
                    const hourMatch = match.match(/(\d{1,2})/);
                    const minuteMatch = match.match(/:(\d{1,2})/);
                    
                    if (hourMatch) {
                        const hour = parseInt(hourMatch[1]);
                        const minute = minuteMatch ? parseInt(minuteMatch[1]) : 0;
                        
                        // Convert to 24-hour format for easier comparison
                        let hour24 = hour;
                        if (match.includes('pm') && hour !== 12) {
                            hour24 += 12;
                        } else if (match.includes('am') && hour === 12) {
                            hour24 = 0;
                        }
                        
                        // Check if time is between 10 AM (10:00) and 5 PM (17:00)
                        if (hour24 >= 10 && hour24 < 17) {
                            isValidTime = true;
                        } else {
                            hasInvalidTime = true;
                        }
                    }
                });
                
                if (hasInvalidTime && !isValidTime) {
                    showNotification('Please specify a pickup time between 10 AM - 5 PM (e.g., "2pm", "2:30pm", "I will pickup at 2pm")', 'error');
                    notesTextarea.style.borderColor = '#dc3545';
                } else if (isValidTime) {
                    notesTextarea.style.borderColor = '#28a745';
                } else {
                    notesTextarea.style.borderColor = '';
                }
            } else {
                // No time pattern found - don't show error immediately
                // Only show error if user has typed a lot but no time
                if (value.length > 20 && !value.includes('pm') && !value.includes('am')) {
                    showNotification('Please include a pickup time between 10 AM - 5 PM (e.g., "2pm", "2:30pm")', 'error');
                    notesTextarea.style.borderColor = '#dc3545';
                } else {
                    notesTextarea.style.borderColor = '';
                }
            }
        }

        function updateTotals(shippingCost) {
            // Get current subtotal
            const subtotalElement = document.querySelector('.order-totals .total-row:first-child span:last-child');
            if (!subtotalElement) {
                console.error('Subtotal element not found');
                return;
            }
            const subtotal = parseFloat(subtotalElement.textContent.replace('RM ', '').replace(',', '')) || 0;
            
            // Update shipping cost display
            const shippingElement = document.querySelector('.order-totals .total-row:nth-child(2) span:last-child');
            if (shippingElement) {
                if (shippingCost > 0) {
                    shippingElement.textContent = 'RM ' + shippingCost.toFixed(2);
                } else {
                    shippingElement.textContent = 'Free';
                }
            }
            
            // Get current voucher discount
            const voucherRow = document.querySelector('.voucher-discount');
            let voucherDiscount = 0;
            if (voucherRow) {
                const discountAmountElement = voucherRow.querySelector('.discount-amount');
                if (discountAmountElement) {
                    voucherDiscount = parseFloat(discountAmountElement.textContent.replace('-RM ', '').replace(',', '')) || 0;
                }
            }
            
            // Update total
            const totalElement = document.querySelector('.order-totals .total-row.total span:last-child');
            if (!totalElement) {
                console.error('Total element not found');
                return;
            }
            
            const total = Math.max(0, subtotal + shippingCost - voucherDiscount);
            totalElement.textContent = 'RM ' + total.toFixed(2);
            
            // Debug log for calculation
            console.log('Total calculation:', {
                subtotal: subtotal,
                shippingCost: shippingCost,
                voucherDiscount: voucherDiscount,
                total: total
            });
            
            // Validate calculation
            const expectedTotal = subtotal + shippingCost - voucherDiscount;
            if (Math.abs(total - expectedTotal) > 0.01) {
                console.error('Total calculation mismatch!', {
                    calculated: total,
                    expected: expectedTotal,
                    difference: Math.abs(total - expectedTotal)
                });
            }
            
            // Update hidden input for server-side processing
            const discountedTotalInput = document.getElementById('discounted_total');
            if (discountedTotalInput) {
                discountedTotalInput.value = total.toFixed(2);
            }
        }

        // Select saved address and populate form
        function selectSavedAddress(address) {
            // Remove selected class from all cards
            document.querySelectorAll('.saved-address-card').forEach(card => {
                card.classList.remove('selected');
            });
            
            // Add selected class to clicked card
            event.currentTarget.classList.add('selected');
            
            // Check the radio button
            document.getElementById('address_' + address.id).checked = true;
            
            // Populate form fields
            document.getElementById('first_name').value = address.first_name;
            document.getElementById('last_name').value = address.last_name;
            document.getElementById('phone').value = address.phone;
            document.getElementById('address').value = address.address;
            document.getElementById('city').value = address.city;
            document.getElementById('state').value = address.state;
            document.getElementById('zip_code').value = address.zip_code;
            document.getElementById('country').value = address.country;
            
            // Show success message
            showNotification('Address selected successfully!', 'success');
        }

        // Show notification function
        function showNotification(message, type = 'info') {
            // Remove existing notifications
            document.querySelectorAll('.checkout-notification').forEach(notification => {
                notification.remove();
            });
            
            // Create notification element
            const notification = document.createElement('div');
            notification.className = `checkout-notification notification-${type}`;
            notification.innerHTML = `
                <i class="fas fa-${type === 'error' ? 'exclamation-circle' : type === 'success' ? 'check-circle' : 'info-circle'}"></i>
                <span>${message}</span>
            `;
            
            // Add styles
            notification.style.cssText = `
                position: fixed;
                top: 20px;
                right: 20px;
                background: ${type === 'error' ? '#fee' : type === 'success' ? '#f0fff4' : '#e3f2fd'};
                color: ${type === 'error' ? '#c53030' : type === 'success' ? '#2f855a' : '#1976d2'};
                border: 1px solid ${type === 'error' ? '#feb2b2' : type === 'success' ? '#9ae6b4' : '#bbdefb'};
                padding: 12px 16px;
                border-radius: 8px;
                display: flex;
                align-items: center;
                gap: 10px;
                font-size: 14px;
                font-weight: 500;
                z-index: 10000;
                animation: slideInRight 0.3s ease;
                max-width: 300px;
                box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            `;
            
            // Add to page
            document.body.appendChild(notification);
            
            // Auto remove after 3 seconds
            setTimeout(() => {
                notification.style.animation = 'slideOutRight 0.3s ease';
                setTimeout(() => {
                    if (notification.parentNode) {
                        notification.parentNode.removeChild(notification);
                    }
                }, 300);
            }, 3000);
        }

        // Form validation
        document.addEventListener('DOMContentLoaded', function() {
            const form = document.querySelector('.checkout-form');
            const submitBtn = document.querySelector('.place-order-btn');
            let isSubmitting = false; // Flag to prevent double submission
            
            // Add animation keyframes for notifications
            if (!document.getElementById('notification-styles')) {
                const style = document.createElement('style');
                style.id = 'notification-styles';
                style.textContent = `
                    @keyframes slideInRight {
                        from { transform: translateX(100%); opacity: 0; }
                        to { transform: translateX(0); opacity: 1; }
                    }
                    @keyframes slideOutRight {
                        from { transform: translateX(0); opacity: 1; }
                        to { transform: translateX(100%); opacity: 0; }
                    }
                `;
                document.head.appendChild(style);
            }
            
            // Add click event listener to submit button
            submitBtn.addEventListener('click', function(e) {
                console.log('Submit button clicked');
                console.log('Form validity:', form.checkValidity());
                
                // Check required fields
                const requiredFields = form.querySelectorAll('[required]');
                requiredFields.forEach(field => {
                    if (!field.value.trim()) {
                        console.log('Missing required field:', field.name || field.id);
                    }
                });
                
                // Check payment method specifically
                const paymentMethod = form.querySelector('input[name="payment_method"]:checked');
                console.log('Selected payment method:', paymentMethod ? paymentMethod.value : 'None selected');
                
                // If no payment method selected, select the default one
                if (!paymentMethod) {
                    console.log('No payment method selected, selecting default');
                    const defaultPayment = form.querySelector('input[name="payment_method"][value="toyyibpay_fpx"]');
                    if (defaultPayment) {
                        defaultPayment.checked = true;
                        console.log('Default payment method selected');
                    }
                }
                
                // Add a timeout to check if form submit event fires
                setTimeout(() => {
                    console.log('Timeout reached - checking if form was submitted');
                    if (submitBtn.innerHTML.includes('Processing')) {
                        console.log('Form is processing - submit event fired');
                    } else {
                        console.log('Form not processing - submit event may not have fired');
                        console.log('Attempting manual submission...');
                        window.manualSubmit();
                    }
                }, 1000);
            });
            
            form.addEventListener('submit', function(e) {
                console.log('Form submit event triggered');
                
                // Prevent double submission
                if (isSubmitting) {
                    console.log('Form is already submitting, preventing duplicate submission');
                    e.preventDefault();
                    return false;
                }
                
                // Check if this is the main checkout form (not a nested form)
                if (!e.target.classList.contains('checkout-form')) {
                    console.log('Ignoring nested form submission');
                    return; // Let the nested form handle its own submission
                }
                
                isSubmitting = true; // Set flag to prevent double submission
                
                // Check if form is valid
                if (!form.checkValidity()) {
                    console.log('Form validation failed');
                    isSubmitting = false; // Reset flag on validation failure
                    e.preventDefault();
                    return false;
                }
                
                console.log('Form is valid, proceeding with submission');
                
                // Check if shipping method is selected
                const shippingSelect = document.getElementById('shipping-method-select');
                if (!shippingSelect || !shippingSelect.value) {
                    console.log('No shipping method selected');
                    alert('Please select a shipping method');
                    isSubmitting = false;
                    e.preventDefault();
                    return false;
                }
                
                // Final check for payment method
                const paymentMethod = form.querySelector('input[name="payment_method"]:checked');
                console.log('Payment method element found:', paymentMethod);
                console.log('All payment method inputs:', form.querySelectorAll('input[name="payment_method"]'));
                console.log('Payment method inputs with their checked state:');
                form.querySelectorAll('input[name="payment_method"]').forEach((input, index) => {
                    console.log(`Input ${index}: value="${input.value}", checked=${input.checked}`);
                });
                
                if (!paymentMethod) {
                    console.log('No payment method selected, preventing submission');
                    console.log('Attempting to select default payment method...');
                    
                    // Try to select the default payment method
                    const defaultPayment = form.querySelector('input[name="payment_method"][value="toyyibpay_fpx"]');
                    if (defaultPayment) {
                        defaultPayment.checked = true;
                        console.log('Default payment method selected:', defaultPayment.value);
                    } else {
                        console.log('No default payment method found');
                        isSubmitting = false; // Reset flag on payment method failure
                        e.preventDefault();
                        return false;
                    }
                } else {
                    console.log('Payment method confirmed:', paymentMethod.value);
                }
                
                // Add loading state
                submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processing...';
                submitBtn.disabled = true;
                
                console.log('Form submission should proceed now...');
            });
            
            // Initialize payment method on page load
            function initializePaymentMethod() {
                const form = document.querySelector('.checkout-form');
                if (form) {
                    const paymentMethod = form.querySelector('input[name="payment_method"]:checked');
                    if (!paymentMethod) {
                        const defaultPayment = form.querySelector('input[name="payment_method"][value="toyyibpay_fpx"]');
                        if (defaultPayment) {
                            defaultPayment.checked = true;
                            console.log('Default payment method initialized on page load');
                        }
                    } else {
                        console.log('Payment method already selected:', paymentMethod.value);
                    }
                }
            }
            
            // Initialize payment method when page loads
            initializePaymentMethod();
            
            // Add a backup manual submission function
            window.manualSubmit = function() {
                console.log('Manual form submission triggered');
                const form = document.querySelector('.checkout-form');
                if (form) {
                    // Ensure payment method is selected
                    const paymentMethod = form.querySelector('input[name="payment_method"]:checked');
                    if (!paymentMethod) {
                        const defaultPayment = form.querySelector('input[name="payment_method"][value="toyyibpay_fpx"]');
                        if (defaultPayment) {
                            defaultPayment.checked = true;
                            console.log('Default payment method selected for manual submission');
                        }
                    }
                    
                    // Submit the form
                    console.log('Submitting form manually...');
                    form.submit();
                }
            };
            
        });

        // Clear all voucher state
        function clearVoucherState() {
            console.log('Clearing all voucher state...');
            
            // Remove all voucher discount elements
            document.querySelectorAll('.voucher-discount').forEach(row => {
                console.log('Removing voucher discount element:', row);
                row.remove();
            });
            
            // Remove applied voucher display
            const appliedVoucher = document.querySelector('.applied-voucher');
            if (appliedVoucher) {
                console.log('Removing applied voucher display');
                appliedVoucher.remove();
            }
            
            // Clear any voucher-related hidden inputs
            const voucherIdInput = document.querySelector('input[name="voucher_id"]');
            if (voucherIdInput) {
                voucherIdInput.value = '';
            }
            
            const voucherDiscountInput = document.querySelector('input[name="voucher_discount_amount"]');
            if (voucherDiscountInput) {
                voucherDiscountInput.value = '';
            }
            
            console.log('Voucher state cleared');
        }

        // Apply voucher function
        function applyVoucher() {
            // Clear any existing voucher state first
            clearVoucherState();
            
            const selectedVoucher = document.querySelector('input[name="selected_voucher"]:checked');
            if (!selectedVoucher) {
                showNotification('Please select a voucher first', 'error');
                return;
            }
            
            const voucherData = {
                voucher_id: selectedVoucher.dataset.voucherId,
                voucher_name: selectedVoucher.dataset.voucherName,
                discount_type: selectedVoucher.dataset.discountType,
                discount_value: parseFloat(selectedVoucher.dataset.discountValue)
            };
            
            // Calculate discount amount
            const subtotalElement = document.querySelector('.order-totals .total-row:first-child span:last-child');
            const subtotal = parseFloat(subtotalElement.textContent.replace('RM ', '').replace(',', ''));
            let discountAmount = 0;
            
            console.log('Voucher calculation:', {
                subtotal: subtotal,
                discount_type: voucherData.discount_type,
                discount_value: voucherData.discount_value
            });
            
            if (voucherData.discount_type === 'percentage') {
                discountAmount = (subtotal * voucherData.discount_value) / 100;
            } else if (voucherData.discount_type === 'fixed_amount') {
                discountAmount = Math.min(voucherData.discount_value, subtotal);
            } else if (voucherData.discount_type === 'free_shipping') {
                const shippingElement = document.querySelector('.order-totals .total-row:nth-child(2) span:last-child');
                if (shippingElement && shippingElement.textContent !== 'Free') {
                    discountAmount = parseFloat(shippingElement.textContent.replace('RM ', '').replace(',', ''));
                }
            }
            
            console.log('Calculated discount amount:', discountAmount);
            
            // Check if discount amount is valid
            if (discountAmount <= 0) {
                showNotification('This voucher does not provide any discount for your current order.', 'warning');
                return;
            }
            
            // Save voucher to session via AJAX
            fetch('ajax/apply-voucher.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(voucherData)
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Update the UI to show applied voucher
                    updateVoucherDisplay(voucherData, discountAmount);
                    
                    // Update voucher discount display
                    updateVoucherDiscount(discountAmount);
                    
                    // Force total recalculation after a delay
                    setTimeout(() => {
                        console.log('Force recalculating totals after voucher application...');
                        const shippingElement = document.querySelector('.order-totals .total-row:nth-child(2) span:last-child');
                        const currentShippingCost = shippingElement && shippingElement.textContent !== 'Free' ? 
                            parseFloat(shippingElement.textContent.replace('RM ', '').replace(',', '')) : 0;
                        updateTotals(currentShippingCost);
                    }, 200);
                    
                    showNotification('Voucher applied successfully!', 'success');
                } else {
                    showNotification('Error applying voucher: ' + data.message, 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showNotification('Error applying voucher. Please try again.', 'error');
            });
        }
        
        // Update voucher display
        function updateVoucherDisplay(voucherData, discountAmount) {
            const voucherSection = document.querySelector('.voucher-section');
            voucherSection.innerHTML = `
                <div class="applied-voucher">
                    <div class="voucher-info">
                        <i class="fas fa-check-circle"></i>
                        <span class="voucher-name">${voucherData.voucher_name}</span>
                        <span class="voucher-discount">
                            ${voucherData.discount_type === 'percentage' ? voucherData.discount_value + '% OFF' : 
                              voucherData.discount_type === 'fixed_amount' ? 'RM ' + voucherData.discount_value + ' OFF' : 
                              'Free Shipping'}
                        </span>
                    </div>
                    <button type="button" class="remove-voucher-btn" onclick="removeAppliedVoucher()">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
            `;
        }
        
        // Update voucher discount display
        function updateVoucherDiscount(discountAmount) {
            console.log('updateVoucherDiscount called with amount:', discountAmount);
            
            // Remove existing voucher discount rows
            const existingRows = document.querySelectorAll('.voucher-discount');
            console.log('Removing existing voucher discount rows:', existingRows.length);
            existingRows.forEach(row => row.remove());
            
            if (discountAmount > 0) {
                console.log('Adding voucher discount row with amount:', discountAmount);
                // Add voucher discount row
                const voucherRow = document.createElement('div');
                voucherRow.className = 'total-row voucher-discount';
                voucherRow.innerHTML = `
                    <div class="voucher-info">
                        <span>Voucher Discount</span>
                    </div>
                    <span class="discount-amount">-RM ${discountAmount.toFixed(2)}</span>
                `;
                
                // Insert before total row
                const totalRow = document.querySelector('.order-totals .total-row.total');
                if (totalRow) {
                    totalRow.parentNode.insertBefore(voucherRow, totalRow);
                    console.log('Voucher discount row added successfully');
                } else {
                    console.error('Total row not found for voucher discount insertion');
                }
            } else {
                console.log('No voucher discount to add (amount is 0 or negative)');
            }
            
            // Recalculate totals - get current shipping cost with a small delay
            setTimeout(() => {
                const shippingElement = document.querySelector('.order-totals .total-row:nth-child(2) span:last-child');
                const currentShippingCost = shippingElement && shippingElement.textContent !== 'Free' ? 
                    parseFloat(shippingElement.textContent.replace('RM ', '').replace(',', '')) : 0;
                updateTotals(currentShippingCost);
            }, 100);
        }

        // Remove applied voucher function
        function removeAppliedVoucher() {
            // Clear voucher from session via AJAX
            fetch('ajax/remove-voucher.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({})
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Fetch fresh voucher data from server
                    fetch('ajax/get-available-vouchers.php', {
                        method: 'GET',
                        headers: {
                            'Content-Type': 'application/json',
                        }
                    })
                    .then(response => response.json())
                    .then(voucherData => {
                        if (voucherData.success && voucherData.vouchers) {
                            // Remove voucher display and reset totals
                            const voucherSection = document.querySelector('.voucher-section');
                            
                            // Restore voucher selection with fresh data
                            let voucherHtml = '<div class="voucher-options">';
                            voucherData.vouchers.forEach(voucher => {
                                voucherHtml += `
                                    <div class="voucher-option">
                                        <input type="radio" id="voucher_${voucher.voucher_id}" name="selected_voucher" value="${voucher.voucher_id}" class="voucher-radio" 
                                               data-voucher-id="${voucher.voucher_id}"
                                               data-voucher-name="${voucher.voucher_name}"
                                               data-discount-type="${voucher.discount_type}"
                                               data-discount-value="${voucher.discount_value}">
                                        <label for="voucher_${voucher.voucher_id}" class="voucher-label">
                                            <div class="voucher-card">
                                                <div class="voucher-card-header">
                                                    <span class="voucher-name">${voucher.voucher_name}</span>
                                                    <span class="voucher-discount">
                                                        ${voucher.discount_type === 'percentage' ? voucher.discount_value + '% OFF' : 
                                                          voucher.discount_type === 'fixed_amount' ? 'RM ' + voucher.discount_value + ' OFF' : 
                                                          'Free Shipping'}
                                                    </span>
                                                </div>
                                                <div class="voucher-card-description">
                                                    ${voucher.voucher_description}
                                                </div>
                                            </div>
                                        </label>
                                    </div>
                                `;
                            });
                            voucherHtml += '</div>';
                            voucherHtml += `
                                <button type="button" class="apply-voucher-btn" onclick="applyVoucher()">
                                    <i class="fas fa-ticket-alt"></i>
                                    Apply Selected Voucher
                                </button>
                            `;
                            
                            voucherSection.innerHTML = voucherHtml;
                            
                            // Remove voucher discount and reset totals
                            updateVoucherDiscount(0);
                            
                            // Force recalculation of totals
                            const shippingElement = document.querySelector('.order-totals .total-row:nth-child(2) span:last-child');
                            const currentShippingCost = shippingElement && shippingElement.textContent !== 'Free' ? 
                                parseFloat(shippingElement.textContent.replace('RM ', '').replace(',', '')) : 0;
                            updateTotals(currentShippingCost);
                            
                            showNotification('Voucher removed', 'info');
                        } else {
                            showNotification('Error fetching fresh voucher data', 'error');
                        }
                    })
                    .catch(error => {
                        console.error('Error fetching vouchers:', error);
                        showNotification('Error fetching fresh voucher data', 'error');
                    });
                } else {
                    showNotification('Error removing voucher: ' + data.message, 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showNotification('Error removing voucher. Please try again.', 'error');
            });
        }
    </script>
    
    <style>
        /* Shipping dropdown styles */
        .form-control {
            width: 100%;
            padding: 12px 16px;
            border: 2px solid #ddd;
            border-radius: 8px;
            font-size: 16px;
            background-color: white;
            transition: border-color 0.3s ease, box-shadow 0.3s ease;
        }
        
        .form-control:focus {
            outline: none;
            border-color: #8B4513;
            box-shadow: 0 0 0 3px rgba(139, 69, 19, 0.1);
        }
        
        .form-control option:disabled {
            color: #999;
            background-color: #f5f5f5;
        }
        
        /* Voucher Section Styles */
        .voucher-section {
            background: #f8f9fa;
            border: 1px solid #e9ecef;
            border-radius: 8px;
            padding: 1rem;
        }
        
        .voucher-subtitle {
            color: #666;
            font-size: 0.9rem;
            margin: 0 0 1rem 0;
        }
        
        .voucher-input-group {
            display: flex;
            gap: 0.5rem;
            margin-bottom: 1rem;
        }
        
        .voucher-input {
            flex: 1;
            padding: 0.75rem;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 0.9rem;
        }
        
        .voucher-input:focus {
            outline: none;
            border-color: #1A4D2A;
            box-shadow: 0 0 0 2px rgba(26, 77, 42, 0.1);
        }
        
        .apply-voucher-btn {
            background: linear-gradient(135deg, #1A4D2A, #2d5a3d);
            color: white;
            border: none;
            padding: 0.75rem 1rem;
            border-radius: 6px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 0.5rem;
            white-space: nowrap;
        }
        
        .apply-voucher-btn:hover {
            background: linear-gradient(135deg, #2d5a3d, #1A4D2A);
            transform: translateY(-1px);
        }
        
        .apply-voucher-btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none;
        }
        
        .voucher-message {
            padding: 0.75rem;
            border-radius: 6px;
            font-size: 0.9rem;
            font-weight: 500;
            margin-bottom: 1rem;
        }
        
        .voucher-message.success {
            background:rgb(0, 0, 0);
            color: #155724;
            border: 1px solidrgb(0, 0, 0);
        }
        
        .voucher-message.error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .voucher-message.info {
            background:rgb(0, 0, 0);
            color: #0c5460;
            border: 1px solid #bee5eb;
        }
        
        .applied-voucher {
            background: linear-gradient(135deg, #f8fff9, #f0fff4);
            border: 2px solid #28a745;
            border-radius: 12px;
            padding: 1.2rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1rem;
            box-shadow: 0 4px 12px rgba(40, 167, 69, 0.15);
        }
        
        .voucher-info {
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .voucher-info i {
            color: #28a745;
            font-size: 1.2rem;
        }
        
        .voucher-name {
            font-weight: 700;
            color: #1a4d2a;
            font-size: 1.1rem;
        }
        
        .voucher-discount {
            background: linear-gradient(135deg, #28a745, #20c997);
            color: black;
            padding: 0.4rem 0.8rem;
            border-radius: 15px;
            font-size: 0.9rem;
            font-weight: 700;
            box-shadow: 0 2px 6px rgba(40, 167, 69, 0.3);
            text-shadow: 0 1px 2px rgba(0, 0, 0, 0.2);
            border: 1px solid rgba(255, 255, 255, 0.2);
        }
        
        .remove-voucher-btn {
            background: #dc3545;
            color: white;
            border: none;
            width: 32px;
            height: 32px;
            border-radius: 50%;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.3s ease;
        }
        
        .remove-voucher-btn:hover {
            background: #c82333;
            transform: scale(1.1);
        }
        
        .no-voucher-message {
            background:rgb(9, 26, 44);
            border: 2px dashed #ddd;
            border-radius: 8px;
            padding: 20px;
            text-align: center;
            color: #666;
        }
        
        .no-voucher-message i {
            font-size: 2rem;
            color: #ddd;
            margin-bottom: 10px;
            display: block;
        }
        
        .no-voucher-message p {
            margin: 0;
            font-size: 0.9rem;
        }
        
        /* Voucher Selection Styles */
        .voucher-selection {
            margin-top: 1rem;
        }
        
        .voucher-discount-row {
            color: #28a745;
        }
        
        .voucher-options {
            display: flex;
            flex-direction: column;
            gap: 1rem;
            margin-bottom: 1.5rem;
        }
        
        .voucher-option {
            position: relative;
        }
        
        .voucher-radio {
            position: absolute;
            opacity: 0;
            pointer-events: none;
        }
        
        .voucher-label {
            display: block;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .voucher-card {
            background: white;
            border: 2px solid #e9ecef;
            border-radius: 12px;
            padding: 1rem;
            transition: all 0.3s ease;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
        }
        
        .voucher-card:hover {
            border-color: #1A4D2A;
            box-shadow: 0 4px 12px rgba(26, 77, 42, 0.15);
            transform: translateY(-2px);
        }
        
        .voucher-radio:checked + .voucher-label .voucher-card {
            border-color: #1A4D2A;
            background: linear-gradient(135deg, #f8f9fa, #e9ecef);
            box-shadow: 0 4px 12px rgba(26, 77, 42, 0.2);
        }
        
        .voucher-radio:checked + .voucher-label .voucher-card::before {
            content: '✓';
            position: absolute;
            top: 10px;
            right: 10px;
            background: #1A4D2A;
            color: white;
            width: 24px;
            height: 24px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 14px;
            font-weight: bold;
        }
        
        .voucher-card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 0.5rem;
        }
        
        .voucher-card .voucher-name {
            font-weight: 600;
            color: #333;
            font-size: 1rem;
        }
        
        .voucher-card .voucher-discount {
            background: linear-gradient(135deg, #1A4D2A, #2d5a3d);
            color: white;
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
        }
        
        .voucher-card-description {
            color: #666;
            font-size: 0.9rem;
            line-height: 1.4;
        }
        
        .apply-voucher-btn {
            width: 100%;
            background: linear-gradient(135deg, #1A4D2A, #2d5a3d);
            color: white;
            border: none;
            padding: 12px 20px;
            border-radius: 8px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
            font-size: 1rem;
        }
        
        .apply-voucher-btn:hover {
            background: linear-gradient(135deg, #2d5a3d, #1A4D2A);
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(26, 77, 42, 0.3);
        }
        
        .apply-voucher-btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none;
        }
        
        /* Voucher Discount in Order Summary */
        .voucher-discount {
            background: linear-gradient(135deg, #f0fff4, #e6fffa);
            border: 2px solid #9ae6b4;
            border-radius: 8px;
            padding: 12px 16px;
            margin: 8px 0;
            box-shadow: 0 2px 8px rgba(46, 160, 67, 0.1);
        }
        
        .voucher-discount .voucher-info {
            display: flex;
            flex-direction: column;
            gap: 4px;
        }
        
        .voucher-discount .voucher-info span:first-child {
            font-size: 1rem;
            color: #1a4d2a;
            font-weight: 600;
        }
        
        .voucher-discount .voucher-details {
            display: flex;
            flex-direction: column;
            gap: 4px;
            margin-top: 4px;
        }
        
        .voucher-discount .voucher-name {
            font-size: 0.85rem;
            color: #2f855a;
            font-weight: 500;
        }
        
        .voucher-discount .voucher-type {
            font-size: 0.9rem;
            color: #1a4d2a;
            font-weight: 600;
            background: linear-gradient(135deg, #2f855a, #38a169);
            color: white;
            padding: 4px 10px;
            border-radius: 15px;
            display: inline-block;
            box-shadow: 0 2px 4px rgba(46, 160, 67, 0.3);
        }
        
        .voucher-discount .discount-amount {
            color: #1a4d2a;
            font-weight: 700;
            font-size: 1.2rem;
            text-shadow: 0 1px 2px rgba(0, 0, 0, 0.1);
        }
        
        @media (max-width: 768px) {
            .voucher-input-group {
                flex-direction: column;
            }
            
            .applied-voucher {
                flex-direction: column;
                gap: 1rem;
                text-align: center;
            }
        }
        
        /* Order Remarks Section Styles */
        .remarks-subtitle {
            color: #666;
            font-size: 0.9rem;
            margin: 0 0 1rem 0;
        }
        
        textarea {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 0.95rem;
            font-family: inherit;
            resize: vertical;
            transition: border-color 0.3s ease, box-shadow 0.3s ease;
        }
        
        textarea:focus {
            outline: none;
            border-color: #8B4513;
            box-shadow: 0 0 0 3px rgba(139, 69, 19, 0.1);
        }
        
        textarea::placeholder {
            color: #999;
            font-style: italic;
        }
        
        .form-hint {
            display: block;
            margin-top: 0.5rem;
            font-size: 0.85rem;
            color: #666;
        }
        
        .form-hint i {
            margin-right: 0.25rem;
            color: #8B4513;
        }
        
        /* Complete Order Section at Bottom */
        .complete-order-section {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            background: white;
            border-top: 1px solid #e9ecef;
            padding: 16px 20px;
            box-shadow: 0 -4px 12px rgba(0, 0, 0, 0.1);
            z-index: 1000;
        }
        
        .complete-order-section .place-order-btn {
            width: 100%;
            background: linear-gradient(135deg, #1A4D2A, #2d5a3d);
            color: white;
            border: none;
            padding: 16px 24px;
            border-radius: 12px;
            font-size: 18px;
            font-weight: 700;
            cursor: pointer;
            transition: all 0.3s ease;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            box-shadow: 0 4px 12px rgba(26, 77, 42, 0.3);
        }
        
        .complete-order-section .place-order-btn:hover {
            background: linear-gradient(135deg, #2d5a3d, #1A4D2A);
            transform: translateY(-2px);
            box-shadow: 0 6px 16px rgba(26, 77, 42, 0.4);
        }
        
        .complete-order-section .place-order-btn:active {
            transform: translateY(0);
            box-shadow: 0 2px 8px rgba(26, 77, 42, 0.3);
        }
        
        .complete-order-section .place-order-btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none;
            box-shadow: 0 2px 8px rgba(26, 77, 42, 0.2);
        }
        
        /* Add bottom padding to main content to account for fixed button */
        .checkout-main {
            padding-bottom: 100px;
        }
        
        @media (max-width: 768px) {
            .complete-order-section {
                padding: 12px 16px;
            }
            
            .complete-order-section .place-order-btn {
                padding: 14px 20px;
                font-size: 16px;
            }
        }
    </style>
</body>
</html>