-- Perfume Store Database Schema
-- Run this SQL script to create the database and tables

-- Create database
CREATE DATABASE IF NOT EXISTS perfume_store;
USE perfume_store;

-- Users table
CREATE TABLE IF NOT EXISTS users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    email VARCHAR(191) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    first_name VARCHAR(100) NOT NULL,
    last_name VARCHAR(100) NOT NULL,
    avatar VARCHAR(191),
    phone VARCHAR(20),
    address TEXT,
    city VARCHAR(100),
    state VARCHAR(100),
    zip_code VARCHAR(20),
    country VARCHAR(100) DEFAULT 'Malaysia',
    is_guest BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Categories table
CREATE TABLE IF NOT EXISTS categories (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    slug VARCHAR(100) UNIQUE NOT NULL,
    description TEXT,
    image VARCHAR(191),
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Products table
CREATE TABLE IF NOT EXISTS products (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(191) NOT NULL,
    brand VARCHAR(100) NOT NULL,
    description TEXT,
    price DECIMAL(10,2) NOT NULL,
    currency VARCHAR(3) DEFAULT 'RM',
    volume VARCHAR(50),
    image VARCHAR(191),
    category_id INT,
    rating DECIMAL(2,1) DEFAULT 0.0,
    review_count INT DEFAULT 0,
    stock_quantity INT DEFAULT 0,
    is_bestseller BOOLEAN DEFAULT FALSE,
    is_new_arrival BOOLEAN DEFAULT FALSE,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (category_id) REFERENCES categories(id) ON DELETE SET NULL
);

-- Product sizes table
CREATE TABLE IF NOT EXISTS product_sizes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    product_id INT NOT NULL,
    size VARCHAR(50) NOT NULL,
    price_adjustment DECIMAL(10,2) DEFAULT 0.00,
    stock_quantity INT DEFAULT 0,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE
);

-- Cart table
CREATE TABLE IF NOT EXISTS cart (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT,
    session_id VARCHAR(191),
    product_id INT NOT NULL,
    quantity INT NOT NULL DEFAULT 1,
    size VARCHAR(50),
    price DECIMAL(10,2) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE,
    UNIQUE KEY unique_cart_item (user_id, product_id, size),
    UNIQUE KEY unique_session_cart_item (session_id, product_id, size)
);

-- Orders table
CREATE TABLE IF NOT EXISTS orders (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT,
    order_number VARCHAR(50) UNIQUE NOT NULL,
    total_amount DECIMAL(10,2) NOT NULL,
    currency VARCHAR(3) DEFAULT 'RM',
    status ENUM('pending', 'processing', 'shipped', 'delivered', 'cancelled') DEFAULT 'pending',
    shipping_address TEXT,
    billing_address TEXT,
    payment_method VARCHAR(50),
    payment_status ENUM('pending', 'paid', 'failed', 'refunded') DEFAULT 'pending',
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL
);

-- Order items table
CREATE TABLE IF NOT EXISTS order_items (
    id INT AUTO_INCREMENT PRIMARY KEY,
    order_id INT NOT NULL,
    product_id INT NOT NULL,
    product_name VARCHAR(191) NOT NULL,
    product_price DECIMAL(10,2) NOT NULL,
    quantity INT NOT NULL,
    size VARCHAR(50),
    total_price DECIMAL(10,2) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (order_id) REFERENCES orders(id) ON DELETE CASCADE,
    FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE SET NULL
);

-- Reviews table
CREATE TABLE IF NOT EXISTS reviews (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    product_id INT NOT NULL,
    rating INT NOT NULL CHECK (rating >= 1 AND rating <= 5),
    title VARCHAR(191),
    comment TEXT,
    is_verified BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_product_review (user_id, product_id)
);

-- Wishlist table
CREATE TABLE IF NOT EXISTS wishlist (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    product_id INT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE,
    UNIQUE KEY unique_wishlist_item (user_id, product_id)
);

-- Orders table
CREATE TABLE IF NOT EXISTS orders (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT,
    session_id VARCHAR(191),
    first_name VARCHAR(100) NOT NULL,
    last_name VARCHAR(100) NOT NULL,
    email VARCHAR(191) NOT NULL,
    phone VARCHAR(20) NOT NULL,
    address TEXT NOT NULL,
    city VARCHAR(100) NOT NULL,
    state VARCHAR(100) NOT NULL,
    zip_code VARCHAR(20) NOT NULL,
    country VARCHAR(100) DEFAULT 'Malaysia',
    payment_method VARCHAR(50) NOT NULL,
    notes TEXT,
    subtotal DECIMAL(10,2) NOT NULL,
    shipping DECIMAL(10,2) DEFAULT 0.00,
    total DECIMAL(10,2) NOT NULL,
    status ENUM('pending', 'processing', 'shipped', 'delivered', 'cancelled') DEFAULT 'pending',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL
);

-- Order items table
CREATE TABLE IF NOT EXISTS order_items (
    id INT AUTO_INCREMENT PRIMARY KEY,
    order_id INT NOT NULL,
    product_id INT NOT NULL,
    product_name VARCHAR(191) NOT NULL,
    product_image VARCHAR(191),
    quantity INT NOT NULL,
    size VARCHAR(50),
    price DECIMAL(10,2) NOT NULL,
    subtotal DECIMAL(10,2) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (order_id) REFERENCES orders(id) ON DELETE CASCADE,
    FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE
);

-- Insert sample categories (ignore if already exist)
INSERT IGNORE INTO categories (name, slug, description) VALUES
('Women', 'women', 'Elegant and feminine fragrances for women'),
('Men', 'men', 'Bold and masculine fragrances for men'),
('Unisex', 'unisex', 'Versatile fragrances for everyone');

-- Insert sample products (ignore if already exist)
INSERT IGNORE INTO products (name, brand, description, price, currency, volume, image, category_id, rating, review_count, stock_quantity, is_bestseller, is_new_arrival) VALUES
('ANGEL SPIRIT', 'PARFUME2days', '2Days Angel Spirit - Category: Royal, Notes: Boozy, sweet vanilla, cinnamon and praline, Scent: Amber Vanilla, Season: Cold Season, Recommendation: Outing', 89.00, 'RM', '30 ml', 'images/angel-spirit-300x300.png', 1, 4.5, 85, 50, TRUE, FALSE),
('SUAVE', 'PARFUME2days', 'A sophisticated and elegant fragrance with a unique blend of notes', 69.00, 'RM', '35 ml', 'images/suave.png', 1, 4.2, 72, 30, TRUE, FALSE),
('BVLGARI Rose Goldea', 'BVLGARI', 'A luxurious fragrance that captures the essence of joy and elegance with its unique blend of precious ingredients', 229.00, 'RM', '50 ml', 'images/angel-spirit-300x300.png', 1, 4.5, 85, 25, FALSE, TRUE),
('Chopard Happy Bigaradia', 'Chopard', 'A luxurious fragrance that captures the essence of joy and elegance with its unique blend of precious ingredients', 249.00, 'RM', '50 ml', 'images/suave.png', 1, 4.2, 72, 20, FALSE, TRUE),
('Lancome Idol Idole Nectar', 'Lancome', 'A modern fragrance that embodies confidence and femininity with its sophisticated and alluring scent', 199.00, 'RM', '50 ml', 'images/angel-spirit-300x300.png', 1, 4.3, 58, 35, FALSE, TRUE),
('Christian Dior Happy Hour', 'Christian Dior', 'An elegant and timeless fragrance that celebrates the joy of life with its refined and captivating aroma', 189.00, 'RM', '50 ml', 'images/suave.png', 1, 4.1, 43, 40, FALSE, TRUE),
('Tom Ford Black Orchid', 'Tom Ford', 'A luxurious and mysterious fragrance with deep, rich notes', 299.00, 'RM', '50 ml', 'images/angel-spirit-300x300.png', 1, 4.6, 92, 15, FALSE, FALSE),
('Chanel No. 5', 'Chanel', 'The iconic fragrance that has defined elegance for generations', 179.00, 'RM', '50 ml', 'images/suave.png', 1, 4.4, 156, 60, FALSE, FALSE),
('Dior Sauvage', 'Christian Dior', 'A fresh and woody fragrance for the modern man', 159.00, 'RM', '60 ml', 'images/angel-spirit-300x300.png', 2, 4.7, 203, 45, FALSE, FALSE),
('Versace Eros', 'Versace', 'A bold and seductive fragrance for confident men', 139.00, 'RM', '50 ml', 'images/suave.png', 2, 4.3, 98, 30, FALSE, FALSE);

-- Insert product sizes (ignore if already exist)
INSERT IGNORE INTO product_sizes (product_id, size, price_adjustment, stock_quantity) VALUES
(1, '30ml', 0.00, 50),
(1, '50ml', 20.00, 30),
(2, '35ml', 0.00, 30),
(2, '50ml', 15.00, 25),
(3, '50ml', 0.00, 25),
(3, '100ml', 50.00, 15),
(4, '50ml', 0.00, 20),
(4, '100ml', 60.00, 10),
(5, '50ml', 0.00, 35),
(5, '100ml', 45.00, 20),
(6, '50ml', 0.00, 40),
(6, '100ml', 40.00, 25),
(7, '50ml', 0.00, 15),
(7, '100ml', 80.00, 8),
(8, '50ml', 0.00, 60),
(8, '100ml', 70.00, 30),
(9, '60ml', 0.00, 45),
(9, '100ml', 50.00, 25),
(10, '50ml', 0.00, 30),
(10, '100ml', 40.00, 15);

-- Loyalty and Rewards System Tables

-- User loyalty points table
CREATE TABLE IF NOT EXISTS user_loyalty_points (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    points INT DEFAULT 0,
    tier ENUM('bronze', 'silver', 'gold') DEFAULT 'bronze',
    total_spent DECIMAL(10,2) DEFAULT 0.00,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

-- Points transactions table
CREATE TABLE IF NOT EXISTS points_transactions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    points INT NOT NULL,
    transaction_type ENUM('earned', 'redeemed', 'expired', 'bonus') NOT NULL,
    description VARCHAR(255) NOT NULL,
    order_id INT NULL,
    reference_id VARCHAR(100) NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (order_id) REFERENCES orders(id) ON DELETE SET NULL
);

-- Digital stamps table
CREATE TABLE IF NOT EXISTS digital_stamps (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    stamps_collected INT DEFAULT 0,
    stamps_required INT DEFAULT 10,
    is_completed BOOLEAN DEFAULT FALSE,
    reward_claimed BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

-- Referral codes table
CREATE TABLE IF NOT EXISTS referral_codes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    referral_code VARCHAR(20) UNIQUE NOT NULL,
    points_earned INT DEFAULT 0,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

-- Referral usage table
CREATE TABLE IF NOT EXISTS referral_usage (
    id INT AUTO_INCREMENT PRIMARY KEY,
    referrer_id INT NOT NULL,
    referred_user_id INT NOT NULL,
    referral_code VARCHAR(20) NOT NULL,
    points_awarded INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (referrer_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (referred_user_id) REFERENCES users(id) ON DELETE CASCADE
);

-- Insert sample user (for testing) - ignore if already exist
INSERT IGNORE INTO users (email, password, first_name, last_name, phone, address, city, state, zip_code) VALUES
('user@example.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'John', 'Doe', '+60123456789', '123 Main Street', 'Kuala Lumpur', 'Kuala Lumpur', '50000');

-- Insert sample loyalty data - ignore if already exist
INSERT IGNORE INTO user_loyalty_points (user_id, points, tier, total_spent) VALUES
(1, 1240, 'silver', 1240.00);

INSERT IGNORE INTO digital_stamps (user_id, stamps_collected, stamps_required) VALUES
(1, 3, 10);

INSERT IGNORE INTO referral_codes (user_id, referral_code) VALUES
(1, 'JOHN2024');

-- Insert sample points transactions - ignore if already exist
INSERT IGNORE INTO points_transactions (user_id, points, transaction_type, description, reference_id) VALUES
(1, 56, 'earned', 'Purchase at Perfume Store', 'ORDER001'),
(1, 123, 'earned', 'Purchase at Perfume Store', 'ORDER002'),
(1, 89, 'earned', 'Purchase at Perfume Store', 'ORDER003');

-- Product field presets table
CREATE TABLE IF NOT EXISTS product_field_presets (
    id INT AUTO_INCREMENT PRIMARY KEY,
    field_name VARCHAR(50) NOT NULL,
    field_type ENUM('brand', 'volume', 'size', 'category', 'season', 'notes', 'scent', 'recommendation') NOT NULL,
    preset_value VARCHAR(100) NOT NULL,
    is_active BOOLEAN DEFAULT TRUE,
    sort_order INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY unique_field_preset (field_type, preset_value)
);

-- Insert default presets (ignore if already exist)
INSERT IGNORE INTO product_field_presets (field_type, field_name, preset_value, sort_order) VALUES
-- Brand presets
('brand', 'Brand', 'PARFUME2days', 1),
('brand', 'Brand', 'BVLGARI', 2),
('brand', 'Brand', 'Chopard', 3),
('brand', 'Brand', 'Lancome', 4),
('brand', 'Brand', 'Christian Dior', 5),
('brand', 'Brand', 'Tom Ford', 6),
('brand', 'Brand', 'Chanel', 7),
('brand', 'Brand', 'Versace', 8),
('brand', 'Brand', 'Yves Saint Laurent', 9),
('brand', 'Brand', 'Giorgio Armani', 10),

-- Volume presets
('volume', 'Volume', '30ml', 1),
('volume', 'Volume', '35ml', 2),
('volume', 'Volume', '50ml', 3),
('volume', 'Volume', '60ml', 4),
('volume', 'Volume', '75ml', 5),
('volume', 'Volume', '100ml', 6),
('volume', 'Volume', '125ml', 7),

-- Size presets
('size', 'Size', '30ml', 1),
('size', 'Size', '35ml', 2),
('size', 'Size', '50ml', 3),
('size', 'Size', '60ml', 4),
('size', 'Size', '75ml', 5),
('size', 'Size', '100ml', 6),
('size', 'Size', '125ml', 7),

-- Season presets
('season', 'Season', 'All Season', 1),
('season', 'Season', 'Spring', 2),
('season', 'Season', 'Summer', 3),
('season', 'Season', 'Autumn', 4),
('season', 'Season', 'Winter', 5),
('season', 'Season', 'Cold Season', 6),
('season', 'Season', 'Warm Season', 7),

-- Notes presets
('notes', 'Notes', 'Boozy', 1),
('notes', 'Notes', 'Sweet vanilla', 2),
('notes', 'Notes', 'Cinnamon', 3),
('notes', 'Notes', 'Praline', 4),
('notes', 'Notes', 'Floral', 5),
('notes', 'Notes', 'Woody', 6),
('notes', 'Notes', 'Citrus', 7),
('notes', 'Notes', 'Oriental', 8),
('notes', 'Notes', 'Fresh', 9),
('notes', 'Notes', 'Spicy', 10),

-- Scent presets
('scent', 'Scent', 'Amber Vanilla', 1),
('scent', 'Scent', 'Floral', 2),
('scent', 'Scent', 'Woody', 3),
('scent', 'Scent', 'Citrus', 4),
('scent', 'Scent', 'Oriental', 5),
('scent', 'Scent', 'Fresh', 6),
('scent', 'Scent', 'Spicy', 7),
('scent', 'Scent', 'Aquatic', 8),
('scent', 'Scent', 'Gourmand', 9),

-- Recommendation presets
('recommendation', 'Recommendation', 'Outing', 1),
('recommendation', 'Recommendation', 'Office', 2),
('recommendation', 'Recommendation', 'Date Night', 3),
('recommendation', 'Recommendation', 'Casual', 4),
('recommendation', 'Recommendation', 'Formal', 5),
('recommendation', 'Recommendation', 'Evening', 6),
('recommendation', 'Recommendation', 'Daily Wear', 7);

-- Create indexes for better performance (ignore if already exist)
CREATE INDEX IF NOT EXISTS idx_products_category ON products(category_id);
CREATE INDEX IF NOT EXISTS idx_products_bestseller ON products(is_bestseller);
CREATE INDEX IF NOT EXISTS idx_products_new_arrival ON products(is_new_arrival);
CREATE INDEX IF NOT EXISTS idx_products_active ON products(is_active);
CREATE INDEX IF NOT EXISTS idx_cart_user ON cart(user_id);
CREATE INDEX IF NOT EXISTS idx_cart_session ON cart(session_id);
CREATE INDEX IF NOT EXISTS idx_orders_user ON orders(user_id);
CREATE INDEX IF NOT EXISTS idx_orders_status ON orders(status);
CREATE INDEX IF NOT EXISTS idx_reviews_product ON reviews(product_id);
CREATE INDEX IF NOT EXISTS idx_wishlist_user ON wishlist(user_id);
CREATE INDEX IF NOT EXISTS idx_presets_field_type ON product_field_presets(field_type);
CREATE INDEX IF NOT EXISTS idx_presets_active ON product_field_presets(is_active);