<?php
session_start();
require_once 'models/User.php';
require_once 'models/Order.php';
require_once 'includes/remember-me.php';

// Check for remember me token and auto-login
checkRememberMe();

// Check if user is logged in
if (!isset($_SESSION['user_id']) || $_SESSION['user_id'] === 'guest') {
    header('Location: login.php');
    exit();
}

// Get user data
$user = new User();
$user_data = $user->getById($_SESSION['user_id']);

if (!$user_data) {
    header('Location: login.php');
    exit();
}

// Get user's orders
$order_model = new Order();
$user_orders = $order_model->getUserOrders($_SESSION['user_id']);

// Filter orders by status for different tabs
$processing_orders = array_filter($user_orders, function($order) {
    return in_array($order['status'], ['pending', 'processing', 'shipped']);
});

$delivered_orders = array_filter($user_orders, function($order) {
    return $order['status'] === 'delivered';
});

$cancelled_orders = array_filter($user_orders, function($order) {
    return $order['status'] === 'cancelled';
});
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Delivery Tracking - 2 Days Parfume</title>
    <link rel="stylesheet" href="styles.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="script.js" defer></script>
    <style>
        .delivery-container {
            min-height: 100vh;
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            padding-bottom: 100px;
        }

        .delivery-header {
            background: linear-gradient(135deg, #1A4D2A 0%, #2d5a3d 50%, #1A4D2A 100%);
            padding: 20px 20px 30px 20px;
            text-align: center;
            position: relative;
            overflow: hidden;
        }

        .delivery-header::before {
            content: '';
            position: absolute;
            bottom: -20px;
            left: 0;
            right: 0;
            height: 40px;
            background: linear-gradient(135deg, #1A4D2A 0%, #2d5a3d 50%, #1A4D2A 100%);
            border-radius: 50% 50% 0 0 / 100% 100% 0 0;
        }

        .delivery-title {
            color: #D4AF37;
            font-size: 24px;
            font-weight: bold;
            margin-bottom: 5px;
        }

        .delivery-subtitle {
            color: #F4D03F;
            font-size: 14px;
        }

        .delivery-content {
            padding: 30px 20px;
            margin-top: -20px;
            position: relative;
            z-index: 2;
        }

        .tabs-container {
            background: white;
            border-radius: 15px;
            margin-bottom: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            overflow: hidden;
        }

        .tabs-header {
            display: flex;
            background: #f8f9fa;
            border-bottom: 1px solid #e9ecef;
        }

        .tab-button {
            flex: 1;
            padding: 15px 15px;
            text-align: center;
            background: none;
            border: none;
            font-size: 13px;
            font-weight: 600;
            color: #666;
            cursor: pointer;
            transition: all 0.3s ease;
            position: relative;
        }

        .tab-button.active {
            color: #1A4D2A;
            background: white;
        }

        .tab-button.active::after {
            content: '';
            position: absolute;
            bottom: 0;
            left: 0;
            right: 0;
            height: 3px;
            background: linear-gradient(135deg, #D4AF37, #F4D03F);
        }

        .tab-count {
            display: block;
            font-size: 16px;
            font-weight: bold;
            margin-bottom: 2px;
        }

        .tab-content {
            display: none;
            padding: 20px;
        }

        .tab-content.active {
            display: block;
        }

        .order-card {
            background: white;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 15px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
            border-left: 4px solid #D4AF37;
        }

        .order-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
        }

        .order-number {
            font-size: 16px;
            font-weight: bold;
            color: #1A4D2A;
        }

        .order-date {
            font-size: 12px;
            color: #666;
        }

        .order-status {
            display: inline-flex;
            align-items: center;
            gap: 5px;
            padding: 6px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            text-transform: uppercase;
        }

        .status-pending {
            background: #fff3cd;
            color: #856404;
        }

        .status-processing {
            background: #cce5ff;
            color: #004085;
        }

        .status-shipped {
            background: #d1ecf1;
            color: #0c5460;
        }

        .status-delivered {
            background: #d4edda;
            color: #155724;
        }

        .status-cancelled {
            background: #f8d7da;
            color: #721c24;
        }

        .status-awaiting_payment {
            background: #fff3cd;
            color: #856404;
        }

        .status-paid {
            background: #d4edda;
            color: #155724;
        }

        .status-failed {
            background: #f8d7da;
            color: #721c24;
        }

        .status-refunded {
            background: #d1ecf1;
            color: #0c5460;
        }

        .order-items {
            margin-bottom: 15px;
        }

        .order-item {
            display: flex;
            align-items: center;
            padding: 10px 0;
            border-bottom: 1px solid #f0f0f0;
        }

        .order-item:last-child {
            border-bottom: none;
        }

        .item-image {
            width: 50px;
            height: 50px;
            border-radius: 8px;
            background: #f8f9fa;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 12px;
            color: #666;
        }

        .item-details {
            flex: 1;
        }

        .item-name {
            font-size: 14px;
            font-weight: 600;
            color: #1A4D2A;
            margin-bottom: 2px;
        }

        .item-size {
            font-size: 12px;
            color: #666;
        }

        .item-quantity {
            font-size: 12px;
            color: #666;
            margin-left: auto;
        }

        .order-total {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding-top: 15px;
            border-top: 1px solid #f0f0f0;
        }

        .total-label {
            font-size: 14px;
            font-weight: 600;
            color: #1A4D2A;
        }

        .total-amount {
            font-size: 16px;
            font-weight: bold;
            color: #D4AF37;
        }

        .tracking-number-section {
            margin-top: 15px;
            padding: 12px;
            background: #f8f9fa;
            border-radius: 8px;
            border-left: 3px solid #1A4D2A;
        }

        .tracking-number-label {
            font-size: 12px;
            font-weight: 600;
            color: #1A4D2A;
            margin-bottom: 5px;
            display: flex;
            align-items: center;
            gap: 6px;
        }

        .tracking-number-value {
            font-size: 14px;
            font-weight: bold;
            color: #D4AF37;
            font-family: 'Courier New', monospace;
            letter-spacing: 1px;
        }

        .tracking-section {
            margin-top: 15px;
            padding-top: 15px;
            border-top: 1px solid #f0f0f0;
        }

        .tracking-title {
            font-size: 14px;
            font-weight: 600;
            color: #1A4D2A;
            margin-bottom: 10px;
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .tracking-steps {
            position: relative;
        }

        .tracking-step {
            display: flex;
            align-items: center;
            margin-bottom: 15px;
            position: relative;
        }

        .tracking-step:last-child {
            margin-bottom: 0;
        }

        .step-icon {
            width: 30px;
            height: 30px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 12px;
            margin-right: 12px;
            position: relative;
            z-index: 2;
        }

        .step-icon.completed {
            background: #D4AF37;
            color: white;
        }

        .step-icon.current {
            background: #1A4D2A;
            color: white;
        }

        .step-icon.pending {
            background: #e9ecef;
            color: #666;
        }

        .step-content {
            flex: 1;
        }

        .step-title {
            font-size: 13px;
            font-weight: 600;
            color: #1A4D2A;
            margin-bottom: 2px;
        }

        .step-date {
            font-size: 11px;
            color: #666;
        }

        .tracking-line {
            position: absolute;
            left: 14px;
            top: 30px;
            bottom: -15px;
            width: 2px;
            background: #e9ecef;
        }

        .tracking-step:last-child .tracking-line {
            display: none;
        }

        .empty-state {
            text-align: center;
            padding: 40px 20px;
            color: #666;
        }

        .empty-state i {
            font-size: 48px;
            color: #D4AF37;
            margin-bottom: 15px;
            opacity: 0.5;
        }

        .empty-state h3 {
            font-size: 18px;
            color: #1A4D2A;
            margin-bottom: 10px;
        }

        .empty-state p {
            font-size: 14px;
            line-height: 1.5;
        }

        .action-buttons {
            display: flex;
            gap: 10px;
            margin-top: 15px;
        }

        .btn-track {
            flex: 1;
            padding: 10px;
            border: 2px solid #D4AF37;
            background: white;
            color: #D4AF37;
            border-radius: 8px;
            font-size: 12px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .btn-track:hover {
            background: #D4AF37;
            color: white;
        }

        .btn-details {
            flex: 1;
            padding: 10px;
            border: 2px solid #1A4D2A;
            background: white;
            color: #1A4D2A;
            border-radius: 8px;
            font-size: 12px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .btn-details:hover {
            background: #1A4D2A;
            color: white;
        }

        @media (max-width: 480px) {
            .delivery-content {
                padding: 20px 15px;
            }
            
            .order-card {
                padding: 15px;
            }
            
            .tabs-header {
                flex-wrap: wrap;
            }
            
            .tab-button {
                min-width: 100px;
                font-size: 12px;
                padding: 12px 10px;
            }
        }
    </style>
</head>
<body>
    <div class="delivery-container">
        <!-- Top Navigation -->
        <nav class="top-nav">
            <div class="nav-left">
                <i class="fas fa-arrow-left" onclick="goBack()"></i>
            </div>
            <div class="nav-center">
                <div class="logo">
                    <img src="images/LOGO.png" alt="2 Days Parfume Logo" class="logo-img">
                </div>
            </div>
            <div class="nav-right">
                <i class="fas fa-search" onclick="window.location.href='products.php'"></i>
            </div>
        </nav>

        <!-- Delivery Header -->
        <div class="delivery-header">
            <div class="delivery-title">Delivery Tracking</div>
            <div class="delivery-subtitle">Track your orders in real-time</div>
        </div>

        <!-- Delivery Content -->
        <div class="delivery-content">
            <!-- Debug Info (remove in production) -->
            <?php if (isset($_GET['debug']) && $_GET['debug'] === '1'): ?>
            <div class="debug-info" style="background: #f8f9fa; padding: 15px; border-radius: 8px; margin-bottom: 20px; font-family: monospace; font-size: 12px;">
                <h4>Debug Information:</h4>
                <p><strong>User ID:</strong> <?php echo $_SESSION['user_id']; ?></p>
                <p><strong>Total Orders:</strong> <?php echo count($user_orders); ?></p>
                <p><strong>Processing Orders:</strong> <?php echo count($processing_orders); ?></p>
                <p><strong>Delivered Orders:</strong> <?php echo count($delivered_orders); ?></p>
                <p><strong>Cancelled Orders:</strong> <?php echo count($cancelled_orders); ?></p>
                <p><strong>Recent Order Statuses:</strong></p>
                <ul>
                    <?php foreach (array_slice($user_orders, 0, 3) as $order): ?>
                        <li>Order #<?php echo $order['order_number']; ?>: <?php echo $order['status']; ?> (Updated: <?php echo $order['updated_at']; ?>)</li>
                    <?php endforeach; ?>
                </ul>
            </div>
            <?php endif; ?>
            
            <!-- Tabs Container -->
            <div class="tabs-container">
                <div class="tabs-header">
                    <button class="tab-button active" onclick="switchTab('processing')">
                        <span class="tab-count"><?php echo count($processing_orders); ?></span>
                        Processing
                    </button>
                    <button class="tab-button" onclick="switchTab('delivered')">
                        <span class="tab-count"><?php echo count($delivered_orders); ?></span>
                        Delivered
                    </button>
                    <button class="tab-button" onclick="switchTab('cancelled')">
                        <span class="tab-count"><?php echo count($cancelled_orders); ?></span>
                        Cancelled
                    </button>
                </div>

                <!-- Processing Orders Tab -->
                <div id="processing-tab" class="tab-content active">
                    <?php if (empty($processing_orders)): ?>
                        <div class="empty-state">
                            <i class="fas fa-cog"></i>
                            <h3>No Processing Orders</h3>
                            <p>You don't have any orders being processed at the moment.</p>
                        </div>
                    <?php else: ?>
                        <?php foreach ($processing_orders as $order): ?>
                            <?php include 'includes/order-card.php'; ?>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>

                <!-- Delivered Orders Tab -->
                <div id="delivered-tab" class="tab-content">
                    <?php if (empty($delivered_orders)): ?>
                        <div class="empty-state">
                            <i class="fas fa-check-circle"></i>
                            <h3>No Delivered Orders</h3>
                            <p>You don't have any delivered orders yet.</p>
                        </div>
                    <?php else: ?>
                        <?php foreach ($delivered_orders as $order): ?>
                            <?php include 'includes/order-card.php'; ?>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>

                <!-- Cancelled Orders Tab -->
                <div id="cancelled-tab" class="tab-content">
                    <?php if (empty($cancelled_orders)): ?>
                        <div class="empty-state">
                            <i class="fas fa-times-circle"></i>
                            <h3>No Cancelled Orders</h3>
                            <p>You don't have any cancelled orders.</p>
                        </div>
                    <?php else: ?>
                        <?php foreach ($cancelled_orders as $order): ?>
                            <?php include 'includes/order-card.php'; ?>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Bottom Navigation -->
        <nav class="bottom-nav">
            <div class="nav-item" onclick="window.location.href='index.php'">
                <i class="fas fa-home"></i>
            </div>
            <div class="nav-item" onclick="window.location.href='products.php'">
                <i class="fas fa-box"></i>
            </div>
            <div class="nav-item" onclick="window.location.href='cart.php'">
                <i class="fas fa-shopping-cart"></i>
            </div>
            <div class="nav-item" onclick="window.location.href='profile.php'">
                <i class="far fa-user"></i>
            </div>
        </nav>
    </div>

    <script>
        function goBack() {
            window.history.back();
        }

        function switchTab(tabName) {
            // Remove active class from all tabs and content
            document.querySelectorAll('.tab-button').forEach(btn => btn.classList.remove('active'));
            document.querySelectorAll('.tab-content').forEach(content => content.classList.remove('active'));
            
            // Add active class to selected tab and content
            event.target.classList.add('active');
            document.getElementById(tabName + '-tab').classList.add('active');
        }

        function markDeliverySuccess(orderId) {
            // Create custom mobile-friendly confirmation modal
            const modal = document.createElement('div');
            modal.className = 'confirmation-modal';
            modal.style.cssText = `
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background: rgba(0,0,0,0.5);
                z-index: 10000;
                display: flex;
                align-items: center;
                justify-content: center;
                padding: 20px;
            `;
            
            modal.innerHTML = `
                <div style="
                    background: white;
                    border-radius: 16px;
                    max-width: 320px;
                    width: 100%;
                    box-shadow: 0 10px 30px rgba(0,0,0,0.3);
                    overflow: hidden;
                ">
                    <div style="
                        padding: 25px 20px 20px 20px;
                        text-align: center;
                        background: linear-gradient(135deg, #1A4D2A 0%, #2d5a3d 100%);
                        color: white;
                    ">
                        <div style="
                            width: 60px;
                            height: 60px;
                            background: rgba(255,255,255,0.2);
                            border-radius: 50%;
                            display: flex;
                            align-items: center;
                            justify-content: center;
                            margin: 0 auto 15px auto;
                        ">
                            <i class="fas fa-check-circle" style="font-size: 28px; color: #D4AF37;"></i>
                        </div>
                        <h3 style="margin: 0 0 8px 0; font-size: 18px; font-weight: 600;">Confirm Delivery</h3>
                        <p style="margin: 0; font-size: 14px; opacity: 0.9;">Mark this order as delivered successfully?</p>
                    </div>
                    
                    <div style="padding: 20px;">
                        <div style="display: flex; gap: 12px;">
                            <button onclick="this.closest('.confirmation-modal').remove()" style="
                                flex: 1;
                                padding: 12px;
                                border: 2px solid #e9ecef;
                                background: white;
                                color: #666;
                                border-radius: 8px;
                                font-size: 14px;
                                font-weight: 600;
                                cursor: pointer;
                                transition: all 0.3s ease;
                            " onmouseover="this.style.background='#f8f9fa'" onmouseout="this.style.background='white'">
                                Cancel
                            </button>
                            <button onclick="confirmDelivery(${orderId})" style="
                                flex: 1;
                                padding: 12px;
                                border: 2px solid #1A4D2A;
                                background: #1A4D2A;
                                color: white;
                                border-radius: 8px;
                                font-size: 14px;
                                font-weight: 600;
                                cursor: pointer;
                                transition: all 0.3s ease;
                            " onmouseover="this.style.background='#2d5a3d'" onmouseout="this.style.background='#1A4D2A'">
                                <i class="fas fa-check"></i> Confirm
                            </button>
                        </div>
                    </div>
                </div>
            `;
            
            document.body.appendChild(modal);
        }

        function confirmDelivery(orderId) {
            // Close the modal
            const modal = document.querySelector('.confirmation-modal');
            if (modal) {
                modal.remove();
            }
            
            // Show loading state
            showNotification('Updating order status...', 'info');
            
            // Update order status to delivered
            fetch('ajax/update-order-status.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    order_id: orderId,
                    status: 'delivered'
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    let message = 'Order marked as delivered successfully!';
                    
                    // Show points awarded message if points were earned
                    if (data.points_awarded && data.points_awarded > 0) {
                        message += ` You earned ${data.points_awarded} loyalty points!`;
                    }
                    
                    showNotification(message, 'success');
                    
                    // Show feedback modal after successful delivery
                    setTimeout(() => {
                        showFeedbackModal(orderId);
                    }, 1000);
                } else {
                    showNotification('Error updating order status', 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showNotification('Error updating order status', 'error');
            });
        }

        function viewOrderDetails(orderId) {
            // Create mobile-friendly modal for order details
            const modal = document.createElement('div');
            modal.className = 'order-details-modal';
            modal.style.cssText = `
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background: rgba(0,0,0,0.5);
                z-index: 10000;
                display: flex;
                align-items: flex-end;
                justify-content: center;
                padding: 0;
            `;
            
            modal.innerHTML = `
                <div style="
                    background: white;
                    border-radius: 20px 20px 0 0;
                    width: 100%;
                    max-height: 85vh;
                    overflow-y: auto;
                    position: relative;
                    animation: slideUp 0.3s ease-out;
                ">
                    <div style="
                        padding: 20px 20px 15px 20px;
                        border-bottom: 1px solid #e9ecef;
                        display: flex;
                        justify-content: space-between;
                        align-items: center;
                        background: linear-gradient(135deg, #1A4D2A 0%, #2d5a3d 100%);
                        color: white;
                        border-radius: 20px 20px 0 0;
                    ">
                        <h3 style="margin: 0; font-size: 18px; font-weight: 600;">Order Details</h3>
                        <button onclick="this.closest('.order-details-modal').remove()" style="
                            background: rgba(255,255,255,0.2);
                            border: none;
                            width: 32px;
                            height: 32px;
                            border-radius: 50%;
                            font-size: 16px;
                            cursor: pointer;
                            color: white;
                            display: flex;
                            align-items: center;
                            justify-content: center;
                        ">&times;</button>
                    </div>
                    <div id="order-details-content" style="padding: 20px;">
                        <div style="text-align: center; padding: 40px;">
                            <i class="fas fa-spinner fa-spin" style="font-size: 24px; color: #D4AF37;"></i>
                            <p style="margin-top: 10px; color: #666;">Loading order details...</p>
                        </div>
                    </div>
                </div>
            `;
            
            // Add slide-up animation
            const style = document.createElement('style');
            style.textContent = `
                @keyframes slideUp {
                    from {
                        transform: translateY(100%);
                    }
                    to {
                        transform: translateY(0);
                    }
                }
            `;
            document.head.appendChild(style);
            
            document.body.appendChild(modal);
            
            
            // Close modal when clicking outside
            modal.addEventListener('click', (e) => {
                if (e.target === modal) {
                    modal.remove();
                }
            });
            
            // Load order details
            fetch(`ajax/get-order-details.php?id=${orderId}`)
                .then(response => response.json())
                .then(data => {
                    const content = document.getElementById('order-details-content');
                    if (data.success) {
                        content.innerHTML = data.html;
                    } else {
                        content.innerHTML = `
                            <div style="text-align: center; padding: 20px; color: #666;">
                                <i class="fas fa-exclamation-triangle" style="font-size: 24px; margin-bottom: 10px;"></i>
                                <p>Error loading order details</p>
                            </div>
                        `;
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    document.getElementById('order-details-content').innerHTML = `
                        <div style="text-align: center; padding: 20px; color: #666;">
                            <i class="fas fa-exclamation-triangle" style="font-size: 24px; margin-bottom: 10px;"></i>
                            <p>Error loading order details</p>
                        </div>
                    `;
                });
        }

        function showFeedbackModal(orderId) {
            // Create feedback modal
            const modal = document.createElement('div');
            modal.className = 'feedback-modal';
            modal.style.cssText = `
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background: rgba(0,0,0,0.5);
                z-index: 10000;
                display: flex;
                align-items: center;
                justify-content: center;
                padding: 20px;
            `;
            
            modal.innerHTML = `
                <div style="
                    background: white;
                    border-radius: 16px;
                    max-width: 350px;
                    width: 100%;
                    box-shadow: 0 10px 30px rgba(0,0,0,0.3);
                    overflow: hidden;
                ">
                    <div style="
                        padding: 25px 20px 20px 20px;
                        text-align: center;
                        background: linear-gradient(135deg, #1A4D2A 0%, #2d5a3d 100%);
                        color: white;
                    ">
                        <div style="
                            width: 60px;
                            height: 60px;
                            background: rgba(255,255,255,0.2);
                            border-radius: 50%;
                            display: flex;
                            align-items: center;
                            justify-content: center;
                            margin: 0 auto 15px auto;
                        ">
                            <i class="fas fa-star" style="font-size: 28px; color: #D4AF37;"></i>
                        </div>
                        <h3 style="margin: 0 0 8px 0; font-size: 18px; font-weight: 600;">Rate Your Experience</h3>
                        <p style="margin: 0; font-size: 14px; opacity: 0.9;">How was your delivery experience?</p>
                    </div>
                    
                    <div style="padding: 25px 20px;">
                        <!-- Star Rating -->
                        <div style="text-align: center; margin-bottom: 20px;">
                            <div class="star-rating" style="display: flex; justify-content: center; gap: 8px; margin-bottom: 10px;">
                                <i class="fas fa-star star" data-rating="1" style="
                                    font-size: 32px;
                                    color: #e9ecef;
                                    cursor: pointer;
                                    transition: color 0.2s ease;
                                "></i>
                                <i class="fas fa-star star" data-rating="2" style="
                                    font-size: 32px;
                                    color: #e9ecef;
                                    cursor: pointer;
                                    transition: color 0.2s ease;
                                "></i>
                                <i class="fas fa-star star" data-rating="3" style="
                                    font-size: 32px;
                                    color: #e9ecef;
                                    cursor: pointer;
                                    transition: color 0.2s ease;
                                "></i>
                                <i class="fas fa-star star" data-rating="4" style="
                                    font-size: 32px;
                                    color: #e9ecef;
                                    cursor: pointer;
                                    transition: color 0.2s ease;
                                "></i>
                                <i class="fas fa-star star" data-rating="5" style="
                                    font-size: 32px;
                                    color: #e9ecef;
                                    cursor: pointer;
                                    transition: color 0.2s ease;
                                "></i>
                            </div>
                            <p id="rating-text" style="margin: 0; font-size: 14px; color: #666; font-weight: 600;">Tap to rate</p>
                        </div>
                        
                        <!-- Feedback Text -->
                        <div style="margin-bottom: 20px;">
                            <textarea id="feedback-text" placeholder="Share your feedback about the delivery experience (optional)" style="
                                width: 100%;
                                height: 80px;
                                padding: 12px;
                                border: 2px solid #e9ecef;
                                border-radius: 8px;
                                font-size: 14px;
                                resize: none;
                                font-family: inherit;
                                box-sizing: border-box;
                            "></textarea>
                        </div>
                        
                        <!-- Action Buttons -->
                        <div style="display: flex; gap: 12px;">
                            <button onclick="skipFeedback(${orderId})" style="
                                flex: 1;
                                padding: 12px;
                                border: 2px solid #e9ecef;
                                background: white;
                                color: #666;
                                border-radius: 8px;
                                font-size: 14px;
                                font-weight: 600;
                                cursor: pointer;
                                transition: all 0.3s ease;
                            " onmouseover="this.style.background='#f8f9fa'" onmouseout="this.style.background='white'">
                                Skip
                            </button>
                            <button onclick="submitFeedback(${orderId})" id="submit-feedback-btn" disabled style="
                                flex: 1;
                                padding: 12px;
                                border: 2px solid #e9ecef;
                                background: #e9ecef;
                                color: #999;
                                border-radius: 8px;
                                font-size: 14px;
                                font-weight: 600;
                                cursor: not-allowed;
                                transition: all 0.3s ease;
                            ">
                                <i class="fas fa-paper-plane"></i> Submit
                            </button>
                        </div>
                    </div>
                </div>
            `;
            
            document.body.appendChild(modal);
            
            // Add star rating functionality
            let selectedRating = 0;
            const stars = modal.querySelectorAll('.star');
            const ratingText = modal.querySelector('#rating-text');
            const submitBtn = modal.querySelector('#submit-feedback-btn');
            
            const ratingTexts = {
                0: 'Tap to rate',
                1: 'Poor',
                2: 'Fair', 
                3: 'Good',
                4: 'Very Good',
                5: 'Excellent'
            };
            
            stars.forEach((star, index) => {
                star.addEventListener('click', () => {
                    selectedRating = index + 1;
                    updateStars();
                    updateSubmitButton();
                });
                
                star.addEventListener('mouseenter', () => {
                    if (selectedRating === 0) {
                        highlightStars(index + 1);
                    }
                });
            });
            
            modal.querySelector('.star-rating').addEventListener('mouseleave', () => {
                if (selectedRating === 0) {
                    highlightStars(0);
                }
            });
            
            function updateStars() {
                stars.forEach((star, index) => {
                    if (index < selectedRating) {
                        star.style.color = '#D4AF37';
                    } else {
                        star.style.color = '#e9ecef';
                    }
                });
                ratingText.textContent = ratingTexts[selectedRating];
            }
            
            function highlightStars(count) {
                stars.forEach((star, index) => {
                    if (index < count) {
                        star.style.color = '#D4AF37';
                    } else {
                        star.style.color = '#e9ecef';
                    }
                });
                ratingText.textContent = count > 0 ? ratingTexts[count] : ratingTexts[0];
            }
            
            function updateSubmitButton() {
                if (selectedRating > 0) {
                    submitBtn.disabled = false;
                    submitBtn.style.background = '#1A4D2A';
                    submitBtn.style.borderColor = '#1A4D2A';
                    submitBtn.style.color = 'white';
                    submitBtn.style.cursor = 'pointer';
                } else {
                    submitBtn.disabled = true;
                    submitBtn.style.background = '#e9ecef';
                    submitBtn.style.borderColor = '#e9ecef';
                    submitBtn.style.color = '#999';
                    submitBtn.style.cursor = 'not-allowed';
                }
            }
        }

        function submitFeedback(orderId) {
            // Get the selected rating from the modal
            const modal = document.querySelector('.feedback-modal');
            const stars = modal.querySelectorAll('.star');
            let rating = 0;
            stars.forEach((star, index) => {
                if (star.style.color === 'rgb(212, 175, 55)') {
                    rating = index + 1;
                }
            });
            
            const feedback = document.getElementById('feedback-text').value.trim();
            
            // Close the modal
            if (modal) {
                modal.remove();
            }
            
            // Submit feedback
            const feedbackData = {
                order_id: orderId,
                rating: rating,
                feedback: feedback
            };
            
            console.log('Submitting feedback:', feedbackData);
            
            fetch('ajax/submit-feedback.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(feedbackData)
            })
            .then(response => response.json())
            .then(data => {
                console.log('Feedback response:', data);
                if (data.success) {
                    showNotification('Thank you for your feedback!', 'success');
                    // Refresh the page to show updated status
                    setTimeout(() => {
                        location.reload();
                    }, 1500);
                } else {
                    showNotification(data.message || 'Error submitting feedback', 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showNotification('Error submitting feedback', 'error');
            });
        }

        function skipFeedback(orderId) {
            // Close the modal
            const modal = document.querySelector('.feedback-modal');
            if (modal) {
                modal.remove();
            }
            
            // Refresh the page to show updated status
            setTimeout(() => {
                location.reload();
            }, 500);
        }


        // Auto-refresh tracking data every 30 seconds
        let lastUpdateTime = new Date().toISOString();
        
        setInterval(function() {
            checkForUpdates();
        }, 30000);
        
        function checkForUpdates() {
            fetch(`ajax/get-order-updates.php?last_update=${encodeURIComponent(lastUpdateTime)}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success && data.has_updates) {
                        // Update tab counts
                        Object.keys(data.order_counts).forEach(status => {
                            // Map old status to new tab names
                            let tabName = status;
                            if (status === 'pending') {
                                tabName = 'processing';
                            }
                            
                            const tabButton = document.querySelector(`[onclick="switchTab('${tabName}')"]`);
                            if (tabButton) {
                                const countSpan = tabButton.querySelector('.tab-count');
                                if (countSpan) {
                                    // For processing tab, combine pending, processing, and shipped counts (excluding awaiting_payment)
                                    if (tabName === 'processing') {
                                        const pendingCount = data.order_counts['pending'] || 0;
                                        const processingCount = data.order_counts['processing'] || 0;
                                        const shippedCount = data.order_counts['shipped'] || 0;
                                        countSpan.textContent = pendingCount + processingCount + shippedCount;
                                    } else {
                                        countSpan.textContent = data.order_counts[status];
                                    }
                                }
                            }
                        });
                        
                        // Show notification for updated orders
                        if (data.updated_orders.length > 0) {
                            showNotification(`${data.updated_orders.length} order(s) updated!`, 'success');
                            // Optionally refresh the page to show latest data
                            setTimeout(() => {
                                location.reload();
                            }, 2000);
                        }
                        
                        lastUpdateTime = data.current_time;
                    }
                })
                .catch(error => {
                    console.error('Error checking for updates:', error);
                });
        }

        // Initialize cart badge
        document.addEventListener('DOMContentLoaded', function() {
            updateCartBadge();
        });
    </script>
</body>
</html>