<?php
session_start();
require_once 'config/database.php';
require_once 'config/email.php';

$error_message = '';
$success_message = '';
$email_sent = false;

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email = trim($_POST['email'] ?? '');
    
    if (empty($email)) {
        $error_message = 'Please enter your email address.';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error_message = 'Please enter a valid email address.';
    } else {
        try {
            $db = getDB();
            
            // Check if user exists
            $stmt = $db->prepare("SELECT id, first_name, last_name, email FROM users WHERE email = :email AND is_guest = 0");
            $stmt->bindValue(':email', $email);
            $stmt->execute();
            $user = $stmt->fetch();
            
            if ($user) {
                // Generate secure token
                $token = bin2hex(random_bytes(32));
                $expires_at = date('Y-m-d H:i:s', strtotime('+1 hour'));
                
                // Store token in database
                $stmt = $db->prepare("INSERT INTO password_reset_tokens (user_id, email, token, expires_at) VALUES (:user_id, :email, :token, :expires_at)");
                $stmt->bindValue(':user_id', $user['id']);
                $stmt->bindValue(':email', $email);
                $stmt->bindValue(':token', $token);
                $stmt->bindValue(':expires_at', $expires_at);
                $stmt->execute();
                
                // Generate reset link
                $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
                $host = $_SERVER['HTTP_HOST'];
                $resetLink = $protocol . '://' . $host . '/apps/reset-password.php?token=' . $token;
                
                // Send email
                $userName = $user['first_name'] . ' ' . $user['last_name'];
                $subject = 'Password Reset Request - 2 Days Parfume';
                $body = getPasswordResetEmailTemplate($userName, $resetLink);
                
                $emailResult = sendEmail($email, $subject, $body);
                
                if ($emailResult['success']) {
                    $success_message = 'Password reset instructions have been sent to your email address.';
                    $email_sent = true;
                } else {
                    $error_message = 'Failed to send email. Please try again later or contact support.';
                    error_log("Password reset email failed for {$email}: " . $emailResult['message']);
                }
            } else {
                // Don't reveal if email exists or not (security best practice)
                $success_message = 'If an account exists with this email, you will receive password reset instructions.';
                $email_sent = true;
            }
        } catch (Exception $e) {
            error_log("Password reset error: " . $e->getMessage());
            $error_message = 'An error occurred. Please try again later.';
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Forgot Password - 2 Days Parfume</title>
    <link rel="stylesheet" href="styles.css">
    <link rel="stylesheet" href="auth.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .info-box {
            background: #e3f2fd;
            border-left: 4px solid #2196f3;
            padding: 15px;
            margin: 20px 0;
            border-radius: 4px;
        }
        .info-box i {
            color: #2196f3;
            margin-right: 10px;
        }
        .info-box p {
            margin: 5px 0;
            font-size: 14px;
            color: #1976d2;
        }
        .back-to-login {
            text-align: center;
            margin-top: 20px;
        }
        .back-to-login a {
            color: #666;
            text-decoration: none;
            font-size: 14px;
            display: inline-flex;
            align-items: center;
            gap: 5px;
        }
        .back-to-login a:hover {
            color: #333;
        }
    </style>
</head>
<body>
    <div class="auth-container">
        <!-- Top Navigation -->
        <nav class="top-nav">
            <div class="nav-left">
                <i class="fas fa-arrow-left" onclick="window.location.href='login.php'"></i>
            </div>
            <div class="nav-center">
                <div class="logo">
                    <img src="images/LOGO.png" alt="2 Days Parfume Logo" class="logo-img">
                </div>
            </div>
            <div class="nav-right">
                <i class="fas fa-search" onclick="window.location.href='products.php'"></i>
            </div>
        </nav>

        <!-- Auth Content -->
        <div class="auth-content">
            <?php if (!$email_sent): ?>
            <div class="auth-header">
                <h1 class="auth-title">Forgot Password?</h1>
                <p class="auth-subtitle">Enter your email address and we'll send you instructions to reset your password.</p>
            </div>

            <!-- Error/Success Messages -->
            <?php if ($error_message): ?>
                <div class="alert alert-error">
                    <i class="fas fa-exclamation-circle"></i>
                    <?php echo htmlspecialchars($error_message); ?>
                </div>
            <?php endif; ?>

            <!-- Forgot Password Form -->
            <form class="auth-form" method="POST" action="">
                <div class="form-group">
                    <label for="email" style="display: block; margin-bottom: 8px; font-size: 14px; color: #333;">Email Address</label>
                    <input type="email" name="email" id="email" class="form-input" placeholder="Enter your email" required autofocus>
                </div>

                <button type="submit" class="auth-btn primary">
                    <span>Send Reset Link</span>
                </button>
            </form>

            <div class="info-box">
                <i class="fas fa-info-circle"></i>
                <p><strong>What happens next?</strong></p>
                <p>• You'll receive an email with a password reset link</p>
                <p>• The link will be valid for 1 hour</p>
                <p>• Click the link to create a new password</p>
            </div>

            <?php else: ?>
            <!-- Success State -->
            <div class="auth-header">
                <div style="text-align: center; margin-bottom: 20px;">
                    <i class="fas fa-envelope-circle-check" style="font-size: 64px; color: #4caf50;"></i>
                </div>
                <h1 class="auth-title">Check Your Email</h1>
                <p class="auth-subtitle">We've sent password reset instructions to your email address.</p>
            </div>

            <div class="alert alert-success">
                <i class="fas fa-check-circle"></i>
                <?php echo htmlspecialchars($success_message); ?>
            </div>

            <div class="info-box">
                <i class="fas fa-lightbulb"></i>
                <p><strong>Didn't receive the email?</strong></p>
                <p>• Check your spam or junk folder</p>
                <p>• Make sure you entered the correct email address</p>
                <p>• Wait a few minutes and check again</p>
                <p>• <a href="forgot-password.php" style="color: #2196f3;">Try again</a> or <a href="customer-service.php" style="color: #2196f3;">contact support</a></p>
            </div>
            <?php endif; ?>

            <!-- Back to Login Link -->
            <div class="back-to-login">
                <a href="login.php">
                    <i class="fas fa-arrow-left"></i>
                    Back to Login
                </a>
            </div>
        </div>
    </div>
</body>
</html>

