/**
 * Analytics Tracking System
 * Tracks user interactions and sends data to server
 */

(function() {
    'use strict';
    
    const AnalyticsTracker = {
        endpoint: 'ajax/track-analytics-simple.php',
        sessionStartTime: Date.now(),
        
        // Track event
        track: function(eventType, data = {}) {
            const trackingData = {
                event_type: eventType,
                timestamp: Date.now(),
                ...data
            };
            
            console.log('Analytics tracking event:', eventType, 'data:', trackingData, 'endpoint:', this.endpoint);
            
            // Use sendBeacon for reliability (works even when page is closing)
            if (navigator.sendBeacon) {
                const blob = new Blob([JSON.stringify(trackingData)], { type: 'application/json' });
                const result = navigator.sendBeacon(this.endpoint, blob);
                console.log('sendBeacon result:', result);
            } else {
                // Fallback to fetch
                fetch(this.endpoint, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify(trackingData),
                    keepalive: true
                })
                .then(response => response.json())
                .then(data => console.log('Analytics response:', data))
                .catch(err => console.error('Analytics tracking failed:', err));
            }
        },
        
        // Track page view
        trackPageView: function(pageType = 'other') {
            this.track('page_view', {
                page_url: window.location.href,
                page_type: pageType,
                screen_width: window.innerWidth,
                screen_height: window.innerHeight
            });
        },
        
        // Track product view
        trackProductView: function(productId) {
            this.track('product_view', {
                product_id: productId
            });
        },
        
        // Track add to favorites
        trackAddFavorite: function(productId) {
            console.log('trackAddFavorite called with productId:', productId);
            this.track('add_favorite', {
                product_id: productId
            });
        },
        
        // Track add to cart
        trackAddToCart: function(productId, quantity = 1) {
            console.log('trackAddToCart called with productId:', productId, 'quantity:', quantity);
            this.track('add_to_cart', {
                product_id: productId,
                quantity: quantity
            });
        },
        
        // Track traffic source (on first page load)
        trackTrafficSource: function() {
            const urlParams = new URLSearchParams(window.location.search);
            this.track('traffic_source', {
                utm_source: urlParams.get('utm_source'),
                utm_medium: urlParams.get('utm_medium'),
                utm_campaign: urlParams.get('utm_campaign'),
                landing_page: window.location.pathname
            });
        },
        
        // Initialize tracking
        init: function() {
            // Track traffic source on first visit
            if (!sessionStorage.getItem('analytics_tracked')) {
                this.trackTrafficSource();
                sessionStorage.setItem('analytics_tracked', '1');
            }
            
            // Auto-detect and track page type
            const path = window.location.pathname;
            let pageType = 'other';
            
            if (path === '/' || path === '/index.php') {
                pageType = 'home';
            } else if (path.includes('product-detail') || path.includes('product.php')) {
                pageType = 'product';
            } else if (path.includes('cart')) {
                pageType = 'cart';
            } else if (path.includes('checkout')) {
                pageType = 'checkout';
            } else if (path.includes('products') || path.includes('category')) {
                pageType = 'category';
            }
            
            this.trackPageView(pageType);
            
            // Track page exit (time spent)
            window.addEventListener('beforeunload', () => {
                const duration = Math.floor((Date.now() - this.sessionStartTime) / 1000);
                // Store for next request
                sessionStorage.setItem('last_page_duration', duration);
            });
        }
    };
    
    // Auto-initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => AnalyticsTracker.init());
    } else {
        AnalyticsTracker.init();
    }
    
    // Expose to global scope
    window.AnalyticsTracker = AnalyticsTracker;
    
})();

// Auto-track add to cart and favorites
document.addEventListener('DOMContentLoaded', function() {
    // Track add to cart buttons (still auto-track these)
    document.querySelectorAll('.add-to-cart-btn, [data-action="add-to-cart"]').forEach(button => {
        button.addEventListener('click', function() {
            const productId = this.getAttribute('data-product-id');
            const quantity = this.getAttribute('data-quantity') || 1;
            if (productId) {
                window.AnalyticsTracker.trackAddToCart(productId, quantity);
            }
        });
    });
});
