<?php
require_once __DIR__ . '/../config/database.php';

class FirebaseNotification {
    private $conn;
    
    // Android/Web Keys
    private $androidAppKey;
    private $androidApiKey;
    
    // iOS Keys
    private $iosAppKey;
    private $iosApiKey;
    
    private $apiUrl = 'https://appilix.com/api/push-notification';
    
    public function __construct() {
        $this->conn = getDB();
        
        // Android/Web Keys (Original)
        $this->androidAppKey = '0rb8jz9ktl4cl18i4702dixcya3rf72q5tq3mnhh';
        $this->androidApiKey = 'ypsbicz6mt1hue5g4do2';
        
        // iOS Keys (New)
        $this->iosAppKey = 'sq59rjjalqf715dvybdscahmzf27ickwo0goehgi';
        $this->iosApiKey = 'uwreh1o0dmxt32q517jz';
    }
    
    /**
     * Send push notification to all users (Both iOS and Android)
     */
    public function sendNotificationToAll($title, $body, $adminName = null) {
        try {
            $results = [];
            $successCount = 0;
            $failCount = 0;
            $messages = [];
            
            // Send to Android/Web
            $androidData = [
                'app_key' => $this->androidAppKey,
                'api_key' => $this->androidApiKey,
                'notification_title' => $title,
                'notification_body' => $body
            ];
            
            $androidResult = $this->makeApiCall($androidData, 'Android/Web');
            $results['android'] = $androidResult;
            
            if ($androidResult['success']) {
                $successCount++;
                $messages[] = '✓ Android/Web';
            } else {
                $failCount++;
                $messages[] = '✗ Android/Web: ' . $androidResult['message'];
            }
            
            // Send to iOS
            $iosData = [
                'app_key' => $this->iosAppKey,
                'api_key' => $this->iosApiKey,
                'notification_title' => $title,
                'notification_body' => $body
            ];
            
            $iosResult = $this->makeApiCall($iosData, 'iOS');
            $results['ios'] = $iosResult;
            
            if ($iosResult['success']) {
                $successCount++;
                $messages[] = '✓ iOS';
            } else {
                $failCount++;
                $messages[] = '✗ iOS: ' . $iosResult['message'];
            }
            
            // Determine overall success
            $overallSuccess = $successCount > 0;
            $overallMessage = "Sent to $successCount platform(s). " . implode(', ', $messages);
            
            // Log the notification
            $this->logNotification($title, $body, 'all', $adminName, $overallSuccess, $overallMessage);
            
            return [
                'success' => $overallSuccess,
                'message' => $overallMessage,
                'details' => $results,
                'stats' => [
                    'successful' => $successCount,
                    'failed' => $failCount
                ]
            ];
        } catch (Exception $e) {
            error_log("Firebase notification error: " . $e->getMessage());
            return ['success' => false, 'message' => 'Failed to send notification: ' . $e->getMessage()];
        }
    }
    
    /**
     * Send push notification to specific user (Both iOS and Android)
     */
    public function sendNotificationToUser($title, $body, $userId, $adminName = null) {
        try {
            $results = [];
            $successCount = 0;
            $failCount = 0;
            $messages = [];
            
            // Send to Android/Web with user_identity parameter
            $androidData = [
                'app_key' => $this->androidAppKey,
                'api_key' => $this->androidApiKey,
                'notification_title' => $title,
                'notification_body' => $body,
                'user_identity' => (string)$userId  // Appilix user_identity parameter
            ];
            
            $androidResult = $this->makeApiCall($androidData, 'Android/Web');
            $results['android'] = $androidResult;
            
            if ($androidResult['success']) {
                $successCount++;
                $messages[] = '✓ Android/Web';
            } else {
                $failCount++;
                $messages[] = '✗ Android/Web: ' . $androidResult['message'];
            }
            
            // Send to iOS with user_identity parameter
            $iosData = [
                'app_key' => $this->iosAppKey,
                'api_key' => $this->iosApiKey,
                'notification_title' => $title,
                'notification_body' => $body,
                'user_identity' => (string)$userId  // Appilix user_identity parameter
            ];
            
            $iosResult = $this->makeApiCall($iosData, 'iOS');
            $results['ios'] = $iosResult;
            
            if ($iosResult['success']) {
                $successCount++;
                $messages[] = '✓ iOS';
            } else {
                $failCount++;
                $messages[] = '✗ iOS: ' . $iosResult['message'];
            }
            
            // Determine overall success
            $overallSuccess = $successCount > 0;
            $overallMessage = "Sent to $successCount platform(s). " . implode(', ', $messages);
            
            // Log the notification
            $this->logNotification($title, $body, $userId, $adminName, $overallSuccess, $overallMessage);
            
            return [
                'success' => $overallSuccess,
                'message' => $overallMessage,
                'details' => $results,
                'stats' => [
                    'successful' => $successCount,
                    'failed' => $failCount
                ]
            ];
        } catch (Exception $e) {
            error_log("Firebase notification error: " . $e->getMessage());
            return ['success' => false, 'message' => 'Failed to send notification: ' . $e->getMessage()];
        }
    }
    
    /**
     * Make API call to Appilix
     */
    private function makeApiCall($data, $platform = 'Unknown') {
        $ch = curl_init();
        
        curl_setopt($ch, CURLOPT_URL, $this->apiUrl);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/x-www-form-urlencoded'
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10); // 10 seconds timeout
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5); // 5 seconds to establish connection
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        
        curl_close($ch);
        
        // Log the API call for debugging
        error_log("[$platform] Push Notification API Call - HTTP Code: $httpCode");
        
        if ($error) {
            error_log("[$platform] cURL Error: " . $error);
            return ['success' => false, 'message' => 'cURL Error: ' . $error];
        }
        
        if ($httpCode !== 200) {
            error_log("[$platform] HTTP Error: " . $httpCode);
            return ['success' => false, 'message' => 'HTTP Error: ' . $httpCode];
        }
        
        $decodedResponse = json_decode($response, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            error_log("[$platform] Invalid JSON response");
            return ['success' => false, 'message' => 'Invalid JSON response'];
        }
        
        error_log("[$platform] Notification sent successfully");
        
        return [
            'success' => $httpCode === 200,
            'message' => $decodedResponse['message'] ?? 'Notification sent successfully',
            'response' => $decodedResponse,
            'platform' => $platform
        ];
    }
    
    /**
     * Log notification to database
     */
    private function logNotification($title, $body, $target, $adminName, $success, $message) {
        try {
            $query = "INSERT INTO notification_logs (title, body, target, admin_name, success, message, created_at) 
                      VALUES (:title, :body, :target, :admin_name, :success, :message, NOW())";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindValue(':title', $title);
            $stmt->bindValue(':body', $body);
            $stmt->bindValue(':target', $target);
            $stmt->bindValue(':admin_name', $adminName);
            $stmt->bindValue(':success', $success ? 1 : 0);
            $stmt->bindValue(':message', $message);
            
            return $stmt->execute();
        } catch (Exception $e) {
            error_log("Failed to log notification: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get notification history
     */
    public function getNotificationHistory($limit = 50, $offset = 0) {
        try {
            $query = "SELECT * FROM notification_logs 
                      ORDER BY created_at DESC 
                      LIMIT :limit OFFSET :offset";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
            $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
            $stmt->execute();
            
            return $stmt->fetchAll();
        } catch (Exception $e) {
            error_log("Failed to get notification history: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get total notification count
     */
    public function getTotalNotificationCount() {
        try {
            $query = "SELECT COUNT(*) as total FROM notification_logs";
            $stmt = $this->conn->query($query);
            $result = $stmt->fetch();
            return $result['total'];
        } catch (Exception $e) {
            error_log("Failed to get notification count: " . $e->getMessage());
            return 0;
        }
    }
    
    /**
     * Get notification statistics
     */
    public function getNotificationStats() {
        try {
            $stats = [];
            
            // Total notifications
            $stmt = $this->conn->query("SELECT COUNT(*) as total FROM notification_logs");
            $stats['total'] = $stmt->fetch()['total'];
            
            // Successful notifications
            $stmt = $this->conn->query("SELECT COUNT(*) as total FROM notification_logs WHERE success = 1");
            $stats['successful'] = $stmt->fetch()['total'];
            
            // Failed notifications
            $stmt = $this->conn->query("SELECT COUNT(*) as total FROM notification_logs WHERE success = 0");
            $stats['failed'] = $stmt->fetch()['total'];
            
            // Notifications today
            $stmt = $this->conn->query("SELECT COUNT(*) as total FROM notification_logs WHERE DATE(created_at) = CURDATE()");
            $stats['today'] = $stmt->fetch()['total'];
            
            // Notifications this week
            $stmt = $this->conn->query("SELECT COUNT(*) as total FROM notification_logs WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)");
            $stats['this_week'] = $stmt->fetch()['total'];
            
            return $stats;
        } catch (Exception $e) {
            error_log("Failed to get notification stats: " . $e->getMessage());
            return [
                'total' => 0,
                'successful' => 0,
                'failed' => 0,
                'today' => 0,
                'this_week' => 0
            ];
        }
    }
    
    /**
     * Get all users for targeting
     */
    public function getAllUsers() {
        try {
            $query = "SELECT id, first_name, last_name, email FROM users WHERE is_guest = 0 ORDER BY first_name, last_name";
            $stmt = $this->conn->query($query);
            return $stmt->fetchAll();
        } catch (Exception $e) {
            error_log("Failed to get users: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Test notification API connection (Both platforms)
     */
    public function testConnection() {
        try {
            $results = [];
            $successCount = 0;
            $failCount = 0;
            $messages = [];
            
            // Test Android/Web connection
            $androidData = [
                'app_key' => $this->androidAppKey,
                'api_key' => $this->androidApiKey,
                'notification_title' => 'Test Notification',
                'notification_body' => 'This is a test notification to verify Android/Web API connection.'
            ];
            
            $androidResult = $this->makeApiCall($androidData, 'Android/Web');
            $results['android'] = $androidResult;
            
            if ($androidResult['success']) {
                $successCount++;
                $messages[] = '✓ Android/Web Connected';
            } else {
                $failCount++;
                $messages[] = '✗ Android/Web: ' . $androidResult['message'];
            }
            
            // Test iOS connection
            $iosData = [
                'app_key' => $this->iosAppKey,
                'api_key' => $this->iosApiKey,
                'notification_title' => 'Test Notification',
                'notification_body' => 'This is a test notification to verify iOS API connection.'
            ];
            
            $iosResult = $this->makeApiCall($iosData, 'iOS');
            $results['ios'] = $iosResult;
            
            if ($iosResult['success']) {
                $successCount++;
                $messages[] = '✓ iOS Connected';
            } else {
                $failCount++;
                $messages[] = '✗ iOS: ' . $iosResult['message'];
            }
            
            // Determine overall success
            $overallSuccess = $successCount > 0;
            $overallMessage = "Connection Test: $successCount/$failCount platforms. " . implode(', ', $messages);
            
            return [
                'success' => $overallSuccess,
                'message' => $overallMessage,
                'details' => $results,
                'stats' => [
                    'successful' => $successCount,
                    'failed' => $failCount,
                    'total' => 2
                ]
            ];
        } catch (Exception $e) {
            return ['success' => false, 'message' => 'Connection test failed: ' . $e->getMessage()];
        }
    }
}
?>