<?php
require_once __DIR__ . '/../config/database.php';

class Loyalty {
    private $conn;

    public function __construct() {
        $this->conn = getDB();
    }

    /**
     * Get user loyalty data
     */
    public function getUserLoyaltyData($userId) {
        $query = "SELECT ulp.*, u.first_name, u.last_name, u.email 
                  FROM user_loyalty_points ulp 
                  JOIN users u ON ulp.user_id = u.id 
                  WHERE ulp.user_id = :user_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':user_id', $userId);
        $stmt->execute();
        
        return $stmt->fetch();
    }

    /**
     * Get user digital stamps
     */
    public function getUserStamps($userId) {
        $query = "SELECT * FROM digital_stamps WHERE user_id = :user_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':user_id', $userId);
        $stmt->execute();
        
        return $stmt->fetch();
    }

    /**
     * Get user referral code
     */
    public function getUserReferralCode($userId) {
        $query = "SELECT * FROM referral_codes WHERE user_id = :user_id AND is_active = 1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':user_id', $userId);
        $stmt->execute();
        
        return $stmt->fetch();
    }

    /**
     * Get recent points transactions
     */
    public function getRecentTransactions($userId, $limit = 10) {
        $query = "SELECT * FROM points_transactions 
                  WHERE user_id = :user_id 
                  ORDER BY created_at DESC 
                  LIMIT :limit";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':user_id', $userId);
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll();
    }

    /**
     * Add points to user account
     */
    public function addPoints($userId, $points, $transactionType, $description, $orderId = null, $referenceId = null) {
        try {
            $this->conn->beginTransaction();

            // Add points to user account
            $updateQuery = "UPDATE user_loyalty_points 
                           SET points = points + :points, 
                               total_spent = total_spent + :points,
                               updated_at = CURRENT_TIMESTAMP
                           WHERE user_id = :user_id";
            
            $stmt = $this->conn->prepare($updateQuery);
            $stmt->bindValue(':points', $points);
            $stmt->bindValue(':user_id', $userId);
            $stmt->execute();

            // Record transaction
            $transactionQuery = "INSERT INTO points_transactions 
                                (user_id, points, transaction_type, description, order_id, reference_id) 
                                VALUES (:user_id, :points, :transaction_type, :description, :order_id, :reference_id)";
            
            $stmt = $this->conn->prepare($transactionQuery);
            $stmt->bindValue(':user_id', $userId);
            $stmt->bindValue(':points', $points);
            $stmt->bindValue(':transaction_type', $transactionType);
            $stmt->bindValue(':description', $description);
            $stmt->bindValue(':order_id', $orderId);
            $stmt->bindValue(':reference_id', $referenceId);
            $stmt->execute();

            // Update user tier based on total spent
            $this->updateUserTier($userId);

            $this->conn->commit();
            return true;
        } catch (Exception $e) {
            $this->conn->rollBack();
            return false;
        }
    }

    /**
     * Update user tier based on total spent
     */
    private function updateUserTier($userId) {
        $query = "SELECT total_spent FROM user_loyalty_points WHERE user_id = :user_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':user_id', $userId);
        $stmt->execute();
        $data = $stmt->fetch();

        if ($data) {
            $tier = 'bronze';
            if ($data['total_spent'] >= 2000) {
                $tier = 'gold';
            } elseif ($data['total_spent'] >= 1000) {
                $tier = 'silver';
            }

            $updateQuery = "UPDATE user_loyalty_points SET tier = :tier WHERE user_id = :user_id";
            $stmt = $this->conn->prepare($updateQuery);
            $stmt->bindValue(':tier', $tier);
            $stmt->bindValue(':user_id', $userId);
            $stmt->execute();
        }
    }

    /**
     * Add digital stamp
     */
    public function addStamp($userId) {
        $query = "INSERT INTO digital_stamps (user_id, stamps_collected, stamps_required) 
                  VALUES (:user_id, 1, 10) 
                  ON DUPLICATE KEY UPDATE 
                  stamps_collected = stamps_collected + 1,
                  is_completed = CASE WHEN stamps_collected + 1 >= stamps_required THEN 1 ELSE 0 END";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':user_id', $userId);
        return $stmt->execute();
    }

    /**
     * Create or get referral code
     */
    public function createReferralCode($userId) {
        // Check if user already has a referral code
        $existing = $this->getUserReferralCode($userId);
        if ($existing) {
            return $existing['referral_code'];
        }

        // Generate unique referral code
        $referralCode = strtoupper(substr($this->getUserInitials($userId), 0, 4) . date('Y'));
        
        // Ensure code is unique
        $counter = 1;
        $originalCode = $referralCode;
        while ($this->isReferralCodeExists($referralCode)) {
            $referralCode = $originalCode . $counter;
            $counter++;
        }

        $query = "INSERT INTO referral_codes (user_id, referral_code) VALUES (:user_id, :referral_code)";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':user_id', $userId);
        $stmt->bindValue(':referral_code', $referralCode);
        
        if ($stmt->execute()) {
            return $referralCode;
        }
        return false;
    }

    /**
     * Get user initials for referral code
     */
    private function getUserInitials($userId) {
        $query = "SELECT first_name, last_name FROM users WHERE id = :user_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':user_id', $userId);
        $stmt->execute();
        $user = $stmt->fetch();
        
        if ($user) {
            return substr($user['first_name'], 0, 1) . substr($user['last_name'], 0, 1);
        }
        return 'USER';
    }

    /**
     * Check if referral code exists
     */
    private function isReferralCodeExists($referralCode) {
        $query = "SELECT id FROM referral_codes WHERE referral_code = :referral_code";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':referral_code', $referralCode);
        $stmt->execute();
        return $stmt->fetch() !== false;
    }

    /**
     * Use referral code
     */
    public function useReferralCode($referralCode, $referredUserId) {
        try {
            $this->conn->beginTransaction();

            // Get referrer info
            $query = "SELECT user_id FROM referral_codes WHERE referral_code = :referral_code AND is_active = 1";
            $stmt = $this->conn->prepare($query);
            $stmt->bindValue(':referral_code', $referralCode);
            $stmt->execute();
            $referrer = $stmt->fetch();

            if (!$referrer) {
                throw new Exception('Invalid referral code');
            }

            $referrerId = $referrer['user_id'];

            // Check if user already used a referral code
            $checkQuery = "SELECT id FROM referral_usage WHERE referred_user_id = :referred_user_id";
            $stmt = $this->conn->prepare($checkQuery);
            $stmt->bindValue(':referred_user_id', $referredUserId);
            $stmt->execute();
            
            if ($stmt->fetch()) {
                throw new Exception('User already used a referral code');
            }

            // Award points to referrer
            $this->addPoints($referrerId, 3000, 'bonus', 'Referral bonus - ' . $referralCode);

            // Record referral usage
            $usageQuery = "INSERT INTO referral_usage (referrer_id, referred_user_id, referral_code, points_awarded) 
                          VALUES (:referrer_id, :referred_user_id, :referral_code, 3000)";
            $stmt = $this->conn->prepare($usageQuery);
            $stmt->bindValue(':referrer_id', $referrerId);
            $stmt->bindValue(':referred_user_id', $referredUserId);
            $stmt->bindValue(':referral_code', $referralCode);
            $stmt->execute();

            $this->conn->commit();
            return true;
        } catch (Exception $e) {
            $this->conn->rollBack();
            return false;
        }
    }

    /**
     * Initialize loyalty account for new user
     */
    public function initializeLoyaltyAccount($userId) {
        $query = "INSERT INTO user_loyalty_points (user_id, points, tier, total_spent) 
                  VALUES (:user_id, 0, 'bronze', 0.00)";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':user_id', $userId);
        return $stmt->execute();
    }

    /**
     * Get next reward points needed
     */
    public function getNextRewardPoints($userId) {
        $query = "SELECT points FROM user_loyalty_points WHERE user_id = :user_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':user_id', $userId);
        $stmt->execute();
        $data = $stmt->fetch();

        if ($data) {
            $currentPoints = $data['points'];
            $nextReward = 2000; // Next reward at 2000 points
            return max(0, $nextReward - $currentPoints);
        }
        return 2000;
    }

    /**
     * Calculate points based on price (RM1 = 1 point)
     * Rounds the price to the nearest integer for point calculation
     */
    public function calculatePointsFromPrice($price) {
        // Round the price to nearest integer (RM1 = 1 point)
        return (int) round($price);
    }

    /**
     * Get database connection
     */
    public function getConnection() {
        return $this->conn;
    }

    /**
     * Deduct points from user account
     */
    public function deductPoints($userId, $points, $description) {
        try {
            $this->conn->beginTransaction();

            // Check if user has enough points
            $checkQuery = "SELECT points FROM user_loyalty_points WHERE user_id = :user_id";
            $stmt = $this->conn->prepare($checkQuery);
            $stmt->bindValue(':user_id', $userId);
            $stmt->execute();
            $userData = $stmt->fetch();

            if (!$userData) {
                throw new Exception('User loyalty account not found');
            }

            if ($userData['points'] < $points) {
                throw new Exception('Insufficient points');
            }

            // Deduct points from user account
            $updateQuery = "UPDATE user_loyalty_points 
                           SET points = points - :points, 
                               updated_at = CURRENT_TIMESTAMP
                           WHERE user_id = :user_id";
            
            $stmt = $this->conn->prepare($updateQuery);
            $stmt->bindValue(':points', $points);
            $stmt->bindValue(':user_id', $userId);
            $stmt->execute();

            // Record transaction (negative points for deduction)
            $transactionQuery = "INSERT INTO points_transactions 
                                (user_id, points, transaction_type, description) 
                                VALUES (:user_id, :points, :transaction_type, :description)";
            
            $stmt = $this->conn->prepare($transactionQuery);
            $stmt->bindValue(':user_id', $userId);
            $stmt->bindValue(':points', -$points); // Negative points for deduction
            $stmt->bindValue(':transaction_type', 'redeemed');
            $stmt->bindValue(':description', $description);
            $stmt->execute();

            $this->conn->commit();
            return true;
        } catch (Exception $e) {
            $this->conn->rollBack();
            throw $e;
        }
    }
}
?>