<?php
require_once __DIR__ . '/../config/database.php';

class Product {
    private $conn;
    private $table_name = "products";

    public function __construct() {
        $this->conn = getDB();
    }

    /**
     * Get all products with optional filters including variations and sales
     */
    public function getAll($filters = []) {
        $query = "SELECT p.*, c.name as category_name, c.slug as category_slug,
                         COALESCE(SUM(vv.stock_quantity), p.stock_quantity) as total_stock,
                         MIN(vv.price) as min_price,
                         MAX(vv.price) as max_price,
                         GROUP_CONCAT(CONCAT(vv.version_type, ':', vv.size, ':', vv.price, ':', vv.stock_quantity) SEPARATOR '|') as variations,
                         p.sale_price, p.is_on_sale, p.is_flash_sale, p.flash_sale_started_at, p.flash_sale_duration_hours, p.flash_sale_variations
                  FROM " . $this->table_name . " p 
                  LEFT JOIN categories c ON p.category_id = c.id 
                  LEFT JOIN version_variations vv ON p.id = vv.product_id AND vv.is_active = 1
                  WHERE p.is_active = 1";
        
        $params = [];
        
        if (!empty($filters['category'])) {
            $query .= " AND c.slug = :category";
            $params['category'] = $filters['category'];
        }
        
        if (!empty($filters['search'])) {
            $query .= " AND (p.name LIKE :search OR p.brand LIKE :search OR p.description LIKE :search OR p.scent LIKE :search OR p.notes LIKE :search)";
            $params['search'] = '%' . $filters['search'] . '%';
        }
        
        if (!empty($filters['scent'])) {
            $query .= " AND p.scent = :scent";
            $params['scent'] = $filters['scent'];
        }
        
        if (!empty($filters['notes'])) {
            $query .= " AND p.notes LIKE :notes";
            $params['notes'] = '%' . $filters['notes'] . '%';
        }
        
        if (!empty($filters['season'])) {
            $query .= " AND p.season = :season";
            $params['season'] = $filters['season'];
        }
        
        if (!empty($filters['recommendation'])) {
            $query .= " AND p.recommendation = :recommendation";
            $params['recommendation'] = $filters['recommendation'];
        }
        
        if (!empty($filters['bestsellers'])) {
            $query .= " AND p.is_bestseller = 1";
        }
        
        if (!empty($filters['new_arrivals'])) {
            $query .= " AND p.is_new_arrival = 1";
        }
        
        
        $query .= " GROUP BY p.id HAVING MIN(vv.price) IS NOT NULL ORDER BY p.created_at DESC";
        
        if (!empty($filters['limit'])) {
            $query .= " LIMIT :limit";
            $params['limit'] = (int)$filters['limit'];
        }
        
        $stmt = $this->conn->prepare($query);
        
        foreach ($params as $key => $value) {
            if ($key === 'limit') {
                $stmt->bindValue(':' . $key, $value, PDO::PARAM_INT);
            } else {
                $stmt->bindValue(':' . $key, $value);
            }
        }
        
        $stmt->execute();
        $products = $stmt->fetchAll();
        
        // Process variations and add sale/flash sale status
        foreach ($products as &$product) {
            $product['variations_detail'] = [];
            // Use database is_out_of_stock field if set, otherwise check stock quantity
            $product['is_out_of_stock'] = $product['is_out_of_stock'] || ($product['total_stock'] <= 0);
            
            // Check if flash sale is active and not expired
            $product['sale_active'] = false;
            $product['is_flash_sale_active'] = false;
            
            if ($product['is_flash_sale'] && $product['flash_sale_started_at']) {
                $hoursElapsed = (time() - strtotime($product['flash_sale_started_at'])) / 3600;
                if ($hoursElapsed < $product['flash_sale_duration_hours']) {
                    $product['is_flash_sale_active'] = true;
                    $product['sale_active'] = true;
                }
            } elseif ($product['is_on_sale'] && $product['sale_price']) {
                $product['sale_active'] = true;
            }
            
            // Calculate display price (use sale price if active, otherwise min price)
            if ($product['sale_active'] && $product['sale_price']) {
                $product['display_price'] = $product['sale_price'];
            } else {
                $product['display_price'] = $product['min_price'] ?? 0;
            }
            
            // Handle flash sale variations
            if ($product['is_flash_sale_active'] && !empty($product['flash_sale_variations'])) {
                $flashSaleVariations = json_decode($product['flash_sale_variations'], true);
                if ($flashSaleVariations) {
                    $product['flash_sale_variations_detail'] = $flashSaleVariations;
                }
            }
            
            // Handle null prices for products without variations
            if ($product['min_price'] === null || $product['max_price'] === null) {
                $product['price_range'] = 'Price not set';
            } else {
                // Check display_price_mode setting
                $displayMode = $product['display_price_mode'] ?? 'range';
                
                if ($displayMode === 'single') {
                    // Always show single price (minimum price or sale price)
                    if ($product['sale_active'] && $product['sale_price']) {
                        $product['price_range'] = 'RM ' . number_format($product['sale_price'], 2);
                    } else {
                        $product['price_range'] = 'RM ' . number_format($product['min_price'], 2);
                    }
                } else {
                    // Show range price if min and max are different, otherwise single price
                    if ($product['sale_active'] && $product['sale_price']) {
                        // For sale items, show sale price as single price
                        $product['price_range'] = 'RM ' . number_format($product['sale_price'], 2);
                    } else {
                        $product['price_range'] = ($product['min_price'] != $product['max_price']) ? 
                            'RM ' . number_format($product['min_price'], 2) . ' - RM ' . number_format($product['max_price'], 2) : 
                            'RM ' . number_format($product['min_price'], 2);
                    }
                }
            }
            
            
            if (!empty($product['variations'])) {
                $variations = explode('|', $product['variations']);
                foreach ($variations as $variation) {
                    if (!empty($variation)) {
                        list($version_type, $size, $price, $stock) = explode(':', $variation);
                        $product['variations_detail'][] = [
                            'version_type' => $version_type,
                            'size' => $size,
                            'price' => (float)$price,
                            'stock' => (int)$stock
                        ];
                    }
                }
            }
        }
        
        return $products;
    }

    /**
     * Get product by ID
     */
    public function getById($id) {
        $query = "SELECT p.*, c.name as category_name, c.slug as category_slug,
                         COALESCE(SUM(vv.stock_quantity), p.stock_quantity) as total_stock,
                         MIN(vv.price) as min_price,
                         MAX(vv.price) as max_price,
                         GROUP_CONCAT(CONCAT(vv.version_type, ':', vv.size, ':', vv.price, ':', vv.stock_quantity) SEPARATOR '|') as variations,
                         p.sale_price, p.is_on_sale, p.is_flash_sale, p.flash_sale_started_at, p.flash_sale_duration_hours, p.flash_sale_variations
                  FROM " . $this->table_name . " p 
                  LEFT JOIN categories c ON p.category_id = c.id 
                  LEFT JOIN version_variations vv ON p.id = vv.product_id AND vv.is_active = 1
                  WHERE p.id = :id AND p.is_active = 1
                  GROUP BY p.id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        
        $product = $stmt->fetch();
        
        if ($product) {
            // Process variations and add sale/flash sale status
            $product['variations_detail'] = [];
            // Use database is_out_of_stock field if set, otherwise check stock quantity
            $product['is_out_of_stock'] = $product['is_out_of_stock'] || ($product['total_stock'] <= 0);
            
            // Check if flash sale is active and not expired
            $product['sale_active'] = false;
            $product['is_flash_sale_active'] = false;
            
            if ($product['is_flash_sale'] && $product['flash_sale_started_at']) {
                $hoursElapsed = (time() - strtotime($product['flash_sale_started_at'])) / 3600;
                if ($hoursElapsed < $product['flash_sale_duration_hours']) {
                    $product['is_flash_sale_active'] = true;
                    $product['sale_active'] = true;
                }
            } elseif ($product['is_on_sale'] && $product['sale_price']) {
                $product['sale_active'] = true;
            }
            
            // Calculate display price (use sale price if active, otherwise min price)
            if ($product['sale_active'] && $product['sale_price']) {
                $product['display_price'] = $product['sale_price'];
            } else {
                $product['display_price'] = $product['min_price'] ?? 0;
            }
            
            // Handle flash sale variations
            if ($product['is_flash_sale_active'] && !empty($product['flash_sale_variations'])) {
                $flashSaleVariations = json_decode($product['flash_sale_variations'], true);
                if ($flashSaleVariations) {
                    $product['flash_sale_variations_detail'] = $flashSaleVariations;
                }
            }
            
            // Handle null prices for products without variations
            if ($product['min_price'] === null || $product['max_price'] === null) {
                $product['price_range'] = 'Price not set';
            } else {
                // Check display_price_mode setting
                $displayMode = $product['display_price_mode'] ?? 'range';
                
                if ($displayMode === 'single') {
                    // Always show single price (minimum price or sale price)
                    if ($product['sale_active'] && $product['sale_price']) {
                        $product['price_range'] = 'RM ' . number_format($product['sale_price'], 2);
                    } else {
                        $product['price_range'] = 'RM ' . number_format($product['min_price'], 2);
                    }
                } else {
                    // Show range price if min and max are different, otherwise single price
                    if ($product['sale_active'] && $product['sale_price']) {
                        // For sale items, show sale price as single price
                        $product['price_range'] = 'RM ' . number_format($product['sale_price'], 2);
                    } else {
                        $product['price_range'] = ($product['min_price'] != $product['max_price']) ? 
                            'RM ' . number_format($product['min_price'], 2) . ' - RM ' . number_format($product['max_price'], 2) : 
                            'RM ' . number_format($product['min_price'], 2);
                    }
                }
            }
            
            
            if (!empty($product['variations'])) {
                $variations = explode('|', $product['variations']);
                foreach ($variations as $variation) {
                    if (!empty($variation)) {
                        list($version_type, $size, $price, $stock) = explode(':', $variation);
                        $product['variations_detail'][] = [
                            'version_type' => $version_type,
                            'size' => $size,
                            'price' => (float)$price,
                            'stock' => (int)$stock
                        ];
                    }
                }
            }
        }
        
        return $product;
    }

    /**
     * Get product version variations
     */
    public function getVersionVariations($product_id) {
        $query = "SELECT * FROM version_variations WHERE product_id = :product_id AND is_active = 1 ORDER BY version_type, size";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':product_id', $product_id, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll();
    }

    /**
     * Get best sellers
     */
    public function getBestSellers($limit = 4) {
        return $this->getAll(['bestsellers' => true, 'limit' => $limit]);
    }

    /**
     * Get new arrivals
     */
    public function getNewArrivals($limit = 4) {
        return $this->getAll(['new_arrivals' => true, 'limit' => $limit]);
    }

    /**
     * Get products by category
     */
    public function getByCategory($category_slug, $limit = null) {
        $filters = ['category' => $category_slug];
        if ($limit) {
            $filters['limit'] = $limit;
        }
        return $this->getAll($filters);
    }

    /**
     * Search products
     */
    public function search($search_term, $limit = null) {
        $filters = ['search' => $search_term];
        if ($limit) {
            $filters['limit'] = $limit;
        }
        return $this->getAll($filters);
    }

    /**
     * Update product rating
     */
    public function updateRating($product_id, $rating, $review_count) {
        $query = "UPDATE " . $this->table_name . " 
                  SET rating = :rating, review_count = :review_count 
                  WHERE id = :product_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':rating', $rating);
        $stmt->bindValue(':review_count', $review_count, PDO::PARAM_INT);
        $stmt->bindValue(':product_id', $product_id, PDO::PARAM_INT);
        
        return $stmt->execute();
    }

    /**
     * Update stock quantity
     */
    public function updateStock($product_id, $quantity) {
        $query = "UPDATE " . $this->table_name . " 
                  SET stock_quantity = :quantity 
                  WHERE id = :product_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':quantity', $quantity, PDO::PARAM_INT);
        $stmt->bindValue(':product_id', $product_id, PDO::PARAM_INT);
        
        return $stmt->execute();
    }

    /**
     * Deduct stock for a specific product variation
     */
    public function deductVariationStock($product_id, $size, $version, $quantity) {
        try {
            $this->conn->beginTransaction();
            
            // First, check if the variation exists and has enough stock
            $check_query = "SELECT stock_quantity FROM version_variations 
                           WHERE product_id = :product_id AND size = :size AND version_type = :version";
            
            $check_stmt = $this->conn->prepare($check_query);
            $check_stmt->bindValue(':product_id', $product_id, PDO::PARAM_INT);
            $check_stmt->bindValue(':size', $size);
            $check_stmt->bindValue(':version', $version);
            $check_stmt->execute();
            
            $variation = $check_stmt->fetch();
            
            if (!$variation) {
                error_log("Product::deductVariationStock - Variation not found: product_id=$product_id, size=$size, version=$version");
                $this->conn->rollBack();
                return false;
            }
            
            $current_stock = (int)$variation['stock_quantity'];
            $new_stock = $current_stock - $quantity;
            
            if ($new_stock < 0) {
                error_log("Product::deductVariationStock - Insufficient stock: current=$current_stock, requested=$quantity");
                $this->conn->rollBack();
                return false;
            }
            
            // Update the variation stock
            $update_query = "UPDATE version_variations 
                            SET stock_quantity = :new_stock 
                            WHERE product_id = :product_id AND size = :size AND version_type = :version";
            
            $update_stmt = $this->conn->prepare($update_query);
            $update_stmt->bindValue(':new_stock', $new_stock, PDO::PARAM_INT);
            $update_stmt->bindValue(':product_id', $product_id, PDO::PARAM_INT);
            $update_stmt->bindValue(':size', $size);
            $update_stmt->bindValue(':version', $version);
            
            if (!$update_stmt->execute()) {
                error_log("Product::deductVariationStock - Failed to update variation stock");
                $this->conn->rollBack();
                return false;
            }
            
            $this->conn->commit();
            error_log("Product::deductVariationStock - Successfully deducted $quantity from product_id=$product_id, size=$size, version=$version. New stock: $new_stock");
            return true;
            
        } catch (Exception $e) {
            $this->conn->rollBack();
            error_log("Product::deductVariationStock - Exception: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Deduct stock for multiple order items
     */
    public function deductOrderStock($order_items) {
        $results = [];
        
        foreach ($order_items as $item) {
            $product_id = $item['product_id'];
            $size = $item['size'] ?: '';
            $version = $item['version'] ?: '';
            $quantity = (int)$item['quantity'];
            
            $result = $this->deductVariationStock($product_id, $size, $version, $quantity);
            $results[] = [
                'product_id' => $product_id,
                'size' => $size,
                'version' => $version,
                'quantity' => $quantity,
                'success' => $result
            ];
        }
        
        return $results;
    }

    /**
     * Get all products for catalog display with variations and sales
     */
    public function getAllProducts() {
        return $this->getAll(['limit' => null]);
    }

    /**
     * Get filter options from database
     */
    public function getFilterOptions() {
        $query = "SELECT field_type, field_name, preset_value FROM product_field_presets WHERE is_active = 1 ORDER BY field_type, sort_order";
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        $results = $stmt->fetchAll();
        
        $filters = [];
        foreach ($results as $result) {
            $filters[$result['field_type']][] = $result['preset_value'];
        }
        
        return $filters;
    }

    /**
     * Filter products based on multiple criteria
     */
    public function filterProducts($filters) {
        return $this->getAll($filters);
    }
}
?>