<?php
/**
 * toyyibPay Payment Callback Handler
 * This file receives payment status updates from toyyibPay
 */

require_once 'config/database.php';
require_once 'models/Order.php';
require_once 'models/ToyyibPay.php';
require_once 'models/Loyalty.php';

// Log callback data for debugging
error_log("toyyibPay Callback received: " . print_r($_POST, true));

// Verify that this is a POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo "Method not allowed";
    exit;
}

// Get callback data
$refno = $_POST['refno'] ?? '';
$status = $_POST['status'] ?? '';
$reason = $_POST['reason'] ?? '';
$billcode = $_POST['billcode'] ?? '';
$order_id = $_POST['order_id'] ?? '';
$amount = $_POST['amount'] ?? '';
$transaction_time = $_POST['transaction_time'] ?? '';

// Validate required fields
if (empty($billcode) || empty($status)) {
    error_log("toyyibPay Callback: Missing required fields");
    http_response_code(400);
    echo "Missing required fields";
    exit;
}

try {
    $order = new Order();
    $toyyibpay = new ToyyibPay();
    
    // Find order by external reference number (order_id)
    if (!empty($order_id)) {
        $order_data = $order->getById($order_id);
        
        if (!$order_data) {
            error_log("toyyibPay Callback: Order not found - ID: " . $order_id);
            http_response_code(404);
            echo "Order not found";
            exit;
        }
        
        // Update order status and payment status based on payment status
        $new_status = '';
        $new_payment_status = '';
        
        switch ($status) {
            case '1': // Success
                $new_status = 'processing'; // Payment successful, ready for processing
                $new_payment_status = 'paid';
                break;
            case '2': // Pending
            case '4': // Pending
                $new_status = 'awaiting_payment'; // Still waiting for payment
                $new_payment_status = 'pending';
                break;
            case '3': // Failed
                $new_status = 'awaiting_payment'; // Keep as awaiting payment for retry
                $new_payment_status = 'failed';
                break;
            default:
                $new_status = 'awaiting_payment';
                $new_payment_status = 'pending';
        }
        
        // Update order status
        $update_result = $order->updateStatus($order_id, $new_status);
        
        // Update payment status
        $payment_update_result = $order->updatePaymentInfo($order_id, [
            'status' => $new_payment_status,
            'reference' => $refno,
            'amount' => $amount
        ]);
        
        if ($update_result && $payment_update_result) {
            // Log successful update
            error_log("toyyibPay Callback: Order {$order_id} status updated to {$new_status}, payment status updated to {$new_payment_status}");
            
            // If payment is successful, you might want to:
            // 1. Send confirmation email
            // 2. Update inventory
            // 3. Generate invoice
            // 4. Notify admin
            
            if ($status === '1') {
                // Payment successful - additional processing
                error_log("toyyibPay Callback: Payment successful for order {$order_id}, amount: {$amount}, reference: {$refno}");
                
                // Deduct stock for order items
                try {
                    require_once 'models/Product.php';
                    $product = new Product();
                    
                    $order_items = $order->getOrderItemsForStock($order_id);
                    if (!empty($order_items)) {
                        $stock_results = $product->deductOrderStock($order_items);
                        
                        $success_count = 0;
                        $fail_count = 0;
                        
                        foreach ($stock_results as $result) {
                            if ($result['success']) {
                                $success_count++;
                            } else {
                                $fail_count++;
                                error_log("Stock deduction failed for product_id={$result['product_id']}, size={$result['size']}, version={$result['version']}, quantity={$result['quantity']}");
                            }
                        }
                        
                        error_log("Stock deduction completed for order {$order_id}: {$success_count} successful, {$fail_count} failed");
                        
                        if ($fail_count > 0) {
                            // If any stock deductions failed, you might want to handle this
                            // For now, we'll just log it
                            error_log("Warning: Some stock deductions failed for order {$order_id}");
                        }
                    } else {
                        error_log("No order items found for order {$order_id}");
                    }
                } catch (Exception $e) {
                    error_log("Error deducting stock for order {$order_id}: " . $e->getMessage());
                }
                
                // Award loyalty points for purchased products
                try {
                    $loyalty = new Loyalty();
                    
                    // Get order items with product details for point calculation
                    $order_items = $order->getOrderItemsForPoints($order_id);
                    if (!empty($order_items)) {
                        $total_points = 0;
                        $points_awarded = [];
                        
                        foreach ($order_items as $item) {
                            // Calculate points based on price (RM1 = 1 point) instead of manual point_rewards
                            $item_price = $item['product_price'] ?? 0;
                            if ($item_price > 0) {
                                $points_per_item = $loyalty->calculatePointsFromPrice($item_price);
                                $total_points += $points_per_item * $item['quantity'];
                                $points_awarded[] = [
                                    'product_name' => $item['product_name'],
                                    'quantity' => $item['quantity'],
                                    'points_per_item' => $points_per_item,
                                    'total_points' => $points_per_item * $item['quantity']
                                ];
                            }
                        }
                        
                        if ($total_points > 0) {
                            // Award points to user
                            $user_id = $order_data['user_id'];
                            $description = "Purchase reward for Order #{$order_id}";
                            
                            $points_result = $loyalty->addPoints($user_id, $total_points, 'earned', $description, $order_id, 'ORDER' . $order_id);
                            
                            if ($points_result) {
                                error_log("Loyalty points awarded for order {$order_id}: {$total_points} points to user {$user_id}");
                                
                                // Log detailed points breakdown
                                foreach ($points_awarded as $award) {
                                    error_log("Points breakdown: {$award['product_name']} x{$award['quantity']} = {$award['total_points']} points ({$award['points_per_item']} per item)");
                                }
                            } else {
                                error_log("Failed to award loyalty points for order {$order_id}");
                            }
                        } else {
                            error_log("No loyalty points to award for order {$order_id} (no products with point rewards)");
                        }
                    } else {
                        error_log("No order items found for point calculation for order {$order_id}");
                    }
                } catch (Exception $e) {
                    error_log("Error awarding loyalty points for order {$order_id}: " . $e->getMessage());
                }
            }
            
        } else {
            error_log("toyyibPay Callback: Failed to update order status for ID: " . $order_id . " (order: " . ($update_result ? 'success' : 'failed') . ", payment: " . ($payment_update_result ? 'success' : 'failed') . ")");
        }
    } else {
        error_log("toyyibPay Callback: No order ID provided");
    }
    
    // Always return success to toyyibPay
    http_response_code(200);
    echo "OK";
    
} catch (Exception $e) {
    error_log("toyyibPay Callback Error: " . $e->getMessage());
    http_response_code(500);
    echo "Internal server error";
}

/**
 * Handle successful payment
 */
function handleSuccessfulPayment($order_data, $amount, $refno) {
    try {
        // Log successful payment
        error_log("Payment successful for Order ID: {$order_data['id']}, Amount: {$amount}, Reference: {$refno}");
        
        // You can add additional processing here:
        // - Send confirmation email
        // - Update inventory
        // - Generate invoice
        // - Notify admin
        
        // Example: Send email notification (implement your email function)
        // sendOrderConfirmationEmail($order_data);
        
    } catch (Exception $e) {
        error_log("Error in handleSuccessfulPayment: " . $e->getMessage());
    }
}
?>