<?php
session_start();
require_once 'models/User.php';
require_once 'models/Order.php';
require_once 'models/ToyyibPay.php';
require_once 'config/payment.php';

// Check if user is logged in
if (!isset($_SESSION['user_id']) || $_SESSION['user_id'] === 'guest') {
    header('Location: login.php');
    exit();
}

// Get user data
$user = new User();
$user_data = $user->getById($_SESSION['user_id']);

if (!$user_data) {
    header('Location: login.php');
    exit();
}

// Get awaiting payment orders
$order = new Order();
$awaiting_orders = $order->getByPaymentStatus($_SESSION['user_id'], 'awaiting_payment');

// Handle continue payment
if (isset($_POST['continue_payment']) && isset($_POST['order_id'])) {
    $order_id = (int)$_POST['order_id'];
    
    // Get the order details
    $order_details = $order->getById($order_id);
    
    if ($order_details && $order_details['user_id'] == $_SESSION['user_id'] && $order_details['status'] == 'awaiting_payment') {
        // Check if bill_code exists
        if (!empty($order_details['bill_code'])) {
            // Redirect to existing payment URL
            $toyyibpay = new ToyyibPay();
            $payment_url = "https://dev.toyyibpay.com/" . $order_details['bill_code'];
            header("Location: " . $payment_url);
            exit();
        } else {
            // Create new payment bill
            $toyyibpay = new ToyyibPay();
            
            // Check ToyyibPay configuration
            if (empty(TOYYIBPAY_CATEGORY_CODE)) {
                $error_message = 'Payment system not configured. Please contact support.';
                error_log("toyyibPay Error: Category code not set in pending-payment.php");
            } else {
            
            $bill_data = [
                'bill_name' => 'Order #' . $order_details['order_number'],
                'bill_description' => 'Payment for order #' . $order_details['order_number'],
                'amount' => $order_details['total_amount'],
                'order_id' => $order_id, // Use database ID, not order number
                'customer_name' => $user_data['first_name'] . ' ' . $user_data['last_name'],
                'customer_email' => $user_data['email'],
                'customer_phone' => $user_data['phone'] ?? '',
                'payment_method' => $order_details['payment_method'] ?? 'toyyibpay_fpx'
            ];
            
            $payment_result = $toyyibpay->createBill($bill_data);
            
            // Log the payment result for debugging
            error_log("pending-payment createBill result: " . print_r($payment_result, true));
            
            if ($payment_result && isset($payment_result['bill_code'])) {
                // Update order with bill code
                $order->updatePaymentInfo($order_id, [
                    'bill_code' => $payment_result['bill_code']
                ]);
                
                // Redirect to payment
                header("Location: " . $payment_result['payment_url']);
                exit();
            } else {
                $error_message = "Failed to create payment. Please try again.";
                error_log("pending-payment: Payment creation failed - result: " . print_r($payment_result, true));
            }
            }
        }
    } else {
        $error_message = "Order not found or already processed.";
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Pending Payment - 2 Days Parfume</title>
    <link rel="stylesheet" href="styles.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .pending-payment-container {
            min-height: 100vh;
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            padding-bottom: 100px;
        }

        .top-nav {
            background: linear-gradient(135deg, #1A4D2A 0%, #2d5a3d 50%, #1A4D2A 100%);
            padding: 15px 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            position: relative;
        }

        .nav-left, .nav-right {
            width: 40px;
            height: 40px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: #D4AF37;
            font-size: 18px;
            cursor: pointer;
        }

        .nav-center {
            flex: 1;
            display: flex;
            justify-content: center;
            align-items: center;
        }

        .logo {
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .logo-img {
            height: 30px;
            width: auto;
        }

        .page-header {
            background: linear-gradient(135deg, #1A4D2A 0%, #2d5a3d 50%, #1A4D2A 100%);
            padding: 20px;
            text-align: center;
            color: white;
            position: relative;
        }

        .page-title {
            font-size: 24px;
            font-weight: bold;
            margin-bottom: 5px;
        }

        .page-subtitle {
            font-size: 14px;
            opacity: 0.9;
        }

        .content-section {
            padding: 20px;
        }

        .no-orders {
            text-align: center;
            padding: 40px 20px;
            background: white;
            border-radius: 15px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            margin-bottom: 20px;
        }

        .no-orders-icon {
            font-size: 48px;
            color: #D4AF37;
            margin-bottom: 15px;
        }

        .no-orders-title {
            font-size: 18px;
            color: #1A4D2A;
            margin-bottom: 10px;
            font-weight: 600;
        }

        .no-orders-text {
            color: #666;
            font-size: 14px;
            margin-bottom: 20px;
        }

        .shop-btn {
            background: linear-gradient(135deg, #D4AF37, #F4D03F);
            color: #1A4D2A;
            border: none;
            padding: 12px 30px;
            border-radius: 25px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .shop-btn:hover {
            background: linear-gradient(135deg, #1A4D2A, #2d5a3d);
            color: #D4AF37;
            transform: translateY(-2px);
        }

        .order-card {
            background: white;
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 15px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            border-left: 4px solid #D4AF37;
        }

        .order-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
        }

        .order-number {
            font-size: 16px;
            font-weight: 600;
            color: #1A4D2A;
        }

        .order-date {
            font-size: 12px;
            color: #666;
        }

        .order-amount {
            font-size: 18px;
            font-weight: bold;
            color: #D4AF37;
            margin-bottom: 10px;
        }

        .order-status {
            display: inline-block;
            background: #ffeaa7;
            color: #d63031;
            padding: 4px 12px;
            border-radius: 15px;
            font-size: 12px;
            font-weight: 600;
            margin-bottom: 15px;
        }

        .order-actions {
            display: flex;
            gap: 10px;
        }

        .continue-payment-btn {
            background: linear-gradient(135deg, #D4AF37, #F4D03F);
            color: #1A4D2A;
            border: none;
            padding: 10px 20px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            flex: 1;
        }

        .continue-payment-btn:hover {
            background: linear-gradient(135deg, #1A4D2A, #2d5a3d);
            color: #D4AF37;
            transform: translateY(-2px);
        }

        .cancel-order-btn {
            background: #e74c3c;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .cancel-order-btn:hover {
            background: #c0392b;
            transform: translateY(-2px);
        }

        .error-message {
            background: #e74c3c;
            color: white;
            padding: 15px;
            border-radius: 10px;
            margin-bottom: 20px;
            text-align: center;
        }

        .success-message {
            background: #27ae60;
            color: white;
            padding: 15px;
            border-radius: 10px;
            margin-bottom: 20px;
            text-align: center;
        }

        .bottom-nav {
            position: fixed;
            bottom: 0;
            left: 50%;
            transform: translateX(-50%);
            width: 100%;
            max-width: 375px;
            background: linear-gradient(135deg, #1A4D2A, #2d5a3d);
            display: flex;
            justify-content: space-around;
            align-items: center;
            padding: 5px 20px 5px 20px;
            border-top: 3px solid #D4AF37;
            border-radius: 20px 20px 0 0;
            box-shadow: 0 -4px 15px rgba(26, 77, 42, 0.3);
            z-index: 10;
        }

        .nav-item {
            display: flex;
            flex-direction: column;
            align-items: center;
            padding: 8px;
            color: #D4AF37;
            cursor: pointer;
            transition: all 0.3s ease;
            border-radius: 10px;
            min-width: 50px;
        }

        .nav-item:hover {
            background: rgba(212, 175, 55, 0.1);
            transform: translateY(-2px);
        }

        .nav-item.active {
            background: #D4AF37;
            color: #1A4D2A;
        }

        .nav-item i {
            font-size: 18px;
            margin-bottom: 2px;
        }
    </style>
</head>
<body>
    <div class="pending-payment-container">
        <!-- Top Navigation -->
        <nav class="top-nav">
            <div class="nav-left">
                <i class="fas fa-arrow-left" onclick="goBack()"></i>
            </div>
            <div class="nav-center">
                <div class="logo">
                    <img src="images/LOGO.png" alt="2 Days Parfume Logo" class="logo-img">
                </div>
            </div>
            <div class="nav-right">
                <i class="fas fa-search" onclick="window.location.href='products.php'"></i>
            </div>
        </nav>

        <!-- Page Header -->
        <div class="page-header">
            <div class="page-title">Pending Payment</div>
            <div class="page-subtitle">Complete your payment to confirm your order</div>
        </div>

        <!-- Content Section -->
        <div class="content-section">
            <?php if (isset($error_message)): ?>
                <div class="error-message">
                    <i class="fas fa-exclamation-triangle"></i> <?php echo htmlspecialchars($error_message); ?>
                </div>
            <?php endif; ?>

            <?php if (empty($awaiting_orders)): ?>
                <div class="no-orders">
                    <div class="no-orders-icon">
                        <i class="fas fa-check-circle"></i>
                    </div>
                    <div class="no-orders-title">No Pending Payments</div>
                    <div class="no-orders-text">You have no orders awaiting payment. All your orders are up to date!</div>
                    <button class="shop-btn" onclick="window.location.href='products.php'">
                        <i class="fas fa-shopping-bag"></i> Continue Shopping
                    </button>
                </div>
            <?php else: ?>
                <?php foreach ($awaiting_orders as $order_item): ?>
                    <div class="order-card">
                        <div class="order-header">
                            <div class="order-number">Order #<?php echo htmlspecialchars($order_item['order_number']); ?></div>
                            <div class="order-date"><?php echo date('M d, Y', strtotime($order_item['created_at'])); ?></div>
                        </div>
                        
                        <div class="order-amount">RM <?php echo number_format($order_item['total_amount'], 2); ?></div>
                        
                        <div class="order-status">
                            <i class="fas fa-clock"></i> Awaiting Payment
                        </div>
                        
                        <div class="order-actions">
                            <form method="POST" style="flex: 1;">
                                <input type="hidden" name="order_id" value="<?php echo $order_item['id']; ?>">
                                <button type="submit" name="continue_payment" class="continue-payment-btn">
                                    <i class="fas fa-credit-card"></i> Continue Payment
                                </button>
                            </form>
                            <button class="cancel-order-btn" onclick="cancelOrder(<?php echo $order_item['id']; ?>)">
                                <i class="fas fa-times"></i> Cancel
                            </button>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>

        <!-- Bottom Navigation -->
        <nav class="bottom-nav">
            <div class="nav-item" onclick="window.location.href='index.php'">
                <i class="fas fa-home"></i>
            </div>
            <div class="nav-item" onclick="window.location.href='products.php'">
                <i class="fas fa-box"></i>
            </div>
            <div class="nav-item" onclick="window.location.href='cart.php'">
                <i class="fas fa-shopping-cart"></i>
            </div>
            <div class="nav-item" onclick="window.location.href='profile.php'">
                <i class="far fa-user"></i>
            </div>
        </nav>
    </div>

    <script>
        function goBack() {
            window.history.back();
        }

        function cancelOrder(orderId) {
            if (confirm('Are you sure you want to cancel this order?')) {
                // AJAX call to cancel order
                fetch('ajax/cancel-order.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        order_id: orderId
                    })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showNotification('Order cancelled successfully', 'success');
                        setTimeout(() => {
                            location.reload();
                        }, 1500);
                    } else {
                        showNotification(data.message || 'Failed to cancel order', 'error');
                    }
                })
                .catch(error => {
                    showNotification('Failed to cancel order', 'error');
                });
            }
        }

        function showNotification(message, type = 'info') {
            // Create notification element
            const notification = document.createElement('div');
            notification.className = `notification notification-${type}`;
            notification.innerHTML = `
                <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-circle' : 'info-circle'}"></i>
                ${message}
            `;
            
            // Style the notification
            notification.style.cssText = `
                position: fixed;
                top: 20px;
                left: 50%;
                transform: translateX(-50%);
                background: ${type === 'success' ? '#27ae60' : type === 'error' ? '#e74c3c' : '#3498db'};
                color: white;
                padding: 15px 20px;
                border-radius: 10px;
                box-shadow: 0 4px 15px rgba(0,0,0,0.2);
                z-index: 1000;
                font-size: 14px;
                font-weight: 500;
                max-width: 90%;
                text-align: center;
            `;
            
            document.body.appendChild(notification);
            
            // Remove notification after 3 seconds
            setTimeout(() => {
                notification.remove();
            }, 3000);
        }
    </script>
</body>
</html>