<?php
require_once 'models/Product.php';
require_once 'models/Loyalty.php';
require_once 'config/database.php';
require_once 'includes/auth-check.php';

// Get product ID from URL parameter
$productId = isset($_GET['id']) ? (int)$_GET['id'] : 1;

// Check if user is logged in and if product is in favorites
$isFavorite = false;
if (isset($_SESSION['user_id']) && $_SESSION['user_id'] !== 'guest') {
    $conn = getDB();
    $stmt = $conn->prepare("SELECT COUNT(*) FROM wishlist WHERE user_id = ? AND product_id = ?");
    $stmt->execute([$_SESSION['user_id'], $productId]);
    $isFavorite = $stmt->fetchColumn() > 0;
}

// Get product data from database with variations and sales
$productModel = new Product();
$loyalty = new Loyalty();
$products = $productModel->getAll(['limit' => null]);
$product = null;

// Find the specific product with all its data
foreach ($products as $p) {
    if ($p['id'] == $productId) {
        $product = $p;
        break;
    }
}

if (!$product) {
    // Redirect to products page if product not found
    header('Location: products.php');
    exit();
}

// Get related products (same brand or category)
$relatedProducts = array_filter($products, function($p) use ($product) {
    return $p['id'] != $product['id'] && 
           ($p['brand'] == $product['brand'] || $p['category_name'] == $product['category_name']);
});
$relatedProducts = array_slice($relatedProducts, 0, 4);

// Get delivery feedback for this product
$deliveryFeedback = [];
$overallRating = 0;
$totalReviews = 0;
$totalSold = 0;

try {
    $pdo = getDB();
    
    // First, ensure the delivery_feedback table exists
    $createTableQuery = "
        CREATE TABLE IF NOT EXISTS delivery_feedback (
            id INT AUTO_INCREMENT PRIMARY KEY,
            order_id INT NOT NULL,
            user_id INT NOT NULL,
            rating INT NOT NULL,
            feedback TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            UNIQUE KEY unique_order_feedback (order_id)
        )
    ";
    $pdo->exec($createTableQuery);
    
    // Get feedback for orders that contain this product
    $feedbackQuery = "
        SELECT df.*, u.first_name, u.last_name, u.avatar, o.created_at as order_date
        FROM delivery_feedback df
        JOIN orders o ON df.order_id = o.id
        JOIN order_items oi ON o.id = oi.order_id
        LEFT JOIN users u ON df.user_id = u.id
        WHERE oi.product_id = :product_id
        ORDER BY df.created_at DESC
        LIMIT 10
    ";
    
    $stmt = $pdo->prepare($feedbackQuery);
    $stmt->bindValue(':product_id', $productId);
    $stmt->execute();
    $deliveryFeedback = $stmt->fetchAll();
    
    // Debug: Log the number of feedback records found
    error_log("Product ID: $productId, Feedback records found: " . count($deliveryFeedback));
    
    // If no feedback found, try a simpler query to check if there's any feedback at all
    if (empty($deliveryFeedback)) {
        $simpleQuery = "SELECT COUNT(*) as count FROM delivery_feedback";
        $result = $pdo->query($simpleQuery);
        $row = $result->fetch();
        error_log("Total feedback records in database: " . $row['count']);
    }
    
    // Get actual sales count for this product
    $salesQuery = "
        SELECT SUM(oi.quantity) as total_sold
        FROM order_items oi
        JOIN orders o ON oi.order_id = o.id
        WHERE oi.product_id = :product_id
        AND o.status IN ('delivered', 'completed', 'shipped')
    ";
    
    $stmt = $pdo->prepare($salesQuery);
    $stmt->bindValue(':product_id', $productId);
    $stmt->execute();
    $salesResult = $stmt->fetch();
    $totalSold = $salesResult['total_sold'] ?? 0;
    
    // Ensure we show 0 if no sales data
    if ($totalSold === null || $totalSold === '') {
        $totalSold = 0;
    }
    
    error_log("Product ID: $productId, Total sold: $totalSold");
    
    // Calculate overall rating
    if (!empty($deliveryFeedback)) {
        $totalRating = array_sum(array_column($deliveryFeedback, 'rating'));
        $overallRating = round($totalRating / count($deliveryFeedback), 1);
        $totalReviews = count($deliveryFeedback);
    }
    
} catch (Exception $e) {
    error_log("Error fetching delivery feedback: " . $e->getMessage());
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $product['name']; ?> - Product Details</title>
    <link rel="stylesheet" href="styles.css">
    <link rel="stylesheet" href="product-detail.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="script.js" defer></script>
</head>
<body>
    <!-- Top Navigation -->
    <nav class="top-nav">
        <div class="nav-left">
            <i class="fas fa-arrow-left" onclick="goBack()"></i>
        </div>
        <div class="nav-center">
            <div class="logo">
                <img src="images/LOGO.png" alt="2 Days Parfume Logo" class="logo-img">
            </div>
        </div>
        <div class="nav-right">
                <div class="cart-icon" onclick="window.location.href='cart.php'">
                    <i class="fas fa-shopping-cart"></i>
                    <span class="cart-count">0</span>
            </div>
        </div>
    </nav>

    <!-- Product Detail Content -->
    <main class="product-detail-main">
        <!-- Product Image Section -->
        <section class="product-image-section">
            <div class="product-image-container">
                <div class="image-slider">
                    <div class="slider-wrapper">
                        <!-- Main Image -->
                        <div class="slide active">
                            <img src="<?php echo $product['image']; ?>" alt="<?php echo $product['name']; ?>" class="product-main-image" onerror="this.src='data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMzAwIiBoZWlnaHQ9IjMwMCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cmVjdCB3aWR0aD0iMzAwIiBoZWlnaHQ9IjMwMCIgZmlsbD0iI2Y0ZjRmNCIvPjx0ZXh0IHg9IjE1MCIgeT0iMTUwIiBmb250LWZhbWlseT0iQXJpYWwiIGZvbnQtc2l6ZT0iMjQiIGZpbGw9IiM5OTkiIHRleHQtYW5jaG9yPSJtaWRkbGUiPkltYWdlPC90ZXh0Pjwvc3ZnPg=='">
                        </div>
                        
                        <!-- Second Image (if available) -->
                        <?php if (!empty($product['image_2'])): ?>
                            <div class="slide">
                                <img src="<?php echo $product['image_2']; ?>" alt="<?php echo $product['name']; ?> - Image 2" class="product-main-image" onerror="this.src='data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMzAwIiBoZWlnaHQ9IjMwMCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cmVjdCB3aWR0aD0iMzAwIiBoZWlnaHQ9IjMwMCIgZmlsbD0iI2Y0ZjRmNCIvPjx0ZXh0IHg9IjE1MCIgeT0iMTUwIiBmb250LWZhbWlseT0iQXJpYWwiIGZvbnQtc2l6ZT0iMjQiIGZpbGw9IiM5OTkiIHRleHQtYW5jaG9yPSJtaWRkbGUiPkltYWdlPC90ZXh0Pjwvc3ZnPg=='">
                            </div>
                        <?php endif; ?>
                        
                        <!-- Third Image (if available) -->
                        <?php if (!empty($product['image_3'])): ?>
                            <div class="slide">
                                <img src="<?php echo $product['image_3']; ?>" alt="<?php echo $product['name']; ?> - Image 3" class="product-main-image" onerror="this.src='data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMzAwIiBoZWlnaHQ9IjMwMCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cmVjdCB3aWR0aD0iMzAwIiBoZWlnaHQ9IjMwMCIgZmlsbD0iI2Y0ZjRmNCIvPjx0ZXh0IHg9IjE1MCIgeT0iMTUwIiBmb250LWZhbWlseT0iQXJpYWwiIGZvbnQtc2l6ZT0iMjQiIGZpbGw9IiM5OTkiIHRleHQtYW5jaG9yPSJtaWRkbGUiPkltYWdlPC90ZXh0Pjwvc3ZnPg=='">
                            </div>
                        <?php endif; ?>
                        
                        <!-- Fourth Image (if available) -->
                        <?php if (!empty($product['image_4'])): ?>
                            <div class="slide">
                                <img src="<?php echo $product['image_4']; ?>" alt="<?php echo $product['name']; ?> - Image 4" class="product-main-image" onerror="this.src='data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMzAwIiBoZWlnaHQ9IjMwMCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cmVjdCB3aWR0aD0iMzAwIiBoZWlnaHQ9IjMwMCIgZmlsbD0iI2Y0ZjRmNCIvPjx0ZXh0IHg9IjE1MCIgeT0iMTUwIiBmb250LWZhbWlseT0iQXJpYWwiIGZvbnQtc2l6ZT0iMjQiIGZpbGw9IiM5OTkiIHRleHQtYW5jaG9yPSJtaWRkbGUiPkltYWdlPC90ZXh0Pjwvc3ZnPg=='">
                            </div>
                        <?php endif; ?>
                        
                        <!-- Fifth Image (if available) -->
                        <?php if (!empty($product['image_5'])): ?>
                            <div class="slide">
                                <img src="<?php echo $product['image_5']; ?>" alt="<?php echo $product['name']; ?> - Image 5" class="product-main-image" onerror="this.src='data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMzAwIiBoZWlnaHQ9IjMwMCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cmVjdCB3aWR0aD0iMzAwIiBoZWlnaHQ9IjMwMCIgZmlsbD0iI2Y0ZjRmNCIvPjx0ZXh0IHg9IjE1MCIgeT0iMTUwIiBmb250LWZhbWlseT0iQXJpYWwiIGZvbnQtc2l6ZT0iMjQiIGZpbGw9IiM5OTkiIHRleHQtYW5jaG9yPSJtaWRkbGUiPkltYWdlPC90ZXh0Pjwvc3ZnPg=='">
                            </div>
                        <?php endif; ?>
                        
                        <!-- Sixth Image (if available) -->
                        <?php if (!empty($product['image_6'])): ?>
                            <div class="slide">
                                <img src="<?php echo $product['image_6']; ?>" alt="<?php echo $product['name']; ?> - Image 6" class="product-main-image" onerror="this.src='data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMzAwIiBoZWlnaHQ9IjMwMCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cmVjdCB3aWR0aD0iMzAwIiBoZWlnaHQ9IjMwMCIgZmlsbD0iI2Y0ZjRmNCIvPjx0ZXh0IHg9IjE1MCIgeT0iMTUwIiBmb250LWZhbWlseT0iQXJpYWwiIGZvbnQtc2l6ZT0iMjQiIGZpbGw9IiM5OTkiIHRleHQtYW5jaG9yPSJtaWRkbGUiPkltYWdlPC90ZXh0Pjwvc3ZnPg=='">
                            </div>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Navigation Arrows -->
                    <?php 
                    $totalImages = 1; // Main image
                    if (!empty($product['image_2'])) $totalImages++;
                    if (!empty($product['image_3'])) $totalImages++;
                    if (!empty($product['image_4'])) $totalImages++;
                    if (!empty($product['image_5'])) $totalImages++;
                    if (!empty($product['image_6'])) $totalImages++;
                    
                    if ($totalImages > 1): ?>
                        <button class="slider-arrow slider-prev" onclick="changeProductSlide(-1)">
                            <i class="fas fa-chevron-left"></i>
                        </button>
                        <button class="slider-arrow slider-next" onclick="changeProductSlide(1)">
                            <i class="fas fa-chevron-right"></i>
                        </button>
                    <?php endif; ?>
                    
                    <!-- Navigation Dots -->
                    <?php if ($totalImages > 1): ?>
                        <div class="slider-dots">
                            <?php for ($i = 1; $i <= $totalImages; $i++): ?>
                                <span class="dot <?php echo $i === 1 ? 'active' : ''; ?>" onclick="currentProductSlide(<?php echo $i; ?>)"></span>
                            <?php endfor; ?>
                        </div>
                    <?php endif; ?>
                </div>
                
                <!-- Sale and Flash Sale Badges -->
                <?php if ($product['is_out_of_stock']): ?>
                    <div class="sold-out-badge">
                        <i class="fas fa-times-circle"></i> Sold Out
                    </div>
                <?php elseif (isset($product['flash_sale_active']) && $product['flash_sale_active']): ?>
                    <div class="product-badge flash-sale-badge">
                        <i class="fas fa-bolt"></i> Flash Sale
                    </div>
                <?php elseif (isset($product['sale_active']) && $product['sale_active']): ?>
                    <div class="product-badge sale-badge">
                        Sale
                    </div>
                <?php endif; ?>
                
                <!-- Out of Stock Overlay -->
                <?php if ($product['is_out_of_stock']): ?>
                    <div class="out-of-stock-overlay">
                        <span>Out of Stock</span>
                        <?php if (!empty($product['restock_date'])): ?>
                            <div class="restock-timer" data-restock-date="<?php echo $product['restock_date']; ?>">
                                Back in stock in: <span class="timer-days">00</span>d <span class="timer-hours">00</span>h <span class="timer-minutes">00</span>m
                            </div>
                        <?php elseif (!empty($product['restock_message'])): ?>
                            <div class="restock-message">
                                <?php echo htmlspecialchars($product['restock_message']); ?>
                            </div>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            </div>
        </section>

        <!-- Product Information Section -->
        <section class="product-info-section">
            <div class="product-info-content">
                <h1 class="product-name"><?php echo $product['name']; ?></h1>
                
                <!-- Rating and Sold Counter -->
                <div class="product-rating-sold">
                <div class="product-rating">
                    <div class="stars">
                        <?php 
                        // Use only real rating from delivery feedback, no fallback to hardcoded values
                        $displayRating = $overallRating;
                        $fullStars = floor($displayRating);
                        $hasHalfStar = ($displayRating - $fullStars) >= 0.5;
                        
                        for ($i = 1; $i <= 5; $i++) {
                            if ($i <= $fullStars) {
                                echo '<i class="fas fa-star"></i>';
                            } elseif ($i == $fullStars + 1 && $hasHalfStar) {
                                echo '<i class="fas fa-star-half-alt"></i>';
                            } else {
                                echo '<i class="far fa-star"></i>';
                            }
                        }
                        ?>
                    </div>
                    <span class="rating-text"><?php echo $displayRating; ?> (<?php echo $totalReviews; ?>)</span>
                    </div>
                    <div class="sold-counter">
                        <span class="sold-text"><?php echo $totalSold; ?> sold</span>
                    </div>
                </div>

                <!-- Description -->
                <p class="product-description"><?php echo $product['description']; ?></p>
                
                <!-- Product Details -->
                <div class="product-details-simple">
                    <?php if (!empty($product['brand'])): ?>
                        <span class="detail-text">Brand: <?php echo htmlspecialchars($product['brand']); ?></span>
                    <?php endif; ?>
                    
                    <?php if (!empty($product['scent'])): ?>
                        <span class="detail-text">Scent: <?php echo htmlspecialchars($product['scent']); ?></span>
                    <?php endif; ?>
                    
                    <?php if (!empty($product['notes'])): ?>
                        <span class="detail-text">Notes: <?php echo htmlspecialchars($product['notes']); ?></span>
                    <?php endif; ?>
                    
                    <?php if (!empty($product['season'])): ?>
                        <span class="detail-text">Season: <?php echo htmlspecialchars($product['season']); ?></span>
                    <?php endif; ?>
                    
                    <?php if (!empty($product['recommendation'])): ?>
                        <span class="detail-text">Recommendation: <?php echo htmlspecialchars($product['recommendation']); ?></span>
                    <?php endif; ?>
                </div>

                <!-- Price and Sizes -->
                <div class="product-details">
                    <div class="price-section">
                        <label class="detail-label">Price</label>
                        <div class="price-container">
                            <div class="price-display">
                                <?php 
                                // Check if product has flash sale variations
                                $hasFlashSaleVariations = isset($product['flash_sale_variations_detail']) && !empty($product['flash_sale_variations_detail']);
                                $showSalePrice = false;
                                $salePrice = 0;
                                $originalPrice = 0;
                                
                                if ($hasFlashSaleVariations) {
                                    // Get the first variation's prices for display
                                    $firstVariation = $product['flash_sale_variations_detail'][0];
                                    $salePrice = $firstVariation['sale_price'];
                                    $originalPrice = $firstVariation['original_price'];
                                    $showSalePrice = true;
                                } elseif (isset($product['sale_active']) && $product['sale_active'] && !empty($product['sale_price'])) {
                                    $salePrice = $product['sale_price'];
                                    $originalPrice = $product['display_price'];
                                    $showSalePrice = true;
                                }
                                ?>
                                
                                <?php if ($showSalePrice): ?>
                                    <div class="sale-price">RM <?php echo number_format($salePrice, 2); ?></div>
                                    <div class="original-price">RM <?php echo number_format($originalPrice, 2); ?></div>
                                <?php else: ?>
                                    <div class="price"><?php echo $product['price_range']; ?></div>
                                <?php endif; ?>
                                
                                <?php if (isset($product['flash_sale_active']) && $product['flash_sale_active']): ?>
                                    <div class="flash-sale-timer" data-end-time="<?php echo $product['flash_sale_ends_at']; ?>">
                                        <i class="fas fa-clock"></i> <span class="countdown">Loading...</span>
                                    </div>
                                <?php endif; ?>
                            </div>
                            <button class="share-btn" id="share-btn" title="Share Product">
                                <i class="fas fa-share-alt"></i>
                            </button>
                        </div>
                    </div>
                    
                    <div class="variation-section">
                        <div class="variation-header" onclick="openProductModal()">
                            <div class="variation-title">
                                <span class="variation-label">Select variation</span>
                                <span class="variation-count">(<?php echo count($product['variations_detail']); ?> SIZES)</span>
                            </div>
                            <i class="fas fa-chevron-right variation-arrow"></i>
                        </div>
                        <div class="variation-options">
                            <?php if (!empty($product['variations_detail'])): ?>
                                <?php 
                                // Deduplicate variations by size and filter out invalid prices
                                $unique_variations = [];
                                $seen_sizes = [];
                                foreach ($product['variations_detail'] as $variation) {
                                    // Skip variations with invalid prices or sizes
                                    if (empty($variation['size']) || $variation['price'] <= 0) {
                                        continue;
                                    }
                                    
                                    if (!in_array($variation['size'], $seen_sizes)) {
                                        $unique_variations[] = $variation;
                                        $seen_sizes[] = $variation['size'];
                                    }
                                }
                                
                                // Show only first 3 variations in main view, rest in modal
                                $display_variations = array_slice($unique_variations, 0, 3);
                                $remaining_count = count($unique_variations) - 3;
                                ?>
                                
                                <?php foreach ($display_variations as $index => $variation): ?>
                                    <?php
                                    // Check if this variation is on flash sale
                                    $displayPrice = $variation['price'];
                                    $originalPrice = $variation['price'];
                                    $isOnSale = false;
                                    
                                    if (isset($product['flash_sale_variations_detail']) && !empty($product['flash_sale_variations_detail'])) {
                                        foreach ($product['flash_sale_variations_detail'] as $saleVariation) {
                                            if ($saleVariation['version_type'] === $variation['version_type'] && 
                                                $saleVariation['size'] === $variation['size']) {
                                                $displayPrice = $saleVariation['sale_price'];
                                                $originalPrice = $saleVariation['original_price'];
                                                $isOnSale = true;
                                                break;
                                            }
                                        }
                                    }
                                    ?>
                                    <div class="variation-option <?php echo $index === 0 ? 'selected' : ''; ?>" onclick="openProductModal()">
                                        <div class="option-text">
                                            <span class="option-type"><?php echo htmlspecialchars($variation['version_type'] . ' - ' . $variation['size']); ?></span>
                                            <span class="option-price">
                                                <?php if ($isOnSale): ?>
                                                    <span class="sale-price">RM <?php echo number_format($displayPrice, 2); ?></span>
                                                    <span class="original-price">RM <?php echo number_format($originalPrice, 2); ?></span>
                                                <?php else: ?>
                                                    RM <?php echo number_format($displayPrice, 2); ?>
                                                <?php endif; ?>
                                            </span>
                                        </div>
                                        <div class="option-stock">
                                            <?php if ($variation['stock'] > 0): ?>
                                                <span class="stock-available"><?php echo $variation['stock']; ?> in stock</span>
                                            <?php else: ?>
                                                <span class="stock-out">Out of stock</span>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                                
                                <?php if ($remaining_count > 0): ?>
                                    <div class="variation-option more-options" onclick="openProductModal()">
                                        <div class="option-text">
                                            <span class="option-type">+<?php echo $remaining_count; ?> more sizes</span>
                                            <span class="option-price">View all</span>
                                        </div>
                                    </div>
                                <?php endif; ?>
                            <?php else: ?>
                                <div class="variation-option">
                                    <div class="option-text">
                                        <span class="option-type">No variations available</span>
                                    </div>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <!-- Point Rewards Section -->
                    <?php 
                    // Calculate points based on price (RM1 = 1 point)
                    $calculated_points = 0;
                    if (!empty($product['variations_detail'])) {
                        // Get the first variation's price for point calculation
                        $first_variation = $product['variations_detail'][0];
                        $price = $first_variation['price'] ?? 0;
                        if ($price > 0) {
                            $calculated_points = $loyalty->calculatePointsFromPrice($price);
                        }
                    }
                    ?>
                    <?php if ($calculated_points > 0): ?>
                    <div class="point-rewards-section">
                        <div class="point-rewards-banner">
                            <div class="point-rewards-icon">
                                <i class="fas fa-gift"></i>
                            </div>
                            <div class="point-rewards-content">
                                <div class="point-rewards-title">Earn Loyalty Points!</div>
                                <div class="point-rewards-description">
                                    Buy this product to get <strong><?php echo $calculated_points; ?> points</strong> added to your loyalty account
                                </div>
                            </div>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </section>

        <!-- Comments and Reviews Section -->
        <section class="comments-section">
            <div class="comments-header">
                <h3 class="comments-title">Customer Reviews</h3>
                <div class="reviews-summary">
                    <div class="overall-rating">
                        <span class="rating-number"><?php echo $overallRating; ?></span>
                        <div class="rating-stars">
                            <?php
                            $fullStars = floor($overallRating);
                            $hasHalfStar = ($overallRating - $fullStars) >= 0.5;
                            
                            for ($i = 1; $i <= 5; $i++) {
                                if ($i <= $fullStars) {
                                    echo '<i class="fas fa-star"></i>';
                                } elseif ($i == $fullStars + 1 && $hasHalfStar) {
                                    echo '<i class="fas fa-star-half-alt"></i>';
                                } else {
                                    echo '<i class="far fa-star"></i>';
                                }
                            }
                            ?>
                        </div>
                        <span class="total-reviews">(<?php echo $totalReviews; ?> reviews)</span>
                    </div>
                </div>
            </div>

            <div class="comments-list">
                <?php if (empty($deliveryFeedback)): ?>
                    <div class="no-reviews">
                        <div style="text-align: center; padding: 40px 20px; color: #666;">
                            <i class="fas fa-star" style="font-size: 48px; color: #D4AF37; margin-bottom: 15px; opacity: 0.5;"></i>
                            <h4 style="color: #1A4D2A; margin-bottom: 10px;">No Reviews Yet</h4>
                            <p>Be the first to review this product after your delivery!</p>
                        </div>
                    </div>
                <?php else: ?>
                    <?php foreach ($deliveryFeedback as $index => $review): ?>
                        <div class="comment-item <?php echo $index >= 3 ? 'hidden-review' : ''; ?>">
                            <div class="comment-header">
                                <div class="commenter-info">
                                    <div class="commenter-avatar">
                                        <?php if (!empty($review['avatar'])): ?>
                                            <img src="<?php echo htmlspecialchars($review['avatar']); ?>" alt="User Avatar" onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                                        <?php endif; ?>
                                        <i class="fas fa-user" <?php echo !empty($review['avatar']) ? 'style="display:none;"' : ''; ?>></i>
                                    </div>
                                    <div class="commenter-details">
                                        <span class="commenter-name">
                                            <?php 
                                            $firstName = htmlspecialchars($review['first_name'] ?? 'Customer');
                                            $lastName = htmlspecialchars($review['last_name'] ?? '');
                                            echo $firstName . ($lastName ? ' ' . substr($lastName, 0, 1) . '.' : '');
                                            ?>
                                        </span>
                                        <div class="comment-rating">
                                            <?php
                                            for ($i = 1; $i <= 5; $i++) {
                                                if ($i <= $review['rating']) {
                                                    echo '<i class="fas fa-star"></i>';
                                                } else {
                                                    echo '<i class="far fa-star"></i>';
                                                }
                                            }
                                            ?>
                                        </div>
                                    </div>
                                </div>
                                <span class="comment-date">
                                    <?php 
                                    $reviewDate = new DateTime($review['created_at']);
                                    $now = new DateTime();
                                    $diff = $now->diff($reviewDate);
                                    
                                    if ($diff->days == 0) {
                                        echo 'Today';
                                    } elseif ($diff->days == 1) {
                                        echo '1 day ago';
                                    } elseif ($diff->days < 7) {
                                        echo $diff->days . ' days ago';
                                    } elseif ($diff->days < 30) {
                                        $weeks = floor($diff->days / 7);
                                        echo $weeks . ' week' . ($weeks > 1 ? 's' : '') . ' ago';
                                    } else {
                                        $months = floor($diff->days / 30);
                                        echo $months . ' month' . ($months > 1 ? 's' : '') . ' ago';
                                    }
                                    ?>
                                </span>
                            </div>
                            <div class="comment-content">
                                <?php if (!empty($review['feedback'])): ?>
                                    <p><?php echo htmlspecialchars($review['feedback']); ?></p>
                                <?php else: ?>
                                    <p><em>No additional feedback provided.</em></p>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>

            <?php if (count($deliveryFeedback) > 3): ?>
            <div class="view-more-reviews">
                <button class="view-more-btn">View More Reviews</button>
            </div>
            <?php endif; ?>
        </section>

        <!-- More Related Products Section -->
        <section class="related-products-section">
            <div class="related-products-header">
                <h3 class="related-products-title">More Related Products</h3>
                <button class="view-all-btn" onclick="window.location.href='products.php'">View All</button>
            </div>
            
            <div class="related-products-scroll">
                <div class="related-products-list">
                    <?php foreach ($relatedProducts as $relatedProduct): ?>
                    <div class="related-product-card <?php echo $relatedProduct['is_out_of_stock'] ? 'out-of-stock' : ''; ?>" onclick="viewProduct(<?php echo $relatedProduct['id']; ?>)">
                        <div class="related-product-image">
                            <img src="<?php echo $relatedProduct['image']; ?>" alt="<?php echo $relatedProduct['name']; ?>" onerror="this.src='data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMTAwIiBoZWlnaHQ9IjEwMCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cmVjdCB3aWR0aD0iMTAwIiBoZWlnaHQ9IjEwMCIgZmlsbD0iI2Y0ZjRmNCIvPjx0ZXh0IHg9IjUwIiB5PSI1MCIgZm9udC1mYW1pbHk9IkFyaWFsIiBmb250LXNpemU9IjE0IiBmaWxsPSIjOTk5IiB0ZXh0LWFuY2hvcj0ibWlkZGxlIj5JbWFnZTwvdGV4dD48L3N2Zz4='">
                            
                            <!-- Sale Badges -->
                                <?php if (isset($relatedProduct['flash_sale_active']) && $relatedProduct['flash_sale_active']): ?>
                                <div class="product-badge flash-sale-badge">
                                    <i class="fas fa-bolt"></i>
                                </div>
                            <?php elseif (isset($relatedProduct['sale_active']) && $relatedProduct['sale_active']): ?>
                                <div class="product-badge sale-badge">Sale</div>
                            <?php endif; ?>
                        </div>
                        <div class="related-product-info">
                            <?php
                            $rpNameRaw = $relatedProduct['name'] ?? '';
                            $rpNameEsc = htmlspecialchars($rpNameRaw);
                            if (function_exists('mb_strlen') && function_exists('mb_substr')) {
                                $rpDisplayName = (mb_strlen($rpNameRaw, 'UTF-8') > 15)
                                    ? htmlspecialchars(mb_substr($rpNameRaw, 0, 15, 'UTF-8')) . '…'
                                    : $rpNameEsc;
                            } else {
                                $rpDisplayName = (strlen($rpNameRaw) > 15)
                                    ? htmlspecialchars(substr($rpNameRaw, 0, 15)) . '…'
                                    : $rpNameEsc;
                            }
                            ?>
                            <h4 class="related-product-name" title="<?php echo $rpNameEsc; ?>"><?php echo $rpDisplayName; ?></h4>
                            <p class="related-product-volume">
                                <?php if (!empty($relatedProduct['variations_detail'])): ?>
                                    <?php echo htmlspecialchars($relatedProduct['variations_detail'][0]['size']); ?>
                                <?php else: ?>
                                    No variations
                                <?php endif; ?>
                            </p>
                            <div class="related-product-price-container">
                                <?php if (isset($relatedProduct['sale_active']) && $relatedProduct['sale_active'] && !empty($relatedProduct['sale_price'])): ?>
                                    <p class="related-product-price sale-price">RM <?php echo number_format($relatedProduct['sale_price'], 2); ?></p>
                                    <p class="related-product-price original-price">RM <?php echo number_format($relatedProduct['display_price'], 2); ?></p>
                                <?php else: ?>
                                    <p class="related-product-price"><?php echo $relatedProduct['price_range']; ?></p>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </section>
    </main>

    <!-- Bottom Action Bar -->
    <div class="bottom-action-bar">
        <button class="favorite-btn" id="favorite-btn" data-product-id="<?php echo $productId; ?>" style="<?php echo $isFavorite ? 'color: #ff6b6b;' : ''; ?>">
            <i class="<?php echo $isFavorite ? 'fas' : 'far'; ?> fa-heart"></i>
        </button>
        <?php if ($product['is_out_of_stock']): ?>
            <button class="add-to-cart-btn disabled" disabled title="Out of Stock">
                <i class="fas fa-shopping-cart"></i>
            </button>
            <button class="buy-now-btn disabled" disabled>
                Out of Stock
            </button>
        <?php else: ?>
            <button class="add-to-cart-btn" id="add-to-cart-detail" title="Add to Cart">
                <i class="fas fa-shopping-cart"></i>
            </button>
            <button class="buy-now-btn" id="buy-now-detail">
                Buy Now
            </button>
        <?php endif; ?>
    </div>

        <!-- Product Selection Modal -->
        <div class="product-modal-overlay" id="product-modal">
            <div class="product-modal">
                <div class="modal-header">
                    <div class="modal-close" id="close-modal">
                        <i class="fas fa-times"></i>
                    </div>
                </div>
                
                <div class="modal-content">
                    <div class="modal-product-info">
                        <div class="modal-product-image">
                            <img src="<?php echo $product['image']; ?>" alt="<?php echo $product['name']; ?>" id="modal-product-img">
                        </div>
                        <div class="modal-product-details">
                            <h3 class="modal-product-name" id="modal-product-name"><?php echo $product['name']; ?></h3>
                            <div class="modal-price" id="modal-price"><?php echo $product['price_range']; ?></div>
                            <div class="modal-stock" id="modal-stock">Total Stock: <?php echo $product['total_stock']; ?></div>
                        </div>
                    </div>

                    <div class="modal-options">
                        <!-- Version Selection -->
                        <div class="option-group">
                            <label class="option-label">VERSION</label>
                            <div class="option-buttons" id="version-buttons">
                                <?php 
                                // Get unique version types from variations
                                $available_versions = [];
                                if (!empty($product['variations_detail'])) {
                                    foreach ($product['variations_detail'] as $variation) {
                                        if (!in_array($variation['version_type'], $available_versions)) {
                                            $available_versions[] = $variation['version_type'];
                                        }
                                    }
                                }
                                
                                // Show only available versions
                                $first_version = !empty($available_versions) ? $available_versions[0] : '';
                                foreach ($available_versions as $index => $version): ?>
                                    <button class="option-btn <?php echo $index === 0 ? 'selected' : ''; ?>" 
                                            data-option="version" 
                                            data-value="<?php echo htmlspecialchars($version); ?>">
                                        <?php echo htmlspecialchars($version); ?>
                                    </button>
                                <?php endforeach; ?>
                            </div>
                        </div>

                        <!-- Size Selection -->
                        <div class="option-group">
                            <label class="option-label">SIZE</label>
                            <div class="option-buttons size-options-container" id="size-buttons">
                                <?php if (!empty($product['variations_detail'])): ?>
                                    <?php 
                                    // Filter variations for the first available version
                                    $first_version_variations = array_filter($product['variations_detail'], function($variation) use ($first_version) {
                                        return $variation['version_type'] === $first_version;
                                    });
                                    
                                    // Sort by size for consistent display
                                    usort($first_version_variations, function($a, $b) {
                                        return strcmp($a['size'], $b['size']);
                                    });
                                    ?>
                                    <?php foreach ($first_version_variations as $index => $variation): ?>
                                        <?php
                                        // Check if this variation is on flash sale
                                        $displayPrice = $variation['price'];
                                        $originalPrice = $variation['price'];
                                        $isOnSale = false;
                                        
                                        if (isset($product['flash_sale_variations_detail']) && !empty($product['flash_sale_variations_detail'])) {
                                            foreach ($product['flash_sale_variations_detail'] as $saleVariation) {
                                                if ($saleVariation['version_type'] === $variation['version_type'] && 
                                                    $saleVariation['size'] === $variation['size']) {
                                                    $displayPrice = $saleVariation['sale_price'];
                                                    $originalPrice = $saleVariation['original_price'];
                                                    $isOnSale = true;
                                                    break;
                                                }
                                            }
                                        }
                                        ?>
                                        <button class="option-btn <?php echo $index === 0 ? 'selected' : ''; ?>" 
                                                data-option="size" 
                                                data-value="<?php echo htmlspecialchars($variation['size']); ?>"
                                                data-price="<?php echo $displayPrice; ?>"
                                                data-original-price="<?php echo $originalPrice; ?>"
                                                data-is-sale="<?php echo $isOnSale ? 'true' : 'false'; ?>"
                                                data-stock="<?php echo $variation['stock']; ?>"
                                                data-version="<?php echo htmlspecialchars($variation['version_type']); ?>"
                                                data-size="<?php echo htmlspecialchars($variation['size']); ?>"
                                                <?php echo $variation['stock'] <= 0 ? 'disabled' : ''; ?>>
                                            <div class="size-option-content">
                                                <div class="size-name"><?php echo htmlspecialchars($variation['size']); ?></div>
                                                <div class="size-price">
                                                    <?php if ($isOnSale): ?>
                                                        <span class="sale-price">RM <?php echo number_format($displayPrice, 2); ?></span>
                                                        <span class="original-price">RM <?php echo number_format($originalPrice, 2); ?></span>
                                                    <?php else: ?>
                                                        RM <?php echo number_format($displayPrice, 2); ?>
                                                    <?php endif; ?>
                                                </div>
                                                <div class="size-stock <?php echo $variation['stock'] <= 0 ? 'out-of-stock' : 'in-stock'; ?>">
                                                    <?php echo $variation['stock'] <= 0 ? 'Out of stock' : $variation['stock'] . ' in stock'; ?>
                                                </div>
                                            </div>
                                        </button>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <button class="option-btn disabled" disabled>
                                        No variations available
                                    </button>
                                <?php endif; ?>
                            </div>
                        </div>

                        <!-- Quantity Selection -->
                        <div class="option-group">
                            <label class="option-label">Quantity</label>
                            <div class="quantity-selector">
                                <button class="quantity-btn minus" id="quantity-minus">
                                    <i class="fas fa-minus"></i>
                                </button>
                                <input type="number" class="quantity-input" id="quantity-input" value="1" min="1" max="10">
                                <button class="quantity-btn plus" id="quantity-plus">
                                    <i class="fas fa-plus"></i>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="modal-actions">
                    <?php if ($product['is_out_of_stock']): ?>
                        <button class="modal-add-cart-btn disabled" disabled>
                            <i class="fas fa-times-circle"></i> Out of Stock
                        </button>
                        <button class="modal-buy-now-btn disabled" disabled>
                            <i class="fas fa-times-circle"></i> Out of Stock
                        </button>
                    <?php else: ?>
                        <button class="modal-add-cart-btn" id="modal-add-cart">
                            Add to Cart
                        </button>
                        <button class="modal-buy-now-btn" id="modal-buy-now">
                            Buy Now
                        </button>
                    <?php endif; ?>
                </div>
            </div>
    </div>

    <script>
        function goBack() {
            window.history.back();
        }

        // Size selection in main product detail (now opens modal)
        document.querySelectorAll('.size-btn').forEach(btn => {
            btn.addEventListener('click', function(e) {
                e.preventDefault();
                // Don't change selection here, let modal handle it
            });
        });

        // Favorite toggle
        document.getElementById('favorite-btn').addEventListener('click', function() {
            const productId = this.getAttribute('data-product-id');
            const icon = this.querySelector('i');
            const isCurrentlyFavorite = icon.classList.contains('fas');
            const action = isCurrentlyFavorite ? 'remove' : 'add';
            
            // Show loading state
            this.disabled = true;
            const originalContent = icon.className;
            icon.className = 'fas fa-spinner fa-spin';
            
            // Send request to server
            fetch('ajax/toggle-favorite.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    product_id: parseInt(productId),
                    action: action
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    if (action === 'add') {
                        // Add to favorites
                        icon.className = 'fas fa-heart';
                        this.style.color = '#ff6b6b';
                        showNotification('Added to favorites!', 'success');
                    } else {
                        // Remove from favorites
                        icon.className = 'far fa-heart';
                        this.style.color = '#666666';
                        showNotification('Removed from favorites', 'info');
                    }
                } else {
                    // Restore original state on error
                    icon.className = originalContent;
                    if (data.message === 'Already in favorites') {
                        showNotification('This item is already in your favorites', 'info');
                    } else {
                        showNotification(data.message || 'Error updating favorites', 'error');
                    }
                }
            })
            .catch(error => {
                console.error('Error:', error);
                icon.className = originalContent;
                showNotification('Error updating favorites', 'error');
            })
            .finally(() => {
                this.disabled = false;
            });
        });

        // Open modal for add to cart and buy now
        const addToCartBtn = document.getElementById('add-to-cart-detail');
        const buyNowBtn = document.getElementById('buy-now-detail');
        
        if (addToCartBtn) {
            addToCartBtn.addEventListener('click', function() {
                console.log('Bottom add to cart clicked - opening modal');
                openProductModal();
            });
        }

        if (buyNowBtn) {
            buyNowBtn.addEventListener('click', function() {
                console.log('Bottom buy now clicked - opening modal');
                openProductModal();
            });
        }
        
        // Prevent modal opening for out of stock products
        function openProductModal() {
            <?php if ($product['is_out_of_stock']): ?>
                showNotification('This product is currently out of stock', 'error');
                return;
            <?php endif; ?>
            
            console.log('Opening product modal...');
            initializeModal(); // Initialize modal with current selections
            document.getElementById('product-modal').style.display = 'flex';
            document.body.style.overflow = 'hidden';
            console.log('Modal opened, checking for size buttons...');
            
            // Debug: Check if size buttons exist
            const sizeButtons = document.querySelectorAll('.option-btn[data-option="size"]');
            console.log('Found size buttons:', sizeButtons.length);
            
            const selectedSizeBtn = document.querySelector('.option-btn[data-option="size"].selected');
            console.log('Selected size button:', selectedSizeBtn);
        }


        function closeProductModal() {
            document.getElementById('product-modal').style.display = 'none';
            document.body.style.overflow = 'auto';
        }

        // Close modal events
        const closeModalBtn = document.getElementById('close-modal');
        const productModal = document.getElementById('product-modal');
        
        if (closeModalBtn) {
            closeModalBtn.addEventListener('click', closeProductModal);
        }
        
        if (productModal) {
            productModal.addEventListener('click', function(e) {
                if (e.target === this) {
                    closeProductModal();
                }
            });
        }

        // Share functionality
        const shareBtn = document.getElementById('share-btn');
        if (shareBtn) {
            shareBtn.addEventListener('click', function() {
            const productName = '<?php echo addslashes($product['name']); ?>';
            const productPrice = '<?php echo $product['price']; ?>';
            const productUrl = window.location.href;
            const shareText = `Check out this amazing perfume: ${productName} - RM ${productPrice}`;
            
            if (navigator.share) {
                // Use native share API if available (mobile)
                navigator.share({
                    title: productName,
                    text: shareText,
                    url: productUrl
                }).catch(err => console.log('Error sharing:', err));
            } else {
                // Fallback for desktop - copy to clipboard
                const shareData = `${shareText}\n${productUrl}`;
                
                if (navigator.clipboard) {
                    navigator.clipboard.writeText(shareData).then(() => {
                        showShareNotification('Product link copied to clipboard!');
                    }).catch(err => {
                        fallbackCopyToClipboard(shareData);
                    });
                } else {
                    fallbackCopyToClipboard(shareData);
                }
            }
            });
        }

        // Fallback copy to clipboard function
        function fallbackCopyToClipboard(text) {
            const textArea = document.createElement('textarea');
            textArea.value = text;
            textArea.style.position = 'fixed';
            textArea.style.left = '-999999px';
            textArea.style.top = '-999999px';
            document.body.appendChild(textArea);
            textArea.focus();
            textArea.select();
            
            try {
                document.execCommand('copy');
                showShareNotification('Product link copied to clipboard!');
            } catch (err) {
                console.log('Fallback copy failed:', err);
                showShareNotification('Unable to copy. Please share manually.');
            }
            
            document.body.removeChild(textArea);
        }

        // Show share notification
        function showShareNotification(message, type = 'info') {
            // Create notification element
            const notification = document.createElement('div');
            notification.className = `share-notification ${type}`;
            notification.textContent = message;
            document.body.appendChild(notification);
            
            // Show notification
            setTimeout(() => {
                notification.classList.add('show');
            }, 100);
            
            // Hide notification after 3 seconds
            setTimeout(() => {
                notification.classList.remove('show');
                setTimeout(() => {
                    document.body.removeChild(notification);
                }, 300);
            }, 3000);
        }

        // Update modal buttons based on selected size stock
        function updateModalButtons() {
            const selectedSizeBtn = document.querySelector('.option-btn[data-option="size"].selected');
            const modalAddCartBtn = document.getElementById('modal-add-cart');
            const modalBuyNowBtn = document.getElementById('modal-buy-now');
            
            if (!selectedSizeBtn || !modalAddCartBtn || !modalBuyNowBtn) {
                return;
            }
            
            const selectedStock = parseInt(selectedSizeBtn.getAttribute('data-stock'));
            const quantity = parseInt(document.getElementById('quantity-input').value);
            
            if (selectedStock <= 0) {
                // Out of stock
                modalAddCartBtn.disabled = true;
                modalAddCartBtn.classList.add('disabled');
                modalAddCartBtn.innerHTML = '<i class="fas fa-times-circle"></i> Out of Stock';
                
                modalBuyNowBtn.disabled = true;
                modalBuyNowBtn.classList.add('disabled');
                modalBuyNowBtn.innerHTML = '<i class="fas fa-times-circle"></i> Out of Stock';
            } else if (quantity > selectedStock) {
                // Quantity exceeds stock
                modalAddCartBtn.disabled = true;
                modalAddCartBtn.classList.add('disabled');
                modalAddCartBtn.innerHTML = `<i class="fas fa-exclamation-triangle"></i> Only ${selectedStock} Available`;
                
                modalBuyNowBtn.disabled = true;
                modalBuyNowBtn.classList.add('disabled');
                modalBuyNowBtn.innerHTML = `<i class="fas fa-exclamation-triangle"></i> Only ${selectedStock} Available`;
            } else {
                // In stock
                modalAddCartBtn.disabled = false;
                modalAddCartBtn.classList.remove('disabled');
                modalAddCartBtn.innerHTML = 'Add to Cart';
                
                modalBuyNowBtn.disabled = false;
                modalBuyNowBtn.classList.remove('disabled');
                modalBuyNowBtn.innerHTML = 'Buy Now';
            }
        }

        // View more reviews functionality
        document.addEventListener('DOMContentLoaded', function() {
            const viewMoreBtn = document.querySelector('.view-more-btn');
            if (viewMoreBtn) {
                viewMoreBtn.addEventListener('click', function() {
                    const hiddenReviews = document.querySelectorAll('.hidden-review');
                    
                    if (hiddenReviews.length > 0) {
                        // Show hidden reviews
                        hiddenReviews.forEach(review => {
                            review.style.display = 'block';
                        });
                        
                        this.textContent = 'View Less Reviews';
                        this.classList.add('view-less');
                    } else {
                        // Hide additional reviews
                        const allReviews = document.querySelectorAll('.comment-item');
                        allReviews.forEach((review, index) => {
                            if (index >= 3) {
                                review.style.display = 'none';
                            }
                        });
                        
                        this.textContent = 'View More Reviews';
                        this.classList.remove('view-less');
                    }
                });
            }

            // Modal option selection
            document.querySelectorAll('.option-btn').forEach(btn => {
                btn.addEventListener('click', function() {
                    const option = this.getAttribute('data-option');
                    const value = this.getAttribute('data-value');
                    
                    // Remove selected class from siblings
                    this.parentNode.querySelectorAll('.option-btn').forEach(b => b.classList.remove('selected'));
                    // Add selected class to clicked button
                    this.classList.add('selected');
                    
                    // If version was changed, update size options
                    if (option === 'version') {
                        updateSizeOptionsForVersion(value);
                    }
                    
                    // Update main product variation display
                    const selectedVersion = document.querySelector('.option-btn[data-option="version"].selected').getAttribute('data-value');
                    const selectedSizeBtn = document.querySelector('.option-btn[data-option="size"].selected');
                    const selectedSize = selectedSizeBtn ? selectedSizeBtn.getAttribute('data-value') : '';
                    updateMainProductVariation(selectedVersion, selectedSize);
                    
                    // Update price and buttons based on selection
                    updateModalPrice();
                    updateModalButtons();
                });
            });

            // Quantity controls
            const quantityInput = document.getElementById('quantity-input');
            const quantityMinus = document.getElementById('quantity-minus');
            const quantityPlus = document.getElementById('quantity-plus');

            quantityMinus.addEventListener('click', function() {
                let currentValue = parseInt(quantityInput.value);
                if (currentValue > 1) {
                    quantityInput.value = currentValue - 1;
                    updateModalPrice();
                    updateModalButtons();
                }
            });

            quantityPlus.addEventListener('click', function() {
                let currentValue = parseInt(quantityInput.value);
                if (currentValue < 10) {
                    quantityInput.value = currentValue + 1;
                    updateModalPrice();
                    updateModalButtons();
                }
            });

            quantityInput.addEventListener('change', function() {
                let value = parseInt(this.value);
                if (value < 1) this.value = 1;
                if (value > 10) this.value = 10;
                updateModalPrice();
                updateModalButtons();
            });

            // Modal action buttons
            const modalAddCartBtn = document.getElementById('modal-add-cart');
            if (modalAddCartBtn) {
                modalAddCartBtn.addEventListener('click', function() {
                console.log('Modal add to cart button clicked');
                const selectedSizeBtn = document.querySelector('.option-btn[data-option="size"].selected');
                
                if (!selectedSizeBtn) {
                    showShareNotification('Please select a size first!');
                    return;
                }
                
                // Debug: Log all size buttons and their states
                const allSizeButtons = document.querySelectorAll('.option-btn[data-option="size"]');
                console.log('All size buttons:', allSizeButtons);
                allSizeButtons.forEach((btn, index) => {
                    console.log(`Size button ${index}:`, {
                        element: btn,
                        selected: btn.classList.contains('selected'),
                        size: btn.getAttribute('data-size'),
                        price: btn.getAttribute('data-price'),
                        stock: btn.getAttribute('data-stock')
                    });
                });
                
                const selectedVersion = selectedSizeBtn.getAttribute('data-version');
                const selectedSize = selectedSizeBtn.getAttribute('data-size');
                const selectedPrice = selectedSizeBtn.getAttribute('data-price');
                const selectedStock = parseInt(selectedSizeBtn.getAttribute('data-stock'));
                const quantity = parseInt(document.getElementById('quantity-input').value);
                
                console.log('Selected size button details:', {
                    element: selectedSizeBtn,
                    version: selectedVersion,
                    size: selectedSize,
                    price: selectedPrice,
                    stock: selectedStock,
                    quantity: quantity
                });
                
                // Check if selected size is out of stock
                if (selectedStock <= 0) {
                    showShareNotification('Selected size is out of stock!', 'error');
                    return;
                }
                
                // Check if quantity exceeds available stock
                if (quantity > selectedStock) {
                    showShareNotification(`Only ${selectedStock} items available in stock!`, 'error');
                    return;
                }
                
                console.log('Add to cart data:', {
                    id: <?php echo $productId; ?>,
                    name: '<?php echo $product['name']; ?>',
                    price: selectedPrice,
                    version: selectedVersion,
                    size: selectedSize,
                    qty: quantity,
                    stock: selectedStock
                });
                
                // Track add to cart analytics
                console.log('AnalyticsTracker available:', typeof window.AnalyticsTracker);
                if (window.AnalyticsTracker && typeof window.AnalyticsTracker.trackAddToCart === 'function') {
                    console.log('Tracking add to cart for product:', <?php echo $productId; ?>, 'quantity:', quantity);
                    window.AnalyticsTracker.trackAddToCart(<?php echo $productId; ?>, quantity);
                } else {
                    console.warn('AnalyticsTracker not available or trackAddToCart method missing');
                }
                
                if (window.PerfumeApp && typeof window.PerfumeApp.addToCart === 'function') {
                    window.PerfumeApp.addToCart({
                        id: <?php echo $productId; ?>,
                        name: '<?php echo $product['name']; ?>',
                        price: selectedPrice,
                        image: '<?php echo $product['image']; ?>',
                        version: selectedVersion,
                        size: selectedSize,
                        qty: quantity
                    });
                } else {
                    console.error('PerfumeApp.addToCart not available, trying direct fetch...');
                    
                    // Fallback: Direct fetch to auth-handler.php
                    const formData = new FormData();
                    formData.append('action', 'add_to_cart');
                    formData.append('product_id', <?php echo $productId; ?>);
                    formData.append('quantity', quantity);
                    formData.append('size', selectedSize);
                    formData.append('version', selectedVersion);
                    formData.append('price', selectedPrice);
                    
                    fetch('auth-handler.php', {
                        method: 'POST',
                        body: formData
                    })
                    .then(response => response.json())
                    .then(data => {
                        console.log('Direct fetch response:', data);
                        if (data.success) {
                            showShareNotification('Added to cart!');
                            // Update cart badge if possible
                            if (window.PerfumeApp && typeof window.PerfumeApp.updateCartBadge === 'function') {
                                window.PerfumeApp.updateCartBadge(data.cart_count);
                            }
                        } else {
                            // Check if we need to redirect to signup
                            if (data.redirect === 'signup.php') {
                                showShareNotification('Please sign up to add items to cart');
                                setTimeout(() => {
                                    window.location.href = 'signup.php';
                                }, 1500);
                            } else {
                                showShareNotification('Failed to add to cart: ' + data.message);
                            }
                        }
                    })
                    .catch(error => {
                        console.error('Direct fetch error:', error);
                        showShareNotification('Error adding to cart');
                    });
                }
                
                closeProductModal();
            });
            }

            const modalBuyNowBtn = document.getElementById('modal-buy-now');
            if (modalBuyNowBtn) {
                modalBuyNowBtn.addEventListener('click', function() {
                console.log('Modal buy now button clicked');
                const selectedSizeBtn = document.querySelector('.option-btn[data-option="size"].selected');
                
                if (!selectedSizeBtn) {
                    showShareNotification('Please select a size first!');
                    return;
                }
                
                const selectedVersion = selectedSizeBtn.getAttribute('data-version');
                const selectedSize = selectedSizeBtn.getAttribute('data-size');
                const selectedPrice = selectedSizeBtn.getAttribute('data-price');
                const selectedStock = parseInt(selectedSizeBtn.getAttribute('data-stock'));
                const quantity = parseInt(document.getElementById('quantity-input').value);
                
                // Check if selected size is out of stock
                if (selectedStock <= 0) {
                    showShareNotification('Selected size is out of stock!', 'error');
                    return;
                }
                
                // Check if quantity exceeds available stock
                if (quantity > selectedStock) {
                    showShareNotification(`Only ${selectedStock} items available in stock!`, 'error');
                    return;
                }
                
                console.log('Buy now data:', {
                    id: <?php echo $productId; ?>,
                    name: '<?php echo $product['name']; ?>',
                    price: selectedPrice,
                    version: selectedVersion,
                    size: selectedSize,
                    qty: quantity
                });
                
                // Direct purchase - go straight to checkout without adding to cart
                const directPurchaseData = {
                    product_id: <?php echo $productId; ?>,
                    quantity: quantity,
                    size: selectedSize,
                    version: selectedVersion,
                    price: selectedPrice
                };
                
                closeProductModal();
                setTimeout(() => {
                    window.location.href = 'checkout.php?direct_purchase=' + encodeURIComponent(JSON.stringify(directPurchaseData));
                }, 300);
            });
            }
        });

        // Update size options based on selected version
        function updateSizeOptionsForVersion(selectedVersion) {
            const sizeContainer = document.getElementById('size-buttons');
            const allVariations = <?php echo json_encode($product['variations_detail']); ?>;
            
            // Filter variations for the selected version
            const versionVariations = allVariations.filter(variation => variation.version_type === selectedVersion);
            
            // Sort by size for consistent display
            versionVariations.sort((a, b) => a.size.localeCompare(b.size));
            
            // Clear existing size buttons
            sizeContainer.innerHTML = '';
            
            // Create new size buttons for the selected version
            let firstAvailableIndex = -1;
            versionVariations.forEach((variation, index) => {
                if (firstAvailableIndex === -1 && variation.stock > 0) {
                    firstAvailableIndex = index;
                }
            });
            
            versionVariations.forEach((variation, index) => {
                // Check if this variation is on flash sale
                let displayPrice = variation.price;
                let originalPrice = variation.price;
                let isOnSale = false;
                
                // Check flash sale variations
                <?php if (isset($product['flash_sale_variations_detail']) && !empty($product['flash_sale_variations_detail'])): ?>
                const flashSaleVariations = <?php echo json_encode($product['flash_sale_variations_detail']); ?>;
                flashSaleVariations.forEach(saleVariation => {
                    if (saleVariation.version_type === variation.version_type && 
                        saleVariation.size === variation.size) {
                        displayPrice = saleVariation.sale_price;
                        originalPrice = saleVariation.original_price;
                        isOnSale = true;
                    }
                });
                <?php endif; ?>
                
                const button = document.createElement('button');
                button.className = `option-btn ${index === firstAvailableIndex ? 'selected' : ''}`;
                button.setAttribute('data-option', 'size');
                button.setAttribute('data-value', variation.size);
                button.setAttribute('data-price', displayPrice);
                button.setAttribute('data-original-price', originalPrice);
                button.setAttribute('data-is-sale', isOnSale);
                button.setAttribute('data-stock', variation.stock);
                button.setAttribute('data-version', variation.version_type);
                button.setAttribute('data-size', variation.size);
                
                if (variation.stock <= 0) {
                    button.disabled = true;
                }
                
                const priceHtml = isOnSale ? 
                    `<span class="sale-price">RM ${parseFloat(displayPrice).toFixed(2)}</span>
                     <span class="original-price">RM ${parseFloat(originalPrice).toFixed(2)}</span>` :
                    `RM ${parseFloat(displayPrice).toFixed(2)}`;
                
                button.innerHTML = `
                    <div class="size-option-content">
                        <div class="size-name">${variation.size}</div>
                        <div class="size-price">${priceHtml}</div>
                        <div class="size-stock ${variation.stock <= 0 ? 'out-of-stock' : 'in-stock'}">
                            ${variation.stock <= 0 ? 'Out of stock' : variation.stock + ' in stock'}
                        </div>
                    </div>
                `;
                
                // Add click event listener
                button.addEventListener('click', function() {
                    console.log('Size button clicked:', {
                        size: this.getAttribute('data-size'),
                        price: this.getAttribute('data-price'),
                        stock: this.getAttribute('data-stock'),
                        version: this.getAttribute('data-version')
                    });
                    
                    // Remove selected class from siblings
                    sizeContainer.querySelectorAll('.option-btn').forEach(b => b.classList.remove('selected'));
                    // Add selected class to clicked button
                    this.classList.add('selected');
                    
                    console.log('After selection - all size buttons:');
                    sizeContainer.querySelectorAll('.option-btn').forEach((btn, index) => {
                        console.log(`Size button ${index}:`, {
                            size: btn.getAttribute('data-size'),
                            selected: btn.classList.contains('selected')
                        });
                    });
                    
                    // Update price and buttons
                    updateModalPrice();
                    updateModalButtons();
                });
                
                sizeContainer.appendChild(button);
            });
            
            // If no variations available for this version
            if (versionVariations.length === 0) {
                const button = document.createElement('button');
                button.className = 'option-btn disabled';
                button.disabled = true;
                button.textContent = 'No sizes available for this version';
                sizeContainer.appendChild(button);
            }
        }

        // Update modal price based on selections
        function updateModalPrice() {
            const selectedSizeBtn = document.querySelector('.option-btn[data-option="size"].selected');
            if (selectedSizeBtn) {
                const price = parseFloat(selectedSizeBtn.getAttribute('data-price'));
                const originalPrice = parseFloat(selectedSizeBtn.getAttribute('data-original-price'));
                const isOnSale = selectedSizeBtn.getAttribute('data-is-sale') === 'true';
                const quantity = parseInt(document.getElementById('quantity-input').value);
                const totalPrice = price * quantity;
                const totalOriginalPrice = originalPrice * quantity;
                
                let priceHtml;
                if (isOnSale && originalPrice !== price) {
                    priceHtml = `<span class="sale-price">RM ${totalPrice.toFixed(2)}</span> <span class="original-price">RM ${totalOriginalPrice.toFixed(2)}</span>`;
                } else {
                    priceHtml = `RM ${totalPrice.toFixed(2)}`;
                }
                
                document.getElementById('modal-price').innerHTML = priceHtml;
            }
        }
        
        // Flash sale countdown timer
        function updateFlashSaleTimer() {
            const timerElement = document.querySelector('.flash-sale-timer .countdown');
            if (timerElement) {
                const endTime = parseInt(timerElement.parentElement.getAttribute('data-end-time'));
                const now = Math.floor(Date.now() / 1000);
                const timeLeft = endTime - now;
                
                if (timeLeft > 0) {
                    const hours = Math.floor(timeLeft / 3600);
                    const minutes = Math.floor((timeLeft % 3600) / 60);
                    const seconds = timeLeft % 60;
                    
                    timerElement.textContent = `${hours}h ${minutes}m ${seconds}s`;
                } else {
                    timerElement.textContent = 'Expired';
                    timerElement.parentElement.style.color = '#e74c3c';
                }
            }
        }
        
        // Update flash sale timer every second
        setInterval(updateFlashSaleTimer, 1000);
        updateFlashSaleTimer(); // Initial call

        // Update main product variation display
        function updateMainProductVariation(selectedVersion, selectedSize) {
            // Update selected state of variation options
            const allOptions = document.querySelectorAll('.variation-option');
            allOptions.forEach(option => {
                option.classList.remove('selected');
            });
            
            // Find and select the matching option
            const versionParts = selectedVersion.split(' / ');
            const optionType = versionParts[0];
            const optionIntensity = versionParts[1];
            
            allOptions.forEach(option => {
                const typeSpan = option.querySelector('.option-type');
                const intensitySpan = option.querySelector('.option-intensity');
                
                if (typeSpan && intensitySpan) {
                    const currentType = typeSpan.textContent.trim();
                    const currentIntensity = intensitySpan.textContent.replace('/', '').trim();
                    
                    if (currentType === optionType && currentIntensity === optionIntensity) {
                        option.classList.add('selected');
                    }
                }
            });
        }

        // Initialize modal with current selections
        function initializeModal() {
            // Reset quantity to 1
            document.getElementById('quantity-input').value = 1;
            
            // Ensure the first available size is selected
            const sizeButtons = document.querySelectorAll('.option-btn[data-option="size"]');
            if (sizeButtons.length > 0) {
                // Remove selected class from all size buttons
                sizeButtons.forEach(btn => btn.classList.remove('selected'));
                
                // Find first available size and select it
                let firstAvailableBtn = null;
                for (let btn of sizeButtons) {
                    const stock = parseInt(btn.getAttribute('data-stock'));
                    if (stock > 0) {
                        firstAvailableBtn = btn;
                        break;
                    }
                }
                
                // Select first available size, or first size if none available
                const btnToSelect = firstAvailableBtn || sizeButtons[0];
                btnToSelect.classList.add('selected');
                
                console.log('Modal initialized - selected size:', btnToSelect.getAttribute('data-size'));
            }
            
            // Update modal price and buttons
            updateModalPrice();
            updateModalButtons();
        }

        // Image Slider Functionality
        let productDetailSlideIndex = 1;
        const totalSlides = document.querySelectorAll('.slide').length;
        
        function showProductSlide(n) {
            const slides = document.querySelectorAll('.slide');
            const dots = document.querySelectorAll('.dot');
            
            if (n > totalSlides) {
                productDetailSlideIndex = 1;
            }
            if (n < 1) {
                productDetailSlideIndex = totalSlides;
            }
            
            // Hide all slides
            slides.forEach(slide => {
                slide.classList.remove('active');
            });
            
            // Remove active class from all dots
            dots.forEach(dot => {
                dot.classList.remove('active');
            });
            
            // Show current slide
            if (slides[productDetailSlideIndex - 1]) {
                slides[productDetailSlideIndex - 1].classList.add('active');
            }
            
            // Activate current dot
            if (dots[productDetailSlideIndex - 1]) {
                dots[productDetailSlideIndex - 1].classList.add('active');
            }
            
            // Move slider wrapper
            const sliderWrapper = document.querySelector('.slider-wrapper');
            if (sliderWrapper) {
                sliderWrapper.style.transform = `translateX(-${(productDetailSlideIndex - 1) * 100}%)`;
            }
        }
        
        function changeProductSlide(n) {
            productDetailSlideIndex += n;
            showProductSlide(productDetailSlideIndex);
        }
        
        function currentProductSlide(n) {
            productDetailSlideIndex = n;
            showProductSlide(productDetailSlideIndex);
        }
        
        // Touch/Swipe Support
        let startX = 0;
        let endX = 0;
        const slider = document.querySelector('.image-slider');
        
        if (slider && totalSlides > 1) {
            slider.addEventListener('touchstart', function(e) {
                startX = e.touches[0].clientX;
            });
            
            slider.addEventListener('touchend', function(e) {
                endX = e.changedTouches[0].clientX;
                handleSwipe();
            });
            
            function handleSwipe() {
                const threshold = 50; // Minimum distance for swipe
                const diff = startX - endX;
                
                if (Math.abs(diff) > threshold) {
                    if (diff > 0) {
                        // Swipe left - next slide
                        changeProductSlide(1);
                    } else {
                        // Swipe right - previous slide
                        changeProductSlide(-1);
                    }
                }
            }
        }
        
        // Auto-play slider (optional)
        let autoPlayInterval;
        
        function startAutoPlay() {
            if (totalSlides > 1) {
                autoPlayInterval = setInterval(() => {
                    changeProductSlide(1);
                }, 5000); // Change slide every 5 seconds
            }
        }
        
        function stopAutoPlay() {
            if (autoPlayInterval) {
                clearInterval(autoPlayInterval);
            }
        }
        
        // Start auto-play when page loads
        document.addEventListener('DOMContentLoaded', function() {
            if (totalSlides > 1) {
                startAutoPlay();
                
                // Pause auto-play on hover
                slider.addEventListener('mouseenter', stopAutoPlay);
                slider.addEventListener('mouseleave', startAutoPlay);
            }
        });

        // Notification function
        function showNotification(message, type = 'info') {
            const notification = document.createElement('div');
            notification.style.cssText = `
                position: fixed;
                top: 20px;
                left: 50%;
                transform: translateX(-50%);
                background: ${type === 'info' ? '#1A4D2A' : type === 'success' ? '#28a745' : '#dc3545'};
                color: white;
                padding: 15px 20px;
                border-radius: 8px;
                font-size: 14px;
                z-index: 10000;
                box-shadow: 0 4px 12px rgba(0,0,0,0.2);
                max-width: 90%;
                text-align: center;
            `;
            notification.textContent = message;
            document.body.appendChild(notification);

            setTimeout(() => {
                if (notification.parentNode) {
                    notification.parentNode.removeChild(notification);
                }
            }, 3000);
        }

    </script>

    <!-- Analytics Tracking Script -->
    <script src="js/analytics-tracking.js"></script>
    <script>
        // Track product view when page loads
        document.addEventListener('DOMContentLoaded', function() {
            const productId = <?php echo $productId; ?>;
            if (window.AnalyticsTracker) {
                window.AnalyticsTracker.trackProductView(productId);
            }
            
            // Initialize restock timers
            if (typeof initializeRestockTimers === 'function') {
                initializeRestockTimers();
            }
        });
    </script>
</body>
</html>