<?php
session_start();
require_once 'config/database.php';

$error_message = '';
$success_message = '';
$token_valid = false;
$token = '';
$user_email = '';

// Check if token is provided
if (isset($_GET['token'])) {
    $token = trim($_GET['token']);
    
    try {
        $db = getDB();
        
        // Verify token
        $stmt = $db->prepare("
            SELECT prt.*, u.email, u.first_name, u.last_name 
            FROM password_reset_tokens prt
            JOIN users u ON prt.user_id = u.id
            WHERE prt.token = :token 
            AND prt.used = 0 
            AND prt.expires_at > NOW()
        ");
        $stmt->bindValue(':token', $token);
        $stmt->execute();
        $reset_data = $stmt->fetch();
        
        if ($reset_data) {
            $token_valid = true;
            $user_email = $reset_data['email'];
        } else {
            $error_message = 'Invalid or expired reset link. Please request a new password reset.';
        }
    } catch (Exception $e) {
        error_log("Token verification error: " . $e->getMessage());
        $error_message = 'An error occurred. Please try again.';
    }
} else {
    $error_message = 'No reset token provided.';
}

// Handle password reset form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $token_valid) {
    $new_password = trim($_POST['password'] ?? '');
    $confirm_password = trim($_POST['confirm_password'] ?? '');
    $token = trim($_POST['token'] ?? '');
    
    if (empty($new_password) || empty($confirm_password)) {
        $error_message = 'Please fill in all fields.';
    } elseif (strlen($new_password) < 8) {
        $error_message = 'Password must be at least 8 characters long.';
    } elseif ($new_password !== $confirm_password) {
        $error_message = 'Passwords do not match.';
    } else {
        try {
            $db = getDB();
            
            // Verify token again
            $stmt = $db->prepare("
                SELECT prt.*, u.id as user_id 
                FROM password_reset_tokens prt
                JOIN users u ON prt.user_id = u.id
                WHERE prt.token = :token 
                AND prt.used = 0 
                AND prt.expires_at > NOW()
            ");
            $stmt->bindValue(':token', $token);
            $stmt->execute();
            $reset_data = $stmt->fetch();
            
            if ($reset_data) {
                // Hash new password
                $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
                
                // Update user password
                $stmt = $db->prepare("UPDATE users SET password = :password WHERE id = :user_id");
                $stmt->bindValue(':password', $hashed_password);
                $stmt->bindValue(':user_id', $reset_data['user_id']);
                $stmt->execute();
                
                // Mark token as used
                $stmt = $db->prepare("UPDATE password_reset_tokens SET used = 1 WHERE token = :token");
                $stmt->bindValue(':token', $token);
                $stmt->execute();
                
                $success_message = 'Your password has been reset successfully! Redirecting to login...';
                
                // Redirect to login after 3 seconds
                echo '<script>setTimeout(function(){ window.location.href = "login.php?reset=success"; }, 3000);</script>';
            } else {
                $error_message = 'Invalid or expired reset link.';
                $token_valid = false;
            }
        } catch (Exception $e) {
            error_log("Password reset error: " . $e->getMessage());
            $error_message = 'An error occurred while resetting your password. Please try again.';
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reset Password - 2 Days Parfume</title>
    <link rel="stylesheet" href="styles.css">
    <link rel="stylesheet" href="auth.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .password-requirements {
            background: #f5f5f5;
            border-radius: 8px;
            padding: 15px;
            margin: 15px 0;
            font-size: 13px;
        }
        .password-requirements h4 {
            margin: 0 0 10px 0;
            font-size: 14px;
            color: #333;
        }
        .password-requirements ul {
            margin: 0;
            padding-left: 20px;
        }
        .password-requirements li {
            margin: 5px 0;
            color: #666;
        }
        .password-toggle {
            position: relative;
        }
        .password-toggle .toggle-icon {
            position: absolute;
            right: 15px;
            top: 50%;
            transform: translateY(-50%);
            cursor: pointer;
            color: #999;
        }
        .password-toggle .toggle-icon:hover {
            color: #333;
        }
        .form-input {
            padding-right: 45px;
        }
        .expired-link {
            text-align: center;
            padding: 40px 20px;
        }
        .expired-link i {
            font-size: 64px;
            color: #f44336;
            margin-bottom: 20px;
        }
        .expired-link h2 {
            color: #333;
            margin-bottom: 10px;
        }
        .expired-link p {
            color: #666;
            margin-bottom: 20px;
        }
    </style>
</head>
<body>
    <div class="auth-container">
        <!-- Top Navigation -->
        <nav class="top-nav">
            <div class="nav-left">
                <i class="fas fa-arrow-left" onclick="window.location.href='login.php'"></i>
            </div>
            <div class="nav-center">
                <div class="logo">
                    <img src="images/LOGO.png" alt="2 Days Parfume Logo" class="logo-img">
                </div>
            </div>
            <div class="nav-right">
                <i class="fas fa-search" onclick="window.location.href='products.php'"></i>
            </div>
        </nav>

        <!-- Auth Content -->
        <div class="auth-content">
            <?php if ($success_message): ?>
                <!-- Success State -->
                <div class="auth-header">
                    <div style="text-align: center; margin-bottom: 20px;">
                        <i class="fas fa-check-circle" style="font-size: 64px; color: #4caf50;"></i>
                    </div>
                    <h1 class="auth-title">Password Reset Successful!</h1>
                </div>

                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i>
                    <?php echo htmlspecialchars($success_message); ?>
                </div>

            <?php elseif (!$token_valid): ?>
                <!-- Invalid/Expired Token -->
                <div class="expired-link">
                    <i class="fas fa-exclamation-triangle"></i>
                    <h2>Invalid or Expired Link</h2>
                    <p>This password reset link is invalid or has expired.</p>
                    
                    <?php if ($error_message): ?>
                        <div class="alert alert-error" style="margin: 20px 0;">
                            <i class="fas fa-exclamation-circle"></i>
                            <?php echo htmlspecialchars($error_message); ?>
                        </div>
                    <?php endif; ?>
                    
                    <a href="forgot-password.php" class="auth-btn primary" style="display: inline-block; margin-top: 20px;">
                        <span>Request New Reset Link</span>
                    </a>
                    
                    <div style="margin-top: 20px;">
                        <a href="login.php" style="color: #666; text-decoration: none;">
                            <i class="fas fa-arrow-left"></i> Back to Login
                        </a>
                    </div>
                </div>

            <?php else: ?>
                <!-- Reset Password Form -->
                <div class="auth-header">
                    <h1 class="auth-title">Create New Password</h1>
                    <p class="auth-subtitle">Enter your new password below.</p>
                </div>

                <!-- Error Messages -->
                <?php if ($error_message): ?>
                    <div class="alert alert-error">
                        <i class="fas fa-exclamation-circle"></i>
                        <?php echo htmlspecialchars($error_message); ?>
                    </div>
                <?php endif; ?>

                <!-- Reset Password Form -->
                <form class="auth-form" method="POST" action="">
                    <input type="hidden" name="token" value="<?php echo htmlspecialchars($token); ?>">
                    
                    <div class="form-group">
                        <label for="password" style="display: block; margin-bottom: 8px; font-size: 14px; color: #333;">New Password</label>
                        <div class="password-toggle">
                            <input type="password" name="password" id="password" class="form-input" placeholder="Enter new password" required minlength="8">
                            <i class="fas fa-eye toggle-icon" onclick="togglePassword('password')"></i>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label for="confirm_password" style="display: block; margin-bottom: 8px; font-size: 14px; color: #333;">Confirm Password</label>
                        <div class="password-toggle">
                            <input type="password" name="confirm_password" id="confirm_password" class="form-input" placeholder="Confirm new password" required minlength="8">
                            <i class="fas fa-eye toggle-icon" onclick="togglePassword('confirm_password')"></i>
                        </div>
                    </div>

                    <div class="password-requirements">
                        <h4><i class="fas fa-shield-alt"></i> Password Requirements:</h4>
                        <ul>
                            <li>At least 8 characters long</li>
                            <li>Mix of letters and numbers recommended</li>
                            <li>Avoid using common words or personal information</li>
                        </ul>
                    </div>

                    <button type="submit" class="auth-btn primary">
                        <span>Reset Password</span>
                    </button>
                </form>

                <div style="text-align: center; margin-top: 20px;">
                    <a href="login.php" style="color: #666; text-decoration: none; font-size: 14px;">
                        <i class="fas fa-arrow-left"></i> Back to Login
                    </a>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <script>
        function togglePassword(fieldId) {
            const field = document.getElementById(fieldId);
            const icon = field.parentElement.querySelector('.toggle-icon');
            
            if (field.type === 'password') {
                field.type = 'text';
                icon.classList.remove('fa-eye');
                icon.classList.add('fa-eye-slash');
            } else {
                field.type = 'password';
                icon.classList.remove('fa-eye-slash');
                icon.classList.add('fa-eye');
            }
        }

        // Password strength indicator (optional)
        document.getElementById('password')?.addEventListener('input', function(e) {
            const password = e.target.value;
            // You can add password strength indicator here
        });

        // Confirm password match validation
        document.getElementById('confirm_password')?.addEventListener('input', function(e) {
            const password = document.getElementById('password').value;
            const confirmPassword = e.target.value;
            
            if (confirmPassword && password !== confirmPassword) {
                e.target.setCustomValidity('Passwords do not match');
            } else {
                e.target.setCustomValidity('');
            }
        });
    </script>
</body>
</html>

