<?php
require_once 'includes/auth-check.php';

// Require registered user for Royal Rewards (not guest users)
requireRegisteredUser();

require_once 'models/Loyalty.php';
require_once 'admin/includes/Voucher.php';

$userId = getUserId();

$loyalty = new Loyalty();
$voucherModel = new Voucher();

$loyaltyData = $loyalty->getUserLoyaltyData($userId);
$stampsData = $loyalty->getUserStamps($userId);
$referralCode = $loyalty->getUserReferralCode($userId);
$recentTransactions = $loyalty->getRecentTransactions($userId, 5);
$nextRewardPoints = $loyalty->getNextRewardPoints($userId);

// Get available vouchers for user
$availableVouchers = $voucherModel->getAvailableVouchers($userId, $loyaltyData['points']);
$userRoyalTier = $voucherModel->getUserRoyalTier($loyaltyData['points']);

// Get user's claimed vouchers (both active and used)
$claimedVouchers = $voucherModel->getUserRedeemedVouchers($userId);

// If no loyalty data exists, initialize it
if (!$loyaltyData) {
    $loyalty->initializeLoyaltyAccount($userId);
    $loyaltyData = $loyalty->getUserLoyaltyData($userId);
}

// If no referral code exists, create one
if (!$referralCode) {
    $referralCode = $loyalty->createReferralCode($userId);
    $referralCode = $loyalty->getUserReferralCode($userId);
}

// Handle referral code sharing
if (isset($_POST['share_referral'])) {
    $referralCodeValue = $referralCode['referral_code'];
    // Here you would typically send the referral code via email, SMS, or copy to clipboard
    $shareMessage = "Your referral code: " . $referralCodeValue;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Royalty & Rewards - Perfume Store</title>
    <link rel="stylesheet" href="styles.css">
    <link rel="stylesheet" href="rewards.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="script.js" defer></script>
</head>
<body>
    <!-- Top Navigation -->
    <nav class="top-nav">
        <div class="nav-left">
            <i class="fas fa-arrow-left" onclick="window.location.href='index.php'"></i>
        </div>
        <div class="nav-center">
            <div class="logo">
                <img src="images/LOGO.png" alt="2 Days Parfume Logo" class="logo-img">
            </div>
        </div>
        <div class="nav-right">
            <i class="fas fa-box" onclick="window.location.href='products.php'"></i>
        </div>
    </nav>

    <!-- Main Content -->
    <main class="main-content">
        <!-- Loyalty Balance Card -->
        <section class="loyalty-balance-card">
            <div class="balance-header">
                <h2>Loyalty Balance</h2>
            </div>
            <div class="balance-content">
                <div class="balance-info">
                    <div class="points-display"><?php echo number_format($loyaltyData['points']); ?>pts</div>
                    <div class="next-reward"><?php echo number_format($nextRewardPoints); ?> points till your next reward</div>
                </div>
            </div>
            <div class="balance-divider"></div>
            <div class="user-info">
                <div class="user-name"><?php echo htmlspecialchars($loyaltyData['first_name'] . ' ' . $loyaltyData['last_name']); ?></div>
                <div class="user-id"><?php echo strtoupper(substr($loyaltyData['first_name'], 0, 4)) . ' ' . substr($loyaltyData['id'], -4); ?></div>
            </div>
        </section>

        <!-- Royal Tier Card -->
        <?php if ($userRoyalTier): ?>
        <section class="royal-tier-card">
            <div class="tier-header">
                <div class="tier-icon" style="background: <?php echo $userRoyalTier['tier_color']; ?>">
                    <i class="<?php echo $userRoyalTier['tier_icon']; ?>"></i>
                </div>
                <div class="tier-info">
                    <h3><?php echo $userRoyalTier['tier_name']; ?> Member</h3>
                    <p><?php echo $userRoyalTier['benefits']; ?></p>
                </div>
            </div>
        </section>
        <?php endif; ?>

        <!-- Available Vouchers Card -->
        <?php if (!empty($availableVouchers)): ?>
        <section class="vouchers-card">
            <h3>Available Vouchers</h3>
            <div class="vouchers-grid">
                <?php foreach ($availableVouchers as $voucher): ?>
                    <div class="voucher-item">
                        <?php if (!empty($voucher['voucher_image'])): ?>
                            <div class="voucher-image-container">
                                <img src="<?php echo htmlspecialchars($voucher['voucher_image'] ?? ''); ?>" alt="Voucher">
                                <div class="voucher-overlay">
                                    <div class="voucher-overlay-title"><?php echo htmlspecialchars($voucher['voucher_name'] ?? ''); ?></div>
                                    <div class="voucher-overlay-description"><?php echo htmlspecialchars($voucher['voucher_description'] ?? ''); ?></div>
                                    <div class="voucher-overlay-details">
                                        <span class="voucher-overlay-points"><?php echo number_format($voucher['points_required'] ?? 0); ?> pts</span>
                                        <span class="voucher-overlay-discount">
                                            <?php 
                                            if (($voucher['discount_type'] ?? '') === 'percentage') {
                                                echo ($voucher['discount_value'] ?? 0) . '% OFF';
                                            } elseif (($voucher['discount_type'] ?? '') === 'fixed_amount') {
                                                echo 'RM ' . ($voucher['discount_value'] ?? 0) . ' OFF';
                                            } else {
                                                echo ucfirst(str_replace('_', ' ', $voucher['discount_type'] ?? ''));
                                            }
                                            ?>
                                        </span>
                                    </div>
                                    <button class="voucher-overlay-btn" onclick="redeemVoucher(<?php echo $voucher['id'] ?? 0; ?>)">
                                        <i class="fas fa-ticket-alt"></i>
                                        Redeem
                                    </button>
                                </div>
                            </div>
                        <?php else: ?>
                            <div class="voucher-image-container" style="background: linear-gradient(135deg, #1A4D2A, #2d5a3d); display: flex; flex-direction: column; justify-content: space-between; padding: 20px;">
                                <div style="text-align: center; color: white; flex: 1; display: flex; flex-direction: column; justify-content: center;">
                                    <i class="fas fa-ticket-alt" style="font-size: 3rem; margin-bottom: 15px; opacity: 0.7;"></i>
                                    <div style="font-size: 1.3rem; font-weight: 700; margin-bottom: 8px;"><?php echo htmlspecialchars($voucher['voucher_name'] ?? ''); ?></div>
                                    <div style="font-size: 0.9rem; opacity: 0.9; margin-bottom: 20px;"><?php echo htmlspecialchars($voucher['voucher_description'] ?? ''); ?></div>
                                </div>
                                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px; gap: 10px;">
                                    <span style="background: linear-gradient(135deg, #FDD835, #FBC02D); color: #333; padding: 6px 12px; border-radius: 20px; font-size: 0.8rem; font-weight: 600; box-shadow: 0 2px 8px rgba(0, 0, 0, 0.3);"><?php echo number_format($voucher['points_required'] ?? 0); ?> pts</span>
                                    <span style="background: linear-gradient(135deg, #28a745, #20c997); color: white; padding: 6px 12px; border-radius: 20px; font-size: 0.8rem; font-weight: 600; box-shadow: 0 2px 8px rgba(0, 0, 0, 0.3);">
                                        <?php 
                                        if (($voucher['discount_type'] ?? '') === 'percentage') {
                                            echo ($voucher['discount_value'] ?? 0) . '% OFF';
                                        } elseif (($voucher['discount_type'] ?? '') === 'fixed_amount') {
                                            echo 'RM ' . ($voucher['discount_value'] ?? 0) . ' OFF';
                                        } else {
                                            echo ucfirst(str_replace('_', ' ', $voucher['discount_type'] ?? ''));
                                        }
                                        ?>
                                    </span>
                                </div>
                                <button style="width: 100%; background: linear-gradient(135deg, #28a745, #20c997); color: white; border: none; padding: 10px; border-radius: 8px; font-weight: 600; cursor: pointer; transition: all 0.3s ease; display: flex; align-items: center; justify-content: center; gap: 6px; font-size: 0.9rem; box-shadow: 0 2px 8px rgba(0, 0, 0, 0.3);" onclick="redeemVoucher(<?php echo $voucher['id'] ?? 0; ?>)">
                                    <i class="fas fa-ticket-alt"></i>
                                    Redeem
                                </button>
                            </div>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>
            </div>
        </section>
        <?php endif; ?>

        <!-- Claimed Vouchers Card -->
        <?php if (!empty($claimedVouchers)): ?>
        <section class="vouchers-card">
            <h3>My Claimed Vouchers</h3>
            <div class="vouchers-grid">
                <?php foreach ($claimedVouchers as $voucher): ?>
                    <div class="voucher-item voucher-status-<?php echo $voucher['status']; ?>">
                        <?php if (!empty($voucher['voucher_image'])): ?>
                            <div class="voucher-image-container">
                                <img src="<?php echo htmlspecialchars($voucher['voucher_image'] ?? ''); ?>" alt="Voucher">
                                <div class="voucher-overlay claimed-overlay">
                                    <div class="voucher-overlay-title"><?php echo htmlspecialchars($voucher['voucher_name'] ?? ''); ?></div>
                                    <div class="voucher-overlay-description"><?php echo htmlspecialchars($voucher['voucher_description'] ?? ''); ?></div>
                                    <div class="voucher-overlay-details">
                                        <span class="voucher-overlay-discount">
                                            <?php 
                                            if (($voucher['discount_type'] ?? '') === 'percentage') {
                                                echo ($voucher['discount_value'] ?? 0) . '% OFF';
                                            } elseif (($voucher['discount_type'] ?? '') === 'fixed_amount') {
                                                echo 'RM ' . ($voucher['discount_value'] ?? 0) . ' OFF';
                                            } else {
                                                echo ucfirst(str_replace('_', ' ', $voucher['discount_type'] ?? ''));
                                            }
                                            ?>
                                        </span>
                                    </div>
                                    <div class="claimed-date">
                                        <i class="fas fa-calendar"></i>
                                        Claimed: <?php echo date('M d, Y', strtotime($voucher['redeemed_at'] ?? 'now')); ?>
                                    </div>
                                    <div class="voucher-status">
                                        <?php if ($voucher['status'] === 'active'): ?>
                                            <i class="fas fa-check-circle"></i>
                                            <span>Ready to Use</span>
                                        <?php elseif ($voucher['status'] === 'used'): ?>
                                            <i class="fas fa-times-circle"></i>
                                            <span>Used</span>
                                        <?php else: ?>
                                            <i class="fas fa-clock"></i>
                                            <span><?php echo ucfirst($voucher['status']); ?></span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        <?php else: ?>
                            <div class="voucher-image-container" style="background: linear-gradient(135deg, #1A4D2A, #2d5a3d); display: flex; flex-direction: column; justify-content: space-between; padding: 20px;">
                                <div style="text-align: center; color: white; flex: 1; display: flex; flex-direction: column; justify-content: center;">
                                    <i class="fas fa-ticket-alt" style="font-size: 3rem; margin-bottom: 15px; opacity: 0.7;"></i>
                                    <div style="font-size: 1.3rem; font-weight: 700; margin-bottom: 8px;"><?php echo htmlspecialchars($voucher['voucher_name'] ?? ''); ?></div>
                                    <div style="font-size: 0.9rem; opacity: 0.9; margin-bottom: 20px;"><?php echo htmlspecialchars($voucher['voucher_description'] ?? ''); ?></div>
                                </div>
                                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px; gap: 10px;">
                                    <span style="background: linear-gradient(135deg, #28a745, #20c997); color: white; padding: 6px 12px; border-radius: 20px; font-size: 0.8rem; font-weight: 600; box-shadow: 0 2px 8px rgba(0, 0, 0, 0.3);">
                                        <?php 
                                        if (($voucher['discount_type'] ?? '') === 'percentage') {
                                            echo ($voucher['discount_value'] ?? 0) . '% OFF';
                                        } elseif (($voucher['discount_type'] ?? '') === 'fixed_amount') {
                                            echo 'RM ' . ($voucher['discount_value'] ?? 0) . ' OFF';
                                        } else {
                                            echo ucfirst(str_replace('_', ' ', $voucher['discount_type'] ?? ''));
                                        }
                                        ?>
                                    </span>
                                </div>
                                <div style="background: rgba(255, 255, 255, 0.2); color: white; padding: 8px 12px; border-radius: 15px; font-size: 0.8rem; margin-bottom: 15px; display: flex; align-items: center; justify-content: center; gap: 6px; backdrop-filter: blur(10px);">
                                    <i class="fas fa-calendar"></i>
                                    Claimed: <?php echo date('M d, Y', strtotime($voucher['redeemed_at'] ?? 'now')); ?>
                                </div>
                                <div class="voucher-status-inline voucher-status-<?php echo $voucher['status']; ?>" style="background: rgba(255, 255, 255, 0.2); color: white; padding: 10px; border-radius: 8px; font-weight: 600; display: flex; align-items: center; justify-content: center; gap: 6px; font-size: 0.9rem; backdrop-filter: blur(10px);">
                                    <?php if ($voucher['status'] === 'active'): ?>
                                        <i class="fas fa-check-circle"></i>
                                        Ready to Use
                                    <?php elseif ($voucher['status'] === 'used'): ?>
                                        <i class="fas fa-times-circle"></i>
                                        Used
                                    <?php else: ?>
                                        <i class="fas fa-clock"></i>
                                        <?php echo ucfirst($voucher['status']); ?>
                                    <?php endif; ?>
                                </div>
                            </div>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>
            </div>
        </section>
        <?php endif; ?>

        <!-- Enter Code Card -->
        <section class="enter-code-card">
            <div class="code-header">
                <h3><i class="fas fa-gift"></i> Enter Code</h3>
                <p>Enter a code to earn bonus points!</p>
            </div>
            <div class="code-form">
                <div class="input-group">
                    <input type="text" id="pointCode" placeholder="Enter your code here..." maxlength="50">
                    <button type="button" id="submitCode" onclick="submitPointCode()">
                        <i class="fas fa-paper-plane"></i>
                        Submit
                    </button>
                </div>
                <div id="codeMessage" class="code-message" style="display: none;"></div>
            </div>
        </section>

        <!-- Give and Get Points Card -->
        <!-- <section class="referral-card">
            <h3>Give and get points</h3>
            <p>When a friend uses your code, you will get 3,000 points.</p>
            <form method="POST" style="display: inline;">
                <button type="submit" name="share_referral" class="share-button">
                    <i class="fas fa-share"></i>
                    Share invite code
                </button>
            </form>
            <?php if (isset($shareMessage)): ?>
                <div class="referral-code-display">
                    <strong>Your Referral Code: <?php echo htmlspecialchars($referralCode['referral_code']); ?></strong>
                </div>
            <?php endif; ?>
        </section> -->

        <!-- Recent Activity -->
        <section class="recent-activity">
            <h3>Recent Activity</h3>
            
            <?php if (!empty($recentTransactions)): ?>
                <?php foreach ($recentTransactions as $transaction): ?>
                    <div class="activity-item">
                        <div class="activity-logo">
                            <?php 
                            $icon = 'fas fa-shopping-bag';
                            if ($transaction['transaction_type'] == 'bonus') {
                                $icon = 'fas fa-gift';
                            } elseif ($transaction['transaction_type'] == 'redeemed') {
                                $icon = 'fas fa-ticket-alt';
                            }
                            ?>
                            <i class="<?php echo $icon; ?>"></i>
                        </div>
                        <div class="activity-info">
                            <div class="activity-title">
                                You <?php echo $transaction['transaction_type'] == 'earned' ? 'earned' : $transaction['transaction_type']; ?> 
                                <?php echo number_format($transaction['points']); ?> points
                            </div>
                            <div class="activity-subtitle"><?php echo htmlspecialchars($transaction['description']); ?></div>
                        </div>
                        <div class="activity-points">
                            <?php echo $transaction['transaction_type'] == 'earned' || $transaction['transaction_type'] == 'bonus' ? '+' : '-'; ?><?php echo number_format($transaction['points']); ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php else: ?>
                <div class="no-activity">
                    <p>No recent activity. Start shopping to earn points!</p>
                </div>
            <?php endif; ?>
        </section>
    </main>

    <!-- Bottom Navigation -->
    <nav class="bottom-nav">
        <div class="nav-item" onclick="window.location.href='index.php'">
            <i class="fas fa-home"></i>
        </div>
        <div class="nav-item" onclick="window.location.href='products.php'">
            <i class="fas fa-box"></i>
        </div>
        <div class="nav-item" onclick="window.location.href='cart.php'">
            <i class="fas fa-shopping-cart"></i>
        </div>
        <div class="nav-item active">
            <i class="fas fa-crown"></i>
        </div>
    </nav>

    <script>

        // Voucher redemption function
        function redeemVoucher(voucherId) {
            showRedeemConfirmation(voucherId);
        }

        function showRedeemConfirmation(voucherId) {
            const modalOverlay = document.createElement('div');
            modalOverlay.className = 'redeem-confirmation-overlay';
            modalOverlay.style.cssText = `
                position: fixed; top: 0; left: 0; width: 100%; height: 100%;
                background: rgba(0, 0, 0, 0.6); display: flex; align-items: center;
                justify-content: center; z-index: 10000; animation: fadeIn 0.3s ease;
                padding: 20px; box-sizing: border-box;
            `;

            modalOverlay.innerHTML = `
                <div class="redeem-confirmation-modal" style="
                    background: white; border-radius: 16px; padding: 0; max-width: 400px;
                    width: 100%; box-shadow: 0 20px 40px rgba(0,0,0,0.3);
                    animation: slideInUp 0.3s ease; overflow: hidden;
                ">
                    <div class="modal-header" style="
                        background: linear-gradient(135deg, #1A4D2A, #2d5a3d);
                        color: white; padding: 20px; text-align: center;
                    ">
                        <i class="fas fa-ticket-alt" style="font-size: 2rem; margin-bottom: 10px; display: block;"></i>
                        <h3 style="margin: 0; font-size: 1.3rem; font-weight: 600;">Redeem Voucher</h3>
                    </div>
                    <div class="modal-content" style="padding: 25px; text-align: center;">
                        <p style="margin: 0 0 20px 0; color: #666; line-height: 1.5;">
                            Are you sure you want to redeem this voucher?<br>
                            <strong style="color: #333;">This action cannot be undone.</strong>
                        </p>
                        <div class="modal-actions" style="
                            display: flex; gap: 12px; justify-content: center;
                        ">
                            <button class="btn-cancel" style="
                                background: #f8f9fa; color: #666; border: 1px solid #ddd;
                                padding: 12px 24px; border-radius: 8px; font-weight: 600;
                                cursor: pointer; transition: all 0.3s ease; flex: 1;
                            ">Cancel</button>
                            <button class="btn-confirm" style="
                                background: linear-gradient(135deg, #28a745, #20c997);
                                color: white; border: none; padding: 12px 24px;
                                border-radius: 8px; font-weight: 600; cursor: pointer;
                                transition: all 0.3s ease; flex: 1;
                            ">Redeem Now</button>
                        </div>
                    </div>
                </div>
            `;

            document.body.appendChild(modalOverlay);
            document.body.style.overflow = 'hidden';

            // Add animation styles
            if (!document.getElementById('redeem-modal-styles')) {
                const style = document.createElement('style');
                style.id = 'redeem-modal-styles';
                style.textContent = `
                    @keyframes fadeIn { from { opacity: 0; } to { opacity: 1; } }
                    @keyframes slideInUp { from { transform: translateY(30px); opacity: 0; } to { transform: translateY(0); opacity: 1; } }
                    .btn-cancel:hover { background: #e9ecef !important; }
                    .btn-confirm:hover { background: linear-gradient(135deg, #218838, #1ea085) !important; transform: translateY(-1px); }
                `;
                document.head.appendChild(style);
            }

            // Event listeners
            modalOverlay.querySelector('.btn-cancel').addEventListener('click', () => {
                closeRedeemModal(modalOverlay);
            });

            modalOverlay.querySelector('.btn-confirm').addEventListener('click', () => {
                proceedWithRedemption(voucherId, modalOverlay);
            });

            // Close on overlay click
            modalOverlay.addEventListener('click', (e) => {
                if (e.target === modalOverlay) {
                    closeRedeemModal(modalOverlay);
                }
            });

            // Close on ESC key
            const handleEsc = (e) => {
                if (e.key === 'Escape') {
                    closeRedeemModal(modalOverlay);
                    document.removeEventListener('keydown', handleEsc);
                }
            };
            document.addEventListener('keydown', handleEsc);
        }

        function closeRedeemModal(modalOverlay) {
            modalOverlay.style.animation = 'fadeOut 0.3s ease';
            setTimeout(() => {
                if (modalOverlay.parentNode) {
                    modalOverlay.parentNode.removeChild(modalOverlay);
                }
                document.body.style.overflow = '';
            }, 300);
        }

        function proceedWithRedemption(voucherId, modalOverlay) {
            const confirmBtn = modalOverlay.querySelector('.btn-confirm');
            const originalText = confirmBtn.innerHTML;
            
            // Show loading state
            confirmBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Redeeming...';
            confirmBtn.disabled = true;
            
            fetch('ajax/redeem-voucher.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    voucher_id: voucherId
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Show success state
                    confirmBtn.innerHTML = '<i class="fas fa-check"></i> Redeemed!';
                    confirmBtn.style.background = 'linear-gradient(135deg, #28a745, #20c997)';
                    
                    setTimeout(() => {
                        closeRedeemModal(modalOverlay);
                        location.reload(); // Refresh page to update points and vouchers
                    }, 1500);
            } else {
                    // Show error state
                    confirmBtn.innerHTML = '<i class="fas fa-times"></i> Error';
                    confirmBtn.style.background = '#dc3545';
                    
                    setTimeout(() => {
                        confirmBtn.innerHTML = originalText;
                        confirmBtn.disabled = false;
                        confirmBtn.style.background = 'linear-gradient(135deg, #28a745, #20c997)';
                    }, 2000);
                    
                    // Show error message
                    showNotification('Error: ' + (data.message || 'Failed to redeem voucher'), 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                confirmBtn.innerHTML = '<i class="fas fa-times"></i> Error';
                confirmBtn.style.background = '#dc3545';
                
                setTimeout(() => {
                    confirmBtn.innerHTML = originalText;
                    confirmBtn.disabled = false;
                    confirmBtn.style.background = 'linear-gradient(135deg, #28a745, #20c997)';
                }, 2000);
                
                showNotification('Error redeeming voucher. Please try again.', 'error');
            });
        }

        function showNotification(message, type = 'info') {
            const notification = document.createElement('div');
            notification.className = `notification notification-${type}`;
            notification.style.cssText = `
                position: fixed; top: 20px; right: 20px; z-index: 10001;
                background: ${type === 'error' ? '#fee' : type === 'success' ? '#f0fff4' : '#e3f2fd'};
                color: ${type === 'error' ? '#c53030' : type === 'success' ? '#2f855a' : '#1976d2'};
                border: 1px solid ${type === 'error' ? '#feb2b2' : type === 'success' ? '#9ae6b4' : '#bbdefb'};
                padding: 12px 16px; border-radius: 8px; display: flex;
                align-items: center; gap: 10px; font-size: 14px; font-weight: 500;
                animation: slideInRight 0.3s ease; max-width: 300px;
                box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            `;
            
            notification.innerHTML = `
                <i class="fas fa-${type === 'error' ? 'exclamation-circle' : type === 'success' ? 'check-circle' : 'info-circle'}"></i>
                <span>${message}</span>
            `;
            
            document.body.appendChild(notification);
            
            setTimeout(() => {
                notification.style.animation = 'slideOutRight 0.3s ease';
                setTimeout(() => {
                    if (notification.parentNode) {
                        notification.parentNode.removeChild(notification);
                    }
                }, 300);
            }, 3000);
        }

        // Point code submission function
        function submitPointCode() {
            const codeInput = document.getElementById('pointCode');
            const submitBtn = document.getElementById('submitCode');
            const messageDiv = document.getElementById('codeMessage');
            
            const code = codeInput.value.trim().toUpperCase();
            
            if (!code) {
                showCodeMessage('Please enter a code', 'error');
                return;
            }
            
            // Show loading state
            submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Submitting...';
            submitBtn.disabled = true;
            
            fetch('ajax/enter-point-code.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    code: code
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showCodeMessage(`Success! You earned ${data.points} points!`, 'success');
                    codeInput.value = '';
                    // Refresh the page to update points display
                    setTimeout(() => {
                        location.reload();
                    }, 2000);
                } else {
                    showCodeMessage(data.message || 'Invalid code', 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showCodeMessage('Error submitting code. Please try again.', 'error');
            })
            .finally(() => {
                // Reset button state
                submitBtn.innerHTML = '<i class="fas fa-paper-plane"></i> Submit';
                submitBtn.disabled = false;
            });
        }

        function showCodeMessage(message, type) {
            const messageDiv = document.getElementById('codeMessage');
            messageDiv.textContent = message;
            messageDiv.className = `code-message ${type}`;
            messageDiv.style.display = 'block';
            
            // Auto-hide after 5 seconds
            setTimeout(() => {
                messageDiv.style.display = 'none';
            }, 5000);
        }

        // Allow Enter key to submit code
        document.getElementById('pointCode').addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                submitPointCode();
            }
        });

    </script>
</body>
</html>