// Mobile App JavaScript Functionality

document.addEventListener('DOMContentLoaded', function() {
    // Initialize the app
    initializeApp();
});

function initializeApp() {
    // Add smooth scrolling behavior
    addSmoothScrolling();
    
    // Initialize navigation interactions
    initializeNavigation();
    
    // Add product card interactions
    initializeProductCards();
    
    // Add search functionality
    initializeSearch();
    
    // Add responsive behavior
    initializeResponsive();
    
    // Initialize horizontal scroll containers
    // initializeHorizontalScrollContainers();
    
    // Initialize cart badge on load
    updateCartBadge();
    
    // Initialize authentication
    initializeAuth();
    
    // Initialize banner slider
loadBanners();
    
    // Initialize restock timers after a short delay to ensure DOM is ready
    setTimeout(initializeRestockTimers, 500);
    
    // Also try to initialize when window loads (fallback)
    window.addEventListener('load', function() {
        setTimeout(initializeRestockTimers, 100);
    });
    
    // Protected navigation is handled directly in onclick handlers
}

function addSmoothScrolling() {
    // Smooth scrolling for anchor links
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', function (e) {
            e.preventDefault();
            const target = document.querySelector(this.getAttribute('href'));
            if (target) {
                target.scrollIntoView({
                    behavior: 'smooth',
                    block: 'start'
                });
            }
        });
    });
}

function initializeNavigation() {
    // Bottom navigation active state
    const navItems = document.querySelectorAll('.nav-item');
    
    navItems.forEach(item => {
        item.addEventListener('click', function() {
            // Remove active class from all items
            navItems.forEach(nav => nav.classList.remove('active'));
            
            // Add active class to clicked item
            this.classList.add('active');
            
            // Add haptic feedback simulation
            if (navigator.vibrate) {
                navigator.vibrate(50);
            }
        });
    });
    
    // Top navigation interactions
    const hamburgerMenu = document.getElementById('hamburger-menu');
    const searchIcon = document.querySelector('.nav-right i');
    const mobileMenuOverlay = document.getElementById('mobile-menu-overlay');
    const mobileMenuClose = document.getElementById('mobile-menu-close');
    
    if (hamburgerMenu && mobileMenuOverlay) {
        hamburgerMenu.addEventListener('click', function() {
            mobileMenuOverlay.classList.add('active');
            document.body.style.overflow = 'hidden'; // Prevent background scrolling
        });
    }
    
    if (mobileMenuClose) {
        mobileMenuClose.addEventListener('click', function() {
            closeMobileMenu();
        });
    }
    
    if (mobileMenuOverlay) {
        mobileMenuOverlay.addEventListener('click', function(e) {
            if (e.target === mobileMenuOverlay) {
                closeMobileMenu();
            }
        });
    }
    
    function closeMobileMenu() {
        if (mobileMenuOverlay) {
            mobileMenuOverlay.classList.remove('active');
            document.body.style.overflow = ''; // Restore scrolling
        }
    }
    
    // Handle search icon click
    
    
    // Handle box icon click (products page redirect)
    const boxIcon = document.querySelector('.fa-box');
    if (boxIcon) {
        boxIcon.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            window.location.href = 'products.php';
        });
    }
}

function initializeProductCards() {
    const productCards = document.querySelectorAll('.product-card');
    
    productCards.forEach(card => {
        // Add click interaction
        card.addEventListener('click', function() {
            const productName = this.querySelector('.product-name').textContent;
            showNotification(`Viewing ${productName}`);
            
            // Add visual feedback
            this.style.transform = 'scale(0.98)';
            setTimeout(() => {
                this.style.transform = '';
            }, 150);
        });
        
        // Add hover effects for desktop
        card.addEventListener('mouseenter', function() {
            if (window.innerWidth > 768) {
                this.style.transform = 'translateY(-4px)';
            }
        });
        
        card.addEventListener('mouseleave', function() {
            if (window.innerWidth > 768) {
                this.style.transform = '';
            }
        });
    });
}

function initializeSearch() {
    // Search functionality placeholder
    const searchIcon = document.querySelector('.nav-right i');
    
    if (searchIcon) {
        searchIcon.addEventListener('click', function() {
            // Create search overlay
            createSearchOverlay();
        });
    }
}

function createSearchOverlay() {
    // Remove existing overlay if any
    const existingOverlay = document.querySelector('.search-overlay');
    if (existingOverlay) {
        existingOverlay.remove();
        return;
    }
    
   
    
    // Add styles
    overlay.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0, 0, 0, 0.8);
        z-index: 1000;
        display: flex;
        align-items: flex-start;
        justify-content: center;
        padding-top: 50px;
    `;
    
    const searchContent = overlay.querySelector('.search-content');
    searchContent.style.cssText = `
        background: white;
        border-radius: 12px;
        padding: 20px;
        width: 90%;
        max-width: 400px;
        box-shadow: 0 10px 30px rgba(0, 0, 0, 0.3);
    `;
    
    const searchInput = overlay.querySelector('.search-input');
    searchInput.style.cssText = `
        width: 100%;
        padding: 12px;
        border: 1px solid #ddd;
        border-radius: 8px;
        font-size: 16px;
        margin-bottom: 15px;
    `;
    
    const searchClose = overlay.querySelector('.search-close');
    searchClose.style.cssText = `
        position: absolute;
        top: 10px;
        right: 10px;
        background: none;
        border: none;
        font-size: 24px;
        cursor: pointer;
        color: #666;
    `;
    
    // Add event listeners
    searchClose.addEventListener('click', () => overlay.remove());
    overlay.addEventListener('click', (e) => {
        if (e.target === overlay) overlay.remove();
    });
    
    // Focus on input
    searchInput.focus();
    
    document.body.appendChild(overlay);
}

function initializeResponsive() {
    // Handle window resize
    let resizeTimeout;
    window.addEventListener('resize', function() {
        clearTimeout(resizeTimeout);
        resizeTimeout = setTimeout(handleResize, 250);
    });
    
    // Handle orientation change
    window.addEventListener('orientationchange', function() {
        setTimeout(handleResize, 500);
    });
}

function handleResize() {
    // Adjust layout for different screen sizes
    const width = window.innerWidth;
    
    if (width < 480) {
        // Mobile optimizations
        document.body.classList.add('mobile-view');
    } else {
        document.body.classList.remove('mobile-view');
    }
}

function showNotification(message) {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = 'notification';
    notification.textContent = message;
    
    // Style the notification
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        left: 50%;
        transform: translateX(-50%);
        background: #333;
        color: white;
        padding: 12px 24px;
        border-radius: 25px;
        font-size: 14px;
        z-index: 1001;
        opacity: 0;
        transition: opacity 0.3s ease;
    `;
    
    document.body.appendChild(notification);
    
    // Animate in
    setTimeout(() => {
        notification.style.opacity = '1';
    }, 10);
    
    // Remove after 3 seconds
    setTimeout(() => {
        notification.style.opacity = '0';
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 300);
    }, 3000);
}

// ----------------------
// Cart Management
// ----------------------
function testDatabase() {
    fetch('auth-handler.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: 'action=test_db'
    })
    .then(response => response.json())
    .then(data => {
        console.log('Database test result:', data);
        if (data.success) {
            showNotification('Database connection working!');
        } else {
            showNotification('Database error: ' + data.message, 'error');
        }
    })
    .catch(error => {
        console.error('Database test error:', error);
        showNotification('Database test failed', 'error');
    });
}

function addToCart(product, callback) {
    if (!product || !product.id) {
        console.error('Invalid product data');
        showNotification('Invalid product data', 'error');
        return;
    }
    
    // Extract price as number - handle both "RM 89.00" and "89.00" formats
    let price;
    if (typeof product.price === 'string') {
        price = parseFloat(product.price.replace(/[^\d.-]/g, ''));
    } else {
        price = parseFloat(product.price);
    }
    
    if (isNaN(price)) {
        console.error('Invalid price:', product.price);
        showNotification('Invalid price format', 'error');
        return;
    }
    
    console.log('Adding to cart:', { 
        id: product.id, 
        name: product.name, 
        price: price, 
        version: product.version || 'none',
        size: product.size || 'none',
        qty: product.qty || 1 
    });
    
    const formData = new FormData();
    formData.append('action', 'add_to_cart');
    formData.append('product_id', product.id);
    formData.append('quantity', product.qty || 1);
    formData.append('size', product.size || '');
    formData.append('version', product.version || '');
    formData.append('price', price);
    
    fetch('auth-handler.php', {
        method: 'POST',
        body: formData
    })
    .then(response => {
        console.log('Response status:', response.status);
        console.log('Response headers:', response.headers);
        return response.text(); // Get response as text first
    })
    .then(text => {
        console.log('Raw response:', text);
        try {
            const data = JSON.parse(text);
            console.log('Add to cart response:', data);
            if (data.success) {
                updateCartBadge(data.cart_count);
                showNotification('Added to cart!');
                
                // Call callback with cart item ID if provided
                if (typeof callback === 'function' && data.cart_item_id) {
                    callback(data.cart_item_id);
                }
            } else {
                // Check if we need to redirect to signup
                if (data.redirect === 'signup.php') {
                    showNotification('Please sign up to add items to cart', 'info');
                    setTimeout(() => {
                        window.location.href = 'signup.php';
                    }, 1500);
                } else {
                    showNotification(data.message || 'Failed to add item to cart', 'error');
                }
            }
        } catch (e) {
            console.error('JSON parse error:', e);
            console.error('Response was not JSON:', text);
            showNotification('Server returned invalid response', 'error');
        }
    })
    .catch(error => {
        console.error('Error adding to cart:', error);
        showNotification('Error adding to cart', 'error');
    });
}

function updateCartQuantity(cartItemId, quantity) {
    const formData = new FormData();
    formData.append('action', 'update_cart');
    formData.append('cart_item_id', cartItemId);
    formData.append('quantity', quantity);
    
    fetch('auth-handler.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            updateCartBadge(data.cart_count);
            // Reload cart page if we're on it
            if (window.location.pathname.includes('cart.php')) {
                loadCartItems();
            }
        } else {
            showNotification('Failed to update cart', 'error');
        }
    })
    .catch(error => {
        console.error('Error updating cart:', error);
        showNotification('Error updating cart', 'error');
    });
}

function removeFromCart(cartItemId) {
    const formData = new FormData();
    formData.append('action', 'remove_from_cart');
    formData.append('cart_item_id', cartItemId);
    
    fetch('auth-handler.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            updateCartBadge(data.cart_count);
            // Reload cart page if we're on it
            if (window.location.pathname.includes('cart.php')) {
                loadCartItems();
            }
        } else {
            showNotification('Failed to remove item from cart', 'error');
        }
    })
    .catch(error => {
        console.error('Error removing from cart:', error);
        showNotification('Error removing from cart', 'error');
    });
}

function loadCartItems() {
    fetch('auth-handler.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: 'action=get_cart'
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            displayCartItems(data.items, data.total);
        } else {
            console.error('Failed to load cart items');
        }
    })
    .catch(error => {
        console.error('Error loading cart items:', error);
    });
}

function displayCartItems(items, total) {
    const container = document.getElementById('cart-items-container');
    const emptyCart = document.getElementById('empty-cart');
    const cartSummary = document.getElementById('cart-summary');
    
    if (!container) return;
    
    if (items.length === 0) {
        container.style.display = 'none';
        if (emptyCart) emptyCart.style.display = 'block';
        if (cartSummary) cartSummary.style.display = 'none';
        return;
    }
    
    container.style.display = 'block';
    if (emptyCart) emptyCart.style.display = 'none';
    if (cartSummary) cartSummary.style.display = 'block';
    
    container.innerHTML = '';
    
    items.forEach((item, index) => {
        const cartItem = document.createElement('div');
        cartItem.className = 'cart-item';
        cartItem.innerHTML = `
            <div class="cart-item-image">
                <img src="${item.image}" alt="${item.name}" onerror="this.src='data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMTAwIiBoZWlnaHQ9IjEwMCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cmVjdCB3aWR0aD0iMTAwIiBoZWlnaHQ9IjEwMCIgZmlsbD0iI2Y0ZjRmNCIvPjx0ZXh0IHg9IjUwIiB5PSI1MCIgZm9udC1mYW1pbHk9IkFyaWFsIiBmb250LXNpemU9IjE0IiBmaWxsPSIjOTk5IiB0ZXh0LWFuY2hvcj0ibWlkZGxlIj5JbWFnZTwvdGV4dD48L3N2Zz4='">
            </div>
            <div class="cart-item-details">
                <h3 class="cart-item-name">${item.name}</h3>
                ${item.size ? `<p class="cart-item-size">Size: ${item.size}</p>` : ''}
                <p class="cart-item-price">${item.currency || 'RM'}${parseFloat(item.price).toFixed(2)}</p>
            </div>
            <div class="cart-item-controls">
                <div class="quantity-controls">
                    <button class="quantity-btn" onclick="updateCartQuantity(${parseInt(item.id)}, ${item.quantity - 1})">
                        <i class="fas fa-minus"></i>
                    </button>
                    <span class="quantity">${item.quantity}</span>
                    <button class="quantity-btn" onclick="updateCartQuantity(${parseInt(item.id)}, ${item.quantity + 1})">
                        <i class="fas fa-plus"></i>
                    </button>
                </div>
                <button class="remove-btn" onclick="removeFromCart(${parseInt(item.id)})">
                    <i class="fas fa-trash"></i>
                </button>
            </div>
        `;
        container.appendChild(cartItem);
    });
    
    if (total) {
        const subtotalEl = document.getElementById('subtotal');
        const totalEl = document.getElementById('total');
        if (subtotalEl) subtotalEl.textContent = `${total.currency || 'RM'}${total.total ? parseFloat(total.total).toFixed(2) : '0.00'}`;
        if (totalEl) totalEl.textContent = `${total.currency || 'RM'}${total.total ? parseFloat(total.total).toFixed(2) : '0.00'}`;
    }
}

function updateCartBadge(count) {
    if (count !== undefined) {
        document.querySelectorAll('.cart-count').forEach(el => {
            el.textContent = String(count);
            el.style.display = count > 0 ? 'inline-block' : 'none';
        });
    } else {
        // Load count from server
        fetch('auth-handler.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: 'action=get_cart'
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const count = data.items ? data.items.length : 0;
                document.querySelectorAll('.cart-count').forEach(el => {
                    el.textContent = String(count);
                    el.style.display = count > 0 ? 'inline-block' : 'none';
                });
            }
        })
        .catch(error => {
            console.error('Error updating cart badge:', error);
        });
    }
}

// Add touch interactions for mobile
function addTouchInteractions() {
    let touchStartY = 0;
    let touchStartX = 0;
    
    document.addEventListener('touchstart', function(e) {
        touchStartY = e.touches[0].clientY;
        touchStartX = e.touches[0].clientX;
    }, { passive: true });
    
    document.addEventListener('touchend', function(e) {
        const touchEndY = e.changedTouches[0].clientY;
        const touchEndX = e.changedTouches[0].clientX;
        const deltaY = touchStartY - touchEndY;
        const deltaX = touchStartX - touchEndX;
        
        // Detect swipe gestures
        if (Math.abs(deltaY) > Math.abs(deltaX)) {
            if (deltaY > 50) {
                // Swipe up
                console.log('Swipe up detected');
            } else if (deltaY < -50) {
                // Swipe down
                console.log('Swipe down detected');
            }
        }
    }, { passive: true });
}

// Initialize touch interactions
addTouchInteractions();

// Handle horizontal scroll containers to allow vertical scrolling
function initializeHorizontalScrollContainers() {
    const scrollContainers = document.querySelectorAll('.products-scroll-container');
    
    scrollContainers.forEach(container => {
        // Add a small delay to allow vertical scrolling to work
        container.addEventListener('touchstart', function(e) {
            // Allow the touch to propagate for vertical scrolling
        }, { passive: true });
        
        container.addEventListener('touchmove', function(e) {
            // Only prevent default if we're actually scrolling horizontally
            const scrollLeft = container.scrollLeft;
            const scrollWidth = container.scrollWidth;
            const clientWidth = container.clientWidth;
            
            // If we're at the beginning or end of horizontal scroll, allow vertical
            if ((scrollLeft <= 0 && e.touches[0].clientX > e.touches[0].clientX) ||
                (scrollLeft >= scrollWidth - clientWidth && e.touches[0].clientX < e.touches[0].clientX)) {
                // Allow vertical scrolling
                return;
            }
            
            // Otherwise, allow horizontal scrolling
        }, { passive: true });
    });
}

// Add loading state management
function showLoadingState() {
    const loader = document.createElement('div');
    loader.className = 'loading-overlay';
    loader.innerHTML = '<div class="spinner"></div>';
    
    loader.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(255, 255, 255, 0.9);
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: 9999;
    `;
    
    const spinner = loader.querySelector('.spinner');
    spinner.style.cssText = `
        width: 40px;
        height: 40px;
        border: 4px solid #f3f3f3;
        border-top: 4px solid #333;
        border-radius: 50%;
        animation: spin 1s linear infinite;
    `;
    
    // Add keyframes for spinner animation
    if (!document.querySelector('#spinner-styles')) {
        const style = document.createElement('style');
        style.id = 'spinner-styles';
        style.textContent = `
            @keyframes spin {
                0% { transform: rotate(0deg); }
                100% { transform: rotate(360deg); }
            }
        `;
        document.head.appendChild(style);
    }
    
    document.body.appendChild(loader);
    return loader;
}

function hideLoadingState(loader) {
    if (loader && loader.parentNode) {
        loader.parentNode.removeChild(loader);
    }
}

// Product navigation function
function viewProduct(productId) {
    window.location.href = `product-detail.php?id=${productId}`;
}

// Authentication functions
function initializeAuth() {
    // Check if user is logged in or is a guest
    checkAuthStatus();
}

function checkAuthStatus() {
    // This would typically check session or localStorage
    // For now, we'll assume guest access is always available
    console.log('Authentication initialized');
}

function continueAsGuest() {
    // Set guest session
    fetch('auth-handler.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: 'action=guest_login'
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification('Welcome! You can now browse as a guest.', 'success');
            // Optionally redirect or update UI
            setTimeout(() => {
                window.location.reload();
            }, 1000);
        } else {
            showNotification('Error setting up guest access', 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showNotification('Error setting up guest access', 'error');
    });
}

function logout() {
    fetch('auth-handler.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: 'action=logout'
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification('Logged out successfully', 'success');
            window.location.href = 'index.php';
        } else {
            showNotification('Error logging out', 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showNotification('Error logging out', 'error');
    });
}

// Banner Slider Functions
let currentSlideIndex = 0;
let slideInterval;
let banners = [];

// Load banners dynamically
async function loadBanners() {
    // Check if banner container exists on this page
    const container = document.getElementById('bannerContainer');
    if (!container) {
        console.log('Banner container not found on this page, skipping banner loading');
        return;
    }
    
    try {
        const response = await fetch('ajax/get_banners.php');
        banners = await response.json();
        
        if (banners.length === 0) {
            showDefaultBanners();
            return;
        }
        
        renderBanners();
        await initializeBannerSlider();
    } catch (error) {
        console.error('Error loading banners:', error);
        showDefaultBanners();
    }
}

// Show default banners if no banners are available
function showDefaultBanners() {
    const container = document.getElementById('bannerContainer');
    if (!container) {
        console.log('Banner container not found, cannot show default banners');
        return;
    }
    
    container.innerHTML = `
        <div class="banner-slide active">
            <img src="banner/1.jpg" alt="Banner 1" class="banner-image" onclick="openBannerModal('banner/1.jpg', 'image')">
            <div class="banner-content">
                <h3>Welcome to 2 Days Parfume</h3>
                <p>Discover our premium collection</p>
            </div>
        </div>
        <div class="banner-slide">
            <img src="banner/2.jpg" alt="Banner 2" class="banner-image" onclick="openBannerModal('banner/2.jpg', 'image')">
            <div class="banner-content">
                <h3>New Arrivals</h3>
                <p>Fresh scents for every occasion</p>
            </div>
        </div>
        <div class="banner-slide">
            <img src="banner/3.jpg" alt="Banner 3" class="banner-image" onclick="openBannerModal('banner/3.jpg', 'image')">
            <div class="banner-content">
                <h3>Best Sellers</h3>
                <p>Our most popular fragrances</p>
            </div>
        </div>
    `;
    
    // Generate dots for default banners
    generateDots(3);
    initializeBannerSlider();
}

// Render banners from loaded data
function renderBanners() {
    const container = document.getElementById('bannerContainer');
    if (!container) {
        console.log('Banner container not found, cannot render banners');
        return;
    }
    
    container.innerHTML = '';
    
    banners.forEach((banner, index) => {
        const slide = document.createElement('div');
        slide.className = `banner-slide ${index === 0 ? 'active' : ''}`;
        
        if (banner.type === 'video') {
            slide.innerHTML = `
                <video class="banner-video" autoplay muted loop playsinline onclick="openBannerModal('${banner.path}', 'video')">
                    <source src="${banner.path}" type="video/${getVideoType(banner.filename)}">
                    Your browser does not support the video tag.
                </video>
                <div class="banner-content">
                    <h3>Premium Collection</h3>
                    <p>Experience luxury fragrances</p>
                </div>
            `;
        } else {
            slide.innerHTML = `
                <img src="${banner.path}" alt="${banner.filename}" class="banner-image" onclick="openBannerModal('${banner.path}', 'image')">
                <div class="banner-content">
                    <h3>Premium Collection</h3>
                    <p>Experience luxury fragrances</p>
                </div>
            `;
        }
        
        container.appendChild(slide);
    });
    
    // Generate dots
    generateDots(banners.length);
}

// Generate navigation dots
function generateDots(count) {
    const dotsContainer = document.getElementById('bannerDots');
    dotsContainer.innerHTML = '';
    
    for (let i = 0; i < count; i++) {
        const dot = document.createElement('span');
        dot.className = `dot ${i === 0 ? 'active' : ''}`;
        dot.onclick = () => currentSlide(i + 1);
        dotsContainer.appendChild(dot);
    }
}

// Get video type from filename
function getVideoType(filename) {
    const extension = filename.split('.').pop().toLowerCase();
    const videoTypes = {
        'mp4': 'mp4',
        'webm': 'webm',
        'ogg': 'ogg',
        'avi': 'avi',
        'mov': 'quicktime'
    };
    return videoTypes[extension] || 'mp4';
}

async function initializeBannerSlider() {
    const slides = document.querySelectorAll('.banner-slide');
    const dots = document.querySelectorAll('.dot');
    
    if (slides.length === 0) return;
    
    // Load banner settings
    try {
        const settingsResponse = await fetch('ajax/get_banner_settings.php');
        const settings = await settingsResponse.json();
        window.bannerSlideTiming = settings.slideTiming * 1000; // convert to milliseconds
    } catch (error) {
        console.error('Error loading banner settings:', error);
        window.bannerSlideTiming = 5000; // default 5 seconds
    }
    
    // Auto-slide functionality
    startAutoSlide();
    
    // Touch/swipe support
    addTouchSupport();
    
    // Pause on hover
    const slider = document.querySelector('.banner-slider');
    if (slider) {
        slider.addEventListener('mouseenter', stopAutoSlide);
        slider.addEventListener('mouseleave', startAutoSlide);
    }
}

function showSlide(index) {
    const slides = document.querySelectorAll('.banner-slide');
    const dots = document.querySelectorAll('.dot');
    
    if (slides.length === 0) return;
    
    // Pause all videos in inactive slides
    slides.forEach((slide, i) => {
        const video = slide.querySelector('video');
        if (video) {
            if (i === index) {
                video.play().catch(e => console.log('Video autoplay prevented:', e));
            } else {
                video.pause();
            }
        }
    });
    
    // Hide all slides
    slides.forEach(slide => slide.classList.remove('active'));
    dots.forEach(dot => dot.classList.remove('active'));
    
    // Show current slide
    if (slides[index]) {
        slides[index].classList.add('active');
    }
    if (dots[index]) {
        dots[index].classList.add('active');
    }
    
    currentSlideIndex = index;
}

function nextSlide() {
    const slides = document.querySelectorAll('.banner-slide');
    const nextIndex = (currentSlideIndex + 1) % slides.length;
    showSlide(nextIndex);
}

function prevSlide() {
    const slides = document.querySelectorAll('.banner-slide');
    const prevIndex = (currentSlideIndex - 1 + slides.length) % slides.length;
    showSlide(prevIndex);
}

function currentSlide(index) {
    showSlide(index - 1); // Convert to 0-based index
}

function changeSlide(direction) {
    if (direction === 1) {
        nextSlide();
    } else {
        prevSlide();
    }
}

function startAutoSlide() {
    stopAutoSlide(); // Clear any existing interval
    const timing = window.bannerSlideTiming || 5000; // Use configurable timing or default 5 seconds
    slideInterval = setInterval(nextSlide, timing);
}

function stopAutoSlide() {
    if (slideInterval) {
        clearInterval(slideInterval);
        slideInterval = null;
    }
}

function addTouchSupport() {
    const slider = document.querySelector('.banner-slider');
    if (!slider) return;
    
    let startX = 0;
    let startY = 0;
    let endX = 0;
    let endY = 0;
    
    slider.addEventListener('touchstart', function(e) {
        startX = e.touches[0].clientX;
        startY = e.touches[0].clientY;
    }, { passive: true });
    
    slider.addEventListener('touchend', function(e) {
        endX = e.changedTouches[0].clientX;
        endY = e.changedTouches[0].clientY;
        
        const deltaX = endX - startX;
        const deltaY = endY - startY;
        
        // Only trigger swipe if horizontal movement is greater than vertical
        if (Math.abs(deltaX) > Math.abs(deltaY) && Math.abs(deltaX) > 50) {
            if (deltaX > 0) {
                // Swipe right - previous slide
                prevSlide();
            } else {
                // Swipe left - next slide
                nextSlide();
            }
        }
    }, { passive: true });
}

// Authentication and Protected Navigation Functions
function initializeProtectedNavigation() {
    // Add click handlers for protected navigation links
    const protectedLinks = document.querySelectorAll('[data-protected]');
    protectedLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            const href = this.getAttribute('href') || this.getAttribute('onclick');
            const protectionLevel = this.getAttribute('data-protected');
            
            // Extract URL from onclick attribute if present
            let url = href;
            if (href && href.includes('window.location.href=')) {
                url = href.match(/window\.location\.href=['"]([^'"]+)['"]/)?.[1];
            }
            
            if (protectionLevel === 'auth') {
                checkAuthAndRedirect(url);
            } else if (protectionLevel === 'registered') {
                checkRegisteredUserAndRedirect(url);
            }
        });
    });
    
    // Add click handlers for protected buttons
    const protectedButtons = document.querySelectorAll('[data-protected-action]');
    protectedButtons.forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            const action = this.getAttribute('data-protected-action');
            
            if (action === 'favorite') {
                checkAuthAndShowMessage('Please sign up to add items to favorites');
            } else if (action === 'royalty') {
                checkRegisteredUserAndShowMessage('Please sign up to access Royal Rewards');
            }
        });
    });
}

function checkAuthAndRedirect(url) {
    // Check if user is logged in
    fetch('auth-handler.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: 'action=check_auth'
    })
    .then(response => response.json())
    .then(data => {
        if (data.success && data.logged_in) {
            window.location.href = url;
        } else {
            showNotification('Please sign up to access this feature', 'info');
            setTimeout(() => {
                window.location.href = 'signup.php?redirect=' + encodeURIComponent(url);
            }, 1500);
        }
    })
    .catch(error => {
        console.error('Auth check error:', error);
        showNotification('Please sign up to access this feature', 'info');
        setTimeout(() => {
            window.location.href = 'signup.php?redirect=' + encodeURIComponent(url);
        }, 1500);
    });
}

function checkRegisteredUserAndRedirect(url) {
    // Check if user is a registered user (not guest)
    fetch('auth-handler.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: 'action=check_registered_user'
    })
    .then(response => response.json())
    .then(data => {
        if (data.success && data.is_registered) {
            window.location.href = url;
        } else {
            showNotification('Please sign up to access Royal Rewards', 'info');
            setTimeout(() => {
                window.location.href = 'signup.php?redirect=' + encodeURIComponent(url);
            }, 1500);
        }
    })
    .catch(error => {
        console.error('Registered user check error:', error);
        showNotification('Please sign up to access Royal Rewards', 'info');
        setTimeout(() => {
            window.location.href = 'signup.php?redirect=' + encodeURIComponent(url);
        }, 1500);
    });
}

function checkAuthAndShowMessage(message) {
    showNotification(message, 'info');
    setTimeout(() => {
        window.location.href = 'signup.php';
    }, 1500);
}

function checkRegisteredUserAndShowMessage(message) {
    showNotification(message, 'info');
    setTimeout(() => {
        window.location.href = 'signup.php';
    }, 1500);
}

// Export functions for global access
window.PerfumeApp = {
    showNotification,
    showLoadingState,
    hideLoadingState,
    createSearchOverlay,
    viewProduct,
    continueAsGuest,
    logout,
    addToCart,
    updateCartBadge,
    testDatabase,
    checkAuthAndRedirect,
    checkRegisteredUserAndRedirect
};

// Export banner slider functions globally
window.currentSlide = currentSlide;

// Create banner modal if it doesn't exist
function createBannerModal() {
    const modalHTML = `
        <div id="bannerModal" class="banner-modal">
            <div class="banner-modal-content">
                <span class="banner-modal-close">&times;</span>
                <img id="modalImage" src="" alt="Banner" style="display: none;">
                <video id="modalVideo" src="" controls autoplay loop muted style="display: none;"></video>
            </div>
        </div>
    `;
    
    document.body.insertAdjacentHTML('beforeend', modalHTML);
    
    // Initialize the modal functionality
    const modal = document.getElementById('bannerModal');
    const closeBtn = document.querySelector('.banner-modal-close');
    
    if (closeBtn) {
        closeBtn.onclick = closeBannerModal;
    }
    
    if (modal) {
        modal.onclick = function(event) {
            if (event.target === modal) {
                closeBannerModal();
            }
        };
    }
}

// Banner modal functions
function openBannerModal(path, type) {
    let modal = document.getElementById('bannerModal');
    let modalImage = document.getElementById('modalImage');
    let modalVideo = document.getElementById('modalVideo');
    
    // Create modal elements if they don't exist
    if (!modal) {
        createBannerModal();
        modal = document.getElementById('bannerModal');
        modalImage = document.getElementById('modalImage');
        modalVideo = document.getElementById('modalVideo');
    }
    
    // Check if elements exist
    if (!modal || !modalImage || !modalVideo) {
        console.error('Modal elements not found');
        return;
    }
    
    // Hide both elements first
    modalImage.style.display = 'none';
    modalVideo.style.display = 'none';
    
    if (type === 'video') {
        modalVideo.src = path;
        modalVideo.style.display = 'block';
    } else {
        modalImage.src = path;
        modalImage.style.display = 'block';
    }
    
    modal.style.display = 'block';
    document.body.style.overflow = 'hidden'; // Prevent background scrolling
}

function closeBannerModal() {
    const modal = document.getElementById('bannerModal');
    const modalVideo = document.getElementById('modalVideo');
    
    if (!modal) {
        console.error('Modal element not found');
        return;
    }
    
    modal.style.display = 'none';
    document.body.style.overflow = 'auto'; // Restore scrolling
    
    // Pause video if it's playing
    if (modalVideo && modalVideo.style.display === 'block') {
        modalVideo.pause();
    }
}

// Initialize banner modal functionality
document.addEventListener('DOMContentLoaded', function() {
    const modal = document.getElementById('bannerModal');
    const closeBtn = document.querySelector('.banner-modal-close');
    
    if (closeBtn) {
        closeBtn.onclick = closeBannerModal;
    }
    
    if (modal) {
        modal.onclick = function(event) {
            if (event.target === modal) {
                closeBannerModal();
            }
        };
    }
    
    // Close modal with Escape key
    document.addEventListener('keydown', function(event) {
        if (event.key === 'Escape' && modal && modal.style.display === 'block') {
            closeBannerModal();
        }
    });
});

// Restock Timer Functionality
function initializeRestockTimers() {
    const timers = document.querySelectorAll('.restock-timer');
    
    timers.forEach(timer => {
        const restockDate = timer.getAttribute('data-restock-date');
        if (restockDate) {
            updateTimer(timer, restockDate);
            // Update timer every minute
            setInterval(() => updateTimer(timer, restockDate), 60000);
        }
    });
}

function updateTimer(timerElement, restockDate) {
    const now = new Date().getTime();
    const restockTime = new Date(restockDate).getTime();
    const timeLeft = restockTime - now;
    
    if (timeLeft > 0) {
        const days = Math.floor(timeLeft / (1000 * 60 * 60 * 24));
        const hours = Math.floor((timeLeft % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
        const minutes = Math.floor((timeLeft % (1000 * 60 * 60)) / (1000 * 60));
        
        const daysElement = timerElement.querySelector('.timer-days');
        const hoursElement = timerElement.querySelector('.timer-hours');
        const minutesElement = timerElement.querySelector('.timer-minutes');
        
        if (daysElement) daysElement.textContent = days.toString().padStart(2, '0');
        if (hoursElement) hoursElement.textContent = hours.toString().padStart(2, '0');
        if (minutesElement) minutesElement.textContent = minutes.toString().padStart(2, '0');
    } else {
        // Timer expired - hide timer and show message
        timerElement.innerHTML = '<div class="timer-expired">Back in stock!</div>';
        timerElement.style.background = 'rgba(40, 167, 69, 0.8)';
    }
}

window.changeSlide = changeSlide;

// Manual timer initialization for testing
window.initTimers = initializeRestockTimers;