<?php
/**
 * Forgot Password Setup Script
 * Run this script once to set up the forgot password functionality
 */

require_once 'config/database.php';

echo "<!DOCTYPE html>
<html lang='en'>
<head>
    <meta charset='UTF-8'>
    <meta name='viewport' content='width=device-width, initial-scale=1.0'>
    <title>Forgot Password Setup</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            max-width: 800px;
            margin: 50px auto;
            padding: 20px;
            background: #f5f5f5;
        }
        .container {
            background: white;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        h1 {
            color: #333;
            border-bottom: 3px solid #3498db;
            padding-bottom: 10px;
        }
        .step {
            margin: 20px 0;
            padding: 15px;
            background: #f9f9f9;
            border-left: 4px solid #3498db;
            border-radius: 4px;
        }
        .success {
            color: #27ae60;
            font-weight: bold;
        }
        .error {
            color: #e74c3c;
            font-weight: bold;
        }
        .warning {
            color: #f39c12;
            font-weight: bold;
        }
        .info {
            background: #e3f2fd;
            padding: 15px;
            border-radius: 5px;
            margin: 15px 0;
        }
        code {
            background: #2c3e50;
            color: #ecf0f1;
            padding: 2px 6px;
            border-radius: 3px;
            font-family: 'Courier New', monospace;
        }
        .checklist {
            list-style: none;
            padding: 0;
        }
        .checklist li {
            padding: 8px;
            margin: 5px 0;
        }
        .checklist li:before {
            content: '✓ ';
            color: #27ae60;
            font-weight: bold;
            margin-right: 10px;
        }
        .btn {
            display: inline-block;
            padding: 10px 20px;
            background: #3498db;
            color: white;
            text-decoration: none;
            border-radius: 5px;
            margin: 10px 5px;
        }
        .btn:hover {
            background: #2980b9;
        }
    </style>
</head>
<body>
    <div class='container'>
        <h1>🔐 Forgot Password Setup</h1>
";

$errors = [];
$warnings = [];
$success = [];

// Step 1: Check database connection
echo "<div class='step'>";
echo "<h2>Step 1: Database Connection</h2>";
try {
    $db = getDB();
    echo "<p class='success'>✓ Database connection successful!</p>";
    $success[] = "Database connected";
} catch (Exception $e) {
    echo "<p class='error'>✗ Database connection failed: " . htmlspecialchars($e->getMessage()) . "</p>";
    $errors[] = "Database connection failed";
}
echo "</div>";

// Step 2: Check if table exists
echo "<div class='step'>";
echo "<h2>Step 2: Password Reset Tokens Table</h2>";
if (isset($db)) {
    try {
        $stmt = $db->query("SHOW TABLES LIKE 'password_reset_tokens'");
        $tableExists = $stmt->rowCount() > 0;
        
        if ($tableExists) {
            echo "<p class='success'>✓ Table 'password_reset_tokens' already exists!</p>";
            $success[] = "Table exists";
        } else {
            echo "<p class='warning'>⚠ Table 'password_reset_tokens' does not exist. Creating...</p>";
            
            // Create table
            $sql = "CREATE TABLE IF NOT EXISTS password_reset_tokens (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id INT NOT NULL,
                email VARCHAR(191) NOT NULL,
                token VARCHAR(255) NOT NULL,
                expires_at DATETIME NOT NULL,
                used BOOLEAN DEFAULT FALSE,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
                INDEX idx_token (token),
                INDEX idx_email (email),
                INDEX idx_expires_at (expires_at)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
            
            $db->exec($sql);
            echo "<p class='success'>✓ Table created successfully!</p>";
            $success[] = "Table created";
        }
    } catch (Exception $e) {
        echo "<p class='error'>✗ Error checking/creating table: " . htmlspecialchars($e->getMessage()) . "</p>";
        $errors[] = "Table creation failed";
    }
}
echo "</div>";

// Step 3: Check email configuration
echo "<div class='step'>";
echo "<h2>Step 3: Email Configuration</h2>";
if (file_exists('config/email.php')) {
    echo "<p class='success'>✓ Email configuration file exists!</p>";
    require_once 'config/email.php';
    
    // Check if password is set
    if (SMTP_PASSWORD === "Use the email account's password") {
        echo "<p class='warning'>⚠ SMTP password not configured!</p>";
        echo "<p>Please edit <code>config/email.php</code> and set the SMTP password.</p>";
        $warnings[] = "SMTP password not set";
    } else {
        echo "<p class='success'>✓ SMTP password is configured!</p>";
        $success[] = "SMTP configured";
    }
    
    echo "<div class='info'>";
    echo "<strong>Current Email Settings:</strong><br>";
    echo "Host: <code>" . SMTP_HOST . "</code><br>";
    echo "Port: <code>" . SMTP_PORT . "</code><br>";
    echo "Username: <code>" . SMTP_USERNAME . "</code><br>";
    echo "Encryption: <code>" . SMTP_ENCRYPTION . "</code><br>";
    echo "From: <code>" . SMTP_FROM_EMAIL . "</code>";
    echo "</div>";
} else {
    echo "<p class='error'>✗ Email configuration file not found!</p>";
    $errors[] = "Email config missing";
}
echo "</div>";

// Step 4: Check PHPMailer
echo "<div class='step'>";
echo "<h2>Step 4: PHPMailer</h2>";
if (file_exists('vendor/autoload.php')) {
    require_once 'vendor/autoload.php';
    if (class_exists('PHPMailer\PHPMailer\PHPMailer')) {
        echo "<p class='success'>✓ PHPMailer is installed!</p>";
        $success[] = "PHPMailer available";
    } else {
        echo "<p class='warning'>⚠ PHPMailer not found in vendor directory.</p>";
        echo "<p>Install with: <code>composer require phpmailer/phpmailer</code></p>";
        $warnings[] = "PHPMailer not installed";
    }
} else {
    echo "<p class='warning'>⚠ Composer autoload not found.</p>";
    echo "<p>The system will use PHP's built-in mail() function as fallback.</p>";
    echo "<p>For better email delivery, install PHPMailer:</p>";
    echo "<code>composer require phpmailer/phpmailer</code>";
    $warnings[] = "Using mail() fallback";
}
echo "</div>";

// Step 5: Check required files
echo "<div class='step'>";
echo "<h2>Step 5: Required Files</h2>";
$requiredFiles = [
    'forgot-password.php' => 'Password reset request page',
    'reset-password.php' => 'Password reset form page',
    'config/email.php' => 'Email configuration',
    'login.php' => 'Login page (updated)'
];

$allFilesExist = true;
echo "<ul class='checklist'>";
foreach ($requiredFiles as $file => $description) {
    if (file_exists($file)) {
        echo "<li>{$description}: <code>{$file}</code></li>";
    } else {
        echo "<li style='color: #e74c3c;'>✗ Missing: <code>{$file}</code> - {$description}</li>";
        $allFilesExist = false;
        $errors[] = "Missing file: {$file}";
    }
}
echo "</ul>";

if ($allFilesExist) {
    echo "<p class='success'>✓ All required files are present!</p>";
    $success[] = "All files present";
}
echo "</div>";

// Summary
echo "<div class='step'>";
echo "<h2>📊 Setup Summary</h2>";

if (count($errors) === 0 && count($warnings) === 0) {
    echo "<p class='success' style='font-size: 18px;'>🎉 Setup completed successfully!</p>";
    echo "<p>You can now test the forgot password functionality.</p>";
} else {
    if (count($errors) > 0) {
        echo "<p class='error'>Errors found: " . count($errors) . "</p>";
        echo "<ul>";
        foreach ($errors as $error) {
            echo "<li class='error'>{$error}</li>";
        }
        echo "</ul>";
    }
    
    if (count($warnings) > 0) {
        echo "<p class='warning'>Warnings: " . count($warnings) . "</p>";
        echo "<ul>";
        foreach ($warnings as $warning) {
            echo "<li class='warning'>{$warning}</li>";
        }
        echo "</ul>";
    }
}

echo "<p><strong>Successful checks: " . count($success) . "</strong></p>";
echo "</div>";

// Next Steps
echo "<div class='step'>";
echo "<h2>🎯 Next Steps</h2>";
echo "<ol>";

if (in_array("SMTP password not set", $warnings)) {
    echo "<li>Update SMTP password in <code>config/email.php</code></li>";
}

if (in_array("PHPMailer not installed", $warnings) || in_array("Using mail() fallback", $warnings)) {
    echo "<li>Install PHPMailer for better email delivery: <code>composer require phpmailer/phpmailer</code></li>";
}

echo "<li>Test the forgot password flow:
    <ul>
        <li>Go to <a href='login.php'>login.php</a></li>
        <li>Click 'Forgot Password?'</li>
        <li>Enter your email</li>
        <li>Check your email for reset link</li>
    </ul>
</li>";
echo "<li>Review the setup guide: <code>FORGOT_PASSWORD_SETUP.md</code></li>";
echo "<li>Set up a cron job to clean expired tokens (optional)</li>";
echo "</ol>";
echo "</div>";

// Action Buttons
echo "<div style='text-align: center; margin-top: 30px;'>";
echo "<a href='login.php' class='btn'>Go to Login</a>";
echo "<a href='forgot-password.php' class='btn'>Test Forgot Password</a>";
echo "<a href='FORGOT_PASSWORD_SETUP.md' class='btn' style='background: #27ae60;'>View Setup Guide</a>";
echo "</div>";

echo "
    </div>
</body>
</html>";
?>

