<?php
session_start();
require_once 'models/User.php';
require_once 'includes/remember-me.php';

// Check for remember me token and auto-login
checkRememberMe();

// Check if user is logged in
if (!isset($_SESSION['user_id']) || $_SESSION['user_id'] === 'guest') {
    header('Location: login.php');
    exit();
}

// Get user data
$user = new User();
$user_data = $user->getById($_SESSION['user_id']);

if (!$user_data) {
    header('Location: login.php');
    exit();
}

$error_message = '';
$success_message = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'add_address') {
        $address_name = trim($_POST['address_name'] ?? '');
        $first_name = trim($_POST['first_name'] ?? '');
        $last_name = trim($_POST['last_name'] ?? '');
        $phone = trim($_POST['phone'] ?? '');
        $address = trim($_POST['address'] ?? '');
        $city = trim($_POST['city'] ?? '');
        $state = trim($_POST['state'] ?? '');
        $zip_code = trim($_POST['zip_code'] ?? '');
        $country = trim($_POST['country'] ?? 'Malaysia');
        $is_default = isset($_POST['is_default']) ? 1 : 0;
        
        if (empty($address_name) || empty($first_name) || empty($last_name) || empty($phone) || empty($address) || empty($city) || empty($state) || empty($zip_code)) {
            $error_message = 'All fields are required.';
        } else {
            $address_data = [
                'user_id' => $_SESSION['user_id'],
                'address_name' => $address_name,
                'first_name' => $first_name,
                'last_name' => $last_name,
                'phone' => $phone,
                'address' => $address,
                'city' => $city,
                'state' => $state,
                'zip_code' => $zip_code,
                'country' => $country,
                'is_default' => $is_default
            ];
            
            if ($user->addShippingAddress($address_data)) {
                $success_message = 'Shipping address added successfully!';
            } else {
                $error_message = 'Failed to add shipping address. Please try again.';
            }
        }
    } elseif ($action === 'update_address') {
        $address_id = $_POST['address_id'] ?? '';
        $address_name = trim($_POST['address_name'] ?? '');
        $first_name = trim($_POST['first_name'] ?? '');
        $last_name = trim($_POST['last_name'] ?? '');
        $phone = trim($_POST['phone'] ?? '');
        $address = trim($_POST['address'] ?? '');
        $city = trim($_POST['city'] ?? '');
        $state = trim($_POST['state'] ?? '');
        $zip_code = trim($_POST['zip_code'] ?? '');
        $country = trim($_POST['country'] ?? 'Malaysia');
        $is_default = isset($_POST['is_default']) ? 1 : 0;
        
        if (empty($address_name) || empty($first_name) || empty($last_name) || empty($phone) || empty($address) || empty($city) || empty($state) || empty($zip_code)) {
            $error_message = 'All fields are required.';
        } else {
            $address_data = [
                'address_name' => $address_name,
                'first_name' => $first_name,
                'last_name' => $last_name,
                'phone' => $phone,
                'address' => $address,
                'city' => $city,
                'state' => $state,
                'zip_code' => $zip_code,
                'country' => $country,
                'is_default' => $is_default
            ];
            
            if ($user->updateShippingAddress($address_id, $address_data)) {
                $success_message = 'Shipping address updated successfully!';
            } else {
                $error_message = 'Failed to update shipping address. Please try again.';
            }
        }
    } elseif ($action === 'delete_address') {
        $address_id = $_POST['address_id'] ?? '';
        
        if ($user->deleteShippingAddress($address_id, $_SESSION['user_id'])) {
            $success_message = 'Shipping address deleted successfully!';
        } else {
            $error_message = 'Failed to delete shipping address. Please try again.';
        }
    } elseif ($action === 'set_default') {
        $address_id = $_POST['address_id'] ?? '';
        
        if ($user->setDefaultShippingAddress($address_id, $_SESSION['user_id'])) {
            $success_message = 'Default address updated successfully!';
        } else {
            $error_message = 'Failed to update default address. Please try again.';
        }
    }
}

// Get user's shipping addresses
$shipping_addresses = $user->getShippingAddresses($_SESSION['user_id']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Shipping Address - 2 Days Parfume</title>
    <link rel="stylesheet" href="styles.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="script.js" defer></script>
    <style>
        .shipping-address-container {
            min-height: 100vh;
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            padding-bottom: 100px;
        }

        .shipping-address-header {
            background: linear-gradient(135deg, #1A4D2A 0%, #2d5a3d 50%, #1A4D2A 100%);
            padding: 20px 20px 30px 20px;
            text-align: center;
            position: relative;
            overflow: hidden;
        }

        .shipping-address-header::before {
            content: '';
            position: absolute;
            bottom: -20px;
            left: 0;
            right: 0;
            height: 40px;
            background: linear-gradient(135deg, #1A4D2A 0%, #2d5a3d 50%, #1A4D2A 100%);
            border-radius: 50% 50% 0 0 / 100% 100% 0 0;
        }

        .shipping-address-title {
            color: #D4AF37;
            font-size: 24px;
            font-weight: bold;
            margin-bottom: 5px;
        }

        .shipping-address-subtitle {
            color: #F4D03F;
            font-size: 14px;
        }

        .shipping-address-content {
            padding: 30px 20px;
            margin-top: -20px;
            position: relative;
            z-index: 2;
        }

        .addresses-container {
            background: white;
            border-radius: 15px;
            padding: 25px;
            margin-bottom: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }

        .section-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }

        .section-title {
            font-size: 18px;
            font-weight: bold;
            color: #1A4D2A;
        }

        .btn-add {
            background: linear-gradient(135deg, #D4AF37, #F4D03F);
            border: none;
            border-radius: 8px;
            color: #1A4D2A;
            padding: 8px 16px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 5px;
        }

        .btn-add:hover {
            background: linear-gradient(135deg, #1A4D2A, #2d5a3d);
            color: #D4AF37;
            transform: translateY(-1px);
        }

        .address-card {
            border: 2px solid #e9ecef;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 15px;
            transition: all 0.3s ease;
            position: relative;
        }

        .address-card:hover {
            border-color: #D4AF37;
            box-shadow: 0 2px 8px rgba(212, 175, 55, 0.1);
        }

        .address-card.default {
            border-color: #D4AF37;
            background: linear-gradient(135deg, rgba(212, 175, 55, 0.05), rgba(244, 208, 63, 0.05));
        }

        .address-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
        }

        .address-name {
            font-size: 16px;
            font-weight: bold;
            color: #1A4D2A;
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .default-badge {
            background: #D4AF37;
            color: white;
            padding: 2px 8px;
            border-radius: 12px;
            font-size: 11px;
            font-weight: 600;
        }

        .address-actions {
            display: flex;
            gap: 8px;
        }

        .btn-action {
            padding: 6px 12px;
            border: none;
            border-radius: 6px;
            font-size: 12px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 4px;
        }

        .btn-edit {
            background: #e3f2fd;
            color: #1976d2;
        }

        .btn-edit:hover {
            background: #1976d2;
            color: white;
        }

        .btn-delete {
            background: #ffebee;
            color: #d32f2f;
        }

        .btn-delete:hover {
            background: #d32f2f;
            color: white;
        }

        .btn-default {
            background: #f3e5f5;
            color: #7b1fa2;
        }

        .btn-default:hover {
            background: #7b1fa2;
            color: white;
        }

        .address-details {
            color: #666;
            line-height: 1.6;
        }

        .address-details .name {
            font-weight: 600;
            color: #1A4D2A;
            margin-bottom: 5px;
        }

        .address-details .phone {
            color: #D4AF37;
            font-weight: 500;
            margin-bottom: 5px;
        }

        .address-details .address {
            margin-bottom: 5px;
        }

        .address-details .location {
            font-size: 13px;
        }

        .empty-state {
            text-align: center;
            padding: 40px 20px;
            color: #666;
        }

        .empty-state i {
            font-size: 48px;
            color: #D4AF37;
            margin-bottom: 15px;
            opacity: 0.5;
        }

        .empty-state h3 {
            font-size: 18px;
            color: #1A4D2A;
            margin-bottom: 10px;
        }

        .empty-state p {
            font-size: 14px;
            line-height: 1.5;
            margin-bottom: 20px;
        }

        .btn-primary {
            background: linear-gradient(135deg, #D4AF37, #F4D03F);
            border: none;
            border-radius: 8px;
            color: #1A4D2A;
            padding: 12px 24px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }

        .btn-primary:hover {
            background: linear-gradient(135deg, #1A4D2A, #2d5a3d);
            color: #D4AF37;
            transform: translateY(-1px);
        }

        .alert {
            padding: 12px 16px;
            border-radius: 8px;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
            font-size: 14px;
            font-weight: 500;
        }

        .alert-error {
            background: #fee;
            color: #c53030;
            border: 1px solid #feb2b2;
        }

        .alert-success {
            background: #f0fff4;
            color: #2f855a;
            border: 1px solid #9ae6b4;
        }

        .alert i {
            font-size: 16px;
        }

        /* Modal Styles */
        .modal {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0,0,0,0.5);
            animation: fadeIn 0.3s ease;
        }

        .modal-content {
            background-color: white;
            margin: 5% auto;
            padding: 0;
            border-radius: 15px;
            width: 90%;
            max-width: 500px;
            max-height: 90vh;
            overflow-y: auto;
            animation: slideIn 0.3s ease;
        }

        .modal-header {
            background: linear-gradient(135deg, #1A4D2A, #2d5a3d);
            color: #D4AF37;
            padding: 20px;
            border-radius: 15px 15px 0 0;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .modal-title {
            font-size: 18px;
            font-weight: bold;
            margin: 0;
        }

        .modal-close {
            color: #D4AF37;
            font-size: 24px;
            font-weight: bold;
            cursor: pointer;
            transition: color 0.3s ease;
        }

        .modal-close:hover {
            color: #F4D03F;
        }

        .modal-body {
            padding: 25px;
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-group label {
            display: block;
            margin-bottom: 8px;
            color: #1A4D2A;
            font-weight: 600;
            font-size: 14px;
        }

        .form-group input,
        .form-group select,
        .form-group textarea {
            width: 100%;
            padding: 12px 16px;
            border: 2px solid #e9ecef;
            border-radius: 10px;
            font-size: 16px;
            background: #f8f9fa;
            color: #1A4D2A;
            transition: all 0.3s ease;
            box-sizing: border-box;
        }

        .form-group input:focus,
        .form-group select:focus,
        .form-group textarea:focus {
            outline: none;
            border-color: #D4AF37;
            background: white;
            box-shadow: 0 0 0 3px rgba(212, 175, 55, 0.1);
        }

        .form-group textarea {
            resize: vertical;
            min-height: 80px;
        }

        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
        }

        .checkbox-group {
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .checkbox-group input[type="checkbox"] {
            width: auto;
            margin: 0;
        }

        .modal-footer {
            padding: 20px 25px;
            border-top: 1px solid #e9ecef;
            display: flex;
            gap: 10px;
            justify-content: flex-end;
        }

        .btn-modal {
            padding: 12px 24px;
            border: none;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .btn-save {
            background: linear-gradient(135deg, #D4AF37, #F4D03F);
            color: #1A4D2A;
        }

        .btn-save:hover {
            background: linear-gradient(135deg, #1A4D2A, #2d5a3d);
            color: #D4AF37;
            transform: translateY(-1px);
        }

        .btn-cancel-modal {
            background: transparent;
            border: 2px solid #D4AF37;
            color: #D4AF37;
        }

        .btn-cancel-modal:hover {
            background: #D4AF37;
            color: white;
        }

        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }

        @keyframes slideIn {
            from { transform: translateY(-50px); opacity: 0; }
            to { transform: translateY(0); opacity: 1; }
        }

        @media (max-width: 480px) {
            .shipping-address-content {
                padding: 20px 15px;
            }
            
            .addresses-container {
                padding: 20px;
            }
            
            .address-header {
                flex-direction: column;
                align-items: flex-start;
                gap: 10px;
            }
            
            .address-actions {
                align-self: flex-end;
            }
            
            .modal-content {
                width: 95%;
                margin: 10% auto;
            }
            
            .form-row {
                grid-template-columns: 1fr;
            }
            
            .modal-footer {
                flex-direction: column;
            }
        }
    </style>
</head>
<body>
    <div class="shipping-address-container">
        <!-- Top Navigation -->
        <nav class="top-nav">
            <div class="nav-left">
                <i class="fas fa-arrow-left" onclick="goBack()"></i>
            </div>
            <div class="nav-center">
                <div class="logo">
                    <img src="images/LOGO.png" alt="2 Days Parfume Logo" class="logo-img">
                </div>
            </div>
            <div class="nav-right">
                <i class="fas fa-search" onclick="window.location.href='products.php'"></i>
            </div>
        </nav>

        <!-- Shipping Address Header -->
        <div class="shipping-address-header">
            <div class="shipping-address-title">Shipping Address</div>
            <div class="shipping-address-subtitle">Manage your delivery addresses</div>
        </div>

        <!-- Shipping Address Content -->
        <div class="shipping-address-content">
            <?php if ($error_message): ?>
                <div class="alert alert-error">
                    <i class="fas fa-exclamation-circle"></i>
                    <?php echo htmlspecialchars($error_message); ?>
                </div>
            <?php endif; ?>

            <?php if ($success_message): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i>
                    <?php echo htmlspecialchars($success_message); ?>
                </div>
            <?php endif; ?>

            <div class="addresses-container">
                <div class="section-header">
                    <div class="section-title">Your Addresses</div>
                    <button class="btn-add" onclick="showAddAddressForm()">
                        <i class="fas fa-plus"></i>
                        Add Address
                    </button>
                </div>

                <?php if (empty($shipping_addresses)): ?>
                    <div class="empty-state">
                        <i class="fas fa-map-marker-alt"></i>
                        <h3>No Shipping Addresses</h3>
                        <p>You haven't added any shipping addresses yet. Add your first address to get started with your orders.</p>
                        <button class="btn-primary" onclick="showAddAddressForm()">
                            <i class="fas fa-plus"></i>
                            Add Your First Address
                        </button>
                    </div>
                <?php else: ?>
                    <?php foreach ($shipping_addresses as $address): ?>
                    <div class="address-card <?php echo $address['is_default'] ? 'default' : ''; ?>">
                        <div class="address-header">
                            <div class="address-name">
                                <i class="fas fa-map-marker-alt"></i>
                                <?php echo htmlspecialchars($address['address_name']); ?>
                                <?php if ($address['is_default']): ?>
                                    <span class="default-badge">Default</span>
                                <?php endif; ?>
                            </div>
                            <div class="address-actions">
                                <button class="btn-action btn-edit" onclick="editAddress(<?php echo $address['id']; ?>)">
                                    <i class="fas fa-edit"></i>
                                    Edit
                                </button>
                                <?php if (!$address['is_default']): ?>
                                    <button class="btn-action btn-default" onclick="setDefault(<?php echo $address['id']; ?>)">
                                        <i class="fas fa-star"></i>
                                        Set Default
                                    </button>
                                <?php endif; ?>
                                <button class="btn-action btn-delete" onclick="deleteAddress(<?php echo $address['id']; ?>)">
                                    <i class="fas fa-trash"></i>
                                    Delete
                                </button>
                            </div>
                        </div>
                        <div class="address-details">
                            <div class="name"><?php echo htmlspecialchars($address['first_name'] . ' ' . $address['last_name']); ?></div>
                            <div class="phone"><?php echo htmlspecialchars($address['phone']); ?></div>
                            <div class="address"><?php echo htmlspecialchars($address['address']); ?></div>
                            <div class="location"><?php echo htmlspecialchars($address['city'] . ', ' . $address['state'] . ' ' . $address['zip_code'] . ', ' . $address['country']); ?></div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>

        <!-- Add/Edit Address Modal -->
        <div id="addressModal" class="modal">
            <div class="modal-content">
                <div class="modal-header">
                    <h3 class="modal-title" id="modalTitle">Add New Address</h3>
                    <span class="modal-close" onclick="closeModal()">&times;</span>
                </div>
                <form id="addressForm" method="POST" action="">
                    <input type="hidden" name="action" id="formAction" value="add_address">
                    <input type="hidden" name="address_id" id="addressId" value="">
                    
                    <div class="modal-body">
                        <div class="form-group">
                            <label for="address_name">Address Name *</label>
                            <input type="text" id="address_name" name="address_name" required 
                                   placeholder="e.g., Home, Office, Work">
                        </div>

                        <div class="form-row">
                            <div class="form-group">
                                <label for="first_name">First Name *</label>
                                <input type="text" id="first_name" name="first_name" required>
                            </div>
                            <div class="form-group">
                                <label for="last_name">Last Name *</label>
                                <input type="text" id="last_name" name="last_name" required>
                            </div>
                        </div>

                        <div class="form-group">
                            <label for="phone">Phone Number *</label>
                            <input type="tel" id="phone" name="phone" required 
                                   placeholder="e.g., +60123456789">
                        </div>

                        <div class="form-group">
                            <label for="address">Address *</label>
                            <textarea id="address" name="address" required 
                                      placeholder="Enter your full address"></textarea>
                        </div>

                        <div class="form-row">
                            <div class="form-group">
                                <label for="city">City *</label>
                                <input type="text" id="city" name="city" required>
                            </div>
                            <div class="form-group">
                                <label for="state">State *</label>
                                <input type="text" id="state" name="state" required>
                            </div>
                        </div>

                        <div class="form-row">
                            <div class="form-group">
                                <label for="zip_code">Postal Code *</label>
                                <input type="text" id="zip_code" name="zip_code" required>
                            </div>
                            <div class="form-group">
                                <label for="country">Country *</label>
                                <select id="country" name="country" required>
                                    <option value="Malaysia">Malaysia</option>
                                    <option value="Singapore">Singapore</option>
                                    <option value="Thailand">Thailand</option>
                                    <option value="Indonesia">Indonesia</option>
                                    <option value="Philippines">Philippines</option>
                                    <option value="Other">Other</option>
                                </select>
                            </div>
                        </div>

                        <div class="form-group">
                            <div class="checkbox-group">
                                <input type="checkbox" id="is_default" name="is_default" value="1">
                                <label for="is_default">Set as default address</label>
                            </div>
                        </div>
                    </div>

                    <div class="modal-footer">
                        <button type="button" class="btn-modal btn-cancel-modal" onclick="closeModal()">
                            <i class="fas fa-times"></i>
                            Cancel
                        </button>
                        <button type="submit" class="btn-modal btn-save">
                            <i class="fas fa-save"></i>
                            Save Address
                        </button>
                    </div>
                </form>
            </div>
        </div>

        <!-- Bottom Navigation -->
        <nav class="bottom-nav">
            <div class="nav-item" onclick="window.location.href='index.php'">
                <i class="fas fa-home"></i>
            </div>
            <div class="nav-item" onclick="window.location.href='products.php'">
                <i class="fas fa-box"></i>
            </div>
            <div class="nav-item" onclick="window.location.href='cart.php'">
                <i class="fas fa-shopping-cart"></i>
            </div>
            <div class="nav-item" onclick="window.location.href='profile.php'">
                <i class="far fa-user"></i>
            </div>
        </nav>
    </div>

    <script>
        // Address data for editing
        const addressData = <?php echo json_encode($shipping_addresses); ?>;

        function goBack() {
            window.history.back();
        }

        function showAddAddressForm() {
            // Reset form for adding new address
            document.getElementById('modalTitle').textContent = 'Add New Address';
            document.getElementById('formAction').value = 'add_address';
            document.getElementById('addressId').value = '';
            document.getElementById('addressForm').reset();
            document.getElementById('is_default').checked = false;
            
            // Show modal
            document.getElementById('addressModal').style.display = 'block';
        }

        function editAddress(addressId) {
            // Find address data
            const address = addressData.find(addr => addr.id == addressId);
            if (!address) {
                showNotification('Address not found!');
                return;
            }

            // Set form for editing
            document.getElementById('modalTitle').textContent = 'Edit Address';
            document.getElementById('formAction').value = 'update_address';
            document.getElementById('addressId').value = addressId;
            
            // Fill form with address data
            document.getElementById('address_name').value = address.address_name;
            document.getElementById('first_name').value = address.first_name;
            document.getElementById('last_name').value = address.last_name;
            document.getElementById('phone').value = address.phone;
            document.getElementById('address').value = address.address;
            document.getElementById('city').value = address.city;
            document.getElementById('state').value = address.state;
            document.getElementById('zip_code').value = address.zip_code;
            document.getElementById('country').value = address.country;
            document.getElementById('is_default').checked = address.is_default == 1;
            
            // Show modal
            document.getElementById('addressModal').style.display = 'block';
        }

        function closeModal() {
            document.getElementById('addressModal').style.display = 'none';
        }

        function deleteAddress(addressId) {
            if (confirm('Are you sure you want to delete this address?')) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.innerHTML = `
                    <input type="hidden" name="action" value="delete_address">
                    <input type="hidden" name="address_id" value="${addressId}">
                `;
                document.body.appendChild(form);
                form.submit();
            }
        }

        function setDefault(addressId) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.innerHTML = `
                <input type="hidden" name="action" value="set_default">
                <input type="hidden" name="address_id" value="${addressId}">
            `;
            document.body.appendChild(form);
            form.submit();
        }

        // Close modal when clicking outside
        window.onclick = function(event) {
            const modal = document.getElementById('addressModal');
            if (event.target == modal) {
                closeModal();
            }
        }

        // Form validation
        document.getElementById('addressForm').addEventListener('submit', function(e) {
            const requiredFields = ['address_name', 'first_name', 'last_name', 'phone', 'address', 'city', 'state', 'zip_code'];
            let isValid = true;

            requiredFields.forEach(fieldId => {
                const field = document.getElementById(fieldId);
                if (!field.value.trim()) {
                    field.style.borderColor = '#d32f2f';
                    isValid = false;
                } else {
                    field.style.borderColor = '#e9ecef';
                }
            });

            if (!isValid) {
                e.preventDefault();
                showNotification('Please fill in all required fields.');
            }
        });

        // Show notification function
        function showNotification(message, type = 'info') {
            // Create notification element
            const notification = document.createElement('div');
            notification.className = `notification notification-${type}`;
            notification.innerHTML = `
                <i class="fas fa-${type === 'error' ? 'exclamation-circle' : type === 'success' ? 'check-circle' : 'info-circle'}"></i>
                <span>${message}</span>
            `;
            
            // Add styles
            notification.style.cssText = `
                position: fixed;
                top: 20px;
                right: 20px;
                background: ${type === 'error' ? '#fee' : type === 'success' ? '#f0fff4' : '#e3f2fd'};
                color: ${type === 'error' ? '#c53030' : type === 'success' ? '#2f855a' : '#1976d2'};
                border: 1px solid ${type === 'error' ? '#feb2b2' : type === 'success' ? '#9ae6b4' : '#bbdefb'};
                padding: 12px 16px;
                border-radius: 8px;
                display: flex;
                align-items: center;
                gap: 10px;
                font-size: 14px;
                font-weight: 500;
                z-index: 10000;
                animation: slideInRight 0.3s ease;
                max-width: 300px;
            `;
            
            // Add animation keyframes
            if (!document.getElementById('notification-styles')) {
                const style = document.createElement('style');
                style.id = 'notification-styles';
                style.textContent = `
                    @keyframes slideInRight {
                        from { transform: translateX(100%); opacity: 0; }
                        to { transform: translateX(0); opacity: 1; }
                    }
                    @keyframes slideOutRight {
                        from { transform: translateX(0); opacity: 1; }
                        to { transform: translateX(100%); opacity: 0; }
                    }
                `;
                document.head.appendChild(style);
            }
            
            // Add to page
            document.body.appendChild(notification);
            
            // Auto remove after 3 seconds
            setTimeout(() => {
                notification.style.animation = 'slideOutRight 0.3s ease';
                setTimeout(() => {
                    if (notification.parentNode) {
                        notification.parentNode.removeChild(notification);
                    }
                }, 300);
            }, 3000);
        }

        // Initialize cart badge
        document.addEventListener('DOMContentLoaded', function() {
            updateCartBadge();
        });
    </script>
</body>
</html>