<?php
/**
 * Test Simple Analytics System
 */

require_once 'config/database.php';

echo "<h2>Simple Analytics System Test</h2>";
echo "<style>
    body { font-family: Arial, sans-serif; padding: 20px; }
    .success { background: #d4edda; padding: 15px; border-left: 4px solid #28a745; margin: 10px 0; }
    .error { background: #f8d7da; padding: 15px; border-left: 4px solid #dc3545; margin: 10px 0; }
    table { border-collapse: collapse; width: 100%; margin: 20px 0; }
    th, td { border: 1px solid #ddd; padding: 12px; text-align: left; }
    th { background-color: #4CAF50; color: white; }
    tr:nth-child(even) { background-color: #f2f2f2; }
    .highlight { background-color: #fff3cd !important; font-weight: bold; }
</style>";

try {
    $conn = getDB();
    
    // Check if table exists
    $stmt = $conn->query("SHOW TABLES LIKE 'product_analytics_simple'");
    if ($stmt->rowCount() == 0) {
        echo "<div class='error'>";
        echo "<h3>❌ Table Not Found</h3>";
        echo "<p>Please run <a href='setup-simple-analytics.php'>setup-simple-analytics.php</a> first</p>";
        echo "</div>";
        exit;
    }
    
    // Test: Simulate tracking events
    echo "<h3>Testing Tracking...</h3>";
    
    $testProductId = 22; // Change to your product ID
    
    // Simulate view
    $stmt = $conn->prepare("
        INSERT INTO product_analytics_simple (product_id, views, favorites, add_to_cart)
        VALUES (?, 1, 0, 0)
        ON DUPLICATE KEY UPDATE 
            views = views + 1,
            last_viewed = NOW(),
            conversion_rate = ROUND((add_to_cart / views) * 100, 2)
    ");
    $stmt->execute([$testProductId]);
    
    // Simulate favorite
    $stmt = $conn->prepare("
        UPDATE product_analytics_simple 
        SET favorites = favorites + 1, last_favorited = NOW()
        WHERE product_id = ?
    ");
    $stmt->execute([$testProductId]);
    
    // Simulate add to cart
    $stmt = $conn->prepare("
        UPDATE product_analytics_simple 
        SET add_to_cart = add_to_cart + 1, 
            last_added_to_cart = NOW(),
            conversion_rate = ROUND((add_to_cart / views) * 100, 2)
        WHERE product_id = ?
    ");
    $stmt->execute([$testProductId]);
    
    echo "<div class='success'>";
    echo "<p>✅ Simulated: 1 view, 1 favorite, 1 add_to_cart for product #$testProductId</p>";
    echo "</div>";
    
    // Show all data
    echo "<h3>All Analytics Data:</h3>";
    $stmt = $conn->query("
        SELECT 
            pas.product_id,
            p.name,
            pas.views,
            pas.favorites,
            pas.add_to_cart,
            pas.conversion_rate,
            pas.last_viewed,
            pas.updated_at
        FROM product_analytics_simple pas
        LEFT JOIN products p ON pas.product_id = p.id
        ORDER BY pas.updated_at DESC
    ");
    
    $products = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if (count($products) > 0) {
        echo "<table>";
        echo "<tr>
                <th>Product ID</th>
                <th>Name</th>
                <th>Views</th>
                <th>Favorites</th>
                <th>Add to Cart</th>
                <th>Conversion %</th>
                <th>Last Viewed</th>
                <th>Updated</th>
              </tr>";
        
        foreach ($products as $product) {
            $isTest = ($product['product_id'] == $testProductId);
            $rowClass = $isTest ? 'highlight' : '';
            
            echo "<tr class='$rowClass'>";
            echo "<td>{$product['product_id']}" . ($isTest ? " 🎯" : "") . "</td>";
            echo "<td>" . htmlspecialchars($product['name'] ?? 'N/A') . "</td>";
            echo "<td><strong>{$product['views']}</strong></td>";
            echo "<td><strong>{$product['favorites']}</strong></td>";
            echo "<td><strong>{$product['add_to_cart']}</strong></td>";
            echo "<td>{$product['conversion_rate']}%</td>";
            echo "<td>" . ($product['last_viewed'] ?? 'N/A') . "</td>";
            echo "<td>{$product['updated_at']}</td>";
            echo "</tr>";
        }
        echo "</table>";
        
        echo "<p><strong>Total products tracked:</strong> " . count($products) . "</p>";
    }
    
    // Comparison with old system
    $stmt = $conn->query("SHOW TABLES LIKE 'product_analytics'");
    if ($stmt->rowCount() > 0) {
        echo "<hr>";
        echo "<h3>📊 Comparison: Old vs New System</h3>";
        
        // Old system row count
        $stmt = $conn->query("SELECT COUNT(*) as count FROM product_analytics");
        $oldCount = $stmt->fetch()['count'];
        
        // New system row count
        $stmt = $conn->query("SELECT COUNT(*) as count FROM product_analytics_simple");
        $newCount = $stmt->fetch()['count'];
        
        echo "<div class='success'>";
        echo "<table style='width: auto;'>";
        echo "<tr><th>System</th><th>Rows in Database</th><th>Efficiency</th></tr>";
        echo "<tr><td>Old (Event-Based)</td><td><strong>$oldCount rows</strong></td><td>❌ Many rows</td></tr>";
        echo "<tr class='highlight'><td>New (Counter-Based)</td><td><strong>$newCount rows</strong></td><td>✅ Much smaller!</td></tr>";
        echo "</table>";
        echo "<p><strong>Space saved:</strong> " . ($oldCount - $newCount) . " rows eliminated!</p>";
        echo "</div>";
    }
    
} catch (Exception $e) {
    echo "<div class='error'>";
    echo "<h3>❌ Error:</h3>";
    echo "<p>" . $e->getMessage() . "</p>";
    echo "</div>";
}
?>
