<?php
/**
 * ToyyibPay Connection Test
 * This file helps diagnose ToyyibPay API connection issues
 */

require_once 'config/payment.php';
require_once 'models/ToyyibPay.php';

// Enable error display for testing
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

echo "<h1>ToyyibPay Connection Test</h1>";
echo "<style>
    body { font-family: Arial, sans-serif; padding: 20px; background: #f5f5f5; }
    .test-section { background: white; padding: 20px; margin: 10px 0; border-radius: 5px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
    .success { color: green; }
    .error { color: red; }
    .warning { color: orange; }
    pre { background: #f0f0f0; padding: 10px; border-radius: 3px; overflow-x: auto; }
</style>";

// Test 1: Configuration Check
echo "<div class='test-section'>";
echo "<h2>1. Configuration Check</h2>";
echo "<p><strong>Environment:</strong> " . PAYMENT_ENVIRONMENT . "</p>";
echo "<p><strong>API URL:</strong> " . TOYYIBPAY_API_URL . "</p>";
echo "<p><strong>Category Code:</strong> " . (TOYYIBPAY_CATEGORY_CODE ? '✓ Set' : '<span class="error">✗ Not Set</span>') . "</p>";
echo "<p><strong>Secret Key:</strong> " . (TOYYIBPAY_USER_SECRET_KEY ? '✓ Set (' . substr(TOYYIBPAY_USER_SECRET_KEY, 0, 10) . '...)' : '<span class="error">✗ Not Set</span>') . "</p>";
echo "<p><strong>Return URL:</strong> " . TOYYIBPAY_RETURN_URL . "</p>";
echo "<p><strong>Callback URL:</strong> " . TOYYIBPAY_CALLBACK_URL . "</p>";
echo "</div>";

// Test 2: cURL Extension Check
echo "<div class='test-section'>";
echo "<h2>2. cURL Extension Check</h2>";
if (function_exists('curl_version')) {
    $curl_version = curl_version();
    echo "<p class='success'>✓ cURL is installed</p>";
    echo "<p><strong>Version:</strong> " . $curl_version['version'] . "</p>";
    echo "<p><strong>SSL Version:</strong> " . $curl_version['ssl_version'] . "</p>";
} else {
    echo "<p class='error'>✗ cURL is NOT installed</p>";
}
echo "</div>";

// Test 3: API Connection Test
echo "<div class='test-section'>";
echo "<h2>3. API Connection Test</h2>";
$toyyibpay = new ToyyibPay();

// Test category details
echo "<h3>Testing Category Access...</h3>";
$category_result = $toyyibpay->getCategoryDetails();
echo "<pre>";
print_r($category_result);
echo "</pre>";

if ($category_result && !isset($category_result[0]['error'])) {
    echo "<p class='success'>✓ Category access successful</p>";
} else {
    echo "<p class='error'>✗ Category access failed</p>";
    if (isset($category_result[0]['error'])) {
        echo "<p class='error'>Error: " . $category_result[0]['error'] . "</p>";
    }
}
echo "</div>";

// Test 4: Test Bill Creation (Small Amount)
echo "<div class='test-section'>";
echo "<h2>4. Test Bill Creation</h2>";
echo "<p>Creating a test bill with RM 1.00...</p>";

$test_bill_data = [
    'bill_name' => 'Test Order #TEST' . time(),
    'bill_description' => 'Test Payment - Do Not Pay',
    'amount' => 1.00,
    'order_id' => 'TEST' . time(),
    'customer_name' => 'Test Customer',
    'customer_email' => 'test@example.com',
    'customer_phone' => '0123456789',
    'payment_method' => 'toyyibpay_fpx'
];

$bill_result = $toyyibpay->createBill($test_bill_data);
echo "<h3>Bill Creation Result:</h3>";
echo "<pre>";
print_r($bill_result);
echo "</pre>";

if ($bill_result && isset($bill_result['bill_code'])) {
    echo "<p class='success'>✓ Test bill created successfully!</p>";
    echo "<p><strong>Bill Code:</strong> " . $bill_result['bill_code'] . "</p>";
    echo "<p><strong>Payment URL:</strong> <a href='" . $bill_result['payment_url'] . "' target='_blank'>" . $bill_result['payment_url'] . "</a></p>";
    echo "<p class='warning'>⚠️ This is a test bill. You can view it but please do not make actual payment.</p>";
} else {
    echo "<p class='error'>✗ Test bill creation failed</p>";
    if (isset($bill_result['error'])) {
        echo "<p class='error'><strong>Error Type:</strong> " . $bill_result['error'] . "</p>";
        echo "<p class='error'><strong>Error Message:</strong> " . $bill_result['message'] . "</p>";
        
        // Provide specific guidance based on error
        if ($bill_result['error'] === 'insufficient_prepaid') {
            echo "<div style='background: #fff3cd; padding: 15px; border-left: 4px solid #ffc107; margin-top: 10px;'>";
            echo "<h4>⚠️ Insufficient Prepaid Balance</h4>";
            echo "<p>Your ToyyibPay account does not have sufficient prepaid balance to create bills.</p>";
            echo "<p><strong>Solution:</strong></p>";
            echo "<ol>";
            echo "<li>Log in to your ToyyibPay account at <a href='https://toyyibpay.com' target='_blank'>https://toyyibpay.com</a></li>";
            echo "<li>Go to 'Prepaid' section</li>";
            echo "<li>Top up your prepaid balance</li>";
            echo "<li>Try again after the balance is topped up</li>";
            echo "</ol>";
            echo "</div>";
        } elseif ($bill_result['error'] === 'api_error') {
            echo "<div style='background: #f8d7da; padding: 15px; border-left: 4px solid #dc3545; margin-top: 10px;'>";
            echo "<h4>❌ API Error</h4>";
            echo "<p>The ToyyibPay API returned an error. This could be due to:</p>";
            echo "<ul>";
            echo "<li>Invalid API credentials (Secret Key or Category Code)</li>";
            echo "<li>Category is inactive or deleted</li>";
            echo "<li>API endpoint issues</li>";
            echo "</ul>";
            echo "<p><strong>Recommended Actions:</strong></p>";
            echo "<ol>";
            echo "<li>Verify your Secret Key in config/payment.php</li>";
            echo "<li>Verify your Category Code in config/payment.php</li>";
            echo "<li>Check if your ToyyibPay account is active</li>";
            echo "<li>Contact ToyyibPay support if the issue persists</li>";
            echo "</ol>";
            echo "</div>";
        }
    }
}
echo "</div>";

// Test 5: Check Recent Error Logs
echo "<div class='test-section'>";
echo "<h2>5. Recent Error Logs</h2>";
echo "<p>Check your PHP error log file for detailed error messages.</p>";
echo "<p><strong>Common log locations:</strong></p>";
echo "<ul>";
echo "<li>Linux/Apache: /var/log/apache2/error.log or /var/log/httpd/error_log</li>";
echo "<li>Windows/XAMPP: C:\\xampp\\apache\\logs\\error.log</li>";
echo "<li>cPanel: public_html/error_log</li>";
echo "</ul>";
echo "<p>Look for entries containing 'toyyibPay' for detailed debugging information.</p>";
echo "</div>";

// Test 6: Recommendations
echo "<div class='test-section'>";
echo "<h2>6. Troubleshooting Recommendations</h2>";
echo "<ol>";
echo "<li><strong>If you see 'insufficient_prepaid' error:</strong> Top up your ToyyibPay prepaid balance</li>";
echo "<li><strong>If you see 'api_error':</strong> Verify your API credentials in config/payment.php</li>";
echo "<li><strong>If you see cURL errors:</strong> Check your server's internet connectivity and SSL certificates</li>";
echo "<li><strong>If category access fails:</strong> Verify the category code is correct and active in your ToyyibPay account</li>";
echo "<li><strong>If all tests pass but checkout still fails:</strong> Check the PHP error logs during actual checkout</li>";
echo "</ol>";
echo "</div>";

echo "<div class='test-section'>";
echo "<p><a href='checkout.php'>← Back to Checkout</a></p>";
echo "</div>";
?>

