<?php
require_once __DIR__ . '/../../config/database.php';

class Voucher {
    private $conn;
    
    public function __construct() {
        $this->conn = getDB();
    }
    
    /**
     * Get all vouchers with pagination
     */
    public function getAllVouchers($limit = 20, $offset = 0, $search = '') {
        $query = "SELECT * FROM vouchers WHERE 1=1";
        $params = [];
        
        if (!empty($search)) {
            $query .= " AND (voucher_name LIKE :search OR voucher_description LIKE :search)";
            $params['search'] = "%$search%";
        }
        
        $query .= " ORDER BY created_at DESC LIMIT :limit OFFSET :offset";
        
        $stmt = $this->conn->prepare($query);
        
        foreach ($params as $key => $value) {
            $stmt->bindValue(':' . $key, $value);
        }
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * Get all point codes with pagination
     */
    public function getAllPointCodes($limit = 20, $offset = 0, $search = '') {
        $query = "SELECT * FROM point_codes WHERE 1=1";
        $params = [];
        
        if (!empty($search)) {
            $query .= " AND (code LIKE :search OR description LIKE :search)";
            $params['search'] = "%$search%";
        }
        
        $query .= " ORDER BY created_at DESC LIMIT :limit OFFSET :offset";
        
        $stmt = $this->conn->prepare($query);
        
        foreach ($params as $key => $value) {
            $stmt->bindValue(':' . $key, $value);
        }
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * Get voucher by ID
     */
    public function getVoucherById($id) {
        $query = "SELECT * FROM vouchers WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    /**
     * Create new voucher
     */
    public function createVoucher($data) {
        $query = "INSERT INTO vouchers (
            voucher_name, voucher_description, voucher_image, points_required,
            discount_type, discount_value, max_redemptions, valid_from, valid_until,
            is_active, created_by
        ) VALUES (
            :voucher_name, :voucher_description, :voucher_image, :points_required,
            :discount_type, :discount_value, :max_redemptions, :valid_from, :valid_until,
            :is_active, :created_by
        )";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':voucher_name', $data['voucher_name']);
        $stmt->bindValue(':voucher_description', $data['voucher_description']);
        $stmt->bindValue(':voucher_image', $data['voucher_image'] ?? null);
        $stmt->bindValue(':points_required', $data['points_required'], PDO::PARAM_INT);
        $stmt->bindValue(':discount_type', $data['discount_type']);
        $stmt->bindValue(':discount_value', $data['discount_value']);
        $stmt->bindValue(':max_redemptions', $data['max_redemptions'] ?? null, PDO::PARAM_INT);
        $stmt->bindValue(':valid_from', $data['valid_from']);
        $stmt->bindValue(':valid_until', $data['valid_until']);
        $stmt->bindValue(':is_active', $data['is_active'] ?? true, PDO::PARAM_BOOL);
        $stmt->bindValue(':created_by', $data['created_by'] ?? null, PDO::PARAM_INT);
        
        return $stmt->execute();
    }
    
    /**
     * Update voucher
     */
    public function updateVoucher($id, $data) {
        $query = "UPDATE vouchers SET 
            voucher_name = :voucher_name,
            voucher_description = :voucher_description,
            voucher_image = :voucher_image,
            points_required = :points_required,
            discount_type = :discount_type,
            discount_value = :discount_value,
            max_redemptions = :max_redemptions,
            valid_from = :valid_from,
            valid_until = :valid_until,
            is_active = :is_active,
            updated_at = CURRENT_TIMESTAMP
        WHERE id = :id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        $stmt->bindValue(':voucher_name', $data['voucher_name']);
        $stmt->bindValue(':voucher_description', $data['voucher_description']);
        $stmt->bindValue(':voucher_image', $data['voucher_image'] ?? null);
        $stmt->bindValue(':points_required', $data['points_required'], PDO::PARAM_INT);
        $stmt->bindValue(':discount_type', $data['discount_type']);
        $stmt->bindValue(':discount_value', $data['discount_value']);
        $stmt->bindValue(':max_redemptions', $data['max_redemptions'] ?? null, PDO::PARAM_INT);
        $stmt->bindValue(':valid_from', $data['valid_from']);
        $stmt->bindValue(':valid_until', $data['valid_until']);
        $stmt->bindValue(':is_active', $data['is_active'] ?? true, PDO::PARAM_BOOL);
        
        return $stmt->execute();
    }
    
    /**
     * Delete voucher
     */
    public function deleteVoucher($id) {
        $query = "DELETE FROM vouchers WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        return $stmt->execute();
    }
    
    /**
     * Get available vouchers for user (based on points and validity)
     */
    public function getAvailableVouchers($userId, $userPoints) {
        $query = "SELECT v.*, 
                    (v.max_redemptions - v.current_redemptions) as remaining_redemptions
                  FROM vouchers v 
                  WHERE v.is_active = 1 
                    AND v.points_required <= :user_points
                    AND v.valid_from <= CURDATE() 
                    AND v.valid_until >= CURDATE()
                    AND (v.max_redemptions IS NULL OR v.current_redemptions < v.max_redemptions)
                    AND v.id NOT IN (
                        SELECT voucher_id FROM user_voucher_redemptions 
                        WHERE user_id = :user_id AND status IN ('active', 'used')
                    )
                  ORDER BY v.points_required ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':user_points', $userPoints, PDO::PARAM_INT);
        $stmt->bindValue(':user_id', $userId, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * Redeem voucher for user
     */
    public function redeemVoucher($userId, $voucherId, $orderId = null) {
        $this->conn->beginTransaction();
        
        try {
            // Get voucher details
            $voucher = $this->getVoucherById($voucherId);
            if (!$voucher || !$voucher['is_active']) {
                throw new Exception('Voucher not available');
            }
            
            // Check if user already redeemed this voucher (active or used)
            $checkQuery = "SELECT id FROM user_voucher_redemptions 
                          WHERE user_id = :user_id AND voucher_id = :voucher_id AND status IN ('active', 'used')";
            $checkStmt = $this->conn->prepare($checkQuery);
            $checkStmt->bindValue(':user_id', $userId, PDO::PARAM_INT);
            $checkStmt->bindValue(':voucher_id', $voucherId, PDO::PARAM_INT);
            $checkStmt->execute();
            
            if ($checkStmt->fetch()) {
                throw new Exception('Voucher already redeemed or used');
            }
            
            // Insert redemption record
            $redemptionQuery = "INSERT INTO user_voucher_redemptions 
                               (user_id, voucher_id, order_id, points_used, status) 
                               VALUES (:user_id, :voucher_id, :order_id, :points_used, 'active')";
            $redemptionStmt = $this->conn->prepare($redemptionQuery);
            $redemptionStmt->bindValue(':user_id', $userId, PDO::PARAM_INT);
            $redemptionStmt->bindValue(':voucher_id', $voucherId, PDO::PARAM_INT);
            $redemptionStmt->bindValue(':order_id', $orderId, PDO::PARAM_INT);
            $redemptionStmt->bindValue(':points_used', $voucher['points_required'], PDO::PARAM_INT);
            $redemptionStmt->execute();
            
            // Update voucher redemption count
            $updateQuery = "UPDATE vouchers SET current_redemptions = current_redemptions + 1 WHERE id = :id";
            $updateStmt = $this->conn->prepare($updateQuery);
            $updateStmt->bindValue(':id', $voucherId, PDO::PARAM_INT);
            $updateStmt->execute();
            
            $this->conn->commit();
            return true;
            
        } catch (Exception $e) {
            $this->conn->rollBack();
            throw $e;
        }
    }
    
    /**
     * Get all royal tiers
     */
    public function getAllRoyalTiers() {
        $query = "SELECT * FROM royal_tiers WHERE is_active = 1 ORDER BY tier_level ASC";
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * Get royal tier by level
     */
    public function getRoyalTierByLevel($level) {
        $query = "SELECT * FROM royal_tiers WHERE tier_level = :level AND is_active = 1";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':level', $level, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    /**
     * Get user's current royal tier based on points
     */
    public function getUserRoyalTier($userPoints) {
        $query = "SELECT * FROM royal_tiers 
                  WHERE min_points <= :points 
                    AND (max_points IS NULL OR max_points >= :points)
                    AND is_active = 1 
                  ORDER BY tier_level DESC 
                  LIMIT 1";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':points', $userPoints, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    /**
     * Get voucher statistics
     */
    public function getVoucherStats() {
        $query = "SELECT 
                    COUNT(*) as total_vouchers,
                    SUM(CASE WHEN is_active = 1 THEN 1 ELSE 0 END) as active_vouchers,
                    SUM(current_redemptions) as total_redemptions,
                    AVG(points_required) as avg_points_required
                  FROM vouchers";
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    /**
     * Check if user has already redeemed a specific voucher
     */
    public function checkUserVoucherRedemption($userId, $voucherId) {
        $query = "SELECT * FROM user_voucher_redemptions 
                  WHERE user_id = :user_id AND voucher_id = :voucher_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':user_id', $userId, PDO::PARAM_INT);
        $stmt->bindValue(':voucher_id', $voucherId, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    /**
     * Get user's redeemed vouchers
     */
    public function getUserRedeemedVouchers($userId, $status = null) {
        $query = "SELECT uvr.*, v.voucher_name, v.voucher_description, v.voucher_image, v.discount_type, v.discount_value
                  FROM user_voucher_redemptions uvr
                  JOIN vouchers v ON uvr.voucher_id = v.id
                  WHERE uvr.user_id = :user_id";
        
        if ($status !== null) {
            $query .= " AND uvr.status = :status";
        }
        
        $query .= " ORDER BY uvr.redeemed_at DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':user_id', $userId, PDO::PARAM_INT);
        
        if ($status !== null) {
            $stmt->bindValue(':status', $status);
        }
        
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * Mark voucher as used
     */
    public function markVoucherAsUsed($userId, $voucherId) {
        $query = "UPDATE user_voucher_redemptions 
                  SET status = 'used' 
                  WHERE user_id = :user_id AND voucher_id = :voucher_id AND status = 'active'";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':user_id', $userId, PDO::PARAM_INT);
        $stmt->bindValue(':voucher_id', $voucherId, PDO::PARAM_INT);
        return $stmt->execute();
    }
    
    /**
     * Check if voucher is already used by user
     */
    public function isVoucherUsed($userId, $voucherId) {
        $query = "SELECT status FROM user_voucher_redemptions 
                  WHERE user_id = :user_id AND voucher_id = :voucher_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':user_id', $userId, PDO::PARAM_INT);
        $stmt->bindValue(':voucher_id', $voucherId, PDO::PARAM_INT);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result && $result['status'] === 'used';
    }

    /**
     * Get point code by ID
     */
    public function getPointCodeById($id) {
        $query = "SELECT * FROM point_codes WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    /**
     * Create new point code
     */
    public function createPointCode($data) {
        $query = "INSERT INTO point_codes (code, points, description, max_uses, valid_from, valid_until, is_active, created_by) 
                  VALUES (:code, :points, :description, :max_uses, :valid_from, :valid_until, :is_active, :created_by)";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':code', $data['code']);
        $stmt->bindValue(':points', $data['points'], PDO::PARAM_INT);
        $stmt->bindValue(':description', $data['description']);
        $stmt->bindValue(':max_uses', $data['max_uses'], PDO::PARAM_INT);
        $stmt->bindValue(':valid_from', $data['valid_from']);
        $stmt->bindValue(':valid_until', $data['valid_until']);
        $stmt->bindValue(':is_active', $data['is_active'], PDO::PARAM_BOOL);
        $stmt->bindValue(':created_by', $data['created_by'], PDO::PARAM_INT);
        
        return $stmt->execute();
    }

    /**
     * Update point code
     */
    public function updatePointCode($id, $data) {
        $query = "UPDATE point_codes 
                  SET code = :code, points = :points, description = :description, 
                      max_uses = :max_uses, valid_from = :valid_from, valid_until = :valid_until, 
                      is_active = :is_active, updated_at = CURRENT_TIMESTAMP
                  WHERE id = :id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        $stmt->bindValue(':code', $data['code']);
        $stmt->bindValue(':points', $data['points'], PDO::PARAM_INT);
        $stmt->bindValue(':description', $data['description']);
        $stmt->bindValue(':max_uses', $data['max_uses'], PDO::PARAM_INT);
        $stmt->bindValue(':valid_from', $data['valid_from']);
        $stmt->bindValue(':valid_until', $data['valid_until']);
        $stmt->bindValue(':is_active', $data['is_active'], PDO::PARAM_BOOL);
        
        return $stmt->execute();
    }

    /**
     * Delete point code
     */
    public function deletePointCode($id) {
        $query = "DELETE FROM point_codes WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        return $stmt->execute();
    }

    /**
     * Get point code statistics
     */
    public function getPointCodeStats() {
        $query = "SELECT 
                    COUNT(*) as total_codes,
                    SUM(CASE WHEN is_active = 1 THEN 1 ELSE 0 END) as active_codes,
                    SUM(current_uses) as total_uses,
                    SUM(points * current_uses) as total_points_awarded
                  FROM point_codes";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
}
?>