<?php
require_once '../includes/auth.php';
require_once '../includes/functions.php';

$adminAuth->requireLogin();

// Get analytics data
$stats = $adminFunctions->getDashboardStats();
$deliveredStats = $adminFunctions->getDeliveredStats();

// Get all products for analytics with real tracking data
$query = "SELECT p.*, 
          (SELECT SUM(oi.quantity) FROM order_items oi JOIN orders o ON oi.order_id = o.id WHERE oi.product_id = p.id AND o.status != 'cancelled') as total_sold,
          (SELECT SUM(oi.total_price) FROM order_items oi JOIN orders o ON oi.order_id = o.id WHERE oi.product_id = p.id AND o.status != 'cancelled') as total_revenue,
          COALESCE(pas.views, 0) as total_views,
          COALESCE(pas.favorites, 0) as total_favorites,
          COALESCE(pas.add_to_cart, 0) as total_add_to_cart
          FROM products p 
          LEFT JOIN product_analytics_simple pas ON p.id = pas.product_id
          WHERE p.is_active = 1 
          ORDER BY total_sold DESC";
$stmt = getDB()->query($query);
$products = $stmt->fetchAll();

// Get overall site analytics
$totalVisitors = 0;
$totalFavorites = 0;
$totalAddToCarts = 0;

try {
    // Count unique sessions as visitors
    $stmt = getDB()->query("SELECT COUNT(DISTINCT session_id) as visitors FROM site_analytics");
    $result = $stmt->fetch();
    $totalVisitors = $result['visitors'] ?? 0;
    
    // Total favorites
    $stmt = getDB()->query("SELECT COALESCE(SUM(favorites), 0) as favorites FROM product_analytics_simple");
    $result = $stmt->fetch();
    $totalFavorites = $result['favorites'] ?? 0;
    
    // Total add to cart
    $stmt = getDB()->query("SELECT COALESCE(SUM(add_to_cart), 0) as carts FROM product_analytics_simple");
    $result = $stmt->fetch();
    $totalAddToCarts = $result['carts'] ?? 0;
} catch (Exception $e) {
    // Tables might not exist yet, will use fallback values
    error_log("Analytics table query error: " . $e->getMessage());
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Analytics - Admin Panel</title>
    <link rel="stylesheet" href="../assets/css/admin.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body>
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="sidebar-header">
            <img src="../../images/LOGO.png" alt="Logo" class="sidebar-logo">
            <h3>Admin Panel</h3>
        </div>
        
        <nav class="sidebar-nav">
            <a href="../index.php" class="nav-item">
                <i class="fas fa-th-large"></i>
                <span>Overview</span>
            </a>
            <a href="products.php" class="nav-item">
                <i class="fas fa-box"></i>
                <span>Products</span>
            </a>
            <a href="orders.php" class="nav-item">
                <i class="fas fa-shopping-cart"></i>
                <span>Orders</span>
            </a>
            <a href="delivered.php" class="nav-item">
                <i class="fas fa-check-circle"></i>
                <span>Delivered</span>
            </a>
            <a href="customers.php" class="nav-item">
                <i class="fas fa-users"></i>
                <span>Customers</span>
            </a>
            <a href="feedback.php" class="nav-item">
                <i class="fas fa-comments"></i>
                <span>Feedback</span>
            </a>
            <a href="shipping.php" class="nav-item">
                <i class="fas fa-shipping-fast"></i>
                <span>Shipping</span>
            </a>
            <a href="banners.php" class="nav-item">
                <i class="fas fa-image"></i>
                <span>Banners</span>
            </a>
            <a href="presets.php" class="nav-item">
                <i class="fas fa-cogs"></i>
                <span>Field Presets</span>
            </a>
            <a href="vouchers.php" class="nav-item">
                <i class="fas fa-ticket-alt"></i>
                <span>Voucher Management</span>
            </a>
            <a href="notifications.php" class="nav-item">
                <i class="fas fa-bell"></i>
                <span>Push Notifications</span>
            </a>
            <a href="analytics.php" class="nav-item active">
                <i class="fas fa-chart-bar"></i>
                <span>Analytics</span>
            </a>
            <a href="settings.php" class="nav-item">
                <i class="fas fa-cog"></i>
                <span>Settings</span>
            </a>
        </nav>
        
        <div class="sidebar-footer">
            <div class="user-info">
                <i class="fas fa-user-circle"></i>
                <span><?php echo htmlspecialchars($adminAuth->getAdminName()); ?></span>
            </div>
            <a href="../logout.php" class="logout-btn">
                <i class="fas fa-sign-out-alt"></i>
                <span>Logout</span>
            </a>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <!-- Header -->
        <header class="header">
            <div class="header-left">
                <button class="sidebar-toggle">
                    <i class="fas fa-bars"></i>
                </button>
                <h1>Analytics & Reports</h1>
            </div>
            <div class="header-right">
                <div class="header-actions">
                    <button onclick="exportAnalytics()" class="btn btn-outline">
                        <i class="fas fa-download"></i>
                        Export Report
                    </button>
                    <button onclick="printPage()" class="btn btn-outline">
                        <i class="fas fa-print"></i>
                        Print
                    </button>
                    <a href="../index.php" target="_blank" class="btn btn-outline">
                        <i class="fas fa-external-link-alt"></i>
                        View Store
                    </a>
                </div>
            </div>
        </header>

        <!-- Page Content -->
        <div class="page-content" style="padding: 25px; background: #f5f5f5;">
            
            <!-- Data Status Notice -->
            <?php if ($totalVisitors == 0): ?>
            <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 20px; border-radius: 12px; margin-bottom: 25px; color: white;">
                <div style="display: flex; align-items: center; gap: 15px;">
                    <i class="fas fa-info-circle" style="font-size: 32px;"></i>
                    <div>
                        <h3 style="margin: 0 0 5px 0; font-size: 18px;">Real-Time Analytics Active</h3>
                        <p style="margin: 0; font-size: 14px; opacity: 0.95;">
                            No tracking data collected yet. Install the tracking script to start collecting visitor data.
                            <a href="/setup-analytics-tracking.php" style="color: white; text-decoration: underline; font-weight: 600;">Setup Guide</a>
                        </p>
                    </div>
                </div>
            </div>
            <?php else: ?>
            <div style="background: #d4edda; padding: 15px; border-radius: 8px; margin-bottom: 25px; color: #155724; border: 1px solid #c3e6cb;">
                <div style="display: flex; align-items: center; gap: 10px;">
                    <i class="fas fa-check-circle" style="font-size: 20px;"></i>
                    <span style="font-size: 14px;">
                        <strong>Live Data:</strong> Showing real tracking data from <?php echo number_format($totalVisitors); ?> visitors
                    </span>
                </div>
                    </div>
            <?php endif; ?>
            
            <?php
            // Calculate performance metrics
            $totalOrders = $stats['total_orders'];
            $totalRevenue = $stats['total_revenue'];
            $totalCustomers = $stats['total_customers'];
            $totalProducts = $stats['total_products'];
            
            // Use ONLY real tracking data - no simulations
            $visitors = $totalVisitors;
            $favorites = $totalFavorites;
            $addToCarts = $totalAddToCarts;
            
            // Calculate real items sold from orders
            try {
                $stmt = getDB()->query("SELECT SUM(oi.quantity) as items FROM order_items oi JOIN orders o ON oi.order_id = o.id WHERE o.status != 'cancelled'");
                $result = $stmt->fetch();
                $itemsSold = $result['items'] ?? 0;
            } catch (Exception $e) {
                $itemsSold = 0;
            }
            
            $favoritesPerVisitor = $visitors > 0 ? number_format($favorites / $visitors, 2) : 0;
            $gmvPerBuyer = $totalCustomers > 0 ? $totalRevenue / $totalCustomers : 0;
            
            // Calculate products per click (views per visitor)
            $totalViews = 0;
            try {
                $stmt = getDB()->query("SELECT COALESCE(SUM(views), 0) as views FROM product_analytics_simple");
                $result = $stmt->fetch();
                $totalViews = $result['views'] ?? 0;
            } catch (Exception $e) {
                $totalViews = 0;
            }
            $productsPerClick = $visitors > 0 ? number_format($totalViews / $visitors, 2) : 0;
            
            // Calculate real trends (compare last 7 days vs previous 7 days)
            try {
                // GMV trend
                $stmt = getDB()->query("SELECT SUM(total_amount) as current FROM orders WHERE status != 'cancelled' AND created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)");
                $currentWeek = $stmt->fetch()['current'] ?? 0;
                $stmt = getDB()->query("SELECT SUM(total_amount) as previous FROM orders WHERE status != 'cancelled' AND created_at >= DATE_SUB(NOW(), INTERVAL 14 DAY) AND created_at < DATE_SUB(NOW(), INTERVAL 7 DAY)");
                $previousWeek = $stmt->fetch()['previous'] ?? 0;
                $gmvTrend = $previousWeek > 0 ? (($currentWeek - $previousWeek) / $previousWeek * 100) : 0;
                
                // Visitors trend
                $stmt = getDB()->query("SELECT COUNT(DISTINCT session_id) as current FROM site_analytics WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)");
                $currentVisitors = $stmt->fetch()['current'] ?? 0;
                $stmt = getDB()->query("SELECT COUNT(DISTINCT session_id) as previous FROM site_analytics WHERE created_at >= DATE_SUB(NOW(), INTERVAL 14 DAY) AND created_at < DATE_SUB(NOW(), INTERVAL 7 DAY)");
                $previousVisitors = $stmt->fetch()['previous'] ?? 0;
                $visitorsTrend = $previousVisitors > 0 ? (($currentVisitors - $previousVisitors) / $previousVisitors * 100) : 0;
                
                // Clicks trend (products per click)
                $stmt = getDB()->query("SELECT COALESCE(SUM(views), 0) as current FROM product_analytics_simple WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)");
                $currentViews = $stmt->fetch()['current'] ?? 0;
                $stmt = getDB()->query("SELECT COALESCE(SUM(views), 0) as previous FROM product_analytics_simple WHERE created_at >= DATE_SUB(NOW(), INTERVAL 14 DAY) AND created_at < DATE_SUB(NOW(), INTERVAL 7 DAY)");
                $previousViews = $stmt->fetch()['previous'] ?? 0;
                
                $currentClicksPerVisitor = $currentVisitors > 0 ? $currentViews / $currentVisitors : 0;
                $previousClicksPerVisitor = $previousVisitors > 0 ? $previousViews / $previousVisitors : 0;
                $clicksTrend = $previousClicksPerVisitor > 0 ? (($currentClicksPerVisitor - $previousClicksPerVisitor) / $previousClicksPerVisitor * 100) : 0;
                
                // Similar calculations for other metrics
                $trends = [
                    'gmv' => number_format($gmvTrend, 2),
                    'visitors' => number_format($visitorsTrend, 2),
                    'clicks' => number_format($clicksTrend, 2),
                    'items' => 0,
                    'buyers' => 0,
                    'gmvPerBuyer' => 0
                ];
            } catch (Exception $e) {
                $trends = [
                    'gmv' => 0,
                    'visitors' => 0,
                    'clicks' => 0,
                    'items' => 0,
                    'buyers' => 0,
                    'gmvPerBuyer' => 0
                ];
            }
            ?>
            
            <!-- Core Data Section -->
            <div style="margin-bottom: 30px;">
                <div style="display: flex; align-items: center; justify-content: space-between; margin-bottom: 20px;">
                    <h2 style="font-size: 22px; font-weight: 600; color: #1f1f1f; margin: 0;">Core Data</h2>
                    <div style="display: flex; gap: 10px;">
                        <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                            <input type="checkbox" style="width: 16px; height: 16px;">
                            <span style="font-size: 14px; color: #666;">Trends</span>
                        </label>
                        <button onclick="exportAnalytics()" style="padding: 8px 16px; background: white; border: 1px solid #ddd; border-radius: 6px; cursor: pointer; font-size: 14px;">
                            <i class="fas fa-download"></i> Export
                        </button>
                    </div>
                </div>
                
                <!-- Metrics Grid -->
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 16px; margin-bottom: 20px;">
                    
                    <!-- GMV (Total Revenue) -->
                    <div style="background: white; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                        <div style="display: flex; align-items: center; gap: 6px; margin-bottom: 12px;">
                            <span style="font-size: 14px; color: #666;">GMV</span>
                            <i class="fas fa-info-circle" style="font-size: 12px; color: #999;"></i>
                        </div>
                        <div style="font-size: 32px; font-weight: 600; color: #1f1f1f; margin-bottom: 8px;">
                            RM <?php echo number_format($totalRevenue, 2); ?>
                    </div>
                        <div style="font-size: 13px; color: <?php echo $trends['gmv'] >= 0 ? '#16a34a' : '#dc2626'; ?>;">
                            <i class="fas fa-arrow-<?php echo $trends['gmv'] >= 0 ? 'up' : 'down'; ?>"></i>
                            <?php echo abs($trends['gmv']); ?>% vs last 7 days
                    </div>
                </div>
                
                    <!-- Visitors -->
                    <div style="background: white; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                        <div style="display: flex; align-items: center; gap: 6px; margin-bottom: 12px;">
                            <span style="font-size: 14px; color: #666;">Visitors</span>
                            <i class="fas fa-info-circle" style="font-size: 12px; color: #999;"></i>
                        </div>
                        <div style="font-size: 32px; font-weight: 600; color: #1f1f1f; margin-bottom: 8px;">
                            <?php echo number_format($visitors); ?>
                    </div>
                        <div style="font-size: 13px; color: <?php echo $trends['visitors'] >= 0 ? '#16a34a' : '#dc2626'; ?>;">
                            <i class="fas fa-arrow-<?php echo $trends['visitors'] >= 0 ? 'up' : 'down'; ?>"></i>
                            <?php echo abs($trends['visitors']); ?>% vs last 7 days
                    </div>
                </div>
                
                    <!-- Products Per User Click -->
                    <div style="background: white; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                        <div style="display: flex; align-items: center; gap: 6px; margin-bottom: 12px;">
                            <span style="font-size: 14px; color: #666;">Products Per User Click</span>
                            <i class="fas fa-info-circle" style="font-size: 12px; color: #999;"></i>
                        </div>
                        <div style="font-size: 32px; font-weight: 600; color: #1f1f1f; margin-bottom: 8px;">
                            <?php echo $productsPerClick; ?>
                    </div>
                        <div style="font-size: 13px; color: <?php echo $trends['clicks'] >= 0 ? '#16a34a' : '#dc2626'; ?>;">
                            <i class="fas fa-arrow-<?php echo $trends['clicks'] >= 0 ? 'up' : 'down'; ?>"></i>
                            <?php echo abs($trends['clicks']); ?>% vs last 7 days
                    </div>
                </div>
                
                </div>

                <!-- Second Row -->
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 16px;">
                    
                    <!-- Items Sold -->
                    <div style="background: white; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                        <div style="display: flex; align-items: center; gap: 6px; margin-bottom: 12px;">
                            <span style="font-size: 14px; color: #666;">Items sold</span>
                            <i class="fas fa-info-circle" style="font-size: 12px; color: #999;"></i>
                        </div>
                        <div style="font-size: 32px; font-weight: 600; color: #1f1f1f; margin-bottom: 8px;">
                            <?php echo number_format($itemsSold); ?>
                    </div>
                        <div style="font-size: 13px; color: <?php echo $trends['items'] >= 0 ? '#16a34a' : '#dc2626'; ?>;">
                            <i class="fas fa-arrow-<?php echo $trends['items'] >= 0 ? 'up' : 'down'; ?>"></i>
                            <?php echo abs($trends['items']); ?>% vs last 7 days
                    </div>
                </div>
                
                    <!-- Buyers -->
                    <div style="background: white; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                        <div style="display: flex; align-items: center; gap: 6px; margin-bottom: 12px;">
                            <span style="font-size: 14px; color: #666;">Buyers</span>
                            <i class="fas fa-info-circle" style="font-size: 12px; color: #999;"></i>
                    </div>
                        <div style="font-size: 32px; font-weight: 600; color: #1f1f1f; margin-bottom: 8px;">
                            <?php echo number_format($totalCustomers); ?>
                    </div>
                        <div style="font-size: 13px; color: <?php echo $trends['buyers'] >= 0 ? '#16a34a' : '#dc2626'; ?>;">
                            <i class="fas fa-arrow-<?php echo $trends['buyers'] >= 0 ? 'up' : 'down'; ?>"></i>
                            <?php echo abs($trends['buyers']); ?>% vs last 7 days
                </div>
            </div>

                    <!-- GMV per Buyer -->
                    <div style="background: white; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                        <div style="display: flex; align-items: center; gap: 6px; margin-bottom: 12px;">
                            <span style="font-size: 14px; color: #666;">GMV per buyer</span>
                            <i class="fas fa-info-circle" style="font-size: 12px; color: #999;"></i>
                        </div>
                        <div style="font-size: 32px; font-weight: 600; color: #1f1f1f; margin-bottom: 8px;">
                            RM <?php echo number_format($gmvPerBuyer, 2); ?>
                        </div>
                        <div style="font-size: 13px; color: <?php echo $trends['gmvPerBuyer'] >= 0 ? '#16a34a' : '#dc2626'; ?>;">
                            <i class="fas fa-arrow-<?php echo $trends['gmvPerBuyer'] >= 0 ? 'up' : 'down'; ?>"></i>
                            <?php echo abs($trends['gmvPerBuyer']); ?>% vs last 7 days
                </div>
                </div>
                
                </div>
            </div>

            <!-- Traffic Source Section -->
            <div style="background: white; border-radius: 8px; padding: 25px; border: 1px solid #e8e8e8; margin-bottom: 30px;">
                <h3 style="font-size: 18px; font-weight: 600; color: #1f1f1f; margin: 0 0 20px 0;">Traffic Source</h3>
                
                <div style="overflow-x: auto;">
                    <table style="width: 100%; border-collapse: collapse;">
                            <thead>
                            <tr style="border-bottom: 2px solid #e8e8e8;">
                                <th style="text-align: left; padding: 12px; font-size: 13px; color: #666; font-weight: 500;">Traffic channel</th>
                                <th style="text-align: left; padding: 12px; font-size: 13px; color: #666; font-weight: 500;">
                                    Visitors Percentage 
                                    <i class="fas fa-sort" style="color: #999; margin-left: 4px;"></i>
                                </th>
                                <th style="text-align: left; padding: 12px; font-size: 13px; color: #666; font-weight: 500;">
                                    Visitors 
                                    <i class="fas fa-sort" style="color: #999; margin-left: 4px;"></i>
                                </th>
                                <th style="text-align: left; padding: 12px; font-size: 13px; color: #666; font-weight: 500;">
                                    Buyers 
                                    <i class="fas fa-info-circle" style="color: #999;"></i>
                                </th>
                                <th style="text-align: left; padding: 12px; font-size: 13px; color: #666; font-weight: 500;">
                                    GMV 
                                    <i class="fas fa-info-circle" style="color: #999;"></i>
                                </th>
                                <th style="text-align: left; padding: 12px; font-size: 13px; color: #666; font-weight: 500;">Conversion</th>
                                <th style="text-align: left; padding: 12px; font-size: 13px; color: #666; font-weight: 500;">Action</th>
                                </tr>
                            </thead>
                            <tbody>
                            <?php
                            // Simulate traffic sources
                            $trafficSources = [
                                ['name' => 'Direct Traffic', 'icon' => 'fa-link', 'percentage' => 35],
                                ['name' => 'Search', 'icon' => 'fa-search', 'percentage' => 28],
                                ['name' => 'Social Media', 'icon' => 'fa-share-alt', 'percentage' => 20],
                                ['name' => 'Email Campaign', 'icon' => 'fa-envelope', 'percentage' => 10],
                                ['name' => 'Referral', 'icon' => 'fa-external-link-alt', 'percentage' => 5],
                                ['name' => 'Others', 'icon' => 'fa-ellipsis-h', 'percentage' => 2]
                            ];
                            
                            foreach ($trafficSources as $source):
                                $sourceVisitors = round($visitors * ($source['percentage'] / 100));
                                $sourceBuyers = round($totalCustomers * ($source['percentage'] / 100));
                                $sourceGMV = $totalRevenue * ($source['percentage'] / 100);
                                $conversion = $sourceVisitors > 0 ? ($sourceBuyers / $sourceVisitors * 100) : 0;
                            ?>
                            <tr style="border-bottom: 1px solid #f0f0f0;">
                                <td style="padding: 16px;">
                                    <div style="display: flex; align-items: center; gap: 10px;">
                                        <i class="fas <?php echo $source['icon']; ?>" style="color: #666;"></i>
                                        <span style="font-size: 14px; color: #1f1f1f;"><?php echo $source['name']; ?></span>
                                    </div>
                                </td>
                                <td style="padding: 16px;">
                                    <span style="font-size: 14px; color: #1f1f1f;"><?php echo $source['percentage']; ?>%</span>
                                </td>
                                <td style="padding: 16px;">
                                    <span style="font-size: 14px; color: #1f1f1f;"><?php echo number_format($sourceVisitors); ?></span>
                                </td>
                                <td style="padding: 16px;">
                                    <span style="font-size: 14px; color: #1f1f1f;"><?php echo number_format($sourceBuyers); ?></span>
                                </td>
                                <td style="padding: 16px;">
                                    <span style="font-size: 14px; color: #1f1f1f;">RM <?php echo number_format($sourceGMV, 2); ?></span>
                                </td>
                                <td style="padding: 16px;">
                                    <span style="font-size: 14px; color: #1f1f1f;"><?php echo number_format($conversion, 2); ?>%</span>
                                </td>
                                <td style="padding: 16px;">
                                    <button style="padding: 6px 12px; background: #f0f0f0; border: none; border-radius: 4px; cursor: pointer; font-size: 13px; color: #0ea5e9;">
                                        View Details <i class="fas fa-chevron-down"></i>
                                    </button>
                                </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
                
            <!-- Additional Insights -->
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(320px, 1fr)); gap: 16px;">
                
                <!-- Product Interaction -->
                <div style="background: white; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                    <h4 style="font-size: 16px; font-weight: 600; color: #1f1f1f; margin: 0 0 15px 0;">Product Interactions</h4>
                    <div style="display: flex; flex-direction: column; gap: 12px;">
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <span style="font-size: 14px; color: #666;">Favorites</span>
                            <span style="font-size: 18px; font-weight: 600; color: #1f1f1f;"><?php echo number_format($favorites); ?></span>
                        </div>
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <span style="font-size: 14px; color: #666;">Add to Cart</span>
                            <span style="font-size: 18px; font-weight: 600; color: #1f1f1f;"><?php echo number_format($addToCarts); ?></span>
                        </div>
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <span style="font-size: 14px; color: #666;">Cart to Order Rate</span>
                            <span style="font-size: 18px; font-weight: 600; color: #16a34a;"><?php echo number_format(($totalOrders / max($addToCarts, 1)) * 100, 1); ?>%</span>
                        </div>
                    </div>
                </div>

                <!-- Order Statistics -->
                <div style="background: white; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                    <h4 style="font-size: 16px; font-weight: 600; color: #1f1f1f; margin: 0 0 15px 0;">Order Statistics</h4>
                    <div style="display: flex; flex-direction: column; gap: 12px;">
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <span style="font-size: 14px; color: #666;">Total Orders</span>
                            <span style="font-size: 18px; font-weight: 600; color: #1f1f1f;"><?php echo number_format($totalOrders); ?></span>
                        </div>
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <span style="font-size: 14px; color: #666;">Pending Orders</span>
                            <span style="font-size: 18px; font-weight: 600; color: #f59e0b;"><?php echo number_format($stats['pending_orders']); ?></span>
                        </div>
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <span style="font-size: 14px; color: #666;">Completion Rate</span>
                            <span style="font-size: 18px; font-weight: 600; color: #16a34a;"><?php echo number_format((($totalOrders - $stats['pending_orders']) / max($totalOrders, 1)) * 100, 1); ?>%</span>
                        </div>
                    </div>
                </div>

                <!-- Product Performance -->
                <div style="background: white; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                    <h4 style="font-size: 16px; font-weight: 600; color: #1f1f1f; margin: 0 0 15px 0;">Product Performance</h4>
                    <div style="display: flex; flex-direction: column; gap: 12px;">
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <span style="font-size: 14px; color: #666;">Active Products</span>
                            <span style="font-size: 18px; font-weight: 600; color: #1f1f1f;"><?php echo number_format($totalProducts); ?></span>
                        </div>
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <span style="font-size: 14px; color: #666;">Low Stock Items</span>
                            <span style="font-size: 18px; font-weight: 600; color: #dc2626;"><?php echo number_format($stats['low_stock']); ?></span>
                        </div>
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <span style="font-size: 14px; color: #666;">Avg Items per Order</span>
                            <span style="font-size: 18px; font-weight: 600; color: #1f1f1f;"><?php echo number_format($itemsSold / max($totalOrders, 1), 1); ?></span>
                        </div>
                    </div>
                </div>

            </div>

            <!-- Product Analytics Section -->
            <div style="margin-top: 30px;">
                <div style="display: flex; align-items: center; justify-content: space-between; margin-bottom: 20px;">
                    <h2 style="font-size: 22px; font-weight: 600; color: #1f1f1f; margin: 0;">Product Performance</h2>
                    <div style="display: flex; gap: 10px; align-items: center;">
                        <button onclick="location.reload()" id="refreshBtn" style="padding: 8px 16px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border: none; border-radius: 6px; font-size: 14px; cursor: pointer; display: flex; align-items: center; gap: 6px; transition: all 0.3s; font-weight: 500;" onmouseover="this.style.transform='scale(1.05)'; this.style.boxShadow='0 4px 12px rgba(102, 126, 234, 0.4)'" onmouseout="this.style.transform='scale(1)'; this.style.boxShadow='none'">
                            <i class="fas fa-sync-alt"></i>
                            Refresh
                        </button>
                        <div id="autoRefreshIndicator" style="display: flex; align-items: center; gap: 6px; padding: 6px 12px; background: #e8f5e8; border: 1px solid #16a34a; border-radius: 6px; font-size: 12px; color: #16a34a;">
                            <div id="refreshDot" style="width: 8px; height: 8px; background: #16a34a; border-radius: 50%; animation: pulse 2s infinite;"></div>
                            Auto-refresh: ON
                        </div>
                        <input type="text" id="productSearch" placeholder="Search products..." style="padding: 8px 12px; border: 1px solid #ddd; border-radius: 6px; font-size: 14px; width: 250px;">
                        <select id="sortProducts" style="padding: 8px 12px; border: 1px solid #ddd; border-radius: 6px; font-size: 14px; cursor: pointer;">
                            <option value="sales">Sort by Sales</option>
                            <option value="revenue">Sort by Revenue</option>
                            <option value="views">Sort by Views</option>
                            <option value="name">Sort by Name</option>
                        </select>
                    </div>
                </div>

                <!-- Products Grid -->
                <div style="display: grid; grid-template-columns: repeat(auto-fill, minmax(280px, 1fr)); gap: 16px;">
                    <?php foreach ($products as $product): 
                        // Use ONLY real tracking data - no simulations
                        $soldQty = $product['total_sold'] ?? 0;
                        $productRevenue = $product['total_revenue'] ?? 0;
                        
                        // Real data only
                        $views = $product['total_views'] ?? 0;
                        $favorites = $product['total_favorites'] ?? 0;
                        $addToCart = $product['total_add_to_cart'] ?? 0;
                        $conversionRate = $views > 0 ? ($soldQty / $views * 100) : 0;
                        
                        $imagePath = !empty($product['image']) ? '../../' . $product['image'] : '../../images/placeholder.svg';
                        $isRealData = $views > 0; // Check if we have real tracking data
                    ?>
                    <div class="product-analytics-card" onclick="showProductDetails(<?php echo $product['id']; ?>)" 
                         style="background: white; border-radius: 10px; border: 1px solid #e8e8e8; overflow: hidden; cursor: pointer; transition: all 0.2s; position: relative;"
                         onmouseover="this.style.boxShadow='0 4px 12px rgba(0,0,0,0.1)'; this.style.transform='translateY(-2px)';"
                         onmouseout="this.style.boxShadow='none'; this.style.transform='translateY(0)';">
                        
                        <!-- Product Image -->
                        <div style="width: 100%; height: 200px; background: #f5f5f5; position: relative; overflow: hidden;">
                            <img src="<?php echo htmlspecialchars($imagePath); ?>"
                                 alt="<?php echo htmlspecialchars($product['name']); ?>"
                                 style="width: 100%; height: 100%; object-fit: cover;">
                            <div class="sales-count" style="position: absolute; top: 10px; right: 10px; background: #16a34a; color: white; padding: 4px 8px; border-radius: 4px; font-size: 12px; font-weight: 600; display: <?php echo $soldQty > 0 ? 'block' : 'none'; ?>;">
                                <span class="sold-qty"><?php echo $soldQty; ?></span> sold
                            </div>
                            <div class="no-data-badge" style="position: absolute; top: 10px; left: 10px; background: #f59e0b; color: white; padding: 4px 8px; border-radius: 4px; font-size: 11px; font-weight: 600; display: <?php echo !$isRealData ? 'block' : 'none'; ?>;">
                                <i class="fas fa-exclamation-circle"></i> No tracking data
                            </div>
                        </div>

                        <!-- Product Info -->
                        <div style="padding: 16px;">
                            <h4 class="product-name" style="font-size: 14px; font-weight: 600; color: #1f1f1f; margin: 0 0 8px 0; overflow: hidden; text-overflow: ellipsis; white-space: nowrap;">
                                <?php echo htmlspecialchars($product['name']); ?>
                            </h4>
                            <p class="product-brand" style="font-size: 12px; color: #666; margin: 0 0 12px 0;">
                                <?php echo htmlspecialchars($product['brand']); ?>
                            </p>

                            <!-- Quick Stats -->
                            <div style="display: flex; flex-direction: column; gap: 8px; border-top: 1px solid #f0f0f0; padding-top: 12px;">
                                <div style="display: flex; justify-content: space-between; align-items: center;">
                                    <span style="font-size: 12px; color: #666;">Views</span>
                                    <span class="views-count" style="font-size: 14px; font-weight: 600; color: #1f1f1f;"><?php echo number_format($views); ?></span>
                                </div>
                                <div style="display: flex; justify-content: space-between; align-items: center;">
                                    <span style="font-size: 12px; color: #666;">Favorites</span>
                                    <span class="favorites-count" style="font-size: 14px; font-weight: 600; color: #1f1f1f;"><?php echo number_format($favorites); ?></span>
                                </div>
                                <div style="display: flex; justify-content: space-between; align-items: center;">
                                    <span style="font-size: 12px; color: #666;">Add to Cart</span>
                                    <span class="cart-count" style="font-size: 14px; font-weight: 600; color: #1f1f1f;"><?php echo number_format($addToCart); ?></span>
                                </div>
                                <div style="display: flex; justify-content: space-between; align-items: center;">
                                    <span style="font-size: 12px; color: #666;">Revenue</span>
                                    <span class="revenue-amount" style="font-size: 14px; font-weight: 600; color: #16a34a;">RM <?php echo number_format($productRevenue, 2); ?></span>
                    </div>
                                <div style="display: flex; justify-content: space-between; align-items: center;">
                                    <span style="font-size: 12px; color: #666;">Conversion</span>
                                    <span class="conversion-rate" style="font-size: 14px; font-weight: 600; color: <?php echo $conversionRate > 2 ? '#16a34a' : '#f59e0b'; ?>;">
                                        <?php echo number_format($conversionRate, 2); ?>%
                                        </span>
                                </div>
                            </div>

                            <!-- View Details Button -->
                            <button style="width: 100%; margin-top: 12px; padding: 8px; background: #f0f0f0; border: none; border-radius: 6px; font-size: 13px; color: #0ea5e9; cursor: pointer; font-weight: 500;"
                                    onmouseover="this.style.background='#e0e0e0'"
                                    onmouseout="this.style.background='#f0f0f0'">
                                View Detailed Analytics <i class="fas fa-arrow-right"></i>
                            </button>
                        </div>
                    </div>
                                <?php endforeach; ?>
                </div>

                <!-- No Products Message -->
                <?php if (empty($products)): ?>
                <div style="background: white; border-radius: 8px; padding: 40px; text-align: center; border: 1px solid #e8e8e8;">
                    <i class="fas fa-box-open" style="font-size: 48px; color: #ccc; margin-bottom: 16px;"></i>
                    <h3 style="font-size: 18px; color: #666; margin: 0;">No products found</h3>
                    <p style="font-size: 14px; color: #999; margin: 8px 0 0 0;">Add products to see their performance analytics</p>
                </div>
                <?php endif; ?>
            </div>

        </div>
                    </div>

    <!-- Product Details Modal -->
    <div id="productModal" style="display: none; position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.5); z-index: 9999; overflow-y: auto; padding: 20px;">
        <div style="max-width: 1000px; margin: 40px auto; background: white; border-radius: 12px; position: relative;">
            <div style="padding: 30px;">
                <button onclick="closeProductDetails()" style="position: absolute; top: 20px; right: 20px; background: #f0f0f0; border: none; width: 36px; height: 36px; border-radius: 50%; cursor: pointer; font-size: 18px;">
                    <i class="fas fa-times"></i>
                </button>
                
                <div id="modalContent">
                    <!-- Content will be loaded here -->
                </div>
            </div>
        </div>
    </div>

    <script src="../assets/js/admin.js"></script>
    <style>
        @keyframes pulse {
            0% { opacity: 1; }
            50% { opacity: 0.5; }
            100% { opacity: 1; }
        }
    </style>
    <script>
        const productsData = <?php echo json_encode($products); ?>;

        function exportAnalytics() {
            // Export analytics data as CSV
            const data = {
                stats: <?php echo json_encode($stats); ?>,
                traffic: 'Traffic data exported'
            };
            
            alert('Analytics export feature - CSV download would happen here');
            console.log('Export data:', data);
        }

        function showProductDetails(productId) {
            const product = productsData.find(p => p.id == productId);
            if (!product) return;

            // Use ONLY real tracking data
            const soldQty = product.total_sold || 0;
            const revenue = product.total_revenue || 0;
            const views = product.total_views || 0;
                const favorites = product.total_favorites || 0;
                const addToCart = product.total_add_to_cart || 0;
                const favoriteRate = views > 0 ? ((favorites / views) * 100).toFixed(2) : 0;
                const cartRate = favorites > 0 ? ((addToCart / favorites) * 100).toFixed(2) : 0;
            const conversionRate = views > 0 ? ((soldQty / views) * 100).toFixed(2) : 0;

            const imagePath = product.image ? '../../' + product.image : '../../images/placeholder.svg';

            const modalContent = `
                <div style="display: flex; gap: 30px; margin-bottom: 30px;">
                    <div style="flex-shrink: 0;">
                        <img src="${imagePath}" alt="${product.name}" style="width: 200px; height: 200px; object-fit: cover; border-radius: 8px; border: 1px solid #e8e8e8;">
                    </div>
                    <div style="flex: 1;">
                        <h2 style="font-size: 24px; font-weight: 600; color: #1f1f1f; margin: 0 0 8px 0;">${product.name}</h2>
                        <p style="font-size: 16px; color: #666; margin: 0 0 16px 0;">${product.brand}</p>
                        <div style="display: flex; gap: 20px; flex-wrap: wrap;">
                            <div>
                                <span style="font-size: 14px; color: #666;">Price:</span>
                                <span style="font-size: 18px; font-weight: 600; color: #1f1f1f; margin-left: 8px;">RM ${parseFloat(product.price).toFixed(2)}</span>
                            </div>
                            <div>
                                <span style="font-size: 14px; color: #666;">Stock:</span>
                                <span style="font-size: 18px; font-weight: 600; color: ${product.stock_quantity < 10 ? '#dc2626' : '#16a34a'}; margin-left: 8px;">${product.stock_quantity}</span>
                            </div>
                        </div>
                    </div>
                </div>

                <h3 style="font-size: 18px; font-weight: 600; color: #1f1f1f; margin: 0 0 20px 0;">Performance Metrics</h3>
                
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 16px; margin-bottom: 30px;">
                    <div style="background: #f9fafb; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                        <div style="font-size: 13px; color: #666; margin-bottom: 8px;">Total Views</div>
                        <div style="font-size: 28px; font-weight: 600; color: #1f1f1f;">${views.toLocaleString()}</div>
                        <div style="font-size: 12px; color: #666; margin-top: 4px;">
                            ${views > 0 ? 'Tracked views' : 'No tracking data yet'}
                        </div>
                    </div>

                    <div style="background: #f9fafb; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                        <div style="font-size: 13px; color: #666; margin-bottom: 8px;">Favorites</div>
                        <div style="font-size: 28px; font-weight: 600; color: #1f1f1f;">${favorites.toLocaleString()}</div>
                        <div style="font-size: 12px; color: #666; margin-top: 4px;">
                            ${favorites > 0 ? 'Tracked favorites' : 'No tracking data yet'}
                        </div>
                    </div>

                    <div style="background: #f9fafb; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                        <div style="font-size: 13px; color: #666; margin-bottom: 8px;">Add to Cart</div>
                        <div style="font-size: 28px; font-weight: 600; color: #1f1f1f;">${addToCart.toLocaleString()}</div>
                        <div style="font-size: 12px; color: #666; margin-top: 4px;">
                            ${addToCart > 0 ? 'Tracked actions' : 'No tracking data yet'}
                        </div>
                    </div>

                    <div style="background: #f9fafb; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                        <div style="font-size: 13px; color: #666; margin-bottom: 8px;">Orders</div>
                        <div style="font-size: 28px; font-weight: 600; color: #1f1f1f;">${soldQty.toLocaleString()}</div>
                        <div style="font-size: 12px; color: ${soldQty > 0 ? '#16a34a' : '#999'}; margin-top: 4px;">
                            ${soldQty > 0 ? 'Completed orders' : 'No sales yet'}
                        </div>
                    </div>

                    <div style="background: #f9fafb; border-radius: 8px; padding: 20px; border: 1px solid #e8e8e8;">
                        <div style="font-size: 13px; color: #666; margin-bottom: 8px;">Revenue</div>
                        <div style="font-size: 28px; font-weight: 600; color: #16a34a;">RM ${parseFloat(revenue).toFixed(2)}</div>
                        <div style="font-size: 12px; color: ${revenue > 0 ? '#16a34a' : '#999'}; margin-top: 4px;">
                            ${revenue > 0 ? 'Total revenue' : 'No revenue yet'}
                        </div>
                    </div>
                </div>

                <h3 style="font-size: 18px; font-weight: 600; color: #1f1f1f; margin: 0 0 20px 0;">Conversion Funnel</h3>
                
                <div style="background: #f9fafb; border-radius: 8px; padding: 25px; border: 1px solid #e8e8e8; margin-bottom: 30px;">
                    <div style="display: flex; flex-direction: column; gap: 20px;">
                        <div>
                            <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
                                <span style="font-size: 14px; color: #1f1f1f; font-weight: 500;">View to Favorite Rate</span>
                                <span style="font-size: 14px; font-weight: 600; color: #1f1f1f;">${favoriteRate}%</span>
                            </div>
                            <div style="width: 100%; height: 8px; background: #e8e8e8; border-radius: 4px; overflow: hidden;">
                                <div style="width: ${favoriteRate}%; height: 100%; background: linear-gradient(90deg, #667eea, #764ba2);"></div>
                            </div>
                        </div>

                        <div>
                            <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
                                <span style="font-size: 14px; color: #1f1f1f; font-weight: 500;">Favorite to Cart Rate</span>
                                <span style="font-size: 14px; font-weight: 600; color: #1f1f1f;">${cartRate}%</span>
                            </div>
                            <div style="width: 100%; height: 8px; background: #e8e8e8; border-radius: 4px; overflow: hidden;">
                                <div style="width: ${cartRate}%; height: 100%; background: linear-gradient(90deg, #16a34a, #059669);"></div>
                            </div>
                        </div>

                        <div>
                            <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
                                <span style="font-size: 14px; color: #1f1f1f; font-weight: 500;">Overall Conversion Rate</span>
                                <span style="font-size: 14px; font-weight: 600; color: #1f1f1f;">${conversionRate}%</span>
                            </div>
                            <div style="width: 100%; height: 8px; background: #e8e8e8; border-radius: 4px; overflow: hidden;">
                                <div style="width: ${conversionRate}%; height: 100%; background: linear-gradient(90deg, #f59e0b, #d97706);"></div>
                            </div>
                        </div>
                    </div>
                </div>

                <div style="display: flex; gap: 12px; justify-content: flex-end;">
                    <button onclick="window.location.href='products.php?edit=${product.id}'" style="padding: 10px 20px; background: #667eea; color: white; border: none; border-radius: 6px; cursor: pointer; font-size: 14px; font-weight: 500;">
                        <i class="fas fa-edit"></i> Edit Product
                    </button>
                    <button onclick="closeProductDetails()" style="padding: 10px 20px; background: #f0f0f0; color: #1f1f1f; border: none; border-radius: 6px; cursor: pointer; font-size: 14px; font-weight: 500;">
                        Close
                    </button>
                </div>
            `;

            document.getElementById('modalContent').innerHTML = modalContent;
            document.getElementById('productModal').style.display = 'block';
            document.body.style.overflow = 'hidden';
        }

        function closeProductDetails() {
            document.getElementById('productModal').style.display = 'none';
            document.body.style.overflow = 'auto';
        }

        // Close modal when clicking outside
        document.getElementById('productModal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeProductDetails();
            }
        });

        // Product search functionality
        document.getElementById('productSearch').addEventListener('input', function(e) {
            const searchTerm = e.target.value.toLowerCase();
            const productCards = document.querySelectorAll('.product-analytics-card');
            
            // Control auto-refresh based on search state
            if (searchTerm.trim() === '') {
                // Search is cleared, resume auto-refresh
                startAutoRefresh();
                updateRefreshIndicator(true);
            } else {
                // Search is active, stop auto-refresh
                stopAutoRefresh();
                updateRefreshIndicator(false);
            }
            
            productCards.forEach(card => {
                const productName = card.querySelector('h4').textContent.toLowerCase();
                const brandName = card.querySelector('p').textContent.toLowerCase();
                
                if (productName.includes(searchTerm) || brandName.includes(searchTerm)) {
                    card.style.display = 'block';
                } else {
                    card.style.display = 'none';
                }
            });
        });

        // Product sorting functionality
        document.getElementById('sortProducts').addEventListener('change', function(e) {
            const sortBy = e.target.value;
            
            // If selecting "Sort by Sales" (default), resume auto-refresh
            if (sortBy === 'sales') {
                currentSortOption = 'sales';
                const searchTerm = document.getElementById('productSearch').value.trim();
                if (searchTerm === '') {
                    startAutoRefresh();
                    updateRefreshIndicator(true);
                }
            } else {
                // Stop auto-refresh for any other sort option
                stopAutoRefresh();
                updateRefreshIndicator(false, 'Sorting Active');
                currentSortOption = sortBy;
            }
            
            // Perform the sort
            sortProductCards(sortBy);
        });

        function sortProductCards(sortBy) {
            const container = document.querySelector('.product-analytics-card')?.parentNode;
            if (!container) return; // Exit if no container found
            
            const cards = Array.from(document.querySelectorAll('.product-analytics-card'));
            if (cards.length === 0) return; // Exit if no cards found
            
            // Sort cards based on the selected criteria
            cards.sort((a, b) => {
                try {
                    switch(sortBy) {
                        case 'sales':
                            const salesElementA = a.querySelector('.sales-count');
                            const salesElementB = b.querySelector('.sales-count');
                            const salesA = salesElementA ? parseInt(salesElementA.textContent?.replace(/,/g, '') || '0') || 0 : 0;
                            const salesB = salesElementB ? parseInt(salesElementB.textContent?.replace(/,/g, '') || '0') || 0 : 0;
                            return salesB - salesA; // Descending order
                            
                        case 'revenue':
                            const revenueElementA = a.querySelector('.revenue-amount');
                            const revenueElementB = b.querySelector('.revenue-amount');
                            const revenueA = revenueElementA ? parseFloat(revenueElementA.textContent?.replace(/[$,]/g, '') || '0') || 0 : 0;
                            const revenueB = revenueElementB ? parseFloat(revenueElementB.textContent?.replace(/[$,]/g, '') || '0') || 0 : 0;
                            return revenueB - revenueA; // Descending order
                            
                        case 'views':
                            const viewsElementA = a.querySelector('.views-count');
                            const viewsElementB = b.querySelector('.views-count');
                            const viewsA = viewsElementA ? parseInt(viewsElementA.textContent?.replace(/,/g, '') || '0') || 0 : 0;
                            const viewsB = viewsElementB ? parseInt(viewsElementB.textContent?.replace(/,/g, '') || '0') || 0 : 0;
                            return viewsB - viewsA; // Descending order
                            
                        case 'name':
                            const nameElementA = a.querySelector('h4');
                            const nameElementB = b.querySelector('h4');
                            const nameA = nameElementA?.textContent?.toLowerCase() || '';
                            const nameB = nameElementB?.textContent?.toLowerCase() || '';
                            return nameA.localeCompare(nameB); // Ascending order
                            
                        default:
                            return 0;
                    }
                } catch (error) {
                    console.error('Error sorting cards:', error);
                    return 0;
                }
            });
            
            // Re-append sorted cards to container
            cards.forEach(card => container.appendChild(card));
        }

        // Auto-refresh functionality for Product Performance section
        let refreshInterval;
        let isRefreshing = false;
        let currentSortOption = 'sales'; // Track current sort option

        function startAutoRefresh() {
            // Don't start if search is active or sorting is active
            const searchTerm = document.getElementById('productSearch').value.trim();
            if (searchTerm !== '' || currentSortOption !== 'sales') {
                return;
            }
            
            // Clear any existing interval
            if (refreshInterval) {
                clearInterval(refreshInterval);
            }
            
            // Refresh every 3 seconds
            refreshInterval = setInterval(refreshProductPerformance, 3000);
        }

        function stopAutoRefresh() {
            if (refreshInterval) {
                clearInterval(refreshInterval);
                refreshInterval = null;
            }
        }

        function updateRefreshIndicator(isOn, reason = '') {
            const indicator = document.getElementById('autoRefreshIndicator');
            const refreshDot = document.getElementById('refreshDot');
            
            if (isOn) {
                indicator.style.background = '#e8f5e8';
                indicator.style.borderColor = '#16a34a';
                indicator.style.color = '#16a34a';
                refreshDot.style.background = '#16a34a';
                refreshDot.style.animation = 'pulse 2s infinite';
                indicator.innerHTML = '<div id="refreshDot" style="width: 8px; height: 8px; background: #16a34a; border-radius: 50%; animation: pulse 2s infinite;"></div>Auto-refresh: ON';
            } else {
                indicator.style.background = '#fef3cd';
                indicator.style.borderColor = '#f59e0b';
                indicator.style.color = '#f59e0b';
                refreshDot.style.background = '#f59e0b';
                refreshDot.style.animation = 'none';
                const reasonText = reason ? ` (${reason})` : ' (Search Active)';
                indicator.innerHTML = `<div id="refreshDot" style="width: 8px; height: 8px; background: #f59e0b; border-radius: 50%;"></div>Auto-refresh: OFF${reasonText}`;
            }
        }

        function refreshProductPerformance() {
            if (isRefreshing) return; // Prevent multiple simultaneous requests
            
            isRefreshing = true;
            
            // Show loading indicator
            const refreshBtn = document.getElementById('refreshBtn');
            const originalText = refreshBtn.innerHTML;
            refreshBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Refreshing...';
            refreshBtn.disabled = true;

            fetch('../ajax/get-product-analytics.php')
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        updateProductPerformance(data.data);
                        console.log('Product performance data refreshed');
                    } else {
                        console.error('Failed to refresh data:', data.message);
                    }
                })
                .catch(error => {
                    console.error('Error refreshing data:', error);
                })
                .finally(() => {
                    isRefreshing = false;
                    // Restore refresh button
                    refreshBtn.innerHTML = originalText;
                    refreshBtn.disabled = false;
                });
        }

        function updateProductPerformance(data) {
            const products = data.products;
            const stats = data.stats;

            // Update the products grid
            const productsGrid = document.querySelector('.page-content > div:last-child .product-analytics-card');
            if (!productsGrid) return;

            const container = productsGrid.parentElement;

            // Update existing cards instead of recreating them
            products.forEach((product, index) => {
                // Use ONLY real tracking data - no simulations
                const soldQty = product.total_sold || 0;
                const productRevenue = product.total_revenue || 0;

                // Real data only
                const views = product.total_views || 0;
                const favorites = product.total_favorites || 0;
                const addToCart = product.total_add_to_cart || 0;
                const conversionRate = views > 0 ? (soldQty / views * 100) : 0;
                const isRealData = views > 0;

                // Find existing card by index
                const existingCards = container.querySelectorAll('.product-analytics-card');
                let productCard = existingCards[index];

                if (!productCard) {
                    // Card doesn't exist, create new one (for new products)
                    const imagePath = product.image ? '../../' + product.image : '../../images/placeholder.svg';

                    productCard = document.createElement('div');
                    productCard.className = 'product-analytics-card';
                    productCard.onclick = () => showProductDetails(product.id);
                    productCard.style.cssText = 'background: white; border-radius: 10px; border: 1px solid #e8e8e8; overflow: hidden; cursor: pointer; transition: all 0.2s; position: relative;';
                    productCard.onmouseover = function() {
                        this.style.boxShadow = '0 4px 12px rgba(0,0,0,0.1)';
                        this.style.transform = 'translateY(-2px)';
                    };
                    productCard.onmouseout = function() {
                        this.style.boxShadow = 'none';
                        this.style.transform = 'translateY(0)';
                    };

                    productCard.innerHTML = `
                        <!-- Product Image -->
                        <div style="width: 100%; height: 200px; background: #f5f5f5; position: relative; overflow: hidden;">
                            <img src="${imagePath}"
                                 alt="${product.name}"
                                 style="width: 100%; height: 100%; object-fit: cover;">
                            <div class="sales-count" style="position: absolute; top: 10px; right: 10px; background: #16a34a; color: white; padding: 4px 8px; border-radius: 4px; font-size: 12px; font-weight: 600; display: ${soldQty > 0 ? 'block' : 'none'};">
                                <span class="sold-qty">${soldQty}</span> sold
                            </div>
                            <div class="no-data-badge" style="position: absolute; top: 10px; left: 10px; background: #f59e0b; color: white; padding: 4px 8px; border-radius: 4px; font-size: 11px; font-weight: 600; display: ${!isRealData ? 'block' : 'none'};">
                                <i class="fas fa-exclamation-circle"></i> No tracking data
                            </div>
                        </div>

                        <!-- Product Info -->
                        <div style="padding: 16px;">
                            <h4 class="product-name" style="font-size: 14px; font-weight: 600; color: #1f1f1f; margin: 0 0 8px 0; overflow: hidden; text-overflow: ellipsis; white-space: nowrap;">
                                ${product.name}
                            </h4>
                            <p class="product-brand" style="font-size: 12px; color: #666; margin: 0 0 12px 0;">
                                ${product.brand}
                            </p>

                            <!-- Quick Stats -->
                            <div style="display: flex; flex-direction: column; gap: 8px; border-top: 1px solid #f0f0f0; padding-top: 12px;">
                                <div style="display: flex; justify-content: space-between; align-items: center;">
                                    <span style="font-size: 12px; color: #666;">Views</span>
                                    <span class="views-count" style="font-size: 14px; font-weight: 600; color: #1f1f1f;">${views.toLocaleString()}</span>
                                </div>
                                <div style="display: flex; justify-content: space-between; align-items: center;">
                                    <span style="font-size: 12px; color: #666;">Favorites</span>
                                    <span class="favorites-count" style="font-size: 14px; font-weight: 600; color: #1f1f1f;">${favorites.toLocaleString()}</span>
                                </div>
                                <div style="display: flex; justify-content: space-between; align-items: center;">
                                    <span style="font-size: 12px; color: #666;">Add to Cart</span>
                                    <span class="cart-count" style="font-size: 14px; font-weight: 600; color: #1f1f1f;">${addToCart.toLocaleString()}</span>
                                </div>
                                <div style="display: flex; justify-content: space-between; align-items: center;">
                                    <span style="font-size: 12px; color: #666;">Revenue</span>
                                    <span class="revenue-amount" style="font-size: 14px; font-weight: 600; color: #16a34a;">RM ${parseFloat(productRevenue).toFixed(2)}</span>
                                </div>
                                <div style="display: flex; justify-content: space-between; align-items: center;">
                                    <span style="font-size: 12px; color: #666;">Conversion</span>
                                    <span class="conversion-rate" style="font-size: 14px; font-weight: 600; color: ${conversionRate > 2 ? '#16a34a' : '#f59e0b'};">
                                        ${conversionRate.toFixed(2)}%
                                    </span>
                                </div>
                            </div>

                            <!-- View Details Button -->
                            <button style="width: 100%; margin-top: 12px; padding: 8px; background: #f0f0f0; border: none; border-radius: 6px; font-size: 13px; color: #0ea5e9; cursor: pointer; font-weight: 500;"
                                    onmouseover="this.style.background='#e0e0e0'"
                                    onmouseout="this.style.background='#f0f0f0'">
                                View Detailed Analytics <i class="fas fa-arrow-right"></i>
                            </button>
                        </div>
                    `;

                    container.appendChild(productCard);
                } else {
                    // Card exists, update only the data (not the image)
                    // Update sold badge
                    const salesCount = productCard.querySelector('.sales-count');
                    const soldQtySpan = productCard.querySelector('.sold-qty');
                    if (salesCount && soldQtySpan) {
                        salesCount.style.display = soldQty > 0 ? 'block' : 'none';
                        soldQtySpan.textContent = soldQty;
                    }

                    // Update no data badge
                    const noDataBadge = productCard.querySelector('.no-data-badge');
                    if (noDataBadge) {
                        noDataBadge.style.display = !isRealData ? 'block' : 'none';
                    }

                    // Update metrics
                    const viewsCount = productCard.querySelector('.views-count');
                    const favoritesCount = productCard.querySelector('.favorites-count');
                    const cartCount = productCard.querySelector('.cart-count');
                    const revenueAmount = productCard.querySelector('.revenue-amount');
                    const conversionRateEl = productCard.querySelector('.conversion-rate');

                    if (viewsCount) viewsCount.textContent = views.toLocaleString();
                    if (favoritesCount) favoritesCount.textContent = favorites.toLocaleString();
                    if (cartCount) cartCount.textContent = addToCart.toLocaleString();
                    if (revenueAmount) revenueAmount.textContent = `RM ${parseFloat(productRevenue).toFixed(2)}`;
                    if (conversionRateEl) {
                        conversionRateEl.textContent = `${conversionRate.toFixed(2)}%`;
                        conversionRateEl.style.color = conversionRate > 2 ? '#16a34a' : '#f59e0b';
                    }
                }
            });

            // Update the productsData for modal functionality
            window.productsData = products;
        }

        // Start auto-refresh when page loads
        document.addEventListener('DOMContentLoaded', function() {
            startAutoRefresh();
        });

        // Stop auto-refresh when page is hidden (tab switch)
        document.addEventListener('visibilitychange', function() {
            if (document.hidden) {
                stopAutoRefresh();
            } else {
                startAutoRefresh();
            }
        });

        // Stop auto-refresh when page is about to unload
        window.addEventListener('beforeunload', function() {
            stopAutoRefresh();
        });
    </script>
</body>
</html>