<?php
require_once '../includes/auth.php';
require_once '../includes/functions.php';

$adminAuth->requireLogin();

// Handle form submissions
if ($_POST) {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'create_flash_sale') {
        $productId = $_POST['product_id'] ?? '';
        $saleType = $_POST['sale_type'] ?? '';
        $saleValue = $_POST['sale_value'] ?? '';
        $duration = $_POST['duration'] ?? 24;
        $selectedVariations = $_POST['selected_variations'] ?? [];
        
        if ($productId && $saleType && $saleValue) {
            try {
                $pdo = getDB();
                
                // Get product and variations
                $stmt = $pdo->prepare("
                    SELECT p.*, vv.id as variation_id, vv.version_type, vv.size, vv.price as variation_price
                    FROM products p 
                    LEFT JOIN version_variations vv ON p.id = vv.product_id AND vv.is_active = 1
                    WHERE p.id = ? AND p.is_active = 1
                ");
                $stmt->execute([$productId]);
                $productData = $stmt->fetchAll();
                
                if ($productData) {
                    $product = $productData[0];
                    $variations = $productData;
                    
                    // Process selected variations
                    $processedVariations = [];
                    foreach ($selectedVariations as $selectedVar) {
                        list($versionType, $size, $originalPrice) = explode(':', $selectedVar);
                        $salePrice = 0;
                        
                        if ($saleType === 'percentage') {
                            $discount = ($originalPrice * $saleValue) / 100;
                            $salePrice = $originalPrice - $discount;
                        } elseif ($saleType === 'fixed') {
                            $salePrice = $saleValue;
                        }
                        
                        $processedVariations[] = [
                            'version_type' => $versionType,
                            'size' => $size,
                            'original_price' => $originalPrice,
                            'sale_price' => $salePrice
                        ];
                    }
                    
                    if (empty($processedVariations)) {
                        $error = "Please select at least one variation for the flash sale.";
                    } else {
                        // Update product with flash sale
                        $stmt = $pdo->prepare("
                            UPDATE products 
                            SET is_flash_sale = 1, 
                                is_on_sale = 1,
                                flash_sale_duration_hours = ?,
                                flash_sale_started_at = NOW()
                            WHERE id = ?
                        ");
                        
                        if ($stmt->execute([$duration, $productId])) {
                            // Store variation-specific sale prices in a JSON field or separate table
                            // For now, we'll store the sale price in the main product table
                            // and handle variations in the display logic
                            $minSalePrice = min(array_column($processedVariations, 'sale_price'));
                            $maxSalePrice = max(array_column($processedVariations, 'sale_price'));
                            
                            $stmt = $pdo->prepare("
                                UPDATE products 
                                SET sale_price = ?,
                                    flash_sale_variations = ?
                                WHERE id = ?
                            ");
                            
                            $variationsJson = json_encode($processedVariations);
                            $stmt->execute([$minSalePrice, $variationsJson, $productId]);
                            
                            $success = "Flash sale created successfully for " . count($processedVariations) . " variation(s)!";
                        } else {
                            $error = "Failed to create flash sale.";
                        }
                    }
                } else {
                    $error = "Product not found.";
                }
            } catch (Exception $e) {
                $error = "Error: " . $e->getMessage();
            }
        } else {
            $error = "Please fill in all required fields.";
        }
    } elseif ($action === 'end_flash_sale') {
        $productId = $_POST['product_id'] ?? '';
        
        if ($productId) {
            try {
                $pdo = getDB();
                
                $stmt = $pdo->prepare("
                    UPDATE products 
                    SET is_flash_sale = 0, 
                        sale_price = NULL, 
                        is_on_sale = 0,
                        flash_sale_started_at = NULL
                    WHERE id = ?
                ");
                
                if ($stmt->execute([$productId])) {
                    $success = "Flash sale ended successfully!";
                } else {
                    $error = "Failed to end flash sale.";
                }
            } catch (Exception $e) {
                $error = "Error: " . $e->getMessage();
            }
        }
    }
}

// Get all products with variations
try {
    $pdo = getDB();
    $stmt = $pdo->prepare("
        SELECT DISTINCT p.id, p.name, p.brand, p.price, p.image, p.category_id, p.is_active,
               c.name as category_name,
               COALESCE(SUM(vv.stock_quantity), p.stock_quantity) as total_stock,
               MIN(vv.price) as min_price,
               MAX(vv.price) as max_price,
               GROUP_CONCAT(CONCAT(vv.version_type, ':', vv.size, ':', vv.price, ':', vv.stock_quantity) SEPARATOR '|') as variations
        FROM products p 
        LEFT JOIN categories c ON p.category_id = c.id 
        LEFT JOIN version_variations vv ON p.id = vv.product_id AND vv.is_active = 1
        WHERE p.is_active = 1 AND p.is_flash_sale = 0
        GROUP BY p.id, p.name, p.brand, p.price, p.image, p.category_id, p.is_active, c.name
        ORDER BY p.name ASC
    ");
    $stmt->execute();
    $products = $stmt->fetchAll();
    
    // Process variations for each product and remove duplicates
    $uniqueProducts = [];
    foreach ($products as $product) {
        // Skip if we've already processed this product
        if (isset($uniqueProducts[$product['id']])) {
            continue;
        }
        
        $product['variations_detail'] = [];
        if (!empty($product['variations'])) {
            $variations = explode('|', $product['variations']);
            foreach ($variations as $variation) {
                if (!empty($variation)) {
                    list($version_type, $size, $price, $stock) = explode(':', $variation);
                    $product['variations_detail'][] = [
                        'version_type' => $version_type,
                        'size' => $size,
                        'price' => (float)$price,
                        'stock' => (int)$stock
                    ];
                }
            }
        }
        
        // Set display price range
        if ($product['min_price'] === null || $product['max_price'] === null) {
            $product['price_range'] = 'RM ' . number_format($product['price'], 2);
        } else {
            $product['price_range'] = ($product['min_price'] != $product['max_price']) ? 
                'RM ' . number_format($product['min_price'], 2) . ' - RM ' . number_format($product['max_price'], 2) : 
                'RM ' . number_format($product['min_price'], 2);
        }
        
        $uniqueProducts[$product['id']] = $product;
    }
    
    // Convert back to indexed array
    $products = array_values($uniqueProducts);
} catch (Exception $e) {
    $error = "Error loading products: " . $e->getMessage();
    $products = [];
}

// Get current flash sales
try {
    $stmt = $pdo->prepare("
        SELECT p.*, c.name as category_name,
               TIMESTAMPDIFF(HOUR, p.flash_sale_started_at, NOW()) as hours_elapsed
        FROM products p 
        LEFT JOIN categories c ON p.category_id = c.id 
        WHERE p.is_flash_sale = 1 AND p.is_active = 1
        ORDER BY p.flash_sale_started_at DESC
    ");
    $stmt->execute();
    $flashSales = $stmt->fetchAll();
} catch (Exception $e) {
    $flashSales = [];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Flash Sale Management - Admin Panel</title>
    <link rel="stylesheet" href="../assets/css/admin.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        .flash-sale-card {
            background: linear-gradient(135deg, #dc2626, #ef4444);
            color: white;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 4px 15px rgba(220, 38, 38, 0.3);
        }
        
        .flash-sale-card h3 {
            margin: 0 0 10px 0;
            font-size: 1.2em;
        }
        
        .flash-sale-card .price-info {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin: 10px 0;
        }
        
        .original-price {
            text-decoration: line-through;
            opacity: 0.8;
        }
        
        .sale-price {
            font-size: 1.5em;
            font-weight: bold;
        }
        
        .time-remaining {
            background: rgba(255, 255, 255, 0.2);
            padding: 8px 12px;
            border-radius: 20px;
            font-size: 0.9em;
        }
        
        .create-flash-sale-form {
            background: white;
            border-radius: 12px;
            padding: 25px;
            margin-bottom: 30px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #374151;
        }
        
        .form-group select,
        .form-group input {
            width: 100%;
            padding: 12px;
            border: 2px solid #e5e7eb;
            border-radius: 8px;
            font-size: 16px;
            transition: border-color 0.3s;
        }
        
        .form-group select:focus,
        .form-group input:focus {
            outline: none;
            border-color: #dc2626;
        }
        
        .sale-type-group {
            display: flex;
            gap: 20px;
            margin-bottom: 20px;
        }
        
        .sale-type-option {
            flex: 1;
            padding: 15px;
            border: 2px solid #e5e7eb;
            border-radius: 8px;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s;
        }
        
        .sale-type-option.selected {
            border-color: #dc2626;
            background-color: #fef2f2;
        }
        
        .sale-type-option input[type="radio"] {
            display: none;
        }
        
        .btn-flash-sale {
            background: linear-gradient(135deg, #dc2626, #ef4444);
            color: white;
            border: none;
            padding: 12px 24px;
            border-radius: 8px;
            font-weight: 600;
            cursor: pointer;
            transition: transform 0.2s;
        }
        
        .btn-flash-sale:hover {
            transform: translateY(-2px);
        }
        
        .btn-end-sale {
            background: #6b7280;
            color: white;
            border: none;
            padding: 8px 16px;
            border-radius: 6px;
            cursor: pointer;
            font-size: 0.9em;
        }
        
        .btn-end-sale:hover {
            background: #4b5563;
        }
        
        .no-flash-sales {
            text-align: center;
            padding: 40px;
            color: #6b7280;
            font-style: italic;
        }
        
        .product-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 20px;
            margin-top: 20px;
        }
        
        .product-card {
            background: white;
            border-radius: 12px;
            padding: 20px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
            border: 2px solid transparent;
            transition: all 0.3s;
        }
        
        .product-card:hover {
            border-color: #dc2626;
            transform: translateY(-2px);
        }
        
        .product-card h4 {
            margin: 0 0 10px 0;
            color: #1f2937;
        }
        
        .product-card .price {
            font-size: 1.2em;
            font-weight: bold;
            color: #059669;
        }
        
        .product-card .category {
            color: #6b7280;
            font-size: 0.9em;
        }
        
        .variations-container {
            background: #f8f9fa;
            border: 1px solid #e9ecef;
            border-radius: 8px;
            padding: 15px;
            max-height: 200px;
            overflow-y: auto;
        }
        
        .variation-item {
            background: white;
            border: 1px solid #dee2e6;
            border-radius: 6px;
            padding: 10px;
            margin-bottom: 8px;
            transition: all 0.2s;
        }
        
        .variation-item:hover {
            border-color: #dc2626;
            box-shadow: 0 2px 4px rgba(220, 38, 38, 0.1);
        }
        
        .variation-item:last-child {
            margin-bottom: 0;
        }
        
        .variation-info {
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-wrap: wrap;
            gap: 8px;
        }
        
        .variation-type {
            background: #dc2626;
            color: white;
            padding: 2px 8px;
            border-radius: 12px;
            font-size: 11px;
            font-weight: bold;
            text-transform: uppercase;
        }
        
        .variation-size {
            color: #374151;
            font-weight: 600;
            font-size: 14px;
        }
        
        .variation-price {
            color: #059669;
            font-weight: bold;
            font-size: 14px;
        }
        
        .variation-stock {
            color: #6b7280;
            font-size: 12px;
            background: #f3f4f6;
            padding: 2px 6px;
            border-radius: 4px;
        }
        
        /* Dropdown styling for better readability */
        #product_id {
            font-size: 14px;
            line-height: 1.4;
        }
        
        #product_id option {
            padding: 8px 12px;
            white-space: normal;
            word-wrap: break-word;
        }
        
        /* Style for variation text in dropdown */
        .variation-text {
            color: #6b7280;
            font-size: 12px;
            font-style: italic;
        }
        
        /* Variation selection styles */
        .variation-item {
            display: flex;
            align-items: center;
            gap: 12px;
        }
        
        .variation-checkbox {
            position: relative;
        }
        
        .variation-checkbox input[type="checkbox"] {
            width: 18px;
            height: 18px;
            cursor: pointer;
            accent-color: #dc2626;
        }
        
        .variation-checkbox label {
            position: absolute;
            top: 0;
            left: 0;
            width: 18px;
            height: 18px;
            cursor: pointer;
        }
        
        .variation-actions {
            margin-top: 10px;
            display: flex;
            gap: 10px;
        }
        
        .btn-select-all, .btn-deselect-all {
            background: #f8f9fa;
            border: 1px solid #dee2e6;
            color: #495057;
            padding: 6px 12px;
            border-radius: 6px;
            cursor: pointer;
            font-size: 12px;
            transition: all 0.2s;
        }
        
        .btn-select-all:hover, .btn-deselect-all:hover {
            background: #e9ecef;
            border-color: #adb5bd;
        }
        
        .btn-select-all {
            color: #28a745;
            border-color: #28a745;
        }
        
        .btn-deselect-all {
            color: #dc3545;
            border-color: #dc3545;
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <!-- Sidebar -->
        <div class="sidebar">
            <div class="sidebar-header">
                <h2>Admin Panel</h2>
            </div>
            <nav class="sidebar-nav">
                <a href="index.php" class="nav-item">
                    <i class="fas fa-tachometer-alt"></i>
                    Dashboard
                </a>
                <a href="products.php" class="nav-item">
                    <i class="fas fa-box"></i>
                    Products
                </a>
                <a href="orders.php" class="nav-item">
                    <i class="fas fa-shopping-cart"></i>
                    Orders
                </a>
                <a href="customers.php" class="nav-item">
                    <i class="fas fa-users"></i>
                    Customers
                </a>
                <a href="analytics.php" class="nav-item">
                    <i class="fas fa-chart-bar"></i>
                    Analytics
                </a>
                <a href="flash-sale.php" class="nav-item active">
                    <i class="fas fa-bolt"></i>
                    Flash Sale
                </a>
                <a href="settings.php" class="nav-item">
                    <i class="fas fa-cog"></i>
                    Settings
                </a>
            </nav>
        </div>

        <!-- Main Content -->
        <div class="main-content">
            <!-- Header -->
            <header class="header">
                <div class="header-left">
                    <button class="sidebar-toggle">
                        <i class="fas fa-bars"></i>
                    </button>
                    <h1>Flash Sale Management</h1>
                </div>
                <div class="header-right">
                    <a href="products.php" class="btn btn-outline">
                        <i class="fas fa-arrow-left"></i>
                        Back to Products
                    </a>
                </div>
            </header>

            <!-- Page Content -->
            <div class="page-content">
                <?php if (isset($success)): ?>
                    <div class="alert" style="background-color: #d4edda; color: #155724; border: 1px solid #c3e6cb;">
                        <i class="fas fa-check-circle"></i>
                        <?php echo htmlspecialchars($success); ?>
                    </div>
                <?php endif; ?>
                
                <?php if (isset($error)): ?>
                    <div class="alert alert-error">
                        <i class="fas fa-exclamation-circle"></i>
                        <?php echo htmlspecialchars($error); ?>
                    </div>
                <?php endif; ?>

                <!-- Create Flash Sale Form -->
                <div class="create-flash-sale-form">
                    <h2><i class="fas fa-plus-circle"></i> Create New Flash Sale</h2>
                    <form method="POST" id="flashSaleForm">
                        <input type="hidden" name="action" value="create_flash_sale">
                        
                        <div class="form-group">
                            <label for="product_id">Select Product</label>
                            <select name="product_id" id="product_id" required>
                                <option value="">Choose a product...</option>
                                <?php foreach ($products as $product): ?>
                                    <option value="<?php echo $product['id']; ?>" 
                                            data-price="<?php echo $product['price']; ?>"
                                            data-min-price="<?php echo $product['min_price'] ?? $product['price']; ?>"
                                            data-max-price="<?php echo $product['max_price'] ?? $product['price']; ?>"
                                            data-variations="<?php echo htmlspecialchars(json_encode($product['variations_detail'])); ?>">
                                        <?php echo htmlspecialchars($product['name']); ?> - <?php echo $product['price_range']; ?>
                                        <?php if (!empty($product['variations_detail'])): ?>
                                            <?php 
                                            $variationTexts = [];
                                            foreach ($product['variations_detail'] as $variation) {
                                                $type = $variation['version_type'] ?? 'Standard';
                                                $size = $variation['size'] ?? 'N/A';
                                                $price = number_format($variation['price'], 2);
                                                $variationTexts[] = "$type $size (RM$price)";
                                            }
                                            echo ' | ' . implode(', ', $variationTexts);
                                            ?>
                                        <?php endif; ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label>Sale Type</label>
                            <div class="sale-type-group">
                                <label class="sale-type-option" for="percentage">
                                    <input type="radio" name="sale_type" value="percentage" id="percentage" required>
                                    <div>
                                        <i class="fas fa-percentage"></i>
                                        <div>Percentage Discount</div>
                                    </div>
                                </label>
                                <label class="sale-type-option" for="fixed">
                                    <input type="radio" name="sale_type" value="fixed" id="fixed" required>
                                    <div>
                                        <i class="fas fa-tag"></i>
                                        <div>Fixed Price</div>
                                    </div>
                                </label>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label for="sale_value" id="sale_value_label">Discount Percentage (%)</label>
                            <input type="number" name="sale_value" id="sale_value" min="1" max="99" step="1" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="duration">Duration (Hours)</label>
                            <select name="duration" id="duration">
                                <option value="1">1 Hour</option>
                                <option value="6">6 Hours</option>
                                <option value="12">12 Hours</option>
                                <option value="24" selected>24 Hours</option>
                                <option value="48">48 Hours</option>
                                <option value="72">72 Hours</option>
                            </select>
                        </div>
                        
                        <!-- Product Variations Display -->
                        <div class="form-group" id="variations-display" style="display: none;">
                            <label>Select Variations for Flash Sale</label>
                            <div class="variations-container" id="variations-container">
                                <!-- Variations will be populated by JavaScript -->
                            </div>
                            <div class="variation-actions" id="variation-actions" style="display: none;">
                                <button type="button" class="btn-select-all" onclick="selectAllVariations()">
                                    <i class="fas fa-check-square"></i> Select All
                                </button>
                                <button type="button" class="btn-deselect-all" onclick="deselectAllVariations()">
                                    <i class="fas fa-square"></i> Deselect All
                                </button>
                            </div>
                        </div>
                        
                        <button type="submit" class="btn-flash-sale">
                            <i class="fas fa-bolt"></i>
                            Create Flash Sale
                        </button>
                    </form>
                </div>

                <!-- Current Flash Sales -->
                <div class="flash-sales-section">
                    <h2><i class="fas fa-fire"></i> Active Flash Sales</h2>
                    
                    <?php if (empty($flashSales)): ?>
                        <div class="no-flash-sales">
                            <i class="fas fa-bolt" style="font-size: 3em; margin-bottom: 20px; color: #d1d5db;"></i>
                            <p>No active flash sales at the moment.</p>
                            <p>Create your first flash sale above to get started!</p>
                        </div>
                    <?php else: ?>
                        <div class="product-grid">
                            <?php foreach ($flashSales as $sale): ?>
                                <div class="flash-sale-card">
                                    <h3><?php echo htmlspecialchars($sale['name']); ?></h3>
                                    <div class="price-info">
                                        <div>
                                            <div class="original-price">RM<?php echo number_format($sale['price'], 2); ?></div>
                                            <div class="sale-price">RM<?php echo number_format($sale['sale_price'], 2); ?></div>
                                        </div>
                                        <div class="time-remaining">
                                            <?php 
                                            $hoursRemaining = $sale['flash_sale_duration_hours'] - $sale['hours_elapsed'];
                                            if ($hoursRemaining > 0) {
                                                echo $hoursRemaining . "h remaining";
                                            } else {
                                                echo "Expired";
                                            }
                                            ?>
                                        </div>
                                    </div>
                                    <div style="margin-top: 15px;">
                                        <form method="POST" style="display: inline;">
                                            <input type="hidden" name="action" value="end_flash_sale">
                                            <input type="hidden" name="product_id" value="<?php echo $sale['id']; ?>">
                                            <button type="submit" class="btn-end-sale" onclick="return confirm('Are you sure you want to end this flash sale?')">
                                                <i class="fas fa-stop"></i>
                                                End Sale
                                            </button>
                                        </form>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Handle sale type selection
        document.querySelectorAll('input[name="sale_type"]').forEach(radio => {
            radio.addEventListener('change', function() {
                // Update visual selection
                document.querySelectorAll('.sale-type-option').forEach(option => {
                    option.classList.remove('selected');
                });
                this.closest('.sale-type-option').classList.add('selected');
                
                // Update form field
                const saleValueInput = document.getElementById('sale_value');
                const saleValueLabel = document.getElementById('sale_value_label');
                
                if (this.value === 'percentage') {
                    saleValueLabel.textContent = 'Discount Percentage (%)';
                    saleValueInput.setAttribute('min', '1');
                    saleValueInput.setAttribute('max', '99');
                    saleValueInput.setAttribute('step', '1');
                    saleValueInput.placeholder = 'Enter percentage (1-99)';
                } else if (this.value === 'fixed') {
                    saleValueLabel.textContent = 'Fixed Sale Price (RM)';
                    saleValueInput.setAttribute('min', '0.01');
                    saleValueInput.setAttribute('step', '0.01');
                    saleValueInput.placeholder = 'Enter fixed price';
                }
            });
        });

        // Update sale value and show variations based on selected product
        document.getElementById('product_id').addEventListener('change', function() {
            const selectedOption = this.options[this.selectedIndex];
            const originalPrice = parseFloat(selectedOption.dataset.price);
            const minPrice = parseFloat(selectedOption.dataset.minPrice);
            const maxPrice = parseFloat(selectedOption.dataset.maxPrice);
            const variations = JSON.parse(selectedOption.dataset.variations || '[]');
            const saleValueInput = document.getElementById('sale_value');
            const selectedSaleType = document.querySelector('input[name="sale_type"]:checked');
            const variationsDisplay = document.getElementById('variations-display');
            const variationsContainer = document.getElementById('variations-container');
            
            // Update sale value constraints
            if (selectedSaleType && selectedSaleType.value === 'fixed') {
                saleValueInput.setAttribute('max', originalPrice);
                saleValueInput.placeholder = `Max: RM${originalPrice.toFixed(2)}`;
            }
            
            // Show/hide variations
            if (variations.length > 0) {
                variationsDisplay.style.display = 'block';
                document.getElementById('variation-actions').style.display = 'block';
                variationsContainer.innerHTML = '';
                
                variations.forEach((variation, index) => {
                    const variationDiv = document.createElement('div');
                    variationDiv.className = 'variation-item';
                    variationDiv.innerHTML = `
                        <div class="variation-checkbox">
                            <input type="checkbox" 
                                   id="variation_${index}" 
                                   name="selected_variations[]" 
                                   value="${variation.version_type || 'Standard'}:${variation.size || 'N/A'}:${variation.price}"
                                   data-price="${variation.price}"
                                   checked>
                            <label for="variation_${index}"></label>
                        </div>
                        <div class="variation-info">
                            <span class="variation-type">${variation.version_type || 'Standard'}</span>
                            <span class="variation-size">${variation.size || 'N/A'}</span>
                            <span class="variation-price">RM ${parseFloat(variation.price).toFixed(2)}</span>
                            <span class="variation-stock">Stock: ${variation.stock}</span>
                        </div>
                    `;
                    variationsContainer.appendChild(variationDiv);
                });
            } else {
                variationsDisplay.style.display = 'none';
                document.getElementById('variation-actions').style.display = 'none';
            }
        });

        // Select all variations
        function selectAllVariations() {
            const checkboxes = document.querySelectorAll('input[name="selected_variations[]"]');
            checkboxes.forEach(checkbox => {
                checkbox.checked = true;
            });
        }
        
        // Deselect all variations
        function deselectAllVariations() {
            const checkboxes = document.querySelectorAll('input[name="selected_variations[]"]');
            checkboxes.forEach(checkbox => {
                checkbox.checked = false;
            });
        }

        // Auto-refresh every 5 minutes to update time remaining
        setInterval(function() {
            location.reload();
        }, 300000); // 5 minutes
    </script>
</body>
</html>