<?php
require_once '../includes/auth.php';
require_once '../includes/functions.php';
require_once '../includes/Voucher.php';

$adminAuth->requireLogin();

$voucherModel = new Voucher();

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'create_point_code':
                $data = [
                    'code' => strtoupper(trim($_POST['code'])),
                    'points' => (int)$_POST['points'],
                    'description' => $_POST['description'],
                    'max_uses' => !empty($_POST['max_uses']) ? (int)$_POST['max_uses'] : null,
                    'valid_from' => $_POST['valid_from'],
                    'valid_until' => $_POST['valid_until'],
                    'is_active' => isset($_POST['is_active']),
                    'created_by' => $_SESSION['admin_id'] ?? 1
                ];
                
                if ($voucherModel->createPointCode($data)) {
                    $success_message = "Point code created successfully!";
                } else {
                    $error_message = "Failed to create point code.";
                }
                break;
                
            case 'update_point_code':
                $id = (int)$_POST['point_code_id'];
                $data = [
                    'code' => strtoupper(trim($_POST['code'])),
                    'points' => (int)$_POST['points'],
                    'description' => $_POST['description'],
                    'max_uses' => !empty($_POST['max_uses']) ? (int)$_POST['max_uses'] : null,
                    'valid_from' => $_POST['valid_from'],
                    'valid_until' => $_POST['valid_until'],
                    'is_active' => isset($_POST['is_active'])
                ];
                
                if ($voucherModel->updatePointCode($id, $data)) {
                    $success_message = "Point code updated successfully!";
                } else {
                    $error_message = "Failed to update point code.";
                }
                break;
                
            case 'delete_point_code':
                $id = (int)$_POST['point_code_id'];
                if ($voucherModel->deletePointCode($id)) {
                    $success_message = "Point code deleted successfully!";
                } else {
                    $error_message = "Failed to delete point code.";
                }
                break;
                
            case 'create_voucher':
                // Handle image upload
                $imagePath = $_POST['voucher_image'] ?? ''; // Default to URL if provided
                
                if (isset($_FILES['voucher_image_file']) && $_FILES['voucher_image_file']['error'] === UPLOAD_ERR_OK) {
                    $uploadDir = __DIR__ . '/../../images/vouchers/';
                    $allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
                    $maxSize = 5 * 1024 * 1024; // 5MB
                    
                    // Create vouchers directory if it doesn't exist
                    if (!is_dir($uploadDir)) {
                        mkdir($uploadDir, 0755, true);
                    }
                    
                    $file = $_FILES['voucher_image_file'];
                    
                    if (in_array($file['type'], $allowedTypes) && $file['size'] <= $maxSize) {
                        $fileName = 'voucher_' . time() . '_' . preg_replace('/[^a-zA-Z0-9._-]/', '', $file['name']);
                        $filePath = $uploadDir . $fileName;
                        
                        if (move_uploaded_file($file['tmp_name'], $filePath)) {
                            $imagePath = 'images/vouchers/' . $fileName;
                        } else {
                            $error_message = 'Error uploading image. Please check directory permissions.';
                        }
                    } else {
                        $error_message = 'Invalid image type or file too large. Only JPG, PNG, GIF, and WebP files up to 5MB are allowed.';
                    }
                }
                
                $data = [
                    'voucher_name' => $_POST['voucher_name'],
                    'voucher_description' => $_POST['voucher_description'],
                    'voucher_image' => $imagePath,
                    'points_required' => (int)$_POST['points_required'],
                    'discount_type' => $_POST['discount_type'],
                    'discount_value' => (float)$_POST['discount_value'],
                    'max_redemptions' => !empty($_POST['max_redemptions']) ? (int)$_POST['max_redemptions'] : null,
                    'valid_from' => $_POST['valid_from'],
                    'valid_until' => $_POST['valid_until'],
                    'is_active' => isset($_POST['is_active']),
                    'created_by' => $_SESSION['admin_id'] ?? 1
                ];
                
                if ($voucherModel->createVoucher($data)) {
                    $success_message = "Voucher created successfully!";
                } else {
                    $error_message = "Failed to create voucher.";
                }
                break;
                
            case 'update_voucher':
                $id = (int)$_POST['voucher_id'];
                
                // Handle image upload for update
                $imagePath = $_POST['voucher_image'] ?? null;
                
                if (isset($_FILES['voucher_image_file']) && $_FILES['voucher_image_file']['error'] === UPLOAD_ERR_OK) {
                    $uploadDir = __DIR__ . '/../../images/vouchers/';
                    $allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
                    $maxSize = 5 * 1024 * 1024; // 5MB
                    
                    // Create vouchers directory if it doesn't exist
                    if (!is_dir($uploadDir)) {
                        mkdir($uploadDir, 0755, true);
                    }
                    
                    $file = $_FILES['voucher_image_file'];
                    
                    if (in_array($file['type'], $allowedTypes) && $file['size'] <= $maxSize) {
                        $fileName = 'voucher_' . time() . '_' . preg_replace('/[^a-zA-Z0-9._-]/', '', $file['name']);
                        $filePath = $uploadDir . $fileName;
                        
                        if (move_uploaded_file($file['tmp_name'], $filePath)) {
                            $imagePath = 'images/vouchers/' . $fileName;
                        } else {
                            $error_message = 'Error uploading image. Please check directory permissions.';
                        }
                    } else {
                        $error_message = 'Invalid image type or file too large. Only JPG, PNG, GIF, and WebP files up to 5MB are allowed.';
                    }
                }
                
                $data = [
                    'voucher_name' => $_POST['voucher_name'],
                    'voucher_description' => $_POST['voucher_description'],
                    'voucher_image' => $imagePath,
                    'points_required' => (int)$_POST['points_required'],
                    'discount_type' => $_POST['discount_type'],
                    'discount_value' => (float)$_POST['discount_value'],
                    'max_redemptions' => !empty($_POST['max_redemptions']) ? (int)$_POST['max_redemptions'] : null,
                    'valid_from' => $_POST['valid_from'],
                    'valid_until' => $_POST['valid_until'],
                    'is_active' => isset($_POST['is_active'])
                ];
                
                if ($voucherModel->updateVoucher($id, $data)) {
                    $success_message = "Voucher updated successfully!";
                } else {
                    $error_message = "Failed to update voucher.";
                }
                break;
                
            case 'delete_voucher':
                $id = (int)$_POST['voucher_id'];
                if ($voucherModel->deleteVoucher($id)) {
                    $success_message = "Voucher deleted successfully!";
                } else {
                    $error_message = "Failed to delete voucher.";
                }
                break;
                
            case 'create_tier':
                $data = [
                    'tier_name' => $_POST['tier_name'],
                    'min_points' => (int)$_POST['min_points'],
                    'max_points' => !empty($_POST['max_points']) ? (int)$_POST['max_points'] : null,
                    'tier_color' => $_POST['tier_color'],
                    'tier_icon' => $_POST['tier_icon'],
                    'benefits' => $_POST['benefits']
                ];
                
                if ($voucherModel->addRoyalTier($data)) {
                    $success_message = "Royal tier created successfully!";
                } else {
                    $error_message = "Failed to create royal tier.";
                }
                break;
                
            case 'update_tier':
                $id = (int)$_POST['tier_id'];
                $data = [
                    'tier_name' => $_POST['tier_name'],
                    'min_points' => (int)$_POST['min_points'],
                    'max_points' => !empty($_POST['max_points']) ? (int)$_POST['max_points'] : null,
                    'tier_color' => $_POST['tier_color'],
                    'tier_icon' => $_POST['tier_icon'],
                    'benefits' => $_POST['benefits']
                ];
                
                if ($voucherModel->updateRoyalTier($id, $data)) {
                    $success_message = "Royal tier updated successfully!";
                } else {
                    $error_message = "Failed to update royal tier.";
                }
                break;
                
            case 'delete_tier':
                $id = (int)$_POST['tier_id'];
                if ($voucherModel->deleteRoyalTier($id)) {
                    $success_message = "Royal tier deleted successfully!";
                } else {
                    $error_message = "Failed to delete royal tier.";
                }
                break;
        }
    }
}

// Get vouchers and stats
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 10;
$offset = ($page - 1) * $limit;
$search = $_GET['search'] ?? '';

$vouchers = $voucherModel->getAllVouchers($limit, $offset, $search);
$stats = $voucherModel->getVoucherStats();
$royalTiers = $voucherModel->getAllRoyalTiers();

// Get point codes
$pointCodes = $voucherModel->getAllPointCodes($limit, $offset, $search);
$pointCodeStats = $voucherModel->getPointCodeStats();

// Get voucher for editing
$editVoucher = null;
if (isset($_GET['edit'])) {
    $editVoucher = $voucherModel->getVoucherById((int)$_GET['edit']);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Voucher Management - Admin Panel</title>
    <link rel="stylesheet" href="../assets/css/admin.css?v=<?php echo time(); ?>">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body>
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="sidebar-header">
            <img src="../../images/LOGO.png" alt="Logo" class="sidebar-logo">
            <h3>Admin Panel</h3>
        </div>
        
        <nav class="sidebar-nav">
            <a href="../index.php" class="nav-item">
                <i class="fas fa-th-large"></i>
                <span>Overview</span>
            </a>
            <a href="products.php" class="nav-item">
                <i class="fas fa-box"></i>
                <span>Products</span>
            </a>
            <a href="orders.php" class="nav-item">
                <i class="fas fa-shopping-cart"></i>
                <span>Orders</span>
            </a>
            <a href="delivered.php" class="nav-item">
                <i class="fas fa-check-circle"></i>
                <span>Delivered</span>
            </a>
            <a href="customers.php" class="nav-item">
                <i class="fas fa-users"></i>
                <span>Customers</span>
            </a>
            <a href="feedback.php" class="nav-item">
                <i class="fas fa-comments"></i>
                <span>Feedback</span>
            </a>
            <a href="shipping.php" class="nav-item">
                <i class="fas fa-shipping-fast"></i>
                <span>Shipping</span>
            </a>
            <a href="banners.php" class="nav-item">
                <i class="fas fa-image"></i>
                <span>Banners</span>
            </a>
            <a href="presets.php" class="nav-item">
                <i class="fas fa-cogs"></i>
                <span>Field Presets</span>
            </a>
            <a href="vouchers.php" class="nav-item active">
                <i class="fas fa-ticket-alt"></i>
                <span>Voucher Management</span>
            </a>
            <a href="notifications.php" class="nav-item">
                <i class="fas fa-bell"></i>
                <span>Push Notifications</span>
            </a>
            <a href="analytics.php" class="nav-item">
                <i class="fas fa-chart-bar"></i>
                <span>Analytics</span>
            </a>
            <a href="settings.php" class="nav-item">
                <i class="fas fa-cog"></i>
                <span>Settings</span>
            </a>
        </nav>
        
        <div class="sidebar-footer">
            <div class="user-info">
                <i class="fas fa-user-circle"></i>
                <span><?php echo htmlspecialchars($adminAuth->getAdminName()); ?></span>
            </div>
            <a href="../logout.php" class="logout-btn">
                <i class="fas fa-sign-out-alt"></i>
                <span>Logout</span>
            </a>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <!-- Header -->
        <header class="header">
            <div class="header-left">
                <button class="sidebar-toggle">
                    <i class="fas fa-bars"></i>
                </button>
                <h1>Voucher Management</h1>
            </div>
            <div class="header-right">
                <div class="header-actions">
                    <button onclick="openAddVoucherModal()" class="btn btn-primary">
                        <i class="fas fa-plus"></i>
                        Add Voucher
                    </button>
                    <a href="../index.php" target="_blank" class="btn btn-outline">
                        <i class="fas fa-external-link-alt"></i>
                        View Store
                    </a>
                </div>
            </div>
        </header>

        <!-- Content -->
        <div class="content">
            <?php if (isset($success_message)): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i> <?php echo $success_message; ?>
                </div>
            <?php endif; ?>

            <?php if (isset($error_message)): ?>
                <div class="alert alert-error">
                    <i class="fas fa-exclamation-circle"></i> <?php echo $error_message; ?>
                </div>
            <?php endif; ?>

            <!-- Statistics Cards -->
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-ticket-alt"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?php echo $stats['total_vouchers']; ?></h3>
                        <p>Total Vouchers</p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-check-circle"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?php echo $stats['active_vouchers']; ?></h3>
                        <p>Active Vouchers</p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-gift"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?php echo $stats['total_redemptions']; ?></h3>
                        <p>Total Redemptions</p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-star"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?php echo number_format($stats['avg_points_required'], 0); ?></h3>
                        <p>Avg Points Required</p>
                    </div>
                </div>
            </div>

            <!-- Point Codes Statistics -->
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-code"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?php echo $pointCodeStats['total_codes']; ?></h3>
                        <p>Total Point Codes</p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-check-circle"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?php echo $pointCodeStats['active_codes']; ?></h3>
                        <p>Active Codes</p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-users"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?php echo $pointCodeStats['total_uses']; ?></h3>
                        <p>Total Uses</p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-coins"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?php echo number_format($pointCodeStats['total_points_awarded'], 0); ?></h3>
                        <p>Points Awarded</p>
                    </div>
                </div>
            </div>

            <!-- Tabs for Vouchers and Point Codes -->
            <div class="tabs-container">
                <div class="tabs">
                    <button class="tab-button active" onclick="switchTab('vouchers')">
                        <i class="fas fa-ticket-alt"></i>
                        Vouchers
                    </button>
                    <button class="tab-button" onclick="switchTab('point-codes')">
                        <i class="fas fa-code"></i>
                        Point Codes
                    </button>
                </div>
            </div>

            <!-- Vouchers Tab Content -->
            <div id="vouchers-tab" class="tab-content active">
                <div class="content-header">
                    <div class="search-box">
                        <i class="fas fa-search"></i>
                        <input type="text" id="searchInput" placeholder="Search vouchers..." value="<?php echo htmlspecialchars($search); ?>">
                    </div>
                    <div class="filter-actions">
                        <button class="btn btn-outline" onclick="searchVouchers()">
                            <i class="fas fa-search"></i>
                            Search
                        </button>
                        <button onclick="openAddVoucherModal()" class="btn btn-primary">
                            <i class="fas fa-plus"></i>
                            Add Voucher
                        </button>
                    </div>
                </div>

                <!-- Vouchers Table -->
                <div class="table-container">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Voucher</th>
                                <th>Points Required</th>
                                <th>Discount</th>
                                <th>Redemptions</th>
                                <th>Valid Until</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($vouchers as $voucher): ?>
                                <tr>
                                    <td>
                                        <div class="voucher-info">
                                            <?php if ($voucher['voucher_image']): ?>
                                                <img src="<?php echo htmlspecialchars($voucher['voucher_image']); ?>" alt="Voucher" class="voucher-thumb">
                                            <?php endif; ?>
                                            <div>
                                                <h4><?php echo htmlspecialchars($voucher['voucher_name']); ?></h4>
                                                <p><?php echo htmlspecialchars($voucher['voucher_description']); ?></p>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <span class="points-badge"><?php echo number_format($voucher['points_required']); ?> pts</span>
                                    </td>
                                    <td>
                                        <span class="discount-badge">
                                            <?php 
                                            if ($voucher['discount_type'] === 'percentage') {
                                                echo $voucher['discount_value'] . '%';
                                            } elseif ($voucher['discount_type'] === 'fixed_amount') {
                                                echo 'RM ' . $voucher['discount_value'];
                                            } else {
                                                echo ucfirst(str_replace('_', ' ', $voucher['discount_type']));
                                            }
                                            ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php echo $voucher['current_redemptions']; ?>
                                        <?php if ($voucher['max_redemptions']): ?>
                                            / <?php echo $voucher['max_redemptions']; ?>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo date('M j, Y', strtotime($voucher['valid_until'])); ?></td>
                                    <td>
                                        <span class="status-badge <?php echo $voucher['is_active'] ? 'active' : 'inactive'; ?>">
                                            <?php echo $voucher['is_active'] ? 'Active' : 'Inactive'; ?>
                                        </span>
                                    </td>
                                    <td>
                                        <div class="action-buttons">
                                            <button onclick="editVoucher(<?php echo $voucher['id']; ?>)" class="btn btn-sm btn-outline">
                                                <i class="fas fa-edit"></i>
                                            </button>
                                            <form method="POST" style="display: inline;" onsubmit="return confirm('Are you sure you want to delete this voucher?')">
                                                <input type="hidden" name="action" value="delete_voucher">
                                                <input type="hidden" name="voucher_id" value="<?php echo $voucher['id']; ?>">
                                                <button type="submit" class="btn btn-sm btn-danger">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </form>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Point Codes Tab Content -->
            <div id="point-codes-tab" class="tab-content">
                <div class="content-header">
                    <div class="search-box">
                        <i class="fas fa-search"></i>
                        <input type="text" id="pointCodeSearchInput" placeholder="Search point codes..." value="<?php echo htmlspecialchars($search); ?>">
                    </div>
                    <div class="filter-actions">
                        <button class="btn btn-outline" onclick="searchPointCodes()">
                            <i class="fas fa-search"></i>
                            Search
                        </button>
                        <button onclick="openAddPointCodeModal()" class="btn btn-primary">
                            <i class="fas fa-plus"></i>
                            Add Point Code
                        </button>
                    </div>
                </div>

                <!-- Point Codes Table -->
                <div class="table-container">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Code</th>
                                <th>Points</th>
                                <th>Description</th>
                                <th>Uses</th>
                                <th>Valid Until</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($pointCodes as $pointCode): ?>
                                <tr>
                                    <td>
                                        <div class="code-info">
                                            <h4><?php echo htmlspecialchars($pointCode['code']); ?></h4>
                                        </div>
                                    </td>
                                    <td>
                                        <span class="points-badge"><?php echo number_format($pointCode['points']); ?> pts</span>
                                    </td>
                                    <td>
                                        <p><?php echo htmlspecialchars($pointCode['description']); ?></p>
                                    </td>
                                    <td>
                                        <?php echo $pointCode['current_uses']; ?>
                                        <?php if ($pointCode['max_uses']): ?>
                                            / <?php echo $pointCode['max_uses']; ?>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo date('M j, Y', strtotime($pointCode['valid_until'])); ?></td>
                                    <td>
                                        <span class="status-badge <?php echo $pointCode['is_active'] ? 'active' : 'inactive'; ?>">
                                            <?php echo $pointCode['is_active'] ? 'Active' : 'Inactive'; ?>
                                        </span>
                                    </td>
                                    <td>
                                        <div class="action-buttons">
                                            <button onclick="editPointCode(<?php echo $pointCode['id']; ?>)" class="btn btn-sm btn-outline">
                                                <i class="fas fa-edit"></i>
                                            </button>
                                            <form method="POST" style="display: inline;" onsubmit="return confirm('Are you sure you want to delete this point code?')">
                                                <input type="hidden" name="action" value="delete_point_code">
                                                <input type="hidden" name="point_code_id" value="<?php echo $pointCode['id']; ?>">
                                                <button type="submit" class="btn btn-sm btn-danger">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </form>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Royal Tiers Section (Locked - Package 3 Feature) -->
            <div class="section locked-section">
                <div class="section-header">
                    <h2>
                        Royal Tiers 
                        <span class="premium-badge">
                            <i class="fas fa-crown"></i> Package 3 Feature
                        </span>
                    </h2>
                    <p>Manage customer loyalty tiers</p>
                    <button onclick="showUpgradeMessage()" class="btn btn-primary btn-locked" style="margin-top: 15px;" disabled>
                        <i class="fas fa-lock"></i>
                        Add New Tier (Locked)
                    </button>
                </div>
                <div class="locked-overlay">
                    <div class="lock-message">
                        <i class="fas fa-lock fa-3x"></i>
                        <h3>Premium Feature Locked</h3>
                        <p>Royal Tiers is available in <strong>Package 3</strong></p>
                        <p>Upgrade your plan to unlock advanced loyalty tier management</p>
                        <button onclick="showUpgradeMessage()" class="btn btn-primary" style="margin-top: 15px;">
                            <i class="fas fa-star"></i>
                            Upgrade to Package 3
                        </button>
                    </div>
                </div>
                <div class="tiers-grid locked-content">
                    <?php 
                    $tierIndex = 0;
                    foreach ($royalTiers as $tier): 
                        $tierIndex++;
                    ?>
                        <div class="tier-card tier-<?php echo strtolower($tier['tier_name']); ?> locked">
                            <div class="tier-icon">
                                <i class="<?php echo $tier['tier_icon']; ?>"></i>
                            </div>
                            <div class="tier-info">
                                <h3><?php echo $tier['tier_name']; ?></h3>
                                <p class="tier-points">
                                    <?php echo number_format($tier['min_points']); ?> - 
                                    <?php echo $tier['max_points'] ? number_format($tier['max_points']) : '∞'; ?> points
                                </p>
                                <p class="tier-benefits"><?php echo $tier['benefits']; ?></p>
                            </div>
                            <div class="tier-actions">
                                <button onclick="showUpgradeMessage()" class="btn btn-sm btn-outline" disabled>
                                    <i class="fas fa-lock"></i>
                                </button>
                                <button onclick="showUpgradeMessage()" class="btn btn-sm btn-danger" disabled>
                                    <i class="fas fa-lock"></i>
                                </button>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
    </div>

    <!-- Add/Edit Voucher Modal -->
    <div id="voucherModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2 id="modalTitle">Add New Voucher</h2>
                <button class="modal-close" onclick="closeVoucherModal()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <form id="voucherForm" method="POST" enctype="multipart/form-data">
                <input type="hidden" name="action" value="create_voucher">
                <input type="hidden" name="voucher_id" id="voucherId">
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="voucher_name">Voucher Name</label>
                        <input type="text" id="voucher_name" name="voucher_name" required>
                    </div>
                    <div class="form-group">
                        <label for="points_required">Points Required</label>
                        <input type="number" id="points_required" name="points_required" min="1" required>
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="voucher_description">Description</label>
                    <textarea id="voucher_description" name="voucher_description" rows="3" required></textarea>
                </div>
                
                <div class="form-group">
                    <label for="voucher_image">Voucher Image</label>
                    <div class="image-upload-section">
                        <div class="image-upload-options">
                            <div class="upload-option">
                                <label class="upload-label">
                                    <input type="radio" name="image_source" value="file" checked>
                                    <span>Upload File</span>
                                </label>
                                <input type="file" id="voucher_image_file" name="voucher_image_file" accept="image/*" style="display: none;">
                                <button type="button" class="btn btn-outline" onclick="document.getElementById('voucher_image_file').click()">
                                    <i class="fas fa-upload"></i>
                                    Choose File
                                </button>
                            </div>
                            <div class="upload-option">
                                <label class="upload-label">
                                    <input type="radio" name="image_source" value="url">
                                    <span>Image URL</span>
                                </label>
                                <input type="url" id="voucher_image" name="voucher_image" placeholder="https://example.com/image.jpg" style="display: none;">
                            </div>
                        </div>
                        <div class="image-preview" id="imagePreview" style="display: none;">
                            <img id="previewImg" src="" alt="Preview">
                            <button type="button" class="remove-image" onclick="removeImage()">
                                <i class="fas fa-times"></i>
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="discount_type">Discount Type</label>
                        <select id="discount_type" name="discount_type" required>
                            <option value="percentage">Percentage</option>
                            <option value="fixed_amount">Fixed Amount</option>
                            <option value="free_shipping">Free Shipping</option>
                            <option value="free_product">Free Product</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="discount_value">Discount Value</label>
                        <input type="number" id="discount_value" name="discount_value" step="0.01" min="0" required>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="max_redemptions">Max Redemptions (optional)</label>
                        <input type="number" id="max_redemptions" name="max_redemptions" min="1">
                    </div>
                    <div class="form-group">
                        <label>
                            <input type="checkbox" id="is_active" name="is_active" checked>
                            Active
                        </label>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="valid_from">Valid From</label>
                        <input type="date" id="valid_from" name="valid_from" required>
                    </div>
                    <div class="form-group">
                        <label for="valid_until">Valid Until</label>
                        <input type="date" id="valid_until" name="valid_until" required>
                    </div>
                </div>
                
                <div class="modal-actions">
                    <button type="button" class="btn btn-outline" onclick="closeVoucherModal()">Cancel</button>
                    <button type="submit" class="btn btn-primary">Save Voucher</button>
                </div>
            </form>
        </div>
    </div>

    <!-- Add/Edit Point Code Modal -->
    <div id="pointCodeModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2 id="pointCodeModalTitle">Add New Point Code</h2>
                <button class="modal-close" onclick="closePointCodeModal()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <form id="pointCodeForm" method="POST">
                <input type="hidden" name="action" value="create_point_code">
                <input type="hidden" name="point_code_id" id="pointCodeId">
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="code">Code</label>
                        <input type="text" id="code" name="code" required placeholder="e.g., WELCOME10" maxlength="50">
                        <small>Code will be automatically converted to uppercase</small>
                    </div>
                    <div class="form-group">
                        <label for="points">Points to Award</label>
                        <input type="number" id="points" name="points" min="1" required placeholder="e.g., 10, 50, 100">
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="description">Description</label>
                    <textarea id="description" name="description" rows="2" required placeholder="Describe what this code does..."></textarea>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="max_uses">Max Uses (optional)</label>
                        <input type="number" id="max_uses" name="max_uses" min="1" placeholder="Leave empty for unlimited">
                    </div>
                    <div class="form-group">
                        <label>
                            <input type="checkbox" id="is_active" name="is_active" checked>
                            Active
                        </label>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="valid_from">Valid From</label>
                        <input type="date" id="valid_from" name="valid_from" required>
                    </div>
                    <div class="form-group">
                        <label for="valid_until">Valid Until</label>
                        <input type="date" id="valid_until" name="valid_until" required>
                    </div>
                </div>
                
                <div class="modal-actions">
                    <button type="button" class="btn btn-outline" onclick="closePointCodeModal()">Cancel</button>
                    <button type="submit" class="btn btn-primary">Save Point Code</button>
                </div>
            </form>
        </div>
    </div>

    <!-- Add/Edit Tier Modal -->
    <div id="tierModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2 id="tierModalTitle">Add New Royal Tier</h2>
                <button class="modal-close" onclick="closeTierModal()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <form id="tierForm" method="POST">
                <input type="hidden" name="action" value="create_tier">
                <input type="hidden" name="tier_id" id="tierId">
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="tier_name">Tier Name</label>
                        <input type="text" id="tier_name" name="tier_name" required placeholder="e.g., Bronze, Silver, Gold">
                    </div>
                    <div class="form-group">
                        <label for="tier_color">Tier Color</label>
                        <input type="color" id="tier_color" name="tier_color" value="#CD7F32" required>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="min_points">Minimum Points</label>
                        <input type="number" id="min_points" name="min_points" min="0" required>
                    </div>
                    <div class="form-group">
                        <label for="max_points">Maximum Points (leave empty for unlimited)</label>
                        <input type="number" id="max_points" name="max_points" min="1">
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="tier_icon">Tier Icon</label>
                    <select id="tier_icon" name="tier_icon" required>
                        <option value="fas fa-medal">Medal</option>
                        <option value="fas fa-award">Award</option>
                        <option value="fas fa-trophy">Trophy</option>
                        <option value="fas fa-crown">Crown</option>
                        <option value="fas fa-gem">Gem</option>
                        <option value="fas fa-star">Star</option>
                        <option value="fas fa-diamond">Diamond</option>
                        <option value="fas fa-fire">Fire</option>
                        <option value="fas fa-bolt">Lightning</option>
                        <option value="fas fa-rocket">Rocket</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="benefits">Benefits Description</label>
                    <textarea id="benefits" name="benefits" rows="3" required placeholder="Describe the benefits for this tier..."></textarea>
                </div>
                
                <div class="modal-actions">
                    <button type="button" class="btn btn-outline" onclick="closeTierModal()">Cancel</button>
                    <button type="submit" class="btn btn-primary">Save Tier</button>
                </div>
            </form>
        </div>
    </div>

    <script>
        // Tab switching functions
        function switchTab(tabName) {
            // Hide all tab contents
            document.querySelectorAll('.tab-content').forEach(tab => {
                tab.classList.remove('active');
            });
            
            // Remove active class from all tab buttons
            document.querySelectorAll('.tab-button').forEach(button => {
                button.classList.remove('active');
            });
            
            // Show selected tab content
            document.getElementById(tabName + '-tab').classList.add('active');
            
            // Add active class to clicked button
            event.target.classList.add('active');
        }

        // Modal functions
        function openAddVoucherModal() {
            document.getElementById('modalTitle').textContent = 'Add New Voucher';
            document.querySelector('input[name="action"]').value = 'create_voucher';
            document.getElementById('voucherForm').reset();
            document.getElementById('voucherId').value = '';
            document.getElementById('valid_from').value = new Date().toISOString().split('T')[0];
            document.getElementById('valid_until').value = new Date(Date.now() + 365 * 24 * 60 * 60 * 1000).toISOString().split('T')[0];
            document.getElementById('voucherModal').style.display = 'flex';
        }

        function openAddPointCodeModal() {
            document.getElementById('pointCodeModalTitle').textContent = 'Add New Point Code';
            document.querySelector('#pointCodeForm input[name="action"]').value = 'create_point_code';
            document.getElementById('pointCodeForm').reset();
            document.getElementById('pointCodeId').value = '';
            document.getElementById('valid_from').value = new Date().toISOString().split('T')[0];
            document.getElementById('valid_until').value = new Date(Date.now() + 365 * 24 * 60 * 60 * 1000).toISOString().split('T')[0];
            document.getElementById('pointCodeModal').style.display = 'flex';
        }

        function editPointCode(pointCodeId) {
            // This would typically fetch point code data via AJAX
            // For now, redirect to edit mode
            window.location.href = 'vouchers.php?edit_point_code=' + pointCodeId;
        }

        function closePointCodeModal() {
            document.getElementById('pointCodeModal').style.display = 'none';
        }

        function searchPointCodes() {
            const searchTerm = document.getElementById('pointCodeSearchInput').value;
            window.location.href = 'vouchers.php?search=' + encodeURIComponent(searchTerm) + '&tab=point-codes';
        }

        function editVoucher(voucherId) {
            // This would typically fetch voucher data via AJAX
            // For now, redirect to edit mode
            window.location.href = 'vouchers.php?edit=' + voucherId;
        }

        function closeVoucherModal() {
            document.getElementById('voucherModal').style.display = 'none';
        }

        function searchVouchers() {
            const searchTerm = document.getElementById('searchInput').value;
            window.location.href = 'vouchers.php?search=' + encodeURIComponent(searchTerm);
        }

        // Handle edit mode
        <?php if ($editVoucher): ?>
        document.addEventListener('DOMContentLoaded', function() {
            document.getElementById('modalTitle').textContent = 'Edit Voucher';
            document.querySelector('input[name="action"]').value = 'update_voucher';
            document.getElementById('voucherId').value = '<?php echo $editVoucher['id']; ?>';
            document.getElementById('voucher_name').value = '<?php echo htmlspecialchars($editVoucher['voucher_name']); ?>';
            document.getElementById('voucher_description').value = '<?php echo htmlspecialchars($editVoucher['voucher_description']); ?>';
            document.getElementById('voucher_image').value = '<?php echo htmlspecialchars($editVoucher['voucher_image'] ?? ''); ?>';
            document.getElementById('points_required').value = '<?php echo $editVoucher['points_required']; ?>';
            document.getElementById('discount_type').value = '<?php echo $editVoucher['discount_type']; ?>';
            document.getElementById('discount_value').value = '<?php echo $editVoucher['discount_value']; ?>';
            document.getElementById('max_redemptions').value = '<?php echo $editVoucher['max_redemptions'] ?? ''; ?>';
            document.getElementById('valid_from').value = '<?php echo $editVoucher['valid_from']; ?>';
            document.getElementById('valid_until').value = '<?php echo $editVoucher['valid_until']; ?>';
            document.getElementById('is_active').checked = <?php echo $editVoucher['is_active'] ? 'true' : 'false'; ?>;
            document.getElementById('voucherModal').style.display = 'flex';
        });
        <?php endif; ?>

        // Search on Enter key
        document.getElementById('searchInput').addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                searchVouchers();
            }
        });

        // Image upload handling
        document.getElementById('voucher_image_file').addEventListener('change', function(e) {
            const file = e.target.files[0];
            if (file) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    document.getElementById('previewImg').src = e.target.result;
                    document.getElementById('imagePreview').style.display = 'block';
                };
                reader.readAsDataURL(file);
            }
        });

        // Image source radio button handling
        document.querySelectorAll('input[name="image_source"]').forEach(radio => {
            radio.addEventListener('change', function() {
                const fileInput = document.getElementById('voucher_image_file');
                const urlInput = document.getElementById('voucher_image');
                const preview = document.getElementById('imagePreview');
                
                if (this.value === 'file') {
                    fileInput.style.display = 'none';
                    urlInput.style.display = 'none';
                    preview.style.display = 'none';
                } else {
                    fileInput.style.display = 'none';
                    urlInput.style.display = 'block';
                    preview.style.display = 'none';
                }
            });
        });

        // URL input handling
        document.getElementById('voucher_image').addEventListener('input', function(e) {
            const url = e.target.value;
            if (url) {
                document.getElementById('previewImg').src = url;
                document.getElementById('imagePreview').style.display = 'block';
            } else {
                document.getElementById('imagePreview').style.display = 'none';
            }
        });

        function removeImage() {
            document.getElementById('voucher_image_file').value = '';
            document.getElementById('voucher_image').value = '';
            document.getElementById('imagePreview').style.display = 'none';
        }

        // Close modal when clicking outside
        window.addEventListener('click', function(e) {
            const modal = document.getElementById('voucherModal');
            if (e.target === modal) {
                closeVoucherModal();
            }
        });

        // Tier Management Functions
        function openTierModal() {
            document.getElementById('tierModalTitle').textContent = 'Add New Royal Tier';
            document.querySelector('#tierForm input[name="action"]').value = 'create_tier';
            document.getElementById('tierForm').reset();
            document.getElementById('tierId').value = '';
            document.getElementById('tier_color').value = '#CD7F32';
            document.getElementById('tierModal').style.display = 'flex';
        }

        function editTier(tierId) {
            // This would typically fetch tier data via AJAX
            // For now, redirect to edit mode
            window.location.href = 'vouchers.php?edit_tier=' + tierId;
        }

        function deleteTier(tierId) {
            if (confirm('Are you sure you want to delete this royal tier? This action cannot be undone.')) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.innerHTML = `
                    <input type="hidden" name="action" value="delete_tier">
                    <input type="hidden" name="tier_id" value="${tierId}">
                `;
                document.body.appendChild(form);
                form.submit();
            }
        }

        function closeTierModal() {
            document.getElementById('tierModal').style.display = 'none';
        }

        // Handle edit tier mode
        <?php if (isset($_GET['edit_tier'])): ?>
        document.addEventListener('DOMContentLoaded', function() {
            // This would typically fetch tier data via AJAX
            // For now, show the modal with edit mode
            document.getElementById('tierModalTitle').textContent = 'Edit Royal Tier';
            document.querySelector('#tierForm input[name="action"]').value = 'update_tier';
            document.getElementById('tierId').value = '<?php echo (int)$_GET['edit_tier']; ?>';
            document.getElementById('tierModal').style.display = 'flex';
        });
        <?php endif; ?>

        // Close tier modal when clicking outside
        window.addEventListener('click', function(e) {
            const modal = document.getElementById('tierModal');
            if (e.target === modal) {
                closeTierModal();
            }
        });

        // Show upgrade message for locked features
        function showUpgradeMessage() {
            alert('🔒 Premium Feature Locked\n\nRoyal Tiers is a Package 3 exclusive feature.\n\nTo unlock this feature and manage advanced loyalty tiers, please upgrade your subscription to Package 3.\n\nContact support for more information.');
        }

        // Disable tier modal functions for Package 2
        function openTierModal() {
            showUpgradeMessage();
        }

        function editTier(tierId) {
            showUpgradeMessage();
        }

        function deleteTier(tierId) {
            showUpgradeMessage();
        }
    </script>
    <style>
        /* Locked Section Styles */
        .locked-section {
            position: relative;
            opacity: 0.7;
        }

        .locked-overlay {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(3px);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 10;
            border-radius: 12px;
        }

        .lock-message {
            text-align: center;
            padding: 40px;
            max-width: 500px;
        }

        .lock-message i {
            color: #6c757d;
            margin-bottom: 20px;
            animation: lockPulse 2s infinite;
        }

        @keyframes lockPulse {
            0%, 100% { transform: scale(1); opacity: 0.8; }
            50% { transform: scale(1.1); opacity: 1; }
        }

        .lock-message h3 {
            color: #343a40;
            margin-bottom: 15px;
            font-size: 24px;
            font-weight: 700;
        }

        .lock-message p {
            color: #6c757d;
            margin-bottom: 10px;
            font-size: 16px;
            line-height: 1.6;
        }

        .lock-message strong {
            color: #ffc107;
            font-weight: 700;
        }

        .premium-badge {
            display: inline-block;
            background: linear-gradient(135deg, #ffd700, #ffed4e);
            color: #000;
            padding: 6px 15px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: 700;
            margin-left: 10px;
            box-shadow: 0 2px 8px rgba(255, 215, 0, 0.3);
            animation: shimmer 2s infinite;
        }

        @keyframes shimmer {
            0%, 100% { transform: scale(1); box-shadow: 0 2px 8px rgba(255, 215, 0, 0.3); }
            50% { transform: scale(1.05); box-shadow: 0 4px 12px rgba(255, 215, 0, 0.5); }
        }

        .premium-badge i {
            margin-right: 5px;
            color: #000;
        }

        .locked-content {
            filter: blur(2px);
            pointer-events: none;
        }

        .tier-card.locked {
            opacity: 0.6;
            cursor: not-allowed;
        }

        .btn-locked {
            background-color: #6c757d !important;
            cursor: not-allowed !important;
            opacity: 0.6;
        }

        .btn-locked:hover {
            background-color: #5a6268 !important;
        }

        .tier-actions button:disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }

        /* Make sure the section header stays visible */
        .locked-section .section-header {
            position: relative;
            z-index: 1;
            background: white;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 20px;
        }

        /* Upgrade button styling */
        .lock-message .btn {
            margin-top: 20px;
            padding: 12px 30px;
            font-size: 16px;
            font-weight: 600;
            box-shadow: 0 4px 12px rgba(0, 123, 255, 0.3);
            transition: all 0.3s ease;
        }

        .lock-message .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 16px rgba(0, 123, 255, 0.4);
        }

        /* Responsive adjustments */
        @media (max-width: 768px) {
            .lock-message {
                padding: 20px;
            }

            .lock-message h3 {
                font-size: 20px;
            }

            .lock-message p {
                font-size: 14px;
            }

            .premium-badge {
                display: block;
                margin: 10px 0;
                width: fit-content;
            }
        }

        /* Tabs Styles */
        .tabs-container {
            margin: 20px 0;
        }

        .tabs {
            display: flex;
            border-bottom: 2px solid #e9ecef;
            margin-bottom: 20px;
        }

        .tab-button {
            background: none;
            border: none;
            padding: 12px 24px;
            font-size: 16px;
            font-weight: 500;
            color: #6c757d;
            cursor: pointer;
            border-bottom: 3px solid transparent;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .tab-button:hover {
            color: #1A4D2A;
            background: rgba(26, 77, 42, 0.05);
        }

        .tab-button.active {
            color: #1A4D2A;
            border-bottom-color: #1A4D2A;
            background: rgba(26, 77, 42, 0.1);
        }

        .tab-content {
            display: none;
        }

        .tab-content.active {
            display: block;
        }

        /* Point Code Styles */
        .code-info h4 {
            font-family: 'Courier New', monospace;
            background: #f8f9fa;
            padding: 8px 12px;
            border-radius: 6px;
            border: 1px solid #e9ecef;
            color: #1A4D2A;
            font-weight: 600;
            margin: 0;
            display: inline-block;
        }

        .code-info p {
            margin: 8px 0 0 0;
            color: #6c757d;
            font-size: 14px;
        }

        /* Form small text */
        .form-group small {
            display: block;
            margin-top: 4px;
            color: #6c757d;
            font-size: 12px;
        }
    </style>
</body>
</html>