<?php
// Suppress all output except JSON
ob_start();
error_reporting(0);
ini_set('display_errors', 0);

session_start();
require_once '../includes/auth-check.php';
require_once '../admin/includes/Voucher.php';

// Clear any output buffer
ob_clean();

header('Content-Type: application/json');

// Check if user is logged in
if (!isLoggedIn()) {
    echo json_encode(['success' => false, 'message' => 'Please log in to use vouchers']);
    exit;
}

$userId = getUserId();
$input = json_decode(file_get_contents('php://input'), true);

// Handle both voucher code input and voucher data from checkout
if (isset($input['voucher_code'])) {
    // Legacy voucher code input
    $voucherCode = $input['voucher_code'];
    $orderTotal = $input['order_total'] ?? 0;

    if (!$voucherCode) {
        echo json_encode(['success' => false, 'message' => 'Please enter a voucher code']);
        exit;
    }

    try {
        $voucherModel = new Voucher();
        
        // Check if user has redeemed this voucher (only active ones)
        $redeemedVouchers = $voucherModel->getUserRedeemedVouchers($userId, 'active');
        
        // Find the voucher by code (we'll use voucher name as code for simplicity)
        $appliedVoucher = null;
        foreach ($redeemedVouchers as $voucher) {
            if (strtolower($voucher['voucher_name']) === strtolower($voucherCode)) {
                $appliedVoucher = $voucher;
                break;
            }
        }
        
        if (!$appliedVoucher) {
            echo json_encode(['success' => false, 'message' => 'Voucher not found or not redeemed yet']);
            exit;
        }
        
        // Calculate discount
        $discountAmount = 0;
        $newTotal = $orderTotal;
        
        if ($appliedVoucher['discount_type'] === 'percentage') {
            $discountAmount = ($orderTotal * $appliedVoucher['discount_value']) / 100;
            $newTotal = $orderTotal - $discountAmount;
        } elseif ($appliedVoucher['discount_type'] === 'fixed_amount') {
            $discountAmount = min($appliedVoucher['discount_value'], $orderTotal);
            $newTotal = $orderTotal - $discountAmount;
        } elseif ($appliedVoucher['discount_type'] === 'free_shipping') {
            // For free shipping, we'll apply a fixed discount (assuming shipping is RM 10)
            $discountAmount = 10;
            $newTotal = $orderTotal - $discountAmount;
        }
        
        // Ensure total doesn't go below 0
        $newTotal = max(0, $newTotal);
        
        echo json_encode([
            'success' => true,
            'message' => 'Voucher applied successfully',
            'voucher' => [
                'voucher_name' => $appliedVoucher['voucher_name'],
                'discount_type' => $appliedVoucher['discount_type'],
                'discount_value' => $appliedVoucher['discount_value'],
                'discount_amount' => $discountAmount
            ],
            'new_total' => $newTotal,
            'discount_amount' => $discountAmount
        ]);
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Error applying voucher: ' . $e->getMessage()]);
    }
} else {
    // New voucher data from checkout page
    $voucherData = $input;
    
    if (!isset($voucherData['voucher_id']) || !isset($voucherData['voucher_name'])) {
        echo json_encode(['success' => false, 'message' => 'Invalid voucher data']);
        exit;
    }
    
    try {
        // Save voucher to session for checkout processing
        $_SESSION['applied_voucher'] = [
            'voucher_id' => $voucherData['voucher_id'],
            'voucher_name' => $voucherData['voucher_name'],
            'discount_type' => $voucherData['discount_type'],
            'discount_value' => $voucherData['discount_value']
        ];
        
        echo json_encode([
            'success' => true,
            'message' => 'Voucher saved to session successfully'
        ]);
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Error saving voucher: ' . $e->getMessage()]);
    }
}
?>