<?php
session_start();
require_once '../models/Order.php';
require_once '../includes/auth-check.php';

header('Content-Type: application/json');

// Check if user is logged in
if (!isset($_SESSION['user_id']) || $_SESSION['user_id'] === 'guest') {
    echo json_encode(['success' => false, 'message' => 'Not authorized']);
    exit;
}

$order_id = $_GET['id'] ?? null;

if (!$order_id) {
    echo json_encode(['success' => false, 'message' => 'Order ID required']);
    exit;
}

try {
    $order = new Order();
    
    // Get order data
    $order_data = $order->getById($order_id);
    if (!$order_data || $order_data['user_id'] != $_SESSION['user_id']) {
        echo json_encode(['success' => false, 'message' => 'Order not found or not authorized']);
        exit;
    }
    
    // Get order items
    $order_items = $order->getOrderItems($order_id);
    
    // Decode shipping address
    $shipping_address = json_decode($order_data['shipping_address'], true);
    
    ob_start();
    ?>
    <div class="order-details-content">
        <!-- Order Header -->
        <div class="order-header-section">
            <h4 style="color: #1A4D2A; margin-bottom: 10px;">Order #<?php echo htmlspecialchars($order_data['order_number']); ?></h4>
            <div style="display: flex; gap: 10px; margin-bottom: 15px;">
                <span class="status status-<?php echo $order_data['status']; ?>" style="
                    padding: 4px 8px;
                    border-radius: 12px;
                    font-size: 12px;
                    font-weight: 600;
                    text-transform: uppercase;
                ">
                    <?php 
                    switch($order_data['status']) {
                        case 'awaiting_payment': echo 'Awaiting Payment'; break;
                        case 'pending': echo 'Pending'; break;
                        case 'processing': echo 'Processing'; break;
                        case 'shipped': echo 'Shipped'; break;
                        case 'delivered': echo 'Delivered'; break;
                        case 'cancelled': echo 'Cancelled'; break;
                        default: echo ucfirst($order_data['status']);
                    }
                    ?>
                </span>
                <span class="status status-<?php echo $order_data['payment_status'] ?? 'pending'; ?>" style="
                    padding: 4px 8px;
                    border-radius: 12px;
                    font-size: 12px;
                    font-weight: 600;
                    text-transform: uppercase;
                ">
                    Payment: <?php echo ucfirst($order_data['payment_status'] ?? 'pending'); ?>
                </span>
            </div>
            <p style="color: #666; font-size: 14px; margin: 0;">
                <strong>Order Date:</strong> <?php echo date('M j, Y g:i A', strtotime($order_data['created_at'])); ?>
            </p>
        </div>

        <!-- Order Items -->
        <div class="order-items-section" style="margin: 20px 0;">
            <h5 style="color: #1A4D2A; margin-bottom: 15px; border-bottom: 2px solid #D4AF37; padding-bottom: 5px;">Order Items</h5>
            <?php foreach ($order_items as $item): ?>
            <div style="
                display: flex;
                align-items: center;
                padding: 12px 0;
                border-bottom: 1px solid #f0f0f0;
            ">
                <div style="
                    width: 50px;
                    height: 50px;
                    border-radius: 8px;
                    background: #f8f9fa;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    margin-right: 12px;
                    color: #666;
                ">
                    <?php if (!empty($item['product_image'])): ?>
                        <img src="<?php echo htmlspecialchars($item['product_image']); ?>" alt="Product" style="width:100%;height:100%;object-fit:cover;border-radius:8px;">
                    <?php else: ?>
                        <i class="fas fa-box"></i>
                    <?php endif; ?>
                </div>
                <div style="flex: 1;">
                    <div style="font-weight: 600; color: #1A4D2A; margin-bottom: 2px;">
                        <?php echo htmlspecialchars($item['product_name']); ?>
                    </div>
                    <div style="font-size: 12px; color: #666;">
                        Size: <?php echo htmlspecialchars($item['size'] ?? 'Standard'); ?> | 
                        Qty: <?php echo $item['quantity']; ?>
                    </div>
                </div>
                <div style="font-weight: bold; color: #D4AF37;">
                    RM <?php echo number_format($item['total_price'], 2); ?>
                </div>
            </div>
            <?php endforeach; ?>
        </div>

        <!-- Order Summary -->
        <div class="order-summary-section" style="margin: 20px 0;">
            <h5 style="color: #1A4D2A; margin-bottom: 15px; border-bottom: 2px solid #D4AF37; padding-bottom: 5px;">Order Summary</h5>
            <div style="background: #f8f9fa; padding: 15px; border-radius: 8px;">
                <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
                    <span>Subtotal:</span>
                    <span>RM <?php echo number_format($order_data['total_amount'] - ($order_data['shipping'] ?? 0), 2); ?></span>
                </div>
                <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
                    <span>Shipping:</span>
                    <span>RM <?php echo number_format($order_data['shipping'] ?? 0, 2); ?></span>
                </div>
                <div style="display: flex; justify-content: space-between; font-weight: bold; font-size: 16px; color: #1A4D2A; border-top: 1px solid #ddd; padding-top: 8px;">
                    <span>Total:</span>
                    <span>RM <?php echo number_format($order_data['total_amount'], 2); ?></span>
                </div>
            </div>
        </div>

        <!-- Tracking Number -->
        <?php if (!empty($order_data['tracking_number'])): ?>
        <div class="tracking-section" style="margin: 20px 0;">
            <h5 style="color: #1A4D2A; margin-bottom: 15px; border-bottom: 2px solid #D4AF37; padding-bottom: 5px;">Tracking Information</h5>
            <div style="
                background: #e8f5e8;
                padding: 15px;
                border-radius: 8px;
                border-left: 4px solid #1A4D2A;
            ">
                <div style="display: flex; align-items: center; gap: 8px; margin-bottom: 5px;">
                    <i class="fas fa-shipping-fast" style="color: #1A4D2A;"></i>
                    <span style="font-weight: 600; color: #1A4D2A;">Tracking Number:</span>
                </div>
                <div style="
                    font-family: 'Courier New', monospace;
                    font-size: 16px;
                    font-weight: bold;
                    color: #D4AF37;
                    letter-spacing: 1px;
                ">
                    <?php echo htmlspecialchars($order_data['tracking_number']); ?>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <!-- Shipping Address -->
        <?php if ($shipping_address): ?>
        <div class="shipping-section" style="margin: 20px 0;">
            <h5 style="color: #1A4D2A; margin-bottom: 15px; border-bottom: 2px solid #D4AF37; padding-bottom: 5px;">Shipping Address</h5>
            <div style="background: #f8f9fa; padding: 15px; border-radius: 8px;">
                <p style="margin: 0 0 5px 0; font-weight: 600;">
                    <?php echo htmlspecialchars($shipping_address['first_name'] . ' ' . $shipping_address['last_name']); ?>
                </p>
                <p style="margin: 0 0 5px 0;">
                    <?php echo htmlspecialchars($shipping_address['address']); ?>
                </p>
                <p style="margin: 0 0 5px 0;">
                    <?php echo htmlspecialchars($shipping_address['city'] . ', ' . $shipping_address['state'] . ' ' . $shipping_address['zip_code']); ?>
                </p>
                <p style="margin: 0;">
                    <?php echo htmlspecialchars($shipping_address['country']); ?>
                </p>
                <?php if (!empty($shipping_address['phone'])): ?>
                <p style="margin: 10px 0 0 0; color: #666;">
                    <i class="fas fa-phone"></i> <?php echo htmlspecialchars($shipping_address['phone']); ?>
                </p>
                <?php endif; ?>
            </div>
        </div>
        <?php endif; ?>

        <!-- Payment Information -->
        <div class="payment-section" style="margin: 20px 0;">
            <h5 style="color: #1A4D2A; margin-bottom: 15px; border-bottom: 2px solid #D4AF37; padding-bottom: 5px;">Payment Information</h5>
            <div style="background: #f8f9fa; padding: 15px; border-radius: 8px;">
                <p style="margin: 0 0 5px 0;">
                    <strong>Payment Method:</strong> <?php echo htmlspecialchars($order_data['payment_method']); ?>
                </p>
                <p style="margin: 0;">
                    <strong>Payment Status:</strong> 
                    <span class="status status-<?php echo $order_data['payment_status'] ?? 'pending'; ?>" style="
                        padding: 2px 6px;
                        border-radius: 8px;
                        font-size: 11px;
                        font-weight: 600;
                        text-transform: uppercase;
                    ">
                        <?php echo ucfirst($order_data['payment_status'] ?? 'pending'); ?>
                    </span>
                </p>
            </div>
        </div>
    </div>
    <?php
    $html = ob_get_clean();
    
    echo json_encode(['success' => true, 'html' => $html]);
    
} catch (Exception $e) {
    error_log("Error getting order details: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => 'An error occurred']);
}
?>