<?php
session_start();
require_once '../config/database.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Get tracking data
$data = json_decode(file_get_contents('php://input'), true);

if (!$data || !isset($data['event_type'])) {
    echo json_encode(['success' => false, 'error' => 'Invalid data']);
    exit;
}

try {
    $conn = getDB();
    
    // Get session ID - prioritize from request body (for mobile apps), then fall back to PHP session
    if (isset($data['session_id']) && !empty($data['session_id'])) {
        // Mobile app or API request with session_id
        $sessionId = $data['session_id'];
    } else {
        // Web browser using PHP sessions
        if (!isset($_SESSION['analytics_session_id'])) {
            $_SESSION['analytics_session_id'] = uniqid('sess_', true);
        }
        $sessionId = $_SESSION['analytics_session_id'];
    }
    
    // Get user ID - prioritize from request body (for mobile apps), then fall back to PHP session
    if (isset($data['user_id']) && !empty($data['user_id']) && $data['user_id'] !== 'guest') {
        // Mobile app or API request with user_id
        $userId = (int)$data['user_id'];
    } else if (isset($_SESSION['user_id']) && $_SESSION['user_id'] !== 'guest') {
        // Web browser using PHP sessions
        $userId = $_SESSION['user_id'];
    } else {
        $userId = null;
    }
    
    // Get client info - allow override from request body for mobile apps
    $userAgent = $data['user_agent'] ?? $_SERVER['HTTP_USER_AGENT'] ?? null;
    $ipAddress = $data['ip_address'] ?? $_SERVER['REMOTE_ADDR'] ?? null;
    $referrer = $data['referrer'] ?? $_SERVER['HTTP_REFERER'] ?? null;
    
    $eventType = $data['event_type'];
    
    // Log analytics request for debugging (optional - can be disabled in production)
    if (defined('DEBUG_ANALYTICS') && DEBUG_ANALYTICS) {
        error_log("Analytics Track: " . json_encode([
            'event_type' => $eventType,
            'user_id' => $userId,
            'session_id' => $sessionId,
            'product_id' => $data['product_id'] ?? null,
            'source' => isset($data['session_id']) ? 'mobile' : 'web'
        ]));
    }
    
    // Handle different event types
    switch ($eventType) {
        case 'product_view':
        case 'product_click':
        case 'add_to_cart':
            if (!isset($data['product_id'])) {
                throw new Exception('Product ID is required');
            }
            
            $trackType = $eventType === 'product_view' ? 'view' : 
                        ($eventType === 'product_click' ? 'click' : 'add_to_cart');
            
            $stmt = $conn->prepare("
                INSERT INTO product_analytics 
                (product_id, event_type, user_id, session_id, user_agent, ip_address, referrer) 
                VALUES (?, ?, ?, ?, ?, ?, ?)
            ");
            
            $stmt->execute([
                $data['product_id'],
                $trackType,
                $userId,
                $sessionId,
                $userAgent,
                $ipAddress,
                $referrer
            ]);
            break;
            
        case 'page_view':
            if (!isset($data['page_url']) || !isset($data['page_type'])) {
                throw new Exception('Page URL and type are required');
            }
            
            $stmt = $conn->prepare("
                INSERT INTO site_analytics 
                (page_url, page_type, user_id, session_id, user_agent, ip_address, referrer) 
                VALUES (?, ?, ?, ?, ?, ?, ?)
            ");
            
            $stmt->execute([
                $data['page_url'],
                $data['page_type'],
                $userId,
                $sessionId,
                $userAgent,
                $ipAddress,
                $referrer
            ]);
            break;
            
        case 'traffic_source':
            // Detect traffic source
            $sourceType = detectTrafficSource($referrer);
            $sourceName = extractSourceName($referrer);
            
            $stmt = $conn->prepare("
                INSERT INTO traffic_sources 
                (session_id, source_type, source_name, utm_source, utm_medium, utm_campaign, landing_page) 
                VALUES (?, ?, ?, ?, ?, ?, ?)
            ");
            
            $stmt->execute([
                $sessionId,
                $sourceType,
                $sourceName,
                $data['utm_source'] ?? null,
                $data['utm_medium'] ?? null,
                $data['utm_campaign'] ?? null,
                $data['landing_page'] ?? null
            ]);
            break;
            
        default:
            throw new Exception('Invalid event type');
    }
    
    echo json_encode([
        'success' => true, 
        'tracked' => true,
        'user_id' => $userId,
        'session_id' => $sessionId,
        'event_type' => $eventType
    ]);
    
} catch (Exception $e) {
    error_log("Analytics tracking error: " . $e->getMessage());
    error_log("Analytics data: " . json_encode($data));
    echo json_encode([
        'success' => false, 
        'error' => $e->getMessage(),
        'debug_info' => [
            'received_user_id' => $data['user_id'] ?? 'not provided',
            'received_session_id' => $data['session_id'] ?? 'not provided',
            'event_type' => $data['event_type'] ?? 'not provided'
        ]
    ]);
}

function detectTrafficSource($referrer) {
    if (empty($referrer)) {
        return 'direct';
    }
    
    $referrerLower = strtolower($referrer);
    
    // Search engines
    if (strpos($referrerLower, 'google') !== false || 
        strpos($referrerLower, 'bing') !== false || 
        strpos($referrerLower, 'yahoo') !== false) {
        return 'search';
    }
    
    // Social media
    if (strpos($referrerLower, 'facebook') !== false || 
        strpos($referrerLower, 'instagram') !== false || 
        strpos($referrerLower, 'twitter') !== false || 
        strpos($referrerLower, 'tiktok') !== false) {
        return 'social';
    }
    
    // Email
    if (strpos($referrerLower, 'mail') !== false) {
        return 'email';
    }
    
    return 'referral';
}

function extractSourceName($referrer) {
    if (empty($referrer)) {
        return null;
    }
    
    $parsed = parse_url($referrer);
    $host = $parsed['host'] ?? '';
    
    // Remove www. prefix
    $host = preg_replace('/^www\./', '', $host);
    
    return $host;
}
