<?php
/**
 * One-Click Fix: Rename clicks to favorites
 */

require_once 'config/database.php';

echo "<h2>🔧 Fixing Analytics Column</h2>";
echo "<style>
    body { font-family: Arial, sans-serif; padding: 20px; background: #f5f5f5; }
    .success { background: #d4edda; padding: 20px; border-left: 4px solid #28a745; margin: 20px 0; border-radius: 8px; }
    .error { background: #f8d7da; padding: 20px; border-left: 4px solid #dc3545; margin: 20px 0; border-radius: 8px; }
    .info { background: #d1ecf1; padding: 20px; border-left: 4px solid #0c5460; margin: 20px 0; border-radius: 8px; }
    pre { background: white; padding: 15px; border-radius: 4px; overflow-x: auto; border: 1px solid #ddd; }
</style>";

try {
    $conn = getDB();
    
    echo "<div class='info'>";
    echo "<h3>🔄 Step 1: Checking current structure...</h3>";
    
    // Check if clicks column exists
    $stmt = $conn->query("SHOW COLUMNS FROM product_analytics_simple LIKE 'clicks'");
    $hasClicks = $stmt->rowCount() > 0;
    
    // Check if favorites column exists
    $stmt = $conn->query("SHOW COLUMNS FROM product_analytics_simple LIKE 'favorites'");
    $hasFavorites = $stmt->rowCount() > 0;
    
    if ($hasClicks) {
        echo "<p>Found 'clicks' column - needs renaming</p>";
    }
    if ($hasFavorites) {
        echo "<p>Found 'favorites' column - already correct!</p>";
    }
    echo "</div>";
    
    if ($hasClicks && !$hasFavorites) {
        echo "<div class='info'>";
        echo "<h3>🔧 Step 2: Renaming 'clicks' to 'favorites'...</h3>";
        
        // Rename column
        $conn->exec("ALTER TABLE product_analytics_simple CHANGE COLUMN clicks favorites INT DEFAULT 0");
        echo "<p>✅ Renamed: clicks → favorites</p>";
        
        // Try to rename timestamp column too
        try {
            $conn->exec("ALTER TABLE product_analytics_simple CHANGE COLUMN last_clicked last_favorited TIMESTAMP NULL");
            echo "<p>✅ Renamed: last_clicked → last_favorited</p>";
        } catch (Exception $e) {
            echo "<p>⚠️ last_clicked column not found (this is OK)</p>";
        }
        
        // Update index
        try {
            $conn->exec("ALTER TABLE product_analytics_simple DROP INDEX idx_clicks");
            echo "<p>✅ Dropped old index: idx_clicks</p>";
        } catch (Exception $e) {
            echo "<p>⚠️ idx_clicks not found (this is OK)</p>";
        }
        
        try {
            $conn->exec("ALTER TABLE product_analytics_simple ADD INDEX idx_favorites (favorites)");
            echo "<p>✅ Added new index: idx_favorites</p>";
        } catch (Exception $e) {
            echo "<p>⚠️ Index might already exist</p>";
        }
        
        echo "</div>";
        
        echo "<div class='success'>";
        echo "<h3>✅ Success! Column Renamed</h3>";
        echo "<p><strong>Changes made:</strong></p>";
        echo "<ul>";
        echo "<li>✅ Column renamed: <code>clicks</code> → <code>favorites</code></li>";
        echo "<li>✅ Timestamp renamed: <code>last_clicked</code> → <code>last_favorited</code></li>";
        echo "<li>✅ Index updated</li>";
        echo "</ul>";
        echo "<p><strong>Next steps:</strong></p>";
        echo "<ol>";
        echo "<li>Run <a href='debug-analytics-full.php'>debug-analytics-full.php</a> again to verify</li>";
        echo "<li>Visit a product page to test tracking</li>";
        echo "<li>Check admin analytics dashboard</li>";
        echo "</ol>";
        echo "</div>";
        
    } else if ($hasFavorites && !$hasClicks) {
        echo "<div class='success'>";
        echo "<h3>✅ Already Fixed!</h3>";
        echo "<p>Your database already has the 'favorites' column. No changes needed!</p>";
        echo "<p><strong>Next:</strong> Run <a href='debug-analytics-full.php'>debug-analytics-full.php</a> to test everything</p>";
        echo "</div>";
        
    } else if ($hasClicks && $hasFavorites) {
        echo "<div class='error'>";
        echo "<h3>⚠️ Both Columns Exist!</h3>";
        echo "<p>You have BOTH 'clicks' and 'favorites' columns. This shouldn't happen.</p>";
        echo "<p><strong>Manual fix needed:</strong> Run this SQL to remove the old 'clicks' column:</p>";
        echo "<pre>ALTER TABLE product_analytics_simple DROP COLUMN clicks;</pre>";
        echo "</div>";
    }
    
    // Show final structure
    echo "<div class='info'>";
    echo "<h3>📋 Current Table Structure:</h3>";
    $stmt = $conn->query("DESCRIBE product_analytics_simple");
    $columns = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo "<table border='1' cellpadding='10' style='border-collapse: collapse; background: white; width: 100%;'>";
    echo "<tr style='background: #4CAF50; color: white;'><th>Column</th><th>Type</th><th>Null</th><th>Default</th></tr>";
    foreach ($columns as $col) {
        $highlight = ($col['Field'] === 'favorites') ? "style='background: #d4edda; font-weight: bold;'" : "";
        echo "<tr $highlight>";
        echo "<td><strong>{$col['Field']}</strong></td>";
        echo "<td>{$col['Type']}</td>";
        echo "<td>{$col['Null']}</td>";
        echo "<td>{$col['Default']}</td>";
        echo "</tr>";
    }
    echo "</table>";
    echo "</div>";
    
} catch (Exception $e) {
    echo "<div class='error'>";
    echo "<h3>❌ Error:</h3>";
    echo "<p><strong>Message:</strong> " . $e->getMessage() . "</p>";
    echo "<p><strong>What to do:</strong></p>";
    echo "<p>Run this SQL manually in phpMyAdmin:</p>";
    echo "<pre>ALTER TABLE product_analytics_simple CHANGE COLUMN clicks favorites INT DEFAULT 0;</pre>";
    echo "</div>";
}
?>
